/*
 *	SILCKeyExchangeStartPayload.java		2002/11/06
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.lang.*;
import java.security.SecureRandom;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import com.alistairphipps.util.Hex;

/** Class to implement a Key Exchange Start payload
 *
 * @author Alistair K Phipps
 * @version 20021108
 */
public class SILCKeyExchangeStartPayload extends SILCPayload
{
	/** whether receiver of payload does not reply to packet */
	private boolean _bNoReply;

	/** whether perfect forward secrecy should be used in key exchange (if not set, rekeying done with old key) */
	private boolean _bPFS;

	/** whether both parties must perform authentication by providing signed data for the other party to verify - usually only responder provides signature data, unless this is set (then initiator must also provide it) */
	private boolean _bMutualAuth;

	/** cookie that randomises payload so each party cannot determine payload beforehand - must be 16 bytes*/
	private byte[] _ylCookie = new byte[16];

	/** version string */
	private String _strVersion;

	/** key exchange groups (comma separated) */
	private String _strKEGroup;

	/** PKCS algorithms (comma separated) */
	private String _strPKCS;

	/** encryption algorithms (comma separated) */
	private String _strCipher;

	/** hashing algorithms (comma separated) */
	private String _strHash; 

	/** HMAC algorithms (comma separated) */
	private String _strHMAC;

	/** compression algorithms (comma separated */
	private String _strCompression;

	/** Constructor to set up the payload with default values.  Note: this is not very flexible, and should really detect what ciphers/compression methods are supported by the code / plugin modules and output them here.
	 */
	public SILCKeyExchangeStartPayload() 
	{
		_strVersion = "SILC-1.2-0.02.jSilc";
		_strKEGroup = "diffie-hellman-group1";
		_strPKCS = "rsa";
		_strCipher = "aes-128-cbc";	// XXX aes-256-cbc is the required cipher, but our crypto library only supports 128-bit AES
		_strHash = "sha1";
		_strHMAC = "hmac-sha1"; // XXX hmac-sha1-96 is the required HMAC, but our crypto library only supports 160-bit sha1
		_strCompression = "none";
		_bNoReply = false;
		_bPFS = false;
		_bMutualAuth = false;

		// fill in cookie with random number
		SecureRandom random = new SecureRandom();
		random.nextBytes( _ylCookie );
	}

	/** Constructor to set up the payload information from a bytelist - the decrypted payload information received in a packet.
	 * @param yl List of bytes forming received payload
	 */
	public SILCKeyExchangeStartPayload( byte[] yl )
	{
		ByteBuffer yb = ByteBuffer.wrap( yl );
	
		byte yReserved = yb.get(); // reserved
		assert( yReserved == 0 ); // must be 0

		byte yFlags = yb.get(); // flags
		// _bNoReply = ( yFlags & 0x01 ) != 0;
		_bNoReply = false;
		_bPFS = ( yFlags & 0x02 ) != 0;
		_bMutualAuth = ( yFlags & 0x04 ) != 0;
	
		short sPayloadLength = yb.getShort(); // payload length
		assert( ( sPayloadLength & 0xFFFF ) == yl.length ); // should be the same length as our array 

		yb.get( _ylCookie ); // cookie

		_strVersion = readShortAndString( yb );
		_strKEGroup = readShortAndString( yb );
		_strPKCS = readShortAndString( yb );
		_strCipher = readShortAndString( yb );
		_strHash = readShortAndString( yb );
		_strHMAC = readShortAndString( yb );
		_strCompression = readShortAndString( yb );
	}
	
	// doc inherited
	public byte[] toByteList()
	{
		// get byte buffers with UTF-8 encoded versions of strings - need to do this first to get the correct length (as UTF-8 could make the string longer)
		ByteBuffer ybVersion = writeShortAndString( _strVersion );
		ByteBuffer ybKEGroup = writeShortAndString( _strKEGroup );
		ByteBuffer ybPKCS = writeShortAndString( _strPKCS );
		ByteBuffer ybCipher = writeShortAndString( _strCipher );
		ByteBuffer ybHash = writeShortAndString( _strHash );
		ByteBuffer ybHMAC = writeShortAndString( _strHMAC );
		ByteBuffer ybCompression = writeShortAndString( _strCompression );

		// 20 is the length of the payload excluding the variable length fields and shorts for the lengths
		short sPayloadLength = (short)( 20 + ybVersion.limit() + ybKEGroup.limit() + ybPKCS.limit() + ybCipher.limit() + ybHash.limit() + ybHMAC.limit() + ybCompression.limit() );
		ByteBuffer yb = ByteBuffer.allocate( sPayloadLength & 0xFFFF );

		yb.put( (byte)0 ); // reserved
		// yb.put( (byte)( ( _bNoReply ? 1 : 0 ) | ( _bPFS ? 2 : 0 ) | ( _bMutualAuth ? 4 : 0 ) ) ); // flags
		yb.put( (byte)( ( _bPFS ? 2 : 0 ) | ( _bMutualAuth ? 4 : 0 ) ) ); // flags

		yb.putShort( sPayloadLength ); // payload length
		yb.put( _ylCookie ); // cookie
		yb.put( ybVersion );
		yb.put( ybKEGroup );
		yb.put( ybPKCS );
		yb.put( ybCipher );
		yb.put( ybHash );
		yb.put( ybHMAC );
		yb.put( ybCompression );

		return yb.array();
	}

	// doc inherited
	public String toString()
	{
		String strRet;
		strRet = "No Reply: " + Boolean.toString( _bNoReply );
		strRet += "\nPFS: " + Boolean.toString( _bPFS );
		strRet += "\nMutual Authentication: " + Boolean.toString( _bMutualAuth );
		strRet += "\nVersion: " + _strVersion;
		strRet += "\nKE Group: " + _strKEGroup;
		strRet += "\nPKCS: " + _strPKCS;
		strRet += "\nCipher: " + _strCipher;
		strRet += "\nHash: " + _strHash;
		strRet += "\nHMAC: " + _strHMAC;
		strRet += "\nCompression: " + _strCompression;
		strRet += "\nCookie: " + Hex.toString( _ylCookie );
		return strRet;
	}

	public boolean isList()
	{
		return false;
	}
	
	public boolean getMutualAuth()
	{
		return _bMutualAuth;
	}
	
	public byte[] getCookie()
	{
		return _ylCookie;
	}
}
