/*
 *	TCPConnection.java		2002/10/24
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.core;

import java.lang.*;
import java.io.*;
import java.net.*;
import java.util.logging.*;

/** 
 * The TCPConnection class provides a standard TCP connection to a particular host/port with the ability to send and receive raw data.
 * @author Alistair K Phipps
 * @version 2002/10/24
 */
public class TCPConnection
{
	/** logger is used for logging */
	private static Logger logger = Logger.getLogger( "com.alistairphipps.jsilc.core.tcpconnection" );

	/** sock is the socket for the connection to the remote host */
	private Socket sock;

	/** out is the OutputStream for the socket */
	private OutputStream out;

	/** in is the InputStream from the socket */
	private InputStream in;
	
	/**
	 * TCPConnection constructor opens the connection to the specified host/port and creates input/output streams.
	 * @param strName the hostname or IP of the server to connect to.
	 * @param iPort the port number of the server to connect to.
	 * @throws UnknownHostException if the hostname could not be found.
	 * @throws IOException if the output or input streams could not be opened.
	 */
	public TCPConnection( String strName, int iPort ) throws UnknownHostException, IOException
	{
		logger.entering( "TCPConnection", "TCPConnection", strName + ":" + Integer.toString( iPort ) );
		sock = new Socket( strName, iPort );
		out = sock.getOutputStream();
		in = sock.getInputStream();
		logger.exiting( "TCPConnection", "TCPConnection" );
	}

	/**
	 * writeData sends an array of bytes to the remote host.
	 * @param byteArray the array of bytes to send.
	 * @throws IOException if the output stream could not be written to.
	 */
	public void writeData( byte[] byteArray ) throws IOException
	{
		logger.entering( "TCPConnection", "writeData", new Integer( byteArray.length ) );
		out.write( byteArray );
		logger.exiting( "TCPConnection", "writeData" );
	}

	/**
	 * readData reads an array of bytes from the remote host.
	 * @param iNumBytes the number of bytes to receive - will block until gets this many.
	 * @return the array of bytes received.
	 * @throws IOException if the input stream could not be read from.
	 */
	public byte[] readData( int iNumBytes ) throws IOException
	{
		//logger.entering( "TCPConnection", "readData" );
		byte[] byteArray = new byte[ iNumBytes ];
		in.read( byteArray );
		//logger.exiting( "TCPConnection", "readData", ( new Integer( byteArray.length ) ).toString() );
		return byteArray;
	}
}
