/*
 *	SILCKeyExchange1Payload.java		2002/11/13
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.lang.String;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import org.bouncycastle.crypto.*;
import org.bouncycastle.crypto.agreement.*;
import org.bouncycastle.crypto.digests.*;
import org.bouncycastle.crypto.generators.*;
import org.bouncycastle.crypto.params.*;
import org.bouncycastle.crypto.engines.*;
import org.bouncycastle.crypto.encodings.*;
import com.alistairphipps.util.Hex;
import com.alistairphipps.util.UBigInteger;

/** Class to implement a Key Exchange payload as sent by client in a KEY_EXCHANGE_1 packet.
 * @author Alistair Phipps
 * @version 20021113
 */
public class SILCKeyExchange1Payload extends SILCKeyExchangePayload
{
	/** D-H keypair (generated) - _ylPubData is set to public portion of this. */
	private AsymmetricCipherKeyPair _kpDH;

	/** RSA keypair (generated) - _pubkey is set to public portion of this (encoded in SILCPubKey). */
	private AsymmetricCipherKeyPair _kpRSA;

	/** Constructor to set up the payload with default values.
	 * @param ylKES KEY_EXCHANGE_START payload sent by client
	 * @param bMutualAuth Whether mutual authentication was selected in KEY_EXCHANGE_START payload received by client from server
	 */
	public SILCKeyExchange1Payload( byte[] ylKES, boolean bMutualAuth ) throws InvalidCipherTextException
	{
		SecureRandom rand = new SecureRandom();	// XXX: change to SecureRandom
		System.out.println( "Creating DH Key pair" );
		// Diffie-Hellman is e = g^x mod p (which is what we want) - create DH keypair and setup pubdata
		DHKeyPairGenerator dhkpGen = new DHKeyPairGenerator();
		dhkpGen.init( new DHKeyGenerationParameters( rand, new DHParameters( s_biP, s_biG ) ) );
		_kpDH = dhkpGen.generateKeyPair();
		_ylPubData = UBigInteger.toByteArray( ( (DHPublicKeyParameters)_kpDH.getPublic() ).getY() );
		System.out.println( "Creating RSA Key pair" );
		// create RSA key pair
		RSAKeyPairGenerator rsakpg = new RSAKeyPairGenerator();
		// parameters are exponent, random generator, strength, prime certainty
		rsakpg.init( new RSAKeyGenerationParameters( BigInteger.valueOf(0x03), rand, 1024, 25 ) ); 
		AsymmetricCipherKeyPair _kpRSA = rsakpg.generateKeyPair();
		RSAKeyParameters rsapu = (RSAKeyParameters)_kpRSA.getPublic();
		_pubkey = new SILCPubKey( rsapu.getExponent(), rsapu.getModulus() );
		
		System.out.println( "Calculating signed hash" );
		if( bMutualAuth )
			_ylSigData = calculateSig( (RSAKeyParameters)_kpRSA.getPrivate(), calculateHash_i( ylKES ) );
		else
			_ylSigData = new byte[0];
	}

	/** Calculate hash_i for KE_1: hash( KES payload | pub key | e )
	 * @param ylKES KES payload data of INITIATOR (client)
	 * @return SHA-1 hash in the form of a byte list
	 */
	private byte[] calculateHash_i( byte[] ylKES )
	{
		ByteBuffer yb = ByteBuffer.allocate( ylKES.length + _pubkey.toByteList().length + _ylPubData.length );
		yb.put( ylKES );
		yb.put( _pubkey.toByteList() );
		yb.put( _ylPubData );
		Digest digest = new SHA1Digest();
		digest.update( yb.array(), 0, yb.array().length );
		byte[] ylHash = new byte[ digest.getDigestSize() ];
		digest.doFinal( ylHash, 0 );
		// XXX temp
		System.out.println( "***** HASH_I *****: " + com.alistairphipps.util.Hex.toString( ylHash ) );
		return ylHash;
	}

	public DHPrivateKeyParameters getDHPrivateKey()
	{
		return (DHPrivateKeyParameters)_kpDH.getPrivate();
	}

	public SILCPubKey getPublicKey()
	{
		return _pubkey;
	}

	public byte[] getDHPublicData()
	{
		return _ylPubData;
	}
}
