/*
 *	SILCKeyExchangePayload.java		2002/11/11
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.lang.String;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import org.bouncycastle.crypto.*;
import org.bouncycastle.crypto.agreement.*;
import org.bouncycastle.crypto.digests.*;
import org.bouncycastle.crypto.generators.*;
import org.bouncycastle.crypto.params.*;
import org.bouncycastle.crypto.engines.*;
import org.bouncycastle.crypto.encodings.*;
import com.alistairphipps.util.Hex;
import com.alistairphipps.util.UBigInteger;

/** Abstract class that encapsulates a Key Exchange payload.  Concrete subclasses exist for KE payloads sent in KEY_EXCHANGE_1 and received in KEY_EXCHANGE_2 packets.
 * @author Alistair K Phipps
 * @version 20021113
 */
public abstract class SILCKeyExchangePayload extends SILCPayload
{
	/** Public key. */
	protected SILCPubKey _pubkey;

	/** D-H public key. */
	protected byte[] _ylPubData;

	/** Signature data - sig signed by sender.  Only provided by initiator if doing mutual authentication. */
	protected byte[] _ylSigData;
	
	/* OAKLEY dh group 1 (prime) */
	protected BigInteger s_biP = new BigInteger(  "179769313486231590770839156793787453197860296048756011706444423684197180216158519368947833795864925541502180565485980503646440548199239100050792877003355816639229553136239076508735759914822574862575007425302077447712589550957937778424442426617334727629299387668709205606050270810842907692932019128194467627007" );

	/* generator for above (base) */
	protected BigInteger s_biG = BigInteger.valueOf( 2 );

	/** Calculate signature (hash_i encrypted with RSA private key) for KE_1 payload; calculate hash by decrypting signature with provided public key for KE_2 payload
	 * @param rsapv Key parameters for RSA key (Private if signing hash_i, public if decrypting signed hash)
	 * @param yl byte array to encrypt decrypt
	 * @return Byte list containing signature (encrypted hash)
	 */
	protected byte[] calculateSig( RSAKeyParameters rsapv, byte[] yl ) throws InvalidCipherTextException
	{
		AsymmetricBlockCipher eng = new PKCS1Encoding( new RSAEngine() );
		eng.init( rsapv.isPrivate(), rsapv );
		assert( eng.getInputBlockSize() >= yl.length ); // largest size an input block can be must be greater than hash length
		return eng.processBlock(yl, 0, yl.length);
	}
	
	// doc inherited
	public byte[] toByteList()
	{
		byte[] ylPubKey = _pubkey.toByteList();
		ByteBuffer yb = ByteBuffer.allocate( s_yEmptyLength + ylPubKey.length + _ylPubData.length + _ylSigData.length );
		yb.putShort( (short)ylPubKey.length );
		yb.putShort( _pubkey.getType() );
		yb.put( _pubkey.toByteList() );
		yb.putShort( (short)_ylPubData.length );
		yb.put( _ylPubData );
		yb.putShort( (short)_ylSigData.length );
		yb.put( _ylSigData );
		return yb.array();
	}

	// doc inherited
	public String toString()
	{
		String strRet;
		strRet = "Public key type: " + SILCPubKeyType.toString( _pubkey.getType() );
		strRet += "\nPublic data: " + Hex.toString( _ylPubData );
		strRet += "\nSignature data: " + Hex.toString( _ylSigData );
		strRet += "\nPublic key\n==========\n" + _pubkey.toString();
		return strRet;
	}

	// doc inherited
	public boolean isList()
	{
		return false;
	}

	/** Length of payload with empty variable length fields */
	private byte s_yEmptyLength = 8;
}
