/*
 *	SILCPubKey.java		2002/11/11
 *	
 *	Copyright (c) 2002 Alistair K Phipps (jsilc@alistairphipps.com).
 *	All rights reserved.
 */

package com.alistairphipps.jsilc.silcprotocol;

import java.nio.ByteBuffer;
import java.math.BigInteger;
import com.alistairphipps.util.Hex;
import org.bouncycastle.crypto.params.RSAKeyParameters;
import com.alistairphipps.util.UBigInteger;

/** Class to implement a SILC Public Key with RSA type public data
 *
 * @author Alistair K Phipps
 * @version 20021111
 */
public class SILCPubKey extends SILCPayload// TODO: rename to SILCRSAPubKey extends SILCPubKey
{
	/** Algorithm name used (always RSA for this class) */
	private String _strAlgoName;

	/** Identifier for the keypair holder */
	private String _strIdentifier = "UN=test, RN=test";

	/** RSA public key exponent */
	private BigInteger _biE;

	/** RSA public key modulus */
	private BigInteger _biN;

	public SILCPubKey( BigInteger biE, BigInteger biN )
	{
		_strAlgoName = "rsa";
		_strIdentifier = "UN=jsilc,HN=jsilc"; // FIXME - should really be loading in all this stuff and generating once anyway

		_biE = biE;
		_biN = biN;
	}
	
	public SILCPubKey( byte[] yl)
	{
		ByteBuffer yb = ByteBuffer.wrap( yl );
		int iPubKeyLength = yb.getInt();
		assert( ( iPubKeyLength & 0xFFFFFFF ) == yl.length );	// check stored length equals actual length NOTE: have to do + 4 if gets fixed to match spec
		_strAlgoName = readShortAndString( yb );
		assert( _strAlgoName.equals( "rsa" ) ); // we only support RSA
		_strIdentifier = readShortAndString( yb );

		int iELength = yb.getInt();
		byte[] ylE = new byte[ iELength & 0xFFFFFFFF ];
		yb.get( ylE );
		_biE = UBigInteger.fromByteArray( ylE );
		
		int iNLength = yb.getInt();
		byte[] ylN = new byte[ iNLength & 0xFFFFFFFF ];
		yb.get( ylN );
		_biN = UBigInteger.fromByteArray( ylN );
	}
	
	public byte[] toByteList()
	{
		ByteBuffer ybAlgoName = writeShortAndString( _strAlgoName );
		ByteBuffer ybIdentifier = writeShortAndString( _strIdentifier );
		byte[] ylE = UBigInteger.toByteArray( _biE );
		// These numbers have an undocumented minimum byte length of 4
		byte[] ylEPad;
		if( ylE.length < 4 )
			ylEPad = new byte[ 4 - ylE.length ];
		else
			ylEPad = new byte[0];
		byte[] ylN = UBigInteger.toByteArray( _biN );
		byte[] ylNPad;
		if( ylN.length < 4 )
			ylNPad = new byte[ 4 - ylN.length ];
		else
			ylNPad = new byte[0];

		int iPubKeyLength = ybAlgoName.limit() + ybIdentifier.limit() + ylE.length + ylEPad.length + 4 + ylN.length + ylNPad.length + 4 + 4; // length INCLUDING this field - spec says excluding, but spec is wrong - according to pekka
		
		ByteBuffer yb = ByteBuffer.allocate( iPubKeyLength );
		yb.putInt( iPubKeyLength );
		yb.put( ybAlgoName );
		yb.put( ybIdentifier );
		yb.putInt( ylE.length + ylEPad.length );
		yb.put( ylEPad );
		yb.put( ylE );
		yb.putInt( ylN.length + ylNPad.length );
		yb.put( ylNPad );
		yb.put( ylN );
		return yb.array();
	}

	public short getType()
	{
		return SILCPubKeyType.SILC;
	}

	public String toString()
	{
		String strRet;
		strRet = "Algorithm name: " + _strAlgoName;
		strRet += "\nIdentifier: " + _strIdentifier;
		strRet += "\nRSA exponent: " + Hex.toString( UBigInteger.toByteArray( _biE ) );
		strRet += "\nRSA modulus: " + Hex.toString( UBigInteger.toByteArray( _biN ) );
		return strRet;
	}

	public RSAKeyParameters toRSAKeyParameters()
	{
		RSAKeyParameters rsakey = new RSAKeyParameters( false, _biN, _biE );
		return rsakey;
	}

	public boolean isList()
	{
		return false;
	}
}
