/*

  silcmap_html.c

  Author: Pekka Riikonen <priikone@silcnet.org>

  Copyright (C) 2003 - 2004 Pekka Riikonen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

*/

#include "silcincludes.h"
#include "silcclient.h"
#include "silcmap.h"

/* Write the HTML data file of the gathered data from the connection. */

bool silc_map_writehtml(SilcMap map, SilcMapConnection mapconn)
{
  FILE *fp;
  char *hostname;
  char filename[256], line[128];
  int begin;

  /* Generate data filename.  First configure hostname is the filename */
  silc_dlist_start(mapconn->hostnames);
  hostname = silc_dlist_get(mapconn->hostnames);
  memset(filename, 0, sizeof(filename));
  snprintf(filename, sizeof(filename) - 1, "%s_%d.html", hostname,
	   mapconn->port);

  /* Open for writing */
  fp = fopen(filename, "w+");
  if (!fp) {
    fprintf(stderr, "Could not open file '%s'\n", filename);
    return FALSE;
  }

  /* Write the HTML page */

  fprintf(fp, "<!-- Automatically generated by silcmap -->\n");
  fprintf(fp, "<br /><hr ><br />\n");

  /* General stuff */

  fprintf(fp, "<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\">\n");
  silc_dlist_start(mapconn->hostnames);
  while ((hostname = silc_dlist_get(mapconn->hostnames)) != SILC_LIST_END)
    fprintf(fp, "<tr><td><b>Hostname</b></td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", hostname);

  silc_dlist_start(mapconn->ips);
  while ((hostname = silc_dlist_get(mapconn->ips)) != SILC_LIST_END)
    fprintf(fp, "<tr><td><b>IP</b></td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", hostname);

  fprintf(fp, "<tr><td><b>Port</b></td><td>&nbsp;:</td><td>&nbsp;%d</td></tr>\n", mapconn->port);
  fprintf(fp, "<tr><td><b>Country</b></td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", mapconn->country);
  fprintf(fp, "<tr><td><b>City</b></td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", mapconn->city);
  fprintf(fp, "<tr><td><b>Admin</b></td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", mapconn->admin);
  fprintf(fp, "</table>\n");

  /* Public key */
  if (mapconn->public_key) {
    SilcPublicKey public_key;
    SilcPublicKeyIdentifier ident;
    char *fingerprint, *babbleprint;
    unsigned char *pk;
    SilcUInt32 pk_len;
    SilcPKCS pkcs;
    SilcUInt32 key_len = 0;
    FILE *pd;
    unsigned char *pdd;

    fprintf(fp, "&nbsp;<br /><hr ><br />\n");
    fprintf(fp, "<b>Public Key:</b>&nbsp;<br />\n");
    fprintf(fp, "<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\">\n");

    if (silc_pkcs_load_public_key(mapconn->public_key, &public_key,
				  SILC_PKCS_FILE_PEM) == FALSE)
      if (silc_pkcs_load_public_key(mapconn->public_key, &public_key,
				    SILC_PKCS_FILE_BIN) == FALSE) {
	fprintf(stderr, "Could not load public key file `%s'\n",
		mapconn->public_key);
	return FALSE;
      }

    ident = silc_pkcs_decode_identifier(public_key->identifier);
    pk = silc_pkcs_public_key_encode(public_key, &pk_len);
    fingerprint = silc_hash_fingerprint(NULL, pk, pk_len);
    babbleprint = silc_hash_babbleprint(NULL, pk, pk_len);

    if (silc_pkcs_alloc(public_key->name, &pkcs)) {
      key_len = silc_pkcs_public_key_set(pkcs, public_key);
      silc_pkcs_free(pkcs);
    }

    fprintf(fp, "<tr><td>&nbsp;&nbsp;");
    fprintf(fp, "Public key file</td><td>&nbsp;:</td><td>&nbsp;<a href=\"%s\">%s</a></td></tr>\n",
	    mapconn->public_key, mapconn->public_key);
    fprintf(fp, "<tr><td>&nbsp;&nbsp;");
    fprintf(fp, "Algorithm</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", public_key->name);
    if (key_len) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Key length</td><td>&nbsp;:</td><td>&nbsp;%d bits</td></tr>\n", (unsigned int)key_len);
    }
    if (ident->realname) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Real name</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", ident->realname);
    }
    if (ident->username) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Username</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", ident->username);
    }
    if (ident->host) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Hostname</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", ident->host);
    }
    if (ident->email) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Email</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", ident->email);
    }
    if (ident->org) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Organization</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", ident->org);
    }
    if (ident->country) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Country</td><td>&nbsp;:</td><td>&nbsp;%s</td></tr>\n", ident->country);
    }
    fprintf(fp, "<tr><td>&nbsp;&nbsp;");
    fprintf(fp, "Fingerprint</td><td>&nbsp;:</td><td>&nbsp;<tt>%s</tt></td></tr>\n", fingerprint);
    fprintf(fp, "<tr><td>&nbsp;&nbsp;");
    fprintf(fp, "Babbleprint</td><td>&nbsp;:</td><td>&nbsp;<tt>%s</tt></td></tr>\n", babbleprint);
    fprintf(fp, "</table>\n");

    pd = fopen(mapconn->public_key, "r");
    if (!pd)
      return FALSE;

    pk_len = silc_file_size(mapconn->public_key);
    pdd = silc_calloc(pk_len + 2, sizeof(*pdd));
    if (!pdd)
      return FALSE;
    fread(pdd, pk_len, 1, pd);
    pdd[pk_len] = EOF;

    fprintf(fp, "<br /><tt><small>\n");
    begin = 0;
    while ((begin = silc_gets(line, sizeof(line) - 1, pdd, pk_len + 1, 
			      begin)) != EOF)
      fprintf(fp, "%s<br />\n", line);
    fprintf(fp, "</small></tt><br />\n");

    fclose(pd);
    silc_free(pdd);
    silc_free(fingerprint);
    silc_free(babbleprint);
    silc_free(pk);
    silc_pkcs_public_key_free(public_key);
    silc_pkcs_free_identifier(ident);
  }

  /* Description */
  if (mapconn->description && mapconn->description[0]) {
    fprintf(fp, "<hr ><br />\n");
    fprintf(fp, "<b>Description:</b>&nbsp;<br />\n");
    fprintf(fp, "%s<br />&nbsp;<br />\n", mapconn->description);
  }

  /* Status */
  if (mapconn->connect) {
    fprintf(fp, "<hr ><br />\n");
    fprintf(fp, "<b>Server status:</b>&nbsp;<br />\n");
    if (mapconn->down)
      fprintf(fp,
	      "Server is currently down and unreachable. "
	      "Please try again later to connect the server.<br />\n");
    else
      fprintf(fp,
	      "Server is up and running<br />\n");
  }

  if (mapconn->connect && !mapconn->down) {
    int days, hours, mins, secs, uptime;

    uptime = mapconn->data.uptime;
    days = uptime / (24 * 60 * 60);
    uptime -= days * (24 * 60 * 60);
    hours = uptime / (60 * 60);
    uptime -= hours * (60 * 60);
    mins = uptime / 60;
    uptime -= mins * 60;
    secs = uptime;

    /* Statistics */
    fprintf(fp, "<br />\n");
    fprintf(fp, "<b>Server statistics:</b>&nbsp;<br />\n");
    fprintf(fp, "<table cellspacing=\"0\" cellpadding=\"0\" border=\"0\">\n");
    if (mapconn->starttime) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Server start time</td><td>&nbsp;:</td><td>&nbsp;%s<td></tr>\n",
	      silc_get_time(mapconn->data.starttime));
    }
    if (mapconn->uptime) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Server uptime</td><td>&nbsp;:</td><td>&nbsp;%d days %d hours %d mins %d secs<td></tr>\n",
	      days, hours, mins, secs);
    }
    if (mapconn->clients) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Local clients</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.clients);
    }
    if (mapconn->channels) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Local channels</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.channels);
    }
    if (mapconn->server_ops) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Local server operators</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.server_ops);
    }
    if (mapconn->router_ops) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Local router operators</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.router_ops);
    }
    if (mapconn->cell_clients) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Cell clients</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.cell_clients);
    }
    if (mapconn->cell_channels) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Cell channels</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.cell_channels);
    }
    if (mapconn->cell_servers) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "Cell servers</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.cell_servers);
    }
    if (mapconn->all_clients) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "All SILC clients</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.all_clients);
    }
    if (mapconn->all_channels) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "All SILC channels</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.all_channels);
    }
    if (mapconn->all_servers) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "All SILC servers</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.all_servers);
    }
    if (mapconn->all_routers) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "All SILC routers</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.all_routers);
    }
    if (mapconn->all_server_ops) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "All SILC server operators</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.all_server_ops);
    }
    if (mapconn->all_router_ops) {
      fprintf(fp, "<tr><td>&nbsp;&nbsp;");
      fprintf(fp, "All SILC router operators</td><td>&nbsp;:</td><td>&nbsp;%ld</td></tr>\n",
	      (unsigned long)mapconn->data.all_router_ops);
    }
    fprintf(fp, "</table>\n");
  }

  /* motd */
  if (mapconn->motd && mapconn->data.motd) {
    fprintf(fp, "&nbsp;<br /><hr ><br />\n");
    fprintf(fp, "<b>Message of the Day:</b>&nbsp;<br />\n");

    fprintf(fp, "<br /><tt><small>\n");
    begin = 0;
    while ((begin = silc_gets(line, sizeof(line) - 1, mapconn->data.motd,
			      strlen(mapconn->data.motd), begin)) != EOF)
      fprintf(fp, "%s<br />\n", line);
    fprintf(fp, "</small></tt>\n");
  }

  fprintf(fp, "<br />\n");

  fclose(fp);
  return TRUE;
}

/* Write the HTML index file that lists all servers. */

bool silc_map_writehtml_index(SilcMap map)
{
  SilcMapConnection mapconn;
  char *hostname, *ip, *class;
  FILE *fp;

  /* Open for writing */
  fp = fopen(map->writehtml.filename, "w+");
  if (!fp) {
    fprintf(stderr, "Could not open file '%s'\n", map->writehtml.filename);
    return FALSE;
  }

  /* Produce a simple HTML index file of all servers */
  class = map->writehtml.text ? map->writehtml.text : "silcmap";

  fprintf(fp, "<!-- Automatically generated by silcmap -->\n");
  fprintf(fp, "<br />\n");
  fprintf(fp, "<table cellspacing=\"0\" cellpadding=\"0\" "
	  "class=\"%s\" border=\"0\">\n", class);
  fprintf(fp,
	  "<tr>\n"
	  "<td align=\"center\" class=\"%s_header\"><b>Hostname</b></td>\n"
	  "<td align=\"center\" class=\"%s_header\"><b>IPv4 Address</b></td>\n"
	  "<td align=\"center\" class=\"%s_header\"><b>Port</b></td>\n"
	  "<td align=\"center\" class=\"%s_header\"><b>Country</b></td>\n"
	  "<td align=\"center\" class=\"%s_header\"><b>Oper</b></td>\n"
	  "</tr>\n", class, class, class, class, class);

  silc_dlist_start(map->conns);
  while ((mapconn = silc_dlist_get(map->conns)) != SILC_LIST_END) {

    silc_dlist_start(mapconn->hostnames);
    hostname = silc_dlist_get(mapconn->hostnames);
    silc_dlist_start(mapconn->ips);
    ip = silc_dlist_get(mapconn->ips);

    fprintf(fp, "<tr>\n");
    if (mapconn->html_url)
      fprintf(fp,
	      "<td align = \"center\" class=\"%s\">&nbsp;<a href=\"%s\">%s</a></td>\n", class, mapconn->html_url, hostname);
    else
      fprintf(fp,
	      "<td align = \"center\" class=\"%s\">&nbsp;<a href=\"%s_%d.html\">%s</a></td>\n", class, hostname, mapconn->port, hostname);
    fprintf(fp,
	    "<td align = \"center\" class=\"%s\">&nbsp;%s</td>\n"
	    "<td align = \"center\" class=\"%s\">&nbsp;%d</td>\n"
	    "<td align = \"center\" class=\"%s\">&nbsp;%s</td>\n"
	    "<td align = \"center\" class=\"%s\">&nbsp;%s</td>\n"
	    "</tr>\n",
	    class, ip, class, mapconn->port, class,
	    mapconn->country, class, mapconn->admin);
  }

  fprintf(fp, "</table><br />\n");

  return TRUE;
}

/* Creates a HTML map file, which can be used to allow user to click
   URLs on the image at the specified locations. */

bool silc_map_writemaphtml(SilcMap map)
{
  SilcMapConnection mapconn;
  SilcMapCommand cmd, c;
  char *hostname, url[256];
  FILE *fp;
  int i, xx , yy, w, h;

  for (i = 0; i < map->writemaphtml_count; i++) {
    c = &map->writemaphtml[i];
    if (c->alon && c->alat) {
      c->x = silc_map_lon2x(map, c->alon);
      c->y = silc_map_lat2y(map, c->alat);
    }

    /* Open for writing */
    fp = fopen(c->filename, "w+");
    if (!fp) {
      fprintf(stderr, "Could not open file '%s'\n", c->filename);
      return FALSE;
    }

    /* The target may be portion of the original map, so we must make the
       new coordinates relative to the new map. */
    xx = c->x;
    yy = c->y;

    memset(url, 0, sizeof(url));

    fprintf(fp, "<!-- Automatically generated by silcmap -->\n");
    fprintf(fp, "<img src=\"%s\" usemap=\"#map\" class=\"silcmap\">\n",
	    c->text);
    fprintf(fp, "<map name=\"map\">\n");

    silc_dlist_start(map->conns);
    while ((mapconn = silc_dlist_get(map->conns)) != SILC_LIST_END) {
      memset(url, 0, sizeof(url));
      if (mapconn->html_url && mapconn->html_url[0]) {
	silc_strncat(url, sizeof(url), mapconn->html_url,
		     strlen(mapconn->html_url));
      } else {
	silc_dlist_start(mapconn->hostnames);
	hostname = silc_dlist_get(mapconn->hostnames);
	snprintf(url, sizeof(url) - 1, "%s_%d.html", hostname, mapconn->port);
      }

      /* Print the positions of various items on the map into the map file */
      silc_dlist_start(mapconn->commands);
      while ((cmd = silc_dlist_get(mapconn->commands)) != SILC_LIST_END) {
	if (cmd->alon && cmd->alat) {
	  cmd->x = silc_map_lon2x(map, cmd->alon);
	  cmd->y = silc_map_lat2y(map, cmd->alat);
	}

	if (cmd->draw_text) {
	  w = strlen(cmd->text) * 5;
	  h = map->font.height - 2;
	  fprintf(fp,
		  "<area shape=\"rect\" coords=\"%d,%d,%d,%d\" href=\"%s\">\n",
		  (int)(cmd->x - xx), (int)(cmd->y - yy), w, h, url);
	}

	if (cmd->draw_circle) {
	  w = 4;
	  fprintf(fp,
		  "<area shape=\"circle\" coords=\"%d,%d,%d\" href=\"%s\">\n",
		  (int)(cmd->x - xx), (int)(cmd->y - yy), w, url);
	  if (cmd->text) {
	    w = strlen(cmd->text) * 5;
	    h = map->font.height - 2;
	    fprintf(fp,
		    "<area shape=\"rect\" coords=\"%d,%d,%d,%d\" href=\"%s\">\n",
		    (int)(cmd->x - xx + cmd->lposx),
		    (int)(cmd->y - yy - cmd->lposy),
		    (int)(cmd->x - xx + cmd->lposx + w),
		    (int)(cmd->y - yy - cmd->lposy + h), url);
	  }
	}

	if (cmd->draw_rectangle) {
	  w = 7;
	  h = 6;
	  fprintf(fp,
		  "<area shape=\"rect\" coords=\"%d,%d,%d,%d\" href=\"%s\">\n",
		  (int)(cmd->x - xx), (int)(cmd->y - yy),
		  (int)(cmd->x - xx + w), (int)(cmd->y - yy + h), url);
	  if (cmd->text) {
	    w = strlen(cmd->text) * 5;
	    h = map->font.height - 2;
	    fprintf(fp,
		    "<area shape=\"rect\" coords=\"%d,%d,%d,%d\" href=\"%s\">\n",
		    (int)(cmd->x - xx + cmd->lposx),
		    (int)(cmd->y - yy - cmd->lposy),
		    (int)(cmd->x - xx + cmd->lposx + w),
		    (int)(cmd->y - yy - cmd->lposy + h), url);
	  }
	}
      }
    }

    fprintf(fp, "</map>\n");
    fclose(fp);
  }

  return TRUE;
}

/* Writes the server uptime reliablity data file. */

bool silc_map_writerel(SilcMap map, SilcMapConnection mapconn)
{
  FILE *fp;
  int try = 0, success = 0;
  char f[256], *hostname;

  /* Generate data filename */
  memset(f, 0, sizeof(f));
  silc_dlist_start(mapconn->hostnames);
  hostname = silc_dlist_get(mapconn->hostnames);
  snprintf(f, sizeof(f) - 1, "%s_%d.rel", hostname, mapconn->port);

  /* Read the current data */
  fp = fopen(f,  "r");
  if (fp) {
    fscanf(fp, "%d:%d", &try, &success);
    fclose(fp);
  }

  /* Update the data */
  try++;
  success = (mapconn->down == FALSE ? success + 1 : success);

  /* Write the data file */
  fp = fopen(f, "w+");
  if (!fp) {
    fprintf(stderr, "Could not open file '%s'\n", map->writerel.filename);
    return FALSE;
  }
  fprintf(fp, "%d:%d", try, success);

  fclose(fp);
  return TRUE;
}

/* Writes the servers' uptime reliability graph as HTML page. */

bool silc_map_writerelhtml(SilcMap map)
{
  SilcMapConnection mapconn;
  char *hostname, *class;
  FILE *fp, *dp;

  /* Open for writing */
  fp = fopen(map->writerel.filename, "w+");
  if (!fp) {
    fprintf(stderr, "Could not open file '%s'\n", map->writerel.filename);
    return FALSE;
  }

  /* Produce the reliability graph as HTML file. */
  class = map->writerel.text ? map->writerel.text : "silcmap";

  fprintf(fp, "<!-- Automatically generated by silcmap -->\n");
  fprintf(fp, "<br />\n");
  fprintf(fp, "<table cellspacing=\"0\" cellpadding=\"0\" "
	  "class=\"%s\" border=\"0\">\n", class);
  fprintf(fp,
	  "<tr>\n"
	  "<td align=\"center\" class=\"%s_header\"><b>Server</b></td>\n"
	  "<td colspan=\"2\" align=\"center\" class=\"%s_header\"><b>Reliability</b></td>\n"
	  "</tr>\n", class, class);

  silc_dlist_start(map->conns);
  while ((mapconn = silc_dlist_get(map->conns)) != SILC_LIST_END) {
    char f[256];
    int try = 0, success = 0;
    double rel = 0;

    silc_dlist_start(mapconn->hostnames);
    hostname = silc_dlist_get(mapconn->hostnames);

    /* Get the data */
    memset(f, 0, sizeof(f));
    snprintf(f, sizeof(f) - 1, "%s_%d.rel", hostname, mapconn->port);
    dp = fopen(f,  "r");
    if (dp) {
      fscanf(dp, "%d:%d", &try, &success);
      fclose(dp);
    }

    /* Count the reliability */
    if (try)
      rel = ((double)success / (double)try) * (double)160.0;

    fprintf(fp, "<tr>\n");
    if (mapconn->html_url)
      fprintf(fp,
	      "<td align = \"center\" class=\"%s\">&nbsp;<a href=\"%s\">%s</a></td>\n", class, mapconn->html_url, hostname);
    else
      fprintf(fp,
	      "<td align = \"center\" class=\"%s\">&nbsp;<a href=\"%s_%d.html\">%s</a></td>\n", class, hostname, mapconn->port, hostname);
    fprintf(fp,
	    "<td class=\"%s\" width=\"160\">"
	    "<table style=\"border: solid 1px black; width: 160px;\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\" class=\"%s\"><tr>"
	    "<td style=\"width: %fpx; height: 10px;\" bgcolor=\"gray\"></td>"
	    "<td style=\"width: %fpx; height: 10px;\" bgcolor=\"white\"></td>"
	    "</tr></table></td>\n"
	    "<td class=\"%s\">%.2f%% - score: %d</td>\n"
	    "</tr>\n",
	    class, class, rel, 160 - rel, class,
	    ((double)success / (double)try) * (double)100.0, success);
  }

  fprintf(fp, "</table><br />\n");

  return TRUE;
}
