/*

  silcruntime.h

  Author: Pekka Riikonen <priikone@silcnet.org>

  Copyright (C) 2008 Pekka Riikonen

  The contents of this file are subject to one of the Licenses specified 
  in the COPYING file;  You may not use this file except in compliance 
  with the License.

  The software distributed under the License is distributed on an "AS IS"
  basis, in the hope that it will be useful, but WITHOUT WARRANTY OF ANY
  KIND, either expressed or implied.  See the COPYING file for more
  information.

*/

/****h* silcutil/Runtime Toolkit Interface
 *
 * DESCRIPTION
 *
 * The SILC Runtime Toolkit API is used to initialize and uninitialize the
 * Runtime Toolkit.  The application should call silc_runtime_init at the
 * very start of the application and the silc_runtime_uninit at the very end
 * of the application before it is exited.
 *
 ***/

#ifndef SILCRUNTIME_H
#define SILCRUNTIME_H

/* Version check macro.  Use this to check that package is of specific
   version compile time.  Use the __SILC_XXX_VERSION below in comparison.
   Example:

   #if __SILC_RUNTIME_VERSION < SILC_VERSION(1,2,0)
     ...
   #endif
*/
#ifndef SILC_VERSION
#define SILC_VERSION(a, b, c) (((a) << 24) + ((b) << 16) + ((c) << 8)
#endif /* !SILC_VERSION */

/* SILC Runtime Toolkit version */
#define __SILC_RUNTIME_VERSION SILC_VERSION(1,2-beta2,0)

/* Compilation time defines, for third-party software.  Use these to check
   in configuration/compilation time how the SILC Runtime Toolkit was
   configured/compiled. */
#define __SILC_HAVE_PTHREAD 1



/* Types */
#define SILC_SIZEOF_LONG_LONG 8
#define SILC_SIZEOF_LONG 8
#define SILC_SIZEOF_INT 4
#define SILC_SIZEOF_SHORT 2
#define SILC_SIZEOF_CHAR 1
#define SILC_SIZEOF_VOID_P 8

/* Detect OS */
#define SILC_UNIX

#ifdef WIN32
#ifndef SILC_WIN32
#define SILC_WIN32
#undef SILC_UNIX
#endif
#endif

#if defined(__EPOC32__) || defined(__SYMBIAN32__)
#ifndef SILC_SYMBIAN
#define SILC_SYMBIAN
#undef SILC_UNIX
#undef SILC_WIN32
#endif
#endif

#if defined(__MACH__) && defined(__APPLE__)
#ifndef SILC_MACOSX
#define SILC_MACOSX
#undef SILC_WIN32
#undef SILC_SYMBIAN
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* Platform specific includes */
#if defined(SILC_WIN32)
#include <silcwin32.h>
#endif

#if defined(SILC_SYMBIAN)
#include <silcsymbian.h>
#endif

#ifndef DLLAPI
#define DLLAPI
#endif

/* We except all systems to have these include files */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(HAVE_RUNTIMEDEFS_H)

/* Automatically generated configuration header.  These are included only
   when the SILC Runtime Toolkit itself is compiled. */
#ifndef SILC_SYMBIAN
#include "runtimedefs.h"
#else
#include "../../symbian/silcdefs.h"
#endif /* SILC_SYMBIAN */
#include "silcdistdefs.h"
#include "silccompile.h"

#else

/* Autodetect CPU so that inline assembly in headers are enabled */

#if defined(__i386__)
#undef SILC_I386
#undef SILC_I486
#define SILC_I386
#define SILC_I486
#endif /* __i386__ */

#if defined(__i486__)
#undef SILC_I386
#undef SILC_I486
#define SILC_I386
#define SILC_I486
#endif /* __i486__ */

#if defined(__i586__)
#undef SILC_I386
#undef SILC_I486
#undef SILC_I586
#define SILC_I386
#define SILC_I486
#define SILC_I586
#endif /* __i586__ */

#if defined(__i686__)
#undef SILC_I386
#undef SILC_I486
#undef SILC_I586
#undef SILC_I686
#define SILC_I386
#define SILC_I486
#define SILC_I586
#define SILC_I686
#endif /* __i686__ */

#if defined(__x86_64__)
#undef SILC_X86_64
#define SILC_X86_64
#endif /* __x86_64__ */

#if defined(__ia64__)
#undef SILC_IA64
#define SILC_IA64
#endif /* __ia64__ */

#if defined(__ppc__) || defined(__ppc64__)
#undef SILC_POWERPC
#define SILC_POWERPC
#endif /* __ppc__ || __ppc64__ */

#ifndef SILC_ALIGNMENT
#define SILC_ALIGNMENT SILC_SIZEOF_VOID_P
#endif /* SILC_ALIGNMENT */

#endif /* HAVE_RUNTIMEDEFS_H */

/* Feature defines */
#ifdef __SILC_HAVE_PTHREAD
#undef SILC_THREADS
#define SILC_THREADS 1
#endif /* __SILC_HAVE_PTHREAD */

#ifdef __SILC_ENABLE_DEBUG
#undef SILC_DEBUG
#define SILC_DEBUG 1
#endif /* __SILC_ENABLE_DEBUG */

#ifdef __SILC_ENABLE_STACKTRACE
#undef SILC_STACKTRACE
#define SILC_STACKTRACE 1
#endif /* __SILC_ENABLE_STACKTRACE */

/* SILC Runtime Toolkit includes */
#include <silcerrno.h>
#include <silctypes.h>
#include <silcbitops.h>
#include <silcmutex.h>
#include <silcatomic.h>
#include <silcgetopt.h>
#include <silcstack.h>
#include <silcmemory.h>
#include <silclist.h>
#include <silcdlist.h>
#include <silcsnprintf.h>
#include <silctime.h>
#include <silctimer.h>
#include <silccond.h>
#include <silcglobal.h>
#include <silcthread.h>
#include <silcthreadqueue.h>
#include <silcschedule.h>
#include <silclog.h>
#include <silcfileutil.h>
#include <silcdir.h>
#include <silcbuffer.h>
#include <silcbuffmt.h>
#include <silcasync.h>
#include <silcregex.h>
#include <silcenv.h>
#include <silcdll.h>
#include <silchashtable.h>
#include <silcstream.h>
#include <silcnet.h>
#include <silcbase64.h>
#include <silcstrutil.h>
#include <silcutf8.h>
#include <silcstringprep.h>
#include <silcutil.h>
#include <silcconfig.h>
#include <silcfsm.h>
#include <silcsocketstream.h>
#include <silcfdstream.h>
#include <silcmime.h>
#include <silcrand.h>

/* Runtime Toolkit API */

/****f* silcutil/silc_runtime_init
 *
 * SYNOPSIS
 *
 *    static inline void silc_runtime_init(void)
 *
 * DESCRIPTION
 *
 *    Initializes the Runtime Toolkit.  Application should call this as the
 *    very first call to the Runtime Toolkit.  The function initializes some
 *    of the Toolkit features.
 *
 ***/
static inline
void silc_runtime_init(void)
{
  /* For now we don't have much here.  Strictly speaking, calling this
     currently isn't required.  But in the future we may have something that
     really must be initialized at the very start.  Even the Tls doesn't
     actually require initialization since any routine using Tls currently
     takes care of that. */

  /* Initialize Tls for this thread (probably main process thread). */
  silc_thread_tls_init();
}

/****f* silcutil/silc_runtime_uninit
 *
 * SYNOPSIS
 *
 *    static inline void silc_runtime_uninit(void)
 *
 * DESCRIPTION
 *
 *    Uninitializes the Runtime Toolkit.  Application should call this as the
 *    very last call to the Runtime Toolkit.  The function uninitializes and
 *    frees all allocated resources.
 *
 ***/
static inline
void silc_runtime_uninit(void)
{
  /* Uninitialize the whole Tls system (not just one thread). */
  silc_thread_tls_uninit();
}

#ifdef __cplusplus
}
#endif

#endif /* SILCRUNTIME_H */
