/*

  silcbitops.c

  Author: Pekka Riikonen <priikone@silcnet.org>

  Copyright (C) 2007 - 2008 Pekka Riikonen

  The contents of this file are subject to one of the Licenses specified 
  in the COPYING file;  You may not use this file except in compliance 
  with the License.

  The software distributed under the License is distributed on an "AS IS"
  basis, in the hope that it will be useful, but WITHOUT WARRANTY OF ANY
  KIND, either expressed or implied.  See the COPYING file for more
  information.

*/

#include "silcruntime.h"

#define SILC_BIT_POS(bit) (bit / SILC_BIT_SIZE)
#define SILC_BIT_MASK(bit) (1UL << (bit % SILC_BIT_SIZE))

/* Set bit */

SilcBool silc_bit_set(volatile unsigned long *bitmap, SilcUInt32 bitmap_size,
		      SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit);

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return FALSE;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return FALSE;
  }

  bitmap[pos] |= mask;
  return TRUE;
}

/* Clear bit */

SilcBool silc_bit_clear(volatile unsigned long *bitmap, SilcUInt32 bitmap_size,
			SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit);

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return FALSE;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return FALSE;
  }

  bitmap[pos] &= ~mask;
  return TRUE;
}

/* Toggle bit */

SilcBool silc_bit_toggle(volatile unsigned long *bitmap,
			 SilcUInt32 bitmap_size, SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit);

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return FALSE;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return FALSE;
  }

  bitmap[pos] ^= mask;
  return TRUE;
}

/* Set bit and return old value */

int silc_bit_test_and_set(volatile unsigned long *bitmap,
			  SilcUInt32 bitmap_size, SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit), ret;

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return -1;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return -1;
  }

  ret = bitmap[pos];
  bitmap[pos] ^= mask;

  return (ret & mask) != 0;
}

/* Clear bit and return old value */

int silc_bit_test_and_clear(volatile unsigned long *bitmap,
			    SilcUInt32 bitmap_size, SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit), ret;

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return -1;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return -1;
  }

  ret = bitmap[pos];
  bitmap[pos] &= ~mask;

  return (ret & mask) != 0;
}

/* Toggle bit and return old value */

int silc_bit_test_and_toggle(volatile unsigned long *bitmap,
			     SilcUInt32 bitmap_size, SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit), ret;

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return -1;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return -1;
  }

  ret = bitmap[pos];
  bitmap[pos] ^= mask;

  return (ret & mask) != 0;
}

/* Return bit value */

int silc_bit_get(volatile unsigned long *bitmap, SilcUInt32 bitmap_size,
		 SilcUInt32 bit)
{
  SilcUInt32 pos = SILC_BIT_POS(bit);
  unsigned long mask = SILC_BIT_MASK(bit);

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return -1;
  }
  if (pos >= bitmap_size) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return -1;
  }

  return (bitmap[pos] & mask) != 0;
}

/* Return first set bit number */

int silc_bit_ffs(volatile unsigned long *bitmap, SilcUInt32 bitmap_size)
{
  return silc_bit_fns(bitmap, bitmap_size, 0);
}

/* Return first zero bit number */

int silc_bit_ffz(volatile unsigned long *bitmap, SilcUInt32 bitmap_size)
{
  return silc_bit_fnz(bitmap, bitmap_size, 0);
}

/* Return next set bit number */

int silc_bit_fns(volatile unsigned long *bitmap, SilcUInt32 bitmap_size,
		 SilcUInt32 offset)
{
  register SilcUInt32 i;

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return -1;
  }
  if (offset >= bitmap_size * SILC_BIT_SIZE) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return -1;
  }

  for (i = offset; i < bitmap_size * SILC_BIT_SIZE; i++)
    if (bitmap[SILC_BIT_POS(i)] & SILC_BIT_MASK(i))
      return i;

  silc_set_errno(SILC_ERR_NOT_FOUND);
  return -1;
}

/* Return next zero bit number */

int silc_bit_fnz(volatile unsigned long *bitmap, SilcUInt32 bitmap_size,
		 SilcUInt32 offset)
{
  register SilcUInt32 i;

  if (!bitmap) {
    silc_set_errno(SILC_ERR_INVALID_ARGUMENT);
    return -1;
  }
  if (offset >= bitmap_size * SILC_BIT_SIZE) {
    silc_set_errno(SILC_ERR_OVERFLOW);
    return -1;
  }

  for (i = offset; i < bitmap_size * SILC_BIT_SIZE; i++)
    if ((bitmap[SILC_BIT_POS(i)] & SILC_BIT_MASK(i)) == 0)
      return i;

  silc_set_errno(SILC_ERR_NOT_FOUND);
  return -1;
}

/* Clear bitmap */

void silc_bit_clear_bitmap(volatile unsigned long *bitmap,
			   SilcUInt32 bitmap_size)
{
  if (!bitmap)
    return;
  memset((void *)bitmap, 0, bitmap_size * 8);
}
