/*
 * RIPEMD-160 Production Implementation
 * This implementation: (c) 1999 Andreas Steinmetz
 * RIPEMD-160 algorithm by Hans Dobbertin, Antoon Bosselaers and Bart Preneel,
 * see http://www.esat.kuleuven.ac.be/~bosselae/ripemd160.html
 * HMAC algorithm by Krawczyk, et. al., see RFC2104
 * Test cases for HMAC: see RFC2286
 *
 * This code is under the GNU public license
 * for use with the virtual private network daemon (vpnd).
 * The copyright holder will however retain the copyright.
 * In addition to vpnd this code may be used
 * for all GPL/LGPL based open source software products.
 * For all other software products please contact astmail@yahoo.com
 * for a license agreement. There is no guarantee for the fitness
 * and usability of this code for any purpose. The author takes no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 *
 * This implementation is tuned for speed.
 *
 * Note that the implementation is processor independent.
 * It does not depend on any run time library and
 * should be 64 bit clean.
 *
 * Restriction:
 *
 * The maximum data length to be processed is 2^32-1 bytes but
 * this shouldn't be an issue for nearly all applications. If
 * it is an issue you will have to extend the variable 'total'
 * from 32 bits to 64 bits in the RIPEMD-160 routines.
 *
 * Data sizes:
 *
 * data block for RIPEMD-160 transformation	WORD08[64]
 * resulting RIPEMD-160 hash			WORD08[20]
 *
 * WORD08 means an unsigned word of 8 bits length
 * WORD32 means an unsigned word of at least 32 bits length
 *
 * Mode of operation:
 *
 * RMD160_FULL defined:
 *
 * 1. Call rmd160init
 * 2. For all data to be hashed call rmd160next
 * 3. To retrieve the hash call rmd160end
 *
 * RMD160_FAST defined:
 *
 * Call rmd160 to get the hash for the specified data
 *
 * RMD160_HMAC_FULL defined:
 *
 * 1. Call rmd160hmkey once to preprocess the selected key
 * 2. Call rmd160hminit
 * 3. For all data to be hashed call rmd160hmnext
 * 4. To retrieve the hash call rmd160hmend
 * 5. When the preprocessed key is no longer required
 *    reset the RMD160HMDATA structure
 *
 * RMD160_HMAC_FAST defined:
 *
 * 1. Call rmd160hmkey once to preprocess the selected key
 * 2. Call rmd160hmac to get the hash for the specified data
 * 3. When the preprocessed key is no longer required
 *    reset the RMD160HMDATA structure
 */

/* sanity settings */

#if defined(RMD160_HMAC_FULL) || defined(RMD160_HMAC_FAST)
#ifndef RMD160_HMAC
#define RMD160_HMAC
#endif
#ifndef RMD160_PARANOID
#define RMD160_PARANOID
#endif
#endif

/* definitions */

#define RMD160_SIZE 20
#define RMD160(a) WORD08 a[RMD160_SIZE]

/* required types */

typedef struct
{
	WORD32 rmd160[5];	/* current rmd160 result		*/
	WORD32 total;	/* total amount of bytes processed		*/
	union
	{
	  WORD32 l[16];	/* longword buffer for paranoia memory reset	*/
	  WORD08 b[64];	/* buffer for incomplete 64 byte message block	*/
	}bfr;
	WORD08 size;	/* size of data in buffer			*/
} RMD160DATA;

#ifdef RMD160_HMAC
typedef struct
{
	WORD32 irmd160[5];	/* preprocessed HMAC key (inner)	*/
	WORD32 ormd160[5];	/* preprocessed HMAC key (outer)	*/
} RMD160HMDATA;
#endif

/* function prototypes */

#if defined(RMD160_FULL) || defined(RMD160_HMAC)
extern void CRYPTOCALL rmd160init(RMD160DATA *ptr);
extern void CRYPTOCALL rmd160next(WORD08 *data,WORD32 length,RMD160DATA *ptr);
extern void CRYPTOCALL rmd160end(WORD08 *result,RMD160DATA *ptr);
#endif
#ifdef RMD160_FAST
extern void CRYPTOCALL rmd160(WORD08 *data,WORD32 length,WORD08 *result);
#endif
#if defined(RMD160_HMAC_FULL) || defined(RMD160_HMAC_FAST)
extern void CRYPTOCALL rmd160hmkey(WORD08 *key,WORD32 keylength,RMD160HMDATA *ptr);
#endif
#ifdef RMD160_HMAC_FULL
extern void CRYPTOCALL rmd160hminit(RMD160DATA *ptr,RMD160HMDATA *key);
#define rmd160hmnext(a,b,c) rmd160next(a,b,c)
extern void CRYPTOCALL rmd160hmend(WORD08 *result,RMD160DATA *ptr,RMD160HMDATA *key);
#endif
#ifdef RMD160_HMAC_FAST
extern void CRYPTOCALL rmd160hmac(WORD08 *data,WORD32 length,WORD08 *result,RMD160HMDATA *key);
#endif

