/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* whereto, my friend: routing                                                */
/*============================================================================*/

/*
 * route
 *
 * input:  how     - 1=add route, 0=delete route
 *	   dest    - the route destination in network byte order
 *	   mask    - the destination netmask in network byte order or -1 for
 *		     a host route
 *	   gateway - the gateway to the destination in network byte order or
 *		     0 if no gateway
 *	   device  - the device for the route
 *
 * return: 0 in case of success, 1 in case of error
 *
 * This procedure adds/deletes the given route to/from the system routing table.
 * The metric of the route is always 0 (rt_metric=real metric+1).
 */

int route(int how,WORD32 dest,WORD32 mask,WORD32 gateway,WORD08 *device)
{
	struct rtentry rt;	/* used to add routes		*/
	int s;			/* routing socket handle	*/


	/* debug message */

	ENTER("route");

	/* set up routing structure */

	memset(&rt,0,sizeof(rt));
	rt.rt_dev=device;
	if(mask==-1)
	{
		rt.rt_flags=RTF_UP|RTF_HOST|(gateway?RTF_GATEWAY:0);
		mask=htonl(0xffffffff);
	}
	else rt.rt_flags=RTF_UP|(gateway?RTF_GATEWAY:0);
	rt.rt_metric=1;
	((struct sockaddr_in *)(&(rt.rt_dst)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(rt.rt_dst)))->sin_addr.s_addr=dest;
	((struct sockaddr_in *)(&(rt.rt_genmask)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(rt.rt_genmask)))->sin_addr.s_addr=mask;
	((struct sockaddr_in *)(&(rt.rt_gateway)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(rt.rt_gateway)))->sin_addr.s_addr=gateway;

	/* add routing entry, handle errors */

	if((s=socket(AF_INET,SOCK_DGRAM,0))==-1)JUMP("socket",err1);
	if(ioctl(s,how?SIOCADDRT:SIOCDELRT,&rt)==-1)
		JUMP(how?"ioctl(SIOCADDRT)":"ioctl(SIOCDELRT)",err2);
	if(close(s))JUMP("close",err1);

	/* debug message */

	LEAVE("route");

	/* signal success */

	return 0;

	/* error handling */

err2:	close(s);
err1:	return 1;
}

/*
 * peerroute
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0  in case of failure, else > 0
 *
 * This procedure searches the routing tables for a routing entry
 * that points to the given peer. If suche an entry is found
 * parameters for a priority route to the peer are set up.
 */

int peerroute(VPN *anchor)
{
	FILE *fp;		/* routing table access	*/
	int i;			/* counter/index	*/
	char *item;		/* current item		*/
	char *device;		/* peer route device	*/
	WORD32 data[7];		/* routing table data	*/
	WORD32 mask;		/* best routing mask	*/
	char *bfr;		/* input buffer		*/


	/* debug message */

	ENTER("peerroute");

	/* set up buffer pointer */

	bfr=anchor->u2.bfr;

	/* open routing table, handle errors */

	if((fp=fopen(ROUTELIST,"r"))==NULL)RETURN("fopen",0);

	/* reset netmask and first line flag */

	mask=i=0;

	/* for all lines of the routing table */

	while(fgets(bfr,sizeof(anchor->u2.bfr),fp))
	{
		/* skip first line (only contains header) */

		if(!i)
		{
			i++;
			continue;
		}

		/* get pointer to routing table entry 1 */

		if((device=strtok(bfr," \t\r\n"))==NULL)
		{
			fclose(fp);
			RET("device",0);
		}

		/* convert routing entries 2 to 8 to numeric
		   (actually values 5 to 7 are never used) */

		for(i=0;i<7;i++)
		{
			if((item=strtok(NULL," \t\r\n"))==NULL)
			{
				fclose(fp);
				RET("item n",0);
			}

			data[i]=0;
			while(*item)
			{
				data[i]=(data[i]<<4)|(*item>='A'?
					*item-'A'+10:*item-'0');
				item++;
			}
		}

		/* if a route to the peer exists with a 
		   higher or equal precedence
		   than the previous one (if any) */

		if((anchor->peerip&data[6])==(data[0]&data[6]))
			if(ntohl(data[6])>=mask)
		{
			/* set route flag according to
			   route type */

			anchor->peerroute=data[2]&RTF_GATEWAY?2:1;

			/* memorize netmask of route */

			mask=ntohl(data[6]);

			/* memorize gateway (if route
			   through gateway */

			anchor->peergate=data[1];

			/* sanity copy of routing device name */

			strncpy(anchor->peerdev,device,16);
			anchor->peerdev[15]=0;
		}
	}

	/* close routing table */

	fclose(fp);

	/* debug message */

	LEAVE("peerroute");

	/* signal success or error */

	return anchor->peerroute;
}
