/*
 * virtual private network daemon (vpnd)
 *
 * (c) 2001, 2005 Andreas Steinmetz with exceptions as stated below
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 *
 * -------------------------------------------------------------------
 * The following stuff is related to the Solaris specific code taken
 * from pppd (ftp://cs.anu.edu.au/pub/software/ppp/) 2.3.10 and 2.4.1:
 * -------------------------------------------------------------------
 *
 * Copyright (c) 1994 The Australian National University.
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, provided that the above copyright
 * notice appears in all copies.  This software is provided without any
 * warranty, express or implied. The Australian National University
 * makes no representations about the suitability of this software for
 * any purpose.
 *
 * IN NO EVENT SHALL THE AUSTRALIAN NATIONAL UNIVERSITY BE LIABLE TO ANY
 * PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 * THE AUSTRALIAN NATIONAL UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE AUSTRALIAN NATIONAL UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE AUSTRALIAN NATIONAL UNIVERSITY HAS NO
 * OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
 * OR MODIFICATIONS.
 */

#define USE_PPP		/* workaround for OpenBSD 3.0 dup struct vjstat */
#include "vpnd.h"

#if defined(SunOS)
#define _PPPIO(n)     (('p' << 8) + (n))
#define PPPIO_NEWPPA  _PPPIO(130)
#define PPPIO_CFLAGS  _PPPIO(135)
#define PPPIO_XACCM   _PPPIO(138)
#define PPPIO_RACCM   _PPPIO(139)
#define PPPIO_VJINIT  _PPPIO(140)
#define PPPIO_NPMODE  _PPPIO(146)
#define PPP_IP        0x21
#define NPMODE_PASS   0
#define NPMODE_DROP   1
#define COMP_VJC      0x10
#define COMP_VJCCID   0x20
#define DECOMP_VJCCID 0x80
#define DECOMP_VJC    0x40
#endif

/*============================================================================*/
/* modern times: ppp interface creation and deletion                          */
/*============================================================================*/

/*
 * ppphandler
 *
 * input:  smode  - bitmap, see below
 *		    bit 0: 0=use pty, 1=use serial line
 *		    bit 1: 0=create interface, 1=delete interface
 *	   anchor - pointer to vpnd global data
 *
 * return: in interface creation mode
 *
 *	   0 - success
 *	   1 - no tty/pty available
 *	   2 - tty configuration failed
 *	   3 - saving/setting line discipline failed
 *	   4 - setting ppp encapsulation/ifconfig failed
 *	   5 - creating route to partner failed
 *	   6 - creating priority route to peer failed
 *	   7 - creation of additional routes failed
 *	   8 - no icmp socket (with keepalive only)
 *
 *	   in interface deletion mode:
 *
 *	   0 - success
 *	   1 - tty/pty close error
 *	   2 - device config restore failed
 *	   3 - line discipline restore failed
 *	   4 - ifconfig down failed
 *	   5 - deleting route to partner failed
 *	   6 - deletion of priority route or additional routes failed
 *
 * This procedure creates or deletes the proxy ppp interface.
 */

int ppphandler(int smode,VPN *anchor)
{
	int i;			/* general purpose	*/
	int j;			/* general purpose	*/
	int k;			/* general purpose	*/
	int err;		/* return code		*/
	int quiet;		/* >0 for quiet errors	*/
	struct ifreq ifr;	/* used to set up if	*/
	int npi[2];		/* interface setup	*/
#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	struct ifaliasreq ifa;	/* used to set up if	*/
#endif
#if !defined(SunOS)
	struct termios t;	/* pty/tty settings	*/
	char bfr[16];		/* string creation	*/
#else
	char *p;		/* pty name		*/
	struct strioctl str;	/* interface setup	*/
	char npc[2];		/* header compr. setup	*/
#endif


	/* debug message */

	ENTER("ppphandler");

	/* reset return code */

	err=0;

	/* handle deletion requests in error handler tree */

	if(smode&2)goto delete;

	/* silence error handling */

	quiet=1;

	/* if a pseudo terminal has to be used */

	if(!(smode&1))
	{
		/* reset master to 'no device' */

		anchor->pty=-1;

		/* try to get pty/tty device pair (Unix98 style) */

#if defined(SunOS)
		if((anchor->pty=open("/dev/ptmx",O_RDWR))>=0)
		{
			if(unlockpt(anchor->pty)==-1)
			{
				close(anchor->pty);
				anchor->pty=-1;
			}
			else if(!(p=ptsname(anchor->pty)))
			{
				close(anchor->pty);
				anchor->pty=-1;
			}
			else if((anchor->tty=open(p,O_RDWR))==-1)
			{
				close(anchor->pty);
				anchor->pty=-1;
			}
			else if(ioctl(anchor->tty,I_PUSH,"ppp_ahdl")==-1)
			{
				close(anchor->tty);
				close(anchor->pty);
				anchor->pty=-1;
			}
			else if(anchor->cslip)
				if(ioctl(anchor->tty,I_PUSH,"ppp_comp")==-1)
			{
				close(anchor->tty);
				close(anchor->pty);
				anchor->pty=-1;
			}
		}
#elif defined(TIOCGPTN) && defined(TIOCSPTLCK)
		if((anchor->pty=open("/dev/ptmx",O_RDWR))>=0)
		{
			j=0;
			if(!ioctl(anchor->pty,TIOCGPTN,&i)&&
				!ioctl(anchor->pty,TIOCSPTLCK,&j))
			{
				j=1;
				sprintf(bfr,"/dev/pts/%d",i);
				if((anchor->tty=open(bfr,
					O_RDWR|O_NOCTTY|O_NDELAY))==-1)
				{
					close(anchor->pty);
					anchor->pty=-1;
				}
				else ioctl(anchor->pty,TIOCSPTLCK,&j);
			}
			else
			{
				close(anchor->pty);
				anchor->pty=-1;
			}
		}
#endif

		/* try to get pty/tty device pair (BSD style) */
#if !defined(SunOS)
		if(anchor->pty<0)
		    for(anchor->pty=-1,i='p';i<'t'&&anchor->pty<0;i++)
			for(j=0;j<16;j++)
		{
			sprintf(bfr,"/dev/pty%c%x",i,j);
			if((anchor->pty=open(bfr,O_RDWR))>=0)
			{
				bfr[5]='t';
				if((anchor->tty=open(bfr,
					O_RDWR|O_NOCTTY|O_NDELAY))>=0)
					break;
				close(anchor->pty);
				anchor->pty=-1;
			}
		}
#endif
		/* signal error in case of failure */

		if(anchor->pty<0)JUMP("open",err1);

		/* save tty device attributes, react to errors */
#if !defined(SunOS)
		if(tcgetattr(anchor->tty,&(anchor->original))<0)
			JUMP("tcgetattr",err2);

		/* set tty device attributes, react to errors */

		t.c_cflag=CS8|CREAD|HUPCL|CLOCAL;
		t.c_iflag=IGNBRK|IGNPAR;
		t.c_oflag=0;
		t.c_lflag=0;
		for(i=0;i<NCCS;i++)t.c_cc[i]=0;
		t.c_cc[VMIN]=1;
		t.c_cc[VTIME]=0;
		cfsetispeed(&t,B115200);
		cfsetospeed(&t,B115200);
		if(tcsetattr(anchor->tty,TCSAFLUSH,&t)<0)JUMP("tcsetattr",err2);
#endif
	}

	/* in serial line mode use serial line handle */

	else anchor->tty=anchor->serial;

	/* save old line discipline, react to errors */
#if !defined(SunOS)
	if(ioctl(anchor->tty,TIOCGETD,&(anchor->disc))<0)
		JUMP("ioctl(TIOCGETD)",err3);
#endif
	/* set line discipline to PPP, react to errors */

#if defined(SunOS)
	if((anchor->pppfd=open("/dev/ppp",O_RDWR|O_NONBLOCK))<0)
		JUMP("open pppfd /dev/ppp",err3);
	str.ic_cmd=PPPIO_NEWPPA;
	str.ic_timout=0;
	str.ic_len=0;
	str.ic_dp=(void *)(npi);
	if(ioctl(anchor->pppfd,I_STR,&str)<0)JUMP("ioctl(PPPIO_NEWPPA)",err3a);
	anchor->proxy=*npi;
	if((anchor->ifd=open("/dev/ppp",O_RDWR))<0)
		JUMP("open ifd /dev/ppp",err3a);
	if(ioctl(anchor->ifd,I_PUSH,"ip")<0)JUMP("ioctl(I_PUSH ip)",err3b);
	if(ioctl(anchor->ifd,IF_UNITSEL,npi)<0)
		JUMP("ioctl(IF_UNITSEL)",err3b);
	if((anchor->mux=ioctl(anchor->pppfd,I_LINK,anchor->tty))<0)
		JUMP("ioctl(I_LINK)",err3b);
#elif defined(LINUX)
	i=N_PPP;
	if((anchor->proxy=ioctl(anchor->tty,TIOCSETD,&i))<0)
		JUMP("ioctl(TIOCSETD)",err3);
#if defined(PPPIOCGCHAN) && defined(PPPIOCATTCHAN) && defined(PPPIOCNEWUNIT) \
	&& defined(PPPIOCCONNECT)
	if((anchor->pppfd=open("/dev/ppp",O_RDWR|O_NONBLOCK))!=-1)
	{
		if((anchor->ifd=open("/dev/ppp",O_RDWR|O_NONBLOCK))==-1)
			JUMP("open(/dev/ppp)",err4);
		if(ioctl(anchor->tty,PPPIOCGCHAN,&i)==-1)
			JUMP("ioctl(PPPIOCGCHAN)",err4);
		if(ioctl(anchor->pppfd,PPPIOCATTCHAN,&i)==-1)
			JUMP("ioctl(PPPIOCATTCHAN)",err4);
		i=-1;
		if(ioctl(anchor->ifd,PPPIOCNEWUNIT,&i)==-1)
			JUMP("ioctl(PPPIOCNEWUNIT)",err4);
		anchor->proxy=i;
		if(ioctl(anchor->pppfd,PPPIOCCONNECT,&i)==-1)
			JUMP("ioctl(PPPIOCCONNECT)",err4);
#ifdef PPPIOCSMRU
		i=anchor->mtu;
		if(ioctl(anchor->pppfd,PPPIOCSMRU,&i)==-1)
			JUMP("ioctl(PPPIOCSMRU)",err4);
#endif
	}
	else anchor->ifd=-1;
#else
	anchor->pppfd=-1;
	anchor->ifd=-1;
#endif
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	i=PPPDISC;
	if(ioctl(anchor->tty,TIOCSETD,&i)<0)
		JUMP("ioctl(TIOCSETD)",err3);
	if(ioctl(anchor->tty,PPPIOCGUNIT,&(anchor->proxy))<0)
		JUMP("ioctl(PPPIOCGUNIT)",err3);
	anchor->pppfd=-1;
	anchor->ifd=-1;
#endif

	/* setup (clear) async maps */

#if defined(SunOS)
	*npi=0;
	str.ic_cmd=PPPIO_XACCM;
	str.ic_timout=0;
	str.ic_len=sizeof(int);
	str.ic_dp=(void *)(npi);
	if(ioctl(anchor->pppfd,I_STR,&str)<0)JUMP("ioctl(PPPIO_XACCM)",err4);
	*npi=0;
	str.ic_cmd=PPPIO_RACCM;
	str.ic_timout=0;
	str.ic_len=sizeof(int);
	str.ic_dp=(void *)(npi);
	if(ioctl(anchor->pppfd,I_STR,&str)<0)JUMP("ioctl(PPPIO_RACCM)",err4);
#else
	j=(anchor->pppfd==-1?anchor->tty:anchor->pppfd);
	i=0;
	if(ioctl(j,PPPIOCSASYNCMAP,&i)<0)
		JUMP("ioctl(PPPIOCSASYNCMAP)",err4);
	i=0;
	if(ioctl(j,PPPIOCSRASYNCMAP,&i)<0)
		JUMP("ioctl(PPPIOCSRASYNCMAP)",err4);
#endif

	/* enable VJ header compression if not disabled */

#if defined(SunOS)
	if(anchor->cslip)
	{
		npc[0]=1;
		npc[1]=15;
		str.ic_cmd=PPPIO_VJINIT;
		str.ic_timout=0;
		str.ic_len=2*sizeof(char);
		str.ic_dp=(void *)(npc);
		if(ioctl(anchor->pppfd,I_STR,&str)<0)
			JUMP("ioctl(PPPIO_VJINIT)",err4);
		npi[0]=npi[1]=COMP_VJC|DECOMP_VJC|COMP_VJCCID|DECOMP_VJCCID;
		str.ic_cmd=PPPIO_CFLAGS;
		str.ic_timout=0;
		str.ic_len=2*sizeof(int);
		str.ic_dp=(void *)(npi);
		if(ioctl(anchor->pppfd,I_STR,&str)<0)
			JUMP("ioctl(PPPIO_CFLAGS)",err4);
	}
#else
	j=(anchor->ifd==-1?anchor->tty:anchor->ifd);
	i=15;
	if(ioctl(j,PPPIOCSMAXCID,&i)<0)
		JUMP("ioctl(PPPIOCSMAXCID)",err4);
	if(ioctl(j,PPPIOCGFLAGS,&i)<0)
		JUMP("ioctl(PPPIOCGFLAGS)",err4);
	if(anchor->cslip)
	{
		i&=~SC_NO_TCP_CCID;
		i|=SC_COMP_TCP;
	}
	else
	{
		i|=SC_NO_TCP_CCID;
		i&=~SC_COMP_TCP;
	}
	if(ioctl(j,PPPIOCSFLAGS,&i)<0)
		JUMP("ioctl(PPPIOCSFLAGS)",err4);
#endif

	/* get control socket, react to errors */
#if !defined(SunOS)
	if((i=socket(AF_INET,SOCK_DGRAM,0))==-1)JUMP("socket",err4);
#else
	if((i=open("/dev/udp",O_RDWR))==-1)JUMP("open /dev/udp",err4);
#endif
	/* reset error flag */

	j=0;

	/* create interface name */

	memset(&ifr,0,sizeof(ifr));
	sprintf(ifr.ifr_name,"ppp%u",anchor->proxy);
#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	memset(&ifa,0,sizeof(ifa));
	sprintf(ifa.ifra_name,"ppp%u",anchor->proxy);
#endif

	/* assert that we have a point-to-point interface */

#if defined(SunOS)
	if(ioctl(i,SIOCGIFFLAGS,&ifr)<0)j++;
	ifr.ifr_flags|=IFF_POINTOPOINT;
	if(!j)if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)j++;
#endif

	/* set up interface address, react to errors */

#if defined(SunOS)
	ifr.ifr_addr.sa_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
	if(!j)if(ioctl(i,SIOCSIFADDR,&ifr)<0)j++;
#elif defined(LINUX)
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
	if(!j)if(ioctl(i,SIOCSIFADDR,&ifr)<0)j++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->local_ip));
#endif

	/* set up interface peer address, react to errors */

#if defined(SunOS)
	ifr.ifr_addr.sa_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
	if(!j)if(ioctl(i,SIOCSIFDSTADDR,&ifr)<0)j++;
#elif defined(LINUX)
	((struct sockaddr_in *)(&(ifr.ifr_dstaddr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_dstaddr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
	if(!j)if(ioctl(i,SIOCSIFDSTADDR,&ifr)<0)j++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_addr.s_addr=
		getaddr((unsigned char *)(anchor->remote_ip));
#endif

	/* set up interface netmask, react to errors */

#if defined(SunOS)
	ifr.ifr_addr.sa_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCSIFNETMASK,&ifr)<0)j++;
#elif defined(LINUX)
	((struct sockaddr_in *)(&(ifr.ifr_netmask)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(ifr.ifr_netmask)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCSIFNETMASK,&ifr)<0)j++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_len=
		sizeof(struct sockaddr_in);
	((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_addr.s_addr=
		htonl(0xffffffff);
	if(!j)if(ioctl(i,SIOCAIFADDR,&ifa)<0)j++;
#endif

	/* set up interface mtu, react to errors */

#if defined(SunOS)
	ifr.ifr_metric=anchor->mtu;
	if(!j)if(ioctl(i,SIOCSIFMTU,&ifr)<0)j++;
#else
	ifr.ifr_mtu=anchor->mtu;
	if(!j)if(ioctl(i,SIOCSIFMTU,&ifr)<0)j++;
#endif

	/* activate interface, react to errors */

#if defined(SunOS)
	ifr.ifr_flags|=IFF_UP;
#elif defined(LINUX)
	ifr.ifr_flags=IFF_UP|IFF_RUNNING|IFF_POINTOPOINT|IFF_NOARP|
		IFF_MULTICAST;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	ifr.ifr_flags=IFF_UP|IFF_POINTOPOINT|IFF_NOARP|IFF_MULTICAST;
#endif
	if(!j)if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)j++;

	/* close control socket */
 
	if(close(i))ERRNO("close");

	/* signal any errors encountered */

#if defined(LINUX) || defined(SunOS)
	if(j)JUMP("ioctl(SIOCSIFADDR,SIOCSIFDSTADDR,SIOCSIFNETMASK," \
		"SIOCSIFMTU,SIOCSIFFLAGS)",err4);
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	if(j)JUMP("ioctl(SIOCAIFADDR,SIOCSIFMTU,SIOCSIFFLAGS)",err4);
#endif

	/* enable IP traffic over the interface */

#if defined(SunOS)
	npi[0]=PPP_IP;
	npi[1]=NPMODE_PASS;
	str.ic_cmd=PPPIO_NPMODE;
	str.ic_timout=0;
	str.ic_len=2*sizeof(int);
	str.ic_dp=(void *)(&npi);
	if(ioctl(anchor->pppfd,I_STR,&str)<0)JUMP("ioctl(PPPIO_NPMODE)",err4);
#else
	j=(anchor->ifd==-1?anchor->tty:anchor->ifd);
	npi[0]=PPP_IP;
	npi[1]=NPMODE_PASS;
	if(ioctl(j,PPPIOCSNPMODE,&npi)<0)
		JUMP("ioctl(PPPIOCSNPMODE)",err4);
#endif

	/* set up route to partner, react to errors */

#ifdef LINUX
	if(!anchor->autoroute)
	{
		if(route(1,getaddr((unsigned char *)(anchor->remote_ip)),-1,0,
			(unsigned char *)(ifr.ifr_name)))
				GOTO("autoroute",err5);
	}
#endif

	/* set up priority route to peer, react to errors */

	if(anchor->peerroute)
		if(route(1,anchor->peerip,-1,
			anchor->peerroute>1?anchor->peergate:0,
			(unsigned char *)(anchor->peerdev)))
				GOTO("peerroute",err6);

	/* set up all additional routes */

	for(k=0;k<anchor->rmax;k++)
		if(route(1,anchor->rtab[k][0],anchor->rtab[k][1],
			anchor->rtab[k][2],NULL))
		GOTO("routex",err7);

	/* in case of keepalive packages when link idle open icmp socket */

	if(anchor->keepalive)if(icmphandler(anchor))GOTO("icmphandler",err8);

	/* mark ppp link as established */

	anchor->slip=1;

	/* execute ppp up process, if there is any */

	doexec(anchor->slipup,ifr.ifr_name);

	/* signal success */

	goto err0;

	/* done, if no ppp link */

delete:	if(!anchor->slip)GOTO("ppp link not up",err0);

	/* create interface name */

	memset(&ifr,0,sizeof(ifr));
	sprintf(ifr.ifr_name,"ppp%u",anchor->proxy);
#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	memset(&ifa,0,sizeof(ifa));
	sprintf(ifa.ifra_name,"ppp%u",anchor->proxy);
#endif

	/* execute ppp down process, if there is any */

	doexec(anchor->slipdown,ifr.ifr_name);

	/* enable verbose mode */

	quiet=0;

	/* in case of keepalive packages when link idle close icmp socket */

	if(anchor->keepalive)icmphandler(anchor);

	/* error handling */

err8:	if(quiet)err++;
err7:	if(quiet)err++;

	/* delete additional routes */

	for(k=anchor->rmax-1;k>=0;k--)
		if(route(0,anchor->rtab[k][0],anchor->rtab[k][1],
			anchor->rtab[k][2],NULL))if(!quiet++)DEBUG1(3,"routex");

	/* delete priority route to peer */

	if(anchor->peerroute)
		if(route(0,anchor->peerip,-1,
			anchor->peerroute>1?anchor->peergate:0,
			(unsigned char *)(anchor->peerdev)))
				if(!quiet++)DEBUG1(3,"peerroute");

err6:	if(quiet)err++;

	/* delete route to partner, react to errors */

#ifdef LINUX
	if(!anchor->autoroute)
	{
		if(route(0,getaddr((unsigned char *)(anchor->remote_ip)),-1,0,
			(unsigned char *)(ifr.ifr_name)))
				if(!quiet++)DEBUG1(3,"autoroute");
	}

err5:	if(quiet)err++;
#elif defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
	if(quiet)err++;
#endif

	/* disable IP traffic over the interface */

#if defined(SunOS)
	npi[0]=PPP_IP;
	npi[1]=NPMODE_DROP;
	str.ic_cmd=PPPIO_NPMODE;
	str.ic_timout=0;
	str.ic_len=2*sizeof(int);
	str.ic_dp=(void *)(&npi);
	if(ioctl(anchor->pppfd,I_STR,&str)<0)
	{
		if(!quiet++)ERRNO("ioctl(PPPIO_NPMODE)");
	}
#else
	j=(anchor->ifd==-1?anchor->tty:anchor->ifd);
	npi[0]=PPP_IP;
	npi[1]=NPMODE_DROP;
	if(ioctl(j,PPPIOCSNPMODE,&npi)<0)
	{
		if(!quiet++)ERRNO("ioctl(PPPIOCSNPMODE)");
	}
#endif
	/* get control socket, react to errors */

#if defined(SunOS)
	if((i=open("/dev/udp",O_RDWR))==-1)
#else
	if((i=socket(AF_INET,SOCK_DGRAM,0))==-1)
#endif
	{
		if(!quiet++)ERRNO("socket");
	}

	/* if control socket could be created */

	else
	{
		/* deactivate interface, react to errors */

#if defined(FreeBSD) || defined(OSNetBSD) || defined(OSOpenBSD)
		((struct sockaddr_in *)(&(ifa.ifra_addr)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_addr)))
			->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifa.ifra_addr)))
			->sin_addr.s_addr=getaddr(anchor->local_ip);
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))
			->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifa.ifra_broadaddr)))
			->sin_addr.s_addr=getaddr(anchor->remote_ip);
		((struct sockaddr_in *)(&(ifa.ifra_mask)))->sin_len=
			sizeof(struct sockaddr_in);
		((struct sockaddr_in *)(&(ifa.ifra_mask)))
			->sin_addr.s_addr=htonl(0xffffffff);
		if(ioctl(i,SIOCDIFADDR,&ifa)<0)
			ERRNO("ioctl(SIOCDIFADDR)");
#endif

		ifr.ifr_flags=0;
		if(ioctl(i,SIOCSIFFLAGS,&ifr)<0)if(!quiet++)
			ERRNO("ioctl(SIOCSIFFLAGS)");

		/* reset interface address, react to errors */

#if defined(LINUX)
		((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_family=AF_INET;
		((struct sockaddr_in *)(&(ifr.ifr_addr)))->sin_addr.s_addr=0;
		if(ioctl(i,SIOCSIFADDR,&ifr)<0)if(!quiet++)
			ERRNO("ioctl(SIOCSIFADDR)");
#endif

		/* close control socket */
 
		if(close(i))if(!quiet++)ERRNO("close");
	}

err4:	if(quiet)err++;
#if defined(SunOS)
	if(ioctl(anchor->pppfd,I_UNLINK,anchor->mux)<0)
		if(!quiet++)ERRNO("ioctl(I_UNLINK)");
#endif
#if defined(LINUX)
	if(anchor->ifd!=-1)if(close(anchor->ifd))
		if(!quiet++)ERRNO("ifd close");
	if(anchor->pppfd!=-1)if(close(anchor->pppfd))
		if(!quiet++)ERRNO("pppfd close");
#endif
	/* restore original line discipline, react to errors */
#if !defined(SunOS)
	if(ioctl(anchor->tty,TIOCSETD,&anchor->disc)<0)if(!quiet++)
		ERRNO("ioctl(TIOCSETD");
#else
err3b:	if(close(anchor->ifd))if(!quiet++)ERRNO("ifd close");
err3a:	if(close(anchor->pppfd))if(!quiet++)ERRNO("pppfd close");
#endif

err3:	if(quiet)err++;

	/* restore old device characteristics, react to errors */
#if !defined(SunOS)
	if(!(smode&1))if(tcsetattr(anchor->tty,TCSAFLUSH,&(anchor->original))<0)
		if(!quiet++)ERRNO("tcsetattr");

err2:	if(quiet)err++;
#else
	if(quiet)err++;
#endif

	/* close tty and pty */

	if(!(smode&1))
	{
		if(close(anchor->tty))if(!quiet++)ERRNO("tty close");
		if(close(anchor->pty))if(!quiet++)ERRNO("pty close");
	}

err1:	if(quiet)err++;

	/*  mark ppp link as down */

	anchor->slip=0;

	/* debug message */

err0:	LEAVE("ppphandler");

	/* signal success or error */

	return err;
}
