/*
 * Copyright (C) 2000 Avaya Labs, Avaya Inc.
 * Copyright (C) 1999 Bell Labs, Lucent Technologies.
 * Copyright (C) Arash Baratloo, Timothy Tsai, and Navjot Singh.
 *
 * This file is part of the Libsafe library.
 * Libsafe version 2.x: protecting against stack smashing attacks.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, 
 *   visit http://www.research.avayalabs.com/project/libsafe/index.html
 *   or email libsafe@research.avayalabs.com
 */

#include <unistd.h>		/* defines readlink() */
#include <syslog.h>		/* defines syslog(3) */
#include <stdarg.h>		/* defines va_args */
#include <signal.h>		/* defines kill() */
#include <stdio.h>
#include <unistd.h>             /* defines pipe() */
#include <sys/resource.h>	/* defines RLIM_INFINITY */
#include <errno.h>		/* to get extern int errno */
#include "util.h"
#include "log.h"


// Sometimes environ is defined in unistd.h.  If so, then comment out the
// following declaration.
extern char **environ[];

static void *stack_start = NULL;

// If we can't find stack_start after calling find_stack_start() once, then we
// don't need to try it again.  So, we set cant_find_stack=1 to tell
// _libsafe_stackVariableP() that it doesn't need to call find_stack_start() a
// second time.
static int cant_find_stack = 0;

static char *get_exename(char *exename, int size) {
    int res;
    
    /*
     * get the name of the current executable
     */
    if ((res = readlink("/proc/self/exe", exename, size - 1)) == -1)
	exename[0] = '\0';
    else
	exename[res] = '\0';
    return (exename);
}

/*
 * Return the address corresponding to the start of the stack.  We assume that
 * the location corresponding to the start of the stack contains NULL.  We need
 * to do the conforming stack check to make sure that we are actually finding
 * the true start of the stack.  If there are no frame pointers on the stack,
 * then set stack_start=NULL and cant_find_stack=1.
 */
static void *find_stack_start() {
    /*
     * nextfp is used in the check for -fomit-frame-pointer code.
     */
    char *nextfp;

    char *fp;

    for (fp=__builtin_frame_address(0); fp && fp<(char*)0xbffffffb; fp=nextfp) {
	nextfp = *(void **) fp;

	if (nextfp == NULL) {
	    /*
	     * This is the only correct way to end the stack.
	     */
	    return fp;
	}

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    cant_find_stack = 1;
	    return NULL;
	}

	/*
	 * Make sure frame pointers are monotonically increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    cant_find_stack = 1;
	    return NULL;
	}
    }

    /*
     * If we get to this point, it means that we haven't found a NULL frame
     * pointer to indicate the start of the stack.  So, we'll assume that
     * we aren't able to find the start of the stack, and therefore, all
     * libsafe checking will be disabled.
     */
    cant_find_stack = 1;
    return NULL;
}

/*
 * Has _libsafe_die() been called?
 */
static int dying = 0;

/* Given an address 'addr' returns 0 iff the address does not point to a stack
 * variable.  Otherwise, it returns a positive number indicating the number of
 * bytes (distance) between the 'addr' and the frame pointer it resides in.
 * Note: stack grows down, and arrays/structures grow up.
 */
uint _libsafe_stackVariableP(void *addr) {
    /*
     * bufsize is the distance between addr and the end of the stack frame.
     * It's what _libsafe_stackVariableP() is trying to calculate.
     */
    uint bufsize = 0;

    /*
     * (Vandoorselaere Yoann)
     * We have now just one cast.
     */
    void *fp, *sp;
    
    /*
     * nextfp is used in the check for -fomit-frame-pointer code.
     */
    void *nextfp;

    /*
     * If _libsafe_die() has been called, then we don't need to do anymore
     * libsafe checking.
     */
    if (dying)
	return 0;

    /*
     * cant_find_stack==1 means that we tried to find the start of the stack
     * via find_stack_start() and failed.
     */
    if (cant_find_stack)
	return 0;

    /*
     * (Arash Baratloo / Yoann Vandoorselaere)
     * use the stack address of the first declared variable to get the 'sp'
     * address in a portable way.
     */
    sp = &fp;

    /*
     * Stack grows downwards (toward 0x00).  Thus, if the stack pointer is
     * above (>) 'addr', 'addr' can't be on the stack.
     */
    if (sp > addr)
	return 0;

    /*
     * Note: the program name is always stored at 0xbffffffb (documented in the
     * book Linux Kernel).  Search back through the frames to find the frame
     * containing 'addr'.
     */
    fp = __builtin_frame_address(0);

    if (stack_start == NULL)
	stack_start = find_stack_start();

    while ((sp < fp) && (fp <= stack_start)) {
	if (fp > addr) {
	    /*
	     * found the frame -- now check the rest of the stack
	     */
	    bufsize = fp - addr;
	    break;
	}

	nextfp = *(void **) fp;

	/*
	 * The following checks are meant to detect code that doesn't insert
	 * frame pointers onto the stack.  (i.e., code that is compiled with
	 * -fomit-frame-pointer).
	 */

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    return 0;
	}

	/*
	 * Make sure frame pointers are monotonically increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    return 0;
	}

	fp = nextfp;
    }

    /*
     * If we haven't found the correct frame by now, it either means that addr
     * isn't on the stack or that the stack doesn't contain frame pointers.
     * Either way, we will return 0 to bypass checks for addr.
     */
    if (bufsize == 0) {
	return 0;
    }

    /*
     * Now check to make sure that the rest of the stack looks reasonable.
     */
    while ((sp < fp) && (fp <= stack_start)) {
	nextfp = *(void **) fp;

	if (nextfp == NULL) {
	    /*
	     * This is the only correct way to end the stack.
	     */
	    return bufsize;
	}

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    return 0;
	}

	/*
	 * Make sure frame pointers are monotonically * increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    return 0;
	}

	fp = nextfp;
    }

    /*
     * We weren't able to say for sure that the stack contains valid frame
     * pointers, so we will return 0, which means that no check for addr will
     * be done.
     */
    return 0;
}

/*
 * Given an address 'addr' returns 1 iff the address points to a return address
 * or a frame pointer on the stack.  Otherwise, it returns 0.  Note: stack
 * grows down, and arrays/structures grow up.
 */
uint _libsafe_raVariableP(void *addr) {
    /*
     * Does addr point to a return address or a frame pointer on the stack?
     */
    int is_ra = 0;

    /*
     * (Vandoorselaere Yoann)
     * We have now just one cast.
     */
    void *fp, *sp;
    
    /*
     * nextfp is used in the check for -fomit-frame-pointer code.
     */
    void *nextfp;

    /*
     * If _libsafe_die() has been called, then we don't need to do anymore
     * libsafe checking.
     */
    if (dying)
	return 0;

    /*
     * cant_find_stack==1 means that we tried to find the start of the stack
     * via find_stack_start() and failed.
     */
    if (cant_find_stack)
	return 0;

    /*
     * (Arash Baratloo / Yoann Vandoorselaere)
     * use the stack address of the first declared variable to get the 'sp'
     * address in a portable way.
     */
    sp = &fp;

    /*
     * Stack grows downwards (toward 0x00).  Thus, if the stack pointer is
     * above (>) 'addr', 'addr' can't be on the stack.
     */
    if (sp > addr)
	return 0;

    /*
     * Note: the program name is always stored at 0xbffffffb (documented in the
     * book Linux Kernel).  Search back through the frames to find the frame
     * containing 'addr'.
     */
    fp = __builtin_frame_address(0);

    if (stack_start == NULL)
	stack_start = find_stack_start();

    while ((sp < fp) && (fp <= stack_start)) {
	if (fp == addr ||	    /* addr points to a frame pointer */
	    fp + 4 == addr)	    /* addr points to a return address */
	{
	    is_ra = 1;
	    break;
	}

	nextfp = *(void **) fp;

	/*
	 * The following checks are meant to detect code that doesn't insert
	 * frame pointers onto the stack.  (i.e., code that is compiled with
	 * -fomit-frame-pointer).
	 */

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    return 0;
	}

	/*
	 * Make sure frame pointers are monotonically increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    return 0;
	}

	fp = nextfp;
    }

    /*
     * If we haven't found the correct frame by now, it either means that addr
     * isn't on the stack or that the stack doesn't contain frame pointers.
     * Either way, we will return 0 to bypass checks for addr.
     */
    if (is_ra == 0) {
	return 0;
    }

    /*
     * Now check to make sure that the rest of the stack looks reasonable.
     */
    while ((sp < fp) && (fp <= stack_start)) {
	nextfp = *(void **) fp;

	if (nextfp == NULL) {
	    /*
	     * This is the only correct way to end the stack.
	     */
	    return is_ra;
	}

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    return 0;
	}

	/*
	 * Make sure frame pointers are monotonically * increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    return 0;
	}

	fp = nextfp;
    }

    /*
     * We weren't able to say for sure that the stack contains valid frame
     * pointers, so we will return 0.
     */
    return 0;
}

/*
 * If start_addr and end_addr are both on the stack and do not reside in the
 * same stack frame, return 1.  If both start_addr and end_addr reside in the
 * same stack frame or either is not on the stack, return 0.
 */
uint _libsafe_span_stack_frames(void *start_addr, void *end_addr) {
    /*
     * start_fp and end_fp point to the start of the stack frames where
     * start_addr and end_addr reside.  NULL means that we haven't found the
     * stack frame yet.
     */
    void *start_fp=NULL;

    /*
     * This is the return value.
     */
    int spans_stack_frames = 0;

    /*
     * (Vandoorselaere Yoann)
     * We have now just one cast.
     */
    void *fp, *sp;
    
    /*
     * nextfp is used in the check for -fomit-frame-pointer code.
     */
    void *nextfp;

    /*
     * If _libsafe_die() has been called, then we don't need to do anymore
     * libsafe checking.
     */
    if (dying)
	return 0;

    /*
     * cant_find_stack==1 means that we tried to find the start of the stack
     * via find_stack_start() and failed.
     */
    if (cant_find_stack)
	return 0;

    /*
     * Make sure that end_addr is greater than start.  If start_addr==end_addr,
     * then obviously there is no spanning of stack frames.
     */
    if (start_addr >= end_addr)
	return 0;

    /*
     * (Arash Baratloo / Yoann Vandoorselaere)
     * use the stack address of the first declared variable to get the 'sp'
     * address in a portable way.
     */
    sp = &fp;

    /*
     * Stack grows downwards (toward 0x00).  Thus, if the stack pointer is
     * above (>) 'addr', 'addr' can't be on the stack.
     */
    if (sp > start_addr)
	return 0;

    /*
     * Note: the program name is always stored at 0xbffffffb (documented in the
     * book Linux Kernel).  Search back through the frames to find the frame
     * containing 'addr'.
     */
    fp = __builtin_frame_address(0);

    if (stack_start == NULL)
	stack_start = find_stack_start();

    while ((sp < fp) && (fp <= stack_start)) {
	if (fp > start_addr) {
	    /*
	     * found the frame for start_addr -- now check the rest of the
	     * stack
	     */
	    start_fp = fp;
	    break;
	}

	nextfp = *(void **) fp;

	/*
	 * The following checks are meant to detect code that doesn't insert
	 * frame pointers onto the stack.  (i.e., code that is compiled with
	 * -fomit-frame-pointer).
	 */

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    return 0;
	}

	/*
	 * Make sure frame pointers are monotonically increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    return 0;
	}

	fp = nextfp;
    }

    if (start_fp == NULL) {
	/*
	 * start_addr is not on the stack (ie, past the stack).
	 */
	return 0;
    }

    if (end_addr < start_fp) {
	/*
	 * start_addr and end_addr are in the same stack frame.  We still need
	 * to check that the stack is conforming, so don't return yet.
	 */
	spans_stack_frames = 0;
    }
    else {
	/*
	 * If end_addr is past start_fp, then it must be in another stack
	 * frame.  We still need to check that the stack is conforming, so
	 * don't return yet.
	 *
	 * Note that we assume that if start_addr is on the stack, then if
	 * end_addr is past the past, then that means that start_addr and
	 * end_addr span multiple stack frame.
	 */
	spans_stack_frames = 1;

	LOG(2, "addr=(%p,%p) fp=%p stack_start=%p\n", start_addr, end_addr,
		start_fp, stack_start);
    }

    /*
     * Now check to make sure that the rest of the stack looks reasonable.
     */
    while ((sp < fp) && (fp <= stack_start)) {
	nextfp = *(void **) fp;

	if (nextfp == NULL) {
	    /*
	     * This is the only correct way to end the stack.
	     */
	    return spans_stack_frames;
	}

	/*
	 * Make sure frame pointers are word aligned.
	 */
	if ((uint)nextfp & 0x03) {
	    LOG(2, "fp not word aligned; bypass enabled\n");
	    return 0;
	}

	/*
	 * Make sure frame pointers are monotonically * increasing.
	 */
	if (nextfp <= fp) {
	    LOG(2, "fp not monotonically increasing; bypass enabled\n");
	    return 0;
	}

	fp = nextfp;
    }

    /*
     * We weren't able to say for sure that the stack contains valid frame
     * pointers, so we will return 0.
     */
    return 0;
}

#ifdef DUMP_STACK
/* Create a new filename consisting for LIBSAFE_DUMP_STACK_FILE followed by the
 * PID.
 */
void create_dump_stack_filename(char *filename, int size) {
    char    *p, *p2;
    char    buf[10];	// To hold the PID, but with digits reversed.  We
			//	assume that the PID is at most 10 digits.
    int	    pid;
    int	    count=0;	// How many chars we already put into filename

    // strcpy(filename, LIBSAFE_DUMP_STACK_FILE);
    // NOTE:  We can't use strcpy or sprintf, since they will be intercepted by
    // libsafe and thus cause infinite recursion.
    for (p=LIBSAFE_DUMP_STACK_FILE,p2=filename; *p && count<size-1; p++) {
	*p2++ = *p;
	count++;
    }

    // strcat(filename, <getpid()>)
    // NOTE:  We can't use strcpy or sprintf, since they will be intercepted by
    // libsafe and thus cause infinite recursion.
    pid = getpid();
    for (p=buf; pid>0; p++) {
	*p = '0' + (pid % 10);
	pid /= 10;
    }
    for (p--; p>=buf && count<size-1; p--) {
	*p2++ = *p;
	count++;
    }

    *p2 = (char) NULL;
}

/* Print the stack contents to stderr.  Use the same approximations for sp (the
 * top of the stack) that _libsafe_stackVariableP() uses.
 */
void _libsafe_dump_stack(char *file, int linenum) {
    char    *sp;
    FILE    *fp;
    caddr_t current_stack_start;
    char    exename[MAXPATHLEN];

    /*
     * We will dump the stack contents into a file named
     * LIBSAFE_DUMP_STACK_FILE plus the PID of this process.  By tacking the
     * PID onto the filename, we allow a multi-threaded process to create stack
     * dump files that don't overwrite each other.
     */
    int	    filename_size = strlen(LIBSAFE_DUMP_STACK_FILE) + 6;
    char    *filename = alloca(filename_size);
    create_dump_stack_filename(filename, filename_size);
    
    /*
     * (Arash Baratloo / Yoann Vandoorselaere)
     * use the stack address of the first declared variable to get the 'sp'
     * address in a portable way.
     */
    sp = (char*)&sp;

    if ((fp=fopen(filename, "w")) == NULL) {
	/* If we can't open the dump file, then just dump to stderr. */
	fp = stderr;
	LOG(1, "Dumping stack to stderr.\n");
    }
    else {
	LOG(1, "Dumping stack to %s.\n", filename);
    }

    fprintf(fp, "Stack dump:  sp=%p  fp=%p  stack_start=%p\n",
	sp, __builtin_frame_address(0), stack_start);
    fprintf(fp, "Initiating dump from file=%s linenum=%d\n", file, linenum);

    /*
     * get the name of the current executable
     */
    get_exename(exename, MAXPATHLEN);
    fprintf(fp, "Libsafe monitoring %s\n", exename);

    /*
     * This makes the fprintf below line up better.
     */
    sp = (char*)((uint)sp & (~0x7));

    /*
     * Try to find the start of the current stack (in case we are dealing with
     * multiple threads).  If we can determine the start of the stack, then use
     * that value.  Otherwise, use stack_start, which we've already determined.
     */
    current_stack_start = find_stack_start();
    if (cant_find_stack)
	current_stack_start = stack_start;

    /*
     * Print out the contents of the stack in hex, until 0x40 bytes past the
     * start of the stack.  Make sure we don't go past 0xbffffffb in any case
     * to avoid segfaults.
     */
    for (; sp<=(char*)current_stack_start+0x40 && sp<(char*)0xbffffffb; sp+=8) {
	fprintf(fp, "%p:  %02x %02x %02x %02x %02x %02x %02x %02x\n",
	    sp,
	    *sp & 0xff,
	    *(sp+1) & 0xff,
	    *(sp+2) & 0xff,
	    *(sp+3) & 0xff,
	    *(sp+4) & 0xff,
	    *(sp+5) & 0xff,
	    *(sp+6) & 0xff,
	    *(sp+7) & 0xff);
    }

    if (fp != stderr)
	fclose(fp);
}
#endif /* DUMP_STACK */

#ifdef NOTIFY_WITH_EMAIL

#include <netinet/in.h>
#include <arpa/inet.h>
#define __USE_GNU 1		/* defines strnlen() */
#include <ctype.h>
#include <string.h>
#include <netdb.h>
#include <time.h>

/*
* Read a single line of data (until the first newline) from the socket, and
* throw away the data.  The loglevel specifies what to do with the data read.
* <=0 means to throw the data away. >1 means to pass the data to LOG(loglevel,
* ...).
 */
static void clear_one_line(int s, int loglevel) {
    int	    res;
    char    throw_away_buf[500];

    while ((res=recv(s, throw_away_buf, sizeof(throw_away_buf)-1, 0)) > 0) {
	throw_away_buf[res] = (char)NULL;
	if (loglevel > 0)
	    LOG(loglevel, "%s\n", throw_away_buf);
	if (throw_away_buf[res-1] == '\n')
	    break;
    }
}

/*
 * Send the command to the mail server.  If the expected response is received,
 * return 0;  else return -1.  expected should be the first digit of the
 * 3-digit response code.  If expected==-1, then don't check the response.
 */
static int send_command(int s, int expected, char *format, ...) {
    char	response;
    char	command[2048];
    int	len, res;

    va_list args;

    /*
     * Form the command to send
     */
    va_start(args, format);
    len = vsnprintf(command, sizeof(command), format, args);
    va_end(args);

    /*
     * Send the command
     */
    res = send(s, command, len, 0);
    if (res == -1) {
	perror("libsafe:send_command:send()");
	return -1;
    }

    /*
     * Read the response from the mail server.  Make sure that the expected
     * response is received.  We only check the first digit from the 3-digit
     * response code.
     */
    if (expected >= 0) {
	if ((res=recv(s, &response, 1, 0)) > 0) {
	    if ((response - '0') != expected) {
		/*
		 * If we didn't get the expected response code, then read the
		 * full response code so we can print it out.
		 */
		char    full_response[4];

		full_response[0] = response;
		recv(s, &full_response[1], 2, 0);
		full_response[3] = (char)NULL;
		LOG(1, "Sendmail error: received %s, expected %dxx: ",
		    full_response, expected);
		syslog(LOG_CRIT, "Sendmail error: received %s, expected %dxx: ",
			full_response, expected);
		clear_one_line(s, 1);
		return -1;
	    }
	}

	/*
	 * We don't care about the rest of the response, so just read it and
	 * ignore it.
	 */
	clear_one_line(s, 0);
    }

    return 0;
}

/*
 * Send email to the recipient, with the message as part of the subject.
 */
#define MAIL_PORT 25
static void sendmail(char *recipient, char *message) {
    struct sockaddr_in	addr;
    struct hostent	*hostinfo;
    int			s;
    char		hostname[100];
    time_t		t;

    /*
     * Get the name of the local machine.
     */
    if (gethostname(hostname, sizeof(hostname))) {
	strncpy(hostname, "localhost", sizeof(hostname));
    }

    /*
     * Find the current time.  This will be used as the send time for the
     * email.
     */
    time(&t);

    if ((s = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
	perror("libsafe:sendmail:socket()");
	return;
    }

    if ((hostinfo = gethostbyname("localhost")) == NULL) {
	syslog(LOG_CRIT, "%s,%d: gethostbyname: %s\n", __FILE__, __LINE__,
		hstrerror(h_errno));
    return;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sin_addr.s_addr = ((struct in_addr *)(hostinfo->h_addr))->s_addr;
    addr.sin_port        = htons(MAIL_PORT);
    addr.sin_family      = AF_INET;
    if (connect(s, (struct sockaddr *)&addr, sizeof(addr)) == -1) {
	perror("libsafe:sendmail:connect()");
	return;
    }

    /*
     * Read the response (response code 220) that is sent from simply opening
     * the connection.
     */
    clear_one_line(s, 0);
    
    /*
     * Send the commands to the sendmail port.
     */
    if (send_command(s, 2, "HELO %s\r\n", hostname)) return;
    if (send_command(s, 2, "MAIL FROM:<libsafe@%s>\r\n", hostname)) return;
    if (send_command(s, 2, "RCPT TO:<%s>\r\n", recipient)) return;
    if (send_command(s, 3, "DATA\r\n")) return;
    if (send_command(s, -1,
	"Subject: ***** libsafe violation detected *****\r\n"
	"To: %s\r\n"
	"Date: %s\r\n"
	"Libsafe violation detected on %s at %s\r\n"
	"%s\r\n",
	recipient, ctime(&t), hostname, ctime(&t), message))
	    return;
    if (send_command(s, 2, "\r\n.\r\n")) return;
    if (send_command(s, -1, "QUIT\r\n")) return;

    LOG(1, "Sent email to %s\n", recipient);
    syslog(LOG_CRIT, "Sent email to %s\n", recipient);
}
#endif	/* NOTIFY_WITH_EMAIL */

/*
 * This is what is called when a violation is detected.  If you want to add
 * customized actions triggered by detection put them here.  (format,...) is
 * similar to printf() and passed to syslog().
 */
void _libsafe_warn(char *format, ...)
{
    char    exename[MAXPATHLEN];
    va_list args;

    dying = 1;

    /*
     * get the name of the current executable
     */
    get_exename(exename, MAXPATHLEN);
    
    va_start(args, format);

    /*
     * add an entry to syslog()
     */
#ifdef DEBUG_TURN_OFF_SYSLOG
    LOG(1, "Turned off syslog entries for debugging.\n");
#else
    openlog(LIBNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);
    syslog(LOG_CRIT, "%s", VERSION);
    syslog(LOG_CRIT, "Detected an attempt to write across stack boundary.");
    syslog(LOG_CRIT, "Terminating %s.", exename);
    syslog(LOG_CRIT, "    uid=%d  euid=%d  pid=%d", getuid(), geteuid(),
	    getpid());
    {
	/*
	 * Print out the call stack.  We can assume that the stack is a normal
	 * stack, since _libsafe_stackVariableP(), _libsafe_raVariableP(), or
	 * _libsafe_span_stack_frames() had to be called first.
	 */
	caddr_t	fp, ra;
	syslog(LOG_CRIT, "Call stack:\n");
	for (fp=__builtin_frame_address(0); *fp; fp=*(void **)fp) {
	    ra = *((caddr_t*)(fp+4));
	    syslog(LOG_CRIT, "    %p\n", (caddr_t)((uint)ra-5));
	}
    }
    syslog(LOG_CRIT, format, args);
//    closelog();
#endif

    LOG(1, "Detected an attempt to write across stack boundary.\n");
    LOG(1, "Terminating %s.\n", exename);
    LOG(1, "    uid=%d  euid=%d  pid=%d\n", getuid(), geteuid(), getpid());
    {
	/*
	 * Print out the call stack.  We can assume that the stack is a normal
	 * stack, since _libsafe_stackVariableP(), _libsafe_raVariableP(), or
	 * _libsafe_span_stack_frames() had to be called first.
	 */
	caddr_t	fp, ra;
	LOG(1, "Call stack:\n");
	for (fp=__builtin_frame_address(0); *fp; fp=*(void **)fp) {
	    ra = *((caddr_t*)(fp+4));
	    LOG(1, "    %p\n", (caddr_t)((uint)ra-5));
	}
    }
    if (1 <= LOG_LEVEL) {
	vfprintf(stderr, format, args);
	fprintf(stderr, "\n");
    }

    va_end(args);

    /*
     * PUT ANY CUSTOMIZED ACTIONS HERE...
     */

#ifdef DUMP_STACK
    /* Print the contents of the stack */
    _libsafe_dump_stack(__FILE__, __LINE__);
#endif

#ifdef NOTIFY_WITH_EMAIL
    {
	char errmsg[1000];
	char buf[1000];
	char recipient[500];
	FILE *fp;

	/*
	 * Form the descriptive message.
	 */
	snprintf(errmsg, sizeof(errmsg),
	    "Libsafe %s\r\n"
	    "Detected an attempt to write across stack boundary.\r\n"
	    "Terminating %s.\r\n"
	    "    uid=%d  euid=%d  pid=%d\r\n",
	    VERSION,
	    exename,
	    getuid(), geteuid(), getpid());
	{
	    /*
	     * Print out the call stack.  We can assume that the stack is a
	     * normal stack, since _libsafe_stackVariableP(),
	     * _libsafe_raVariableP(), or _libsafe_span_stack_frames() had to
	     * be called first.
	     */
	    caddr_t	fp, ra;
	    snprintf(buf, sizeof(buf), "Call stack:\r\n");
	    strncat(errmsg, buf,
		sizeof(errmsg) - strnlen(errmsg,sizeof(errmsg)) - 1);
	    for (fp=__builtin_frame_address(0); *fp; fp=*(void **)fp) {
		ra = *((caddr_t*)(fp+4));
		snprintf(buf, sizeof(buf), "    %p\r\n", (caddr_t)((uint)ra-5));
		strncat(errmsg, buf,
		    sizeof(errmsg) - strnlen(errmsg,sizeof(errmsg)) - 1);
	    }
	}
#ifdef DUMP_STACK
	{
	    /*
	     * We will dump the stack contents into a file named
	     * LIBSAFE_DUMP_STACK_FILE plus the PID of this process.  By
	     * tacking the PID onto the filename, we allow a multi-threaded
	     * process to create stack dump files that don't overwrite each
	     * other.
	     */
	    int	    filename_size = strlen(LIBSAFE_DUMP_STACK_FILE) + 6;
	    char    *filename = alloca(filename_size);
	    create_dump_stack_filename(filename, filename_size);
	    snprintf(buf, sizeof(buf), "Dumped stack to %s.\r\n", filename);
	    strncat(errmsg, buf,
		sizeof(errmsg) - strnlen(errmsg,sizeof(errmsg)) - 1);
	}
#endif
    
	va_start(args, format);
	vsnprintf(buf, sizeof(buf), format, args);
	va_end(args);
	strncat(errmsg, buf,
	    sizeof(errmsg) - strnlen(errmsg,sizeof(errmsg)) - 1);

	/*
	 * If the mail_list file exists, then send email to all the recipients
	 * listed in that file.  Otherwise, send email to root@localhost.
	 */
	if ((fp = fopen("/etc/libsafe.notify", "r")) == NULL) {
	    sendmail("root@localhost", errmsg);
	} else {
	    while (fgets(recipient, sizeof(recipient), fp)) {
		char *p;

		/*
		 * Chop off any trailing newlines if present
		 */
		for (p=recipient + strnlen(recipient, sizeof(recipient)) - 1;
		     isspace(*p);
		     p--)
		{
		    *p = (char) NULL;
		}
		
		sendmail(recipient, errmsg);
	    }
	    fclose(fp);
	}
    }
#endif	/* NOTIFY_WITH_EMAIL */

    /*
     * Since we are justing doing a warning, set dying=0 to indicate that we
     * should resume libsafe checking now.
     */
    dying = 0;
}


/*
 * This is what is called when a buffer overflow on the stack is detected.  If
 * you want to add customized actions triggered by detection put them here.
 * 'name' is the name of this library, and (format,...) is similar to printf()
 * and passed to syslog().
 */
void _libsafe_die(char *format, ...)
{
    va_list args;

    dying = 1;

    va_start(args, format);
    _libsafe_warn(format, args);
    va_end(args);

#ifdef DUMP_CORE
    /*
     * Kill this process, but generate a core dump in the /tmp directory.  If
     * there is no /tmp directory, the core dump is placed in the current
     * working directory, wherever that is.
     *
     * signal() is needed to disabled any registered handlers for SIGABRT,
     * which is used by abort().  Doing a chdir("/tmp") makes it easier to find
     * where the core dump file is.  However, if /tmp/core already exists and
     * is owned by another user, then no core dump file will be generated.
     */
    signal(SIGABRT, SIG_IGN);
    if (chdir("/tmp")) {
	LOG(1, "Dumping core to /tmp.");
    }
    else {
	char dirname[100];
	getcwd(dirname, sizeof(dirname));
	LOG(1, "Dumping core to %s.\n", dirname);
    }
    {
    /*
     * setrlimit() makes sure that we can produce a core dump file.
     */
    struct rlimit rlim = {0, RLIM_INFINITY};
    setrlimit(RLIMIT_CORE, &rlim);
    }
    abort();
#else
    /*
     * (Vandoorselaere Yoann)
     * let's do that in a cleaner way, don't use code to generate sigsegv cause
     * it can be handled... use _exit().
     */
    _exit(1);
#endif
}


