/*
 * Copyright  2002  Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * priv_client.cc
 * Provides the client half of the process.  Should be considered
 * untrusted by the privman server.
 *
 * $Id: priv_client.cc,v 1.18 2002/08/02 17:52:34 dougk Exp $
 */

#include <stdarg.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
#include <sys/wait.h>

#include "privman.h"
#include "msghdr.h"

#include "types.h"
#include "priv_impl.h"

/* Used as a cache for get/set_item */
static const void      *pam_types[20] = { 0 };
/*                                      // Strings.  void* for strings.
 *  [PAM_SERVICE]           = NULL,
 *  [PAM_USER]              = NULL,
 *  [PAM_TTY]               = NULL,
 *  [PAM_RHOST]             = NULL,
 *  [PAM_CONV]              = NULL,
 *
 *  [PAM_RUSER]             = NULL,
 *  [PAM_USER_PROMPT]       = NULL,
 *  [PAM_FAIL_DELAY]        = NULL      // This is just a function pointer
 */
 
#ifndef CONFIG_PATH
#define CONFIG_PATH="/etc/privman.d"
#endif
static void readConfig(const char *progname) {
    extern FILE        *yyin; /* lex's input */
    char                pathname[PATH_MAX+1] = CONFIG_PATH;
    /* Assigning a string to the buffer null-pads it */
 
    strncpy(pathname+sizeof(CONFIG_PATH)-1,progname,
            sizeof(pathname)-sizeof(CONFIG_PATH));
 
    /* fopen, cause yyin if a FILE* */
    yyin = fopen(pathname, "r");
 
    if (yyin == NULL) {
        fprintf(stderr,"Error: missing privmand configuration file\n");
    } else if (yyparse() != 0) {
        fprintf(stderr,"Error reading privmand configuration file\n");
    } 
    if (yyin != NULL)
        fclose(yyin);
}

/* When requested, calls the PAM conversion function registered by
 * the client.
 */
static void handleConvert(message_t *msg)
{
    struct pam_message        **messages;
    struct pam_response        *resp;
    int                         num_msg;
    int                         i, n, rc;

    num_msg = msg_getInt(msg);
    messages = (struct pam_message**)malloc(sizeof(*messages) * num_msg);
    for (i = 0 ; i < num_msg; ++i) {
        char    buf[PAM_MAX_MSG_SIZE];
        messages[i] = (struct pam_message*)malloc(sizeof(**messages));

        messages[i]->msg_style = msg_getInt(msg);
        msg_getString(msg,buf,sizeof(buf)-1);
        buf[sizeof(buf)-1] = '\0';
        messages[i]->msg = strdup(buf);
    }

    rc = ((struct pam_conv*)pam_types[PAM_CONV])->conv(num_msg,
            (const struct pam_message **)messages, /* C const fun cast */
            &resp, ((struct pam_conv*)pam_types[PAM_CONV])->appdata_ptr);


    msg_clear(msg);

    msg_addInt(msg,rc);
    for (i = 0; i < num_msg; ++i) {
        msg_addString(msg, resp[i].resp);
        msg_addInt(msg, resp[i].resp_retcode);
    }

    n = msg_sendmsg(msg, privmand_fd);
    if (n < 0)
        boom("handleConvert(sendmsg)");

    /* Tear down all the data. */
    for (i = 0; i < num_msg; ++i) {
        free((char*)(messages[i]->msg)); /* const cast */
        free(resp[i].resp);
    }
    free(messages);
    free(resp);
}


static __inline__
void wait_for_debugger(void)
{
#if defined(DEBUG)
    /* Block until the debugger unblocks us.  Gives you a known
     * place to attach the debugger.
     */
    volatile int i = 0;
    fprintf(stderr,"waiting for debugger\n");
    while (i == 0)
        sleep(1);
#endif
}

void socketfun(int sockfds[2], bool server) {
    if (server) {
        close(sockfds[1]); /* We keep [0] */
        privmand_fd = sockfds[0];
    } else {
        close(sockfds[0]); /* we keep [1] */
        privmand_fd = sockfds[1];
    }
}

void setup_child(int sockfds[2],
        void (*fnptr)(const char *), const char *arg)
{
    struct passwd  *pwent;

    wait_for_debugger();

    /* in the child process */
    socketfun(sockfds, false);

    /* Get unpriv_user info, in case chroot changes it.
     * chroot,
     * setuid.
     */

    /* Normalize config. */
    if (config->unpriv_user == "")
        config->unpriv_user = "nobody";
    if (config->unpriv_jail == "")
        config->unpriv_jail = "/";

    /* getpwnam */
    pwent = getpwnam(config->unpriv_user.c_str());
    /* Don't know if its a static pointer, or malloced and I'm allowed
     * to clear it, so just leak it.
     */

    if (pwent == NULL) 
        boom("getpwnam");

    /* chroot */
    if (chroot(config->unpriv_jail.c_str()) < 0) {
        fprintf(stderr, "chroot to %s\n", config->unpriv_jail.c_str());
        boom("setup_child(chroot)");
    }

    /* setuid, drop priviledge. */
    setreuid(pwent->pw_uid,pwent->pw_uid);
    setregid(pwent->pw_gid,pwent->pw_gid);

    /* Call provilded function */
    if (fnptr != NULL) {
        fnptr(arg);
    }

    /* And return to do normal work. */
}

void priv_sep_init(void (*servfn)(void),
    void (*childfn)(const char *), const char *childfn_arg)
{
    int         sockfds[2];

    if (socketpair(AF_LOCAL, SOCK_STREAM, 0, sockfds) < 0)
        boom("socketpair");

    child_pid = fork();
    if (child_pid == 0) {
        setup_child(sockfds, childfn, childfn_arg);
    } else if (child_pid < 0) {
        boom("fork");
    } else {
        /* Parent process */
        socketfun(sockfds, true);

        wait_for_debugger();

        if (servfn != NULL)
            servfn();
        /* Fall out.  If we even get here, we're actually a second
         * child.
         */
    }
}

void priv_init(const char *appname)
{
    /* Read the config now. */
    readConfig(appname);

    priv_sep_init(privman_serv_init, 0, 0);
}


int priv_open(const char *pathname, int flags, ...)
{
    va_list             ap;/*va_start(pathname,ap);va_arg(ap,type);va_end(ap)*/
    message_t          *msg = msg_new();
    int                 n, retval;
    char                cwd[PATH_MAX];

    msg_init(msg, CMD_OPEN);

    msg_addInt(msg,flags);

    if (flags & O_CREAT) {
        va_start(ap, flags);
        msg_addInt(msg,va_arg(ap,int));
        va_end(ap);
    } else {
        msg_addInt(msg,0);
    }

    /* We have to canpath the path, else chdir() messes us up. */
    if (getcwd(cwd, sizeof(cwd)) == NULL) {
        /* Use a token "NFC" value.  we won't work right in this
         * case, but it might work if you didn't chdir
         */
        msg_addString(msg,"");
    } else {
        msg_addString(msg,cwd);
    }
    msg_addString(msg,pathname);
    /* send the message */
    n = msg_sendmsg(msg, privmand_fd);
    if (n < 0) {
        retval = -1;
        goto exit;
    }

    /* listen for responce */
    msg_clear(msg);
    n = msg_recvmsg(msg, privmand_fd);
    if ( n < 0 ) {
        retval = -1;
        goto exit;
    }

    n = msg_getInt(msg);
    if ( n < 0 ) {
        errno = -n;
        retval = -1;
    } else {
        retval = msg_getFd(msg);
    }
exit:
    msg_delete(msg);
    return retval;
}

/* Done in terms of fopen */
FILE* priv_fopen(const char *pathname, const char *mode)
{
    int         fd;
    int         open_mode = 0;

    /* First get the extra flags, then the basic open mode.  The
     * extra flags are purely depending on the base of the fopen
     * mode, while the base open mode is based on the + and the
     * base fopen mode.
     */
    switch(mode[0]) {
    case 'r':
        open_mode |= 0; /* Nothing here.  No creation. */       break;
    case 'w':
        open_mode |= O_CREAT|O_TRUNC;                           break;
    case 'a':
        open_mode |= O_CREAT|O_APPEND;                          break;
    default:
        errno = EINVAL;
        return NULL;
    }
    if (mode[1] == '+') /* '+' or '\0' */
        open_mode |= O_RDWR;
    else
        switch (mode[0]) {
        case 'w':
        case 'a':
            open_mode |= O_WRONLY;                              break;
        case 'r':
            open_mode |= O_RDONLY;                              break;
        }

    /* Punt to previously written code.  Easier. */
    fd = priv_open(pathname, open_mode);
    if (fd < 0)
        return NULL; /* errno should already be set */

   return fdopen(fd, mode);
}

int priv_bind(int sockfd, struct sockaddr *addr, socklen_t addrlen)
{
    message_t  *msg = msg_new();
    int         n;

    msg_addInt(msg,CMD_BIND);
    msg_setFd(msg, sockfd);
    msg_addInt(msg, addrlen);
    msg_addData(msg, addr, addrlen);

    /* send the message */
    n = msg_sendmsg(msg, privmand_fd);
    if (n < 0)
        boom("priv_bind(sendmsg)");

    /* listen for responce */
    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_bind(recvmsg)");
    n = msg_getInt(msg);
    if (n < 0) {
        errno = -n;
        n = -1;
    }

    msg_delete(msg);
    return n;
}

int priv_pam_start(const char *service, const char *user,
                    const struct pam_conv *conv,
                    pam_handle_t **pamh_p)
{
    /* service: string.  Just send.
     * user: string.  Just send (two strings?  eek.  strlen:string
     * conversion function.  Don't bother with.  The other side will
     *  will have to return a call to us.  I think any of the priv_pam
     *  calls will have to listen for it.
     * pamh_p.  Er, I think we'll just pretend this is an opaque.  The
     *  other side will keep a list.
     */
    message_t          *msg = msg_new();
    int                 n, retval = PAM_SYSTEM_ERR;

    msg_addInt(msg, CMD_PAM_START);
    msg_addString(msg, service);
    msg_addString(msg, user);

    /* Save the conversion function for when we have to use it.
     * TBS: handle multiple man sessions/handles.
     */
    pam_types[PAM_CONV] = conv;

    msg_sendmsg(msg, privmand_fd, "priv_pam_start(sendmsg)");

    /* Listen for responce */
    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_pam_start(recvmsg)");

    n = msg_getInt(msg);
    if (n < 0) {
        errno = -retval;
        retval = PAM_PERM_DENIED;
    } else {
        assert( n == PRIV_PAM_RC );
        retval = msg_getInt(msg);
        *pamh_p = (pam_handle_t*)msg_getPtr(msg);
    }

    return retval;
}

static int priv_pam_simple_func(pam_handle_t *pamh, int flags,
        const char *function_name, char function_code)
{
    message_t                  *msg = msg_new();
    int                         rc;
    enum privman_responces      cmd;

    msg_addInt(msg, function_code);
    msg_addPtr (msg, pamh);
    msg_addInt (msg, flags);

    /* send the message */
    msg_sendmsg(msg, privmand_fd, function_name);

    do {
        msg_clear(msg);
        msg_recvmsg(msg, privmand_fd, function_name);

        rc = msg_getInt(msg);
        if (rc < 0) {
            errno = -rc;
            return PAM_PERM_DENIED;
        }

        cmd = (enum privman_responces)rc;
        switch (cmd) {
        case PRIV_PAM_RC:
            rc = msg_getInt(msg);
            break;
        case PRIV_PAM_RUN_CONV:
            handleConvert(msg);
            break;
        case PRIV_NONE:
        case PRIV_SET_COE:
        default:
            boom("priv_pam_simple_func(unexpected responce)");
            break;
        }
    } while (cmd != PRIV_PAM_RC);

    return rc;
}

#define PRIV_PAM_SIMPLE(name,code)                              \
int priv_##name (pam_handle_t *pamh, int flags)                 \
{                                                               \
    return priv_pam_simple_func(pamh, flags,                    \
            __FUNCTION__, code) ;                               \
}

PRIV_PAM_SIMPLE(pam_authenticate,       CMD_PAM_AUTHENTICATE)
PRIV_PAM_SIMPLE(pam_acct_mgmt,          CMD_PAM_ACCT_MGMT)
PRIV_PAM_SIMPLE(pam_end,                CMD_PAM_END)
PRIV_PAM_SIMPLE(pam_setcred,            CMD_PAM_SETCRED)
PRIV_PAM_SIMPLE(pam_chauthtok,          CMD_PAM_CHAUTHTOK)
PRIV_PAM_SIMPLE(pam_open_session,       CMD_PAM_OPEN_SESSION)
PRIV_PAM_SIMPLE(pam_close_session,      CMD_PAM_CLOSE_SESSION)

int priv_pam_set_item(pam_handle_t *pamh, int item_type, const void *item)
{
    message_t          *msg = NULL;
    int                 rc;

    assert(item_type != PAM_CONV); /* handled by pam_start */
    /* Assume that pam_fail_delay is not dynamically loaded. */

    msg = msg_new();
    msg_addInt(msg, CMD_PAM_SET_ITEM);
    msg_addPtr (msg, pamh);
    msg_addInt (msg, item_type);
    if (item_type != PAM_FAIL_DELAY) {
        msg_addString(msg, (char *)item);
    } else {
        msg_addPtr(msg, item);
    }

    /* send the message */
    msg_sendmsg(msg, privmand_fd, "priv_pam_get_item(sendmsg)");

    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_pam_get_item(recvmsg)");

    rc = msg_getInt(msg); /* were we denied at the gate? */
    if (rc < 0) {
        errno = -rc;
        return PAM_PERM_DENIED;
    }

    assert((enum privman_responces)rc == PRIV_PAM_RC);

    rc = msg_getInt(msg); /* and the RC of pam_get_item. */

    /* Wait until success to set the cache. */
    if (rc == PAM_SUCCESS) {
        if (item_type != PAM_FAIL_DELAY) {
            if (pam_types[item_type])
                free(((void*)pam_types[item_type]));
            pam_types[item_type] = strdup((char *)item);
        } else {
            /* FAIL_DELAY */
            pam_types[item_type] = item;
        }
    }

    return rc;
}

int priv_pam_get_item(pam_handle_t *pamh, int item_type, const void **item)
{
    message_t          *msg = NULL;
    int                 rc;

    if (pam_types[item_type] != NULL) {
        *item = pam_types[item_type];
        return PAM_SUCCESS;
    }

    assert(item_type != PAM_CONV); /* handled by pam_start */
    /* Assume that pam_fail_delay is not dynamically loaded. */

    msg = msg_new();
    msg_addInt(msg, CMD_PAM_GET_ITEM);
    msg_addPtr (msg, pamh);
    msg_addInt (msg, item_type);

    /* send the message */
    msg_sendmsg(msg, privmand_fd, "priv_pam_get_item(sendmsg)");

    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_pam_get_item(recvmsg)");

    rc = (enum privman_responces)msg_getInt(msg);
    if (rc < 0) {
        errno = -rc;
        return PAM_PERM_DENIED;
    }
    assert((enum privman_responces)rc == PRIV_PAM_RC);

    rc = msg_getInt(msg);

    if (rc == PAM_SUCCESS) {
        if (item_type == PAM_FAIL_DELAY) {
            pam_types[item_type] = msg_getPtr(msg);
        } else {
#define ITEM_BUF_SIZE 1024
            pam_types[item_type] = malloc(ITEM_BUF_SIZE);
            msg_getString(msg, (char*)(pam_types[item_type]),ITEM_BUF_SIZE-1);
            pam_types[ITEM_BUF_SIZE-1] = '\0';
        }

        *item = pam_types[item_type];
    }
    return rc;
}

int priv_pam_putenv(pam_handle_t *pamh, const char *name_value);
int priv_pam_getenv(pam_handle_t *pamh, const char *name);
/*
PAM:

    To do these right, I need a "map" implimentation.  I'm seriously
    thinking of redoing this in C++ with a C interface, so when I do
    that I'll do these function.

    pam_putenv(pam_handle, "FOO=bar"); "FOO=" for empty, "FOO" to nuke
    pam_getenv(pam_handle, "FOO")
    pam_getenvlist(pam_handle)

*/

pid_t priv_fork(void)
{
    /* Tell the parent to dup.  Parent will fork, and will return the
     * fd we now use to talk to it.
     * We fork, tell parent our new pid.
     * No race condition since parent is ST.  I think
     * XXX Race condition?
     */
    pid_t       retval;
    int         new_fd, n;
    message_t  *msg = msg_new();

    msg_init(msg, CMD_FORK);
    n = msg_sendmsg(msg, privmand_fd);
    if (n < 0) {
        retval = -1;
        goto exit;
    }

    msg_clear(msg);
    n = msg_recvmsg(msg, privmand_fd);
    if (n < 0) {
        retval = -1;
        goto exit;
    }

    n = msg_getInt(msg);
    if (n < 0) {
        errno = -n;
        retval = -1;
        goto exit;
    } else {
        new_fd = msg_getFd(msg);
    }

    retval = fork();
    if (retval > 0) {
        /* Parent */
        close(new_fd); /* don't need it here */
    } else if (retval == 0) {
        /* Child */
        close(privmand_fd);
        privmand_fd = new_fd;
    } else {
        /* error.  Tell new server.*/
        msg_init(msg, CMD_EXIT);
        msg_addInt(msg, -1);
        msg_sendmsg(msg, new_fd); /* Never check an error you can't handle */
        close(new_fd);
    }

exit:
    return retval;
}

/* "daemon".  On the client side, this does some muching with FD's.
 * On the server side, it does a fork();exit() pair in the original
 * process so that it detacches.
 */

int priv_daemon(int nochdir, int noclose)
{
    message_t  *msg = msg_new();
    int         n = 0;

    /* First, tell parent to detach */
    msg_init(msg, CMD_DAEMON);
    msg_sendmsg(msg, privmand_fd, "priv_daemon(sendmsg)");

    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_daemon(sendmsg)");

    n = msg_getInt(msg);
    msg_delete(msg);

    if (n < 0) {
        errno = -n;
        return -1;
    }

#ifdef HAVE_SETSID
    n = setsid();
    if (n < 0)
        return n;
    /* Can't fail, as priv_init() makes us a child */
#endif
    if (!nochdir)
        chdir("/");

    if (!noclose) {
        freopen("/dev/null", "r", stdin);
        freopen("/dev/null", "w", stdout);
        freopen("/dev/null", "a", stderr);
    }

    return 0;
}

/* Exec "filename" as user "user" in chroot jail "root" with args argv,
 * and environment envp
 * TODO: limits.  Feh.
 *
 * filename as found from the root jail.
 * Will be executed with as user, so check permissions.
 *
 * Actually Executes from the parent, then exit's here.
 */
int priv_execve(const char *filename, char * const argv[], char * const envp[],
            const char *user, const char *root)
{
    message_t *msg = msg_new();
    int i, argc, envc;

    msg_init(msg, CMD_EXEC_AS);
    msg_addString(msg, filename);
    msg_addString(msg, user != NULL ? user : "");
    msg_addString(msg, root != NULL ? root : "");

    for (argc = 0; argv[argc] != NULL; ++argc)
        ;
    msg_addInt(msg, argc);
    for (i = 0; i < argc; ++i) {
        msg_addString(msg, argv[i]);
    }

    for (envc = 0; envp[envc] != NULL; ++envc)
        ;
    msg_addInt(msg, envc);
    for (i = 0; i < envc; ++i) {
        msg_addString(msg, envp[i]);
    }

    /* send the message */
    msg_sendmsg(msg, privmand_fd, "priv_execve(sendmsg)");

    /* listen for responce */
    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_execve(recvmsg)");
    i = msg_getInt(msg);
    if (i < 0) {
        errno = -i;
        i = -1;
    } else {
        _exit(0);
    }

    msg_delete(msg);
    return i;
}

/* Creates a clean fork() from the privmand server with state equivilent
 * to the client state when priv_init() was first called.
 *
 * Invokes the specified function, with the string arg provided.
 * chroots to the directory provided, setuid's to the user provided.
 * TODO: limits.  Feh.
 *
 */
int priv_rerunas(void (*fnptr)(const char *), const char *arg,
            const char *user, const char *root)
{
    message_t *msg = msg_new();
    int i;

    msg_init(msg, CMD_RERUN_AS);
    msg_addPtr(msg, (void*)fnptr);
    msg_addString(msg, arg);
    msg_addString(msg, user != NULL ? user : "");
    msg_addString(msg, root != NULL ? root : "");

    /* send the message */
    msg_sendmsg(msg, privmand_fd, "priv_rerunas(sendmsg)");

    /* listen for responce */
    msg_clear(msg);
    msg_recvmsg(msg, privmand_fd, "priv_rerunas(recvmsg)");
    i = msg_getInt(msg);
    if (i < 0) {
        errno = -i;
        i = -1;
    } else {
        /* _exit since the "real program" transitioned to the new one. */
        _exit(0);
    }

    msg_delete(msg);
    return i;
}

