/*
 * Copyright  2002  Networks Associates Technology, Inc.
 * All rights reserved.
 *
 * privman.cc 
 * Provides the server half of the process.  This half retains priviledge
 * and should be nei-invulnerable.
 *
 * $Id: privman.cc,v 1.32 2002/08/30 20:05:07 dougk Exp $
 */
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <limits.h>
#include <fcntl.h>
#include <string.h>
#include <pwd.h>
#include <assert.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <sys/wait.h>
#include <syslog.h>

extern "C" {
    /* sockaddr_in */
#include <security/pam_appl.h>
}

#include <map>

#include "privman.h"
#include "msghdr.h"

#include "types.h"
#include "priv_impl.h"
/* Globals */
int             privmand_fd     = -1;    /* FD to talk to the other process */

pid_t           child_pid       = 0;

/* TBD: multiple conversion functions. */
static       struct pam_conv    pconv;    /* server side. */

static bool     p_wait_on_child = true;

/* The config for this invocation.  Not static so that the parser can
 * find it to set it.  Parser run from priv_init.
 */
config_t               *config;

/* Dispatch table */
std::map< enum commands, void(*)(message_t*) > function_map;

/* PAM conversion function.  We pass the work back to the other
 * side.  The assumption is that the other side is currently
 * waiting in a priv_pam_foo() call, and will recoginize the
 * message.
 *
 * Don't bother passing app_ptr, as it won't be right anyway.  The other
 * side has it.  resp is an out value, not an in value.
 */

#ifdef __cplusplus
#define UNUSED(p)
#else
#define UNUSED(p) p __attribute((unused))
#endif

static int convert_punt(int num_msg,
        const struct pam_message **messages, struct pam_response **resp,
        void * UNUSED(app_ptr))
{
    message_t                  *msg = msg_new();
    struct pam_response        *reply;
    int                         retval, i;

    msg_initResponce(msg, PRIV_PAM_RUN_CONV);
    msg_addInt(msg,num_msg);
    for (i = 0; i < num_msg; ++i) {
        msg_addInt(msg,messages[i]->msg_style);
        msg_addString(msg,messages[i]->msg);
    }

    msg_sendmsg(msg, privmand_fd, "convert_punt(sendmsg)");

    msg_initResponce(msg, 0);
    msg_recvmsg(msg, privmand_fd, "convert_punt(recvmsg)");

    retval = msg_getInt(msg);
    reply = (struct pam_response*)malloc(sizeof(*reply) * num_msg);
    for (i = 0; i < num_msg; ++i) {
        char    buf[PAM_MAX_RESP_SIZE];
        msg_getString(msg, buf, sizeof(buf)-1);
        buf[sizeof(buf)-1] = '\0';

        reply[i].resp = strdup(buf);
        reply[i].resp_retcode = msg_getInt(msg);
    }

    msg_delete(msg);

    *resp = reply;
    return retval;
}

static bool openPerm(const char *path, int flags)
{
    /* MAC check here. TBD: use real globs instead of this hack */
    /* Hack:
     *   1) Path is in set.  Go home happy.
     *   2) /first/bit/of/path/[*] is in set.  Go home happy.
     */
    enum {
        /* Use this enum to index into the list[] array below. */
                            read_only       = 0,
                            read_write      = 1,
                            append_only     = 2,
                            exec            = 3
    }                   type;
    path_list          *list[] = {
                            &(config->open_ro),
                            &(config->open_rw),
                            &(config->open_ao),
                            &(config->exec_list)
                        };
    char                testpath[MAXPATHLEN+1];
    char               *offset;

    /* Determine the type of access. */
    if ((flags & 3) == O_RDONLY)
        type = read_only;
    else if (((flags & 3) == O_WRONLY) && (flags & O_APPEND))
        type = append_only;
    else if (((flags & 3) == O_RDWR) || (flags & 3) == O_WRONLY)
        type = read_write;
    else
        type = exec;

    strncpy(testpath, path, sizeof(testpath)-2);
    testpath[sizeof(testpath)-2] = '\0';
    offset = testpath + strlen(path); /* char* cause that's what rindex
                                       * returns */
    while (offset != NULL) {
        memcpy(testpath, path, offset - testpath);
        if ( *offset == '/' ) { /* We have a directory.  Look for a glob */
            offset[1] = '*'; /* See the -2 above for your "space" question */
            offset[2] = '\0';
        }

        if (list[type]->count(testpath) != 0)
            break;

        /* Chop off the last element, and loop */
        *offset = '\0';
        offset = rindex(testpath, '/');
    }
    if (offset == NULL)
        return false;

    return true;
}

/* Like "realpath", but can cope with a missing file in a non
 * missing directory.  Might not handle dangling symlinks right. XXX
 */
static bool myrealpath(const char *path, char *resolved)
{
    char       *rv;
    char        buf[PATH_MAX+1];
    char        last_elm[PATH_MAX+1];
    char       *last_slash;
    int         n;

    strncpy(buf, path, sizeof(buf)-1);
    buf[sizeof(buf)-1] = '\0';

    /* deletegate to "realpath" */
    rv = realpath(buf, resolved);
    if (rv != NULL || errno != ENOENT)
        return rv != NULL;

    /* Ok, doesn't exist.  Chop off the filename and save it.
     * Huge buffer way to big.  *shrug*
     */
    last_slash = rindex(buf, '/');
    if (last_slash == NULL)
        return false;

    strncpy(last_elm, last_slash, sizeof(last_elm)-1);
    last_elm[sizeof(last_elm)-1] = '\0';

    /* Now chop off the last bit, and try the directory */
    *last_slash = '\0';

    rv = realpath(buf, resolved);
    if (rv == NULL)
        return false;

    /* Add it back, and call it a day */
    n = strlen(resolved);
    strncpy(resolved+n, last_elm, PATH_MAX-n);

    return true;
}
/* Handle an "open file" request */
static void openFile(message_t *msg) {
    char        path[MAXPATHLEN+1];
    char        cwd[MAXPATHLEN+1];

    char        canpath[MAXPATHLEN+1];
    int         flags, mode;
    int         rfd = 0;
    int         n;

    /* Open file specified, return the fd. */
    /* arg1 = open mode
     * arg2 = creation mode (optional)
     * arg3 = pathname
     *
     * Returns
     * arg1 = 0 or -1
     * arg2 = errno (if arg1 < 0)
     */

    flags = msg_getInt(msg);
    mode  = msg_getInt(msg);
    msg_getString(msg, cwd, sizeof(cwd)-1);
    msg_getString(msg, path, sizeof(path)-1);
    path[sizeof(path)-1] = '\0';
    cwd[sizeof(cwd)-1] = '\0';

    /* Canacolize the path, so that a comparison with the
     * allowed list makes sence, and in case the client did a chdir.
     *
     * The client sends us "getcwd()" output, so no trailing '/'.  If
     * the client wants to lie to us for this message, no great foul.
     * He gets no file.
     */

    n = strlen(cwd);
    if (path[0] == '/') { /* absolute or not? */
        /* Abs: nuke cwd with path, then realpath it. */
        strncpy(cwd, path, sizeof(cwd) - n);
    } else {
        cwd[n++] = '/'; /* Path seperator */
        strncpy(cwd + n, path, sizeof(cwd) - n);
    }

    if (!myrealpath(cwd, canpath)) {
        /* Could be lots of reasons.  So lets confuse em with whatever
         * realpath said.
         */
        msg_initResponce(msg, -errno);
    }

    if (!openPerm(canpath, flags)) {
        /* MAC check here. TBD: use real globs instead of this hack */
        msg_initResponce(msg, -EPERM);
        syslog(LOG_NOTICE, "Unauthorized attempt open(%s, %d)", canpath, flags);
    } else {

        /* Ok, now do it. */
        rfd = open(canpath,flags,mode);
        if (rfd < 0) {
            msg_initResponce(msg, -errno);
            perror("msg_open_file(open)");
        } else {
            msg_initResponce(msg, 0);
            msg_setFd(msg,rfd);
        }
    }

    msg_sendmsg(msg, privmand_fd, "openFile(sendmsg)");
    if (rfd)
        close(rfd); /* prevent the leak */
}

static void bindPort(message_t *msg) {
    int                 sockfd;
    struct sockaddr_in *addr;
    socklen_t           addrlen;
    int                 retval;

    addrlen     = msg_getInt(msg);
    addr        = (struct sockaddr_in*)malloc(addrlen);
    msg_getData(msg, addr, addrlen);
    sockfd      = msg_getFd(msg);

    /* Permission check */
    if (addr->sin_family != AF_INET
    || sockfd < 0
    || addrlen < sizeof(sockaddr_in)
    || ( config->bind_port.count(ntohs(addr->sin_port)) == 0
     &&  config->bind_port.count(ntohs(0)) == 0 ) ) { /* wildcard */
        /* Permission denied */
        msg_initResponce(msg, -EPERM);
        syslog(LOG_NOTICE, "Unauthorized attempt bind to port %d",
                addr->sin_port);
    } else {
        retval      = bind(sockfd, (struct sockaddr*)addr, addrlen);

        if (retval < 0)
            retval = -errno; /* since it'll be used for this anyway. */

        msg_initResponce(msg, retval);
    }

    msg_sendmsg(msg, privmand_fd, "bindPort(sendmsg)");
    close(sockfd); /* don't leak this. */
}
    
static void pamStart(message_t *msg)
{
    pam_handle_t       *handle;
    int                 retval;
    char                service[128];
    char                user[128];

    msg_getString(msg, service, sizeof(service)-1);
    service[sizeof(service)-1] = '\0';
    msg_getString(msg, user, sizeof(user)-1);
    user[sizeof(user)-1] = '\0';

    pconv = (struct pam_conv){ convert_punt, 0};

    retval = pam_start(service, user, &pconv, &handle);

    msg_initResponce(msg, PRIV_PAM_RC);
    msg_addInt(msg,retval);
    msg_addPtr(msg, handle);

    msg_sendmsg(msg, privmand_fd, "pamStart(sendmsg)");
}

static void pamSimpleFunc(message_t *msg, int (*func)(pam_handle_t*,int))
{
    pam_handle_t *pamh;
    int flags, rc;

    pamh  = (pam_handle_t*)msg_getPtr(msg);
    flags = msg_getInt(msg);

    rc = func(pamh, flags);

    /* This isn't a call to run the conv func */
    msg_initResponce(msg, PRIV_PAM_RC);
    msg_addInt(msg, rc);
    msg_sendmsg(msg, privmand_fd, "pamSimpleFunc(sendmsg)");
}

static void pamSetItem(message_t *msg)
{
    pam_handle_t       *pamh;
    int                 type, rc;

    pamh = (pam_handle_t*)msg_getPtr(msg);
    type = msg_getInt(msg);

    assert(type != PAM_CONV);

    if (type == PAM_FAIL_DELAY) {
        void   *item = msg_getPtr(msg);
        rc = pam_set_item(pamh, type, item);
    } else {
        char    buf[1024];
        msg_getString(msg, buf, sizeof(buf)-1);
        buf[sizeof(buf)-1] = '\0';
        rc = pam_set_item(pamh, type, buf);
    }

    msg_clear(msg);

    msg_addInt(msg, PRIV_PAM_RC);
    msg_addInt(msg, rc);

    msg_sendmsg(msg, privmand_fd, "pamSetItem(sendmsg)");
}

static void pamGetItem(message_t *msg)
{
    pam_handle_t       *pamh;
    int                 type, rc;
    void               *item;

    pamh = (pam_handle_t*)msg_getPtr(msg);
    type = msg_getInt(msg);

    assert(type != PAM_CONV);

    rc = pam_get_item(pamh, type, (const void **)(&item));

    msg_clear(msg);

    msg_addInt(msg, PRIV_PAM_RC);
    msg_addInt(msg, rc);

    if (rc == PAM_SUCCESS) {
        if (type != PAM_FAIL_DELAY) {
            msg_addString(msg, (char*)item);
        } else {
            msg_addPtr(msg, item);
        }
    }

    msg_sendmsg(msg, privmand_fd, "pamGetItem(sendmsg)");
}

static void exitServer(message_t *UNUSED(msg)) {
    /* Single. */
    _exit(0);
}

static void forkProcess(message_t *msg) {
    /* Client requests fork().
     * We create new pid, hand back to client.
     *    Fork.
     *    Child privmand listens on pid.
     *    We close pid.
     */
    int         fds[2], n;

    if (socketpair(AF_LOCAL, SOCK_STREAM, 0, fds) < 0)
        boom("forkProcess(socketpair)");

    msg_clear(msg);
    msg_addInt(msg, 0); // Return code
    msg_setFd(msg, fds[0]);

    msg_sendmsg(msg, privmand_fd, "forkProcess(sendmsg)");

    close(fds[0]);

    n = fork();
    if (n < 0)
        boom("forkProcess(fork))");
    else if (n > 0) {
        /* Parent.  Close the FD's */
        close(fds[1]);
    } else {
        /* child */
        close(privmand_fd);
        privmand_fd = fds[1];
        p_wait_on_child = false;
    }
}

void daemonProcess(message_t *msg) {
    /* Daemon.
     * 1) Fork, parent quits.  No mucking with global state
     *    and the child should be fine.
     * 2) Setsid.  This detaches us from the controlling terminal,
     *    so ^C won't work.
     * 3) How do I handle waiting on the child?  I don't.  It quits,
     *    I go away.  So set the p_wait_on_child flag.
     */
    int n;

    n = fork();
    if (n == 0) {
        /* child.  See above. */
        setsid(); /* Detach */

        /* Handle stdio.  strace for output? */
        freopen("/dev/null", "r", stdin);
        freopen("/dev/null", "w", stdout);
        freopen("/dev/null", "a", stderr);

        p_wait_on_child = false;
        /* Send a success message */
        msg_clear(msg);
        msg_initResponce(msg, 0);
        msg_sendmsg(msg, privmand_fd, "daemonProcess(sendmsg)");
    } else if (n > 0) {
        /* Parent.  This is easy. */
        _exit(0);
    } else if (n < 0) {
        /* Error.  Panic and fall down? */
        boom("daemonProcess(fork)");
    }
}

void rerunAsProcess(message_t *msg)
{
    char        buf[4096];
    char       *args;
    char       *user;
    char       *root;
    void        (*fnptr)(const char*);

    fnptr = (void (*)(const char*))msg_getPtr(msg);

    msg_getString(msg, buf, sizeof(buf)-1);
    buf[sizeof(buf)-1] = '\0';
    args = strdup(buf);

    msg_getString(msg, buf, sizeof(buf)-1);
    buf[sizeof(buf)-1] = '\0';
    user = strdup(buf);

    msg_getString(msg, buf, sizeof(buf)-1);
    buf[sizeof(buf)-1] = '\0';
    root = strdup(buf);

    /* MAC checks. */
    if (user[0] != '\0' && config->user.count(user) == 0
            && config->user.count("*") == 0) {
        msg_initResponce(msg, -EPERM);
        syslog(LOG_NOTICE, "Unauthorized attempt rerunAs as user %s",
                user);
        msg_sendmsg(msg, privmand_fd, "rerunAsProcess(sendmsg)");

        free(user);
        free(root);
        free(args);
        return;
    }

    /* config hacking.  Sanity checks? */
    config->unpriv_user = user;
    config->unpriv_jail = root;

    /* the previous statement was actually a copy.  config-> has
     * stl strings, not char * pointers.
     */
    free(user); free(root);

    msg_clear(msg);
    msg_initResponce(msg, 0);
    msg_sendmsg(msg, privmand_fd, "rerunAsProcess(sendmsg)");

    /* And if we wait on the child, wait here. */
    if (child_pid != 0 && p_wait_on_child == true)
        waitpid(child_pid, 0, 0);

    /* This will run the client function, then drop back
     * through here, as both the parent and the child.
     * Parent case is handled: globals have been changed, loop
     * continues as before.
     * Child side's a little different.
     */
    priv_sep_init(0, fnptr, args);

    free(args);
}

void execAsProcess(message_t *msg) 
{
    /* filename, user, root, argc, argv, envc, envp */
    char program[MAXPATHLEN];
    char username[32];
    char rootpath[MAXPATHLEN];
    int argc, envc;
    char **argv;
    char **envp;

    struct passwd     * pw;
    struct stat         buf;
    int                 i;
#if 0
    int                 orig_rootfd;
#endif

    msg_getString(msg, program,  sizeof(program) -1);
    program[sizeof(program) -1] = '\0';
    msg_getString(msg, username, sizeof(username)-1);
    program[sizeof(username)-1] = '\0';
    msg_getString(msg, rootpath, sizeof(rootpath)-1);
    program[sizeof(rootpath)-1] = '\0';

    /* Config file MAC */
    if (!openPerm(program, 3)) {
        msg_initResponce(msg, -EPERM);
        syslog(LOG_NOTICE, "Unauthorized attempt execAs %s", program);
        msg_sendmsg(msg, privmand_fd, "execAsProcess(sendmsg)");
        return;
    }
    if (username[0] != '\0' && config->user.count(username) == 0
            && config->user.count("*") == 0) {
        msg_initResponce(msg, -EPERM);
        syslog(LOG_NOTICE, "Unauthorized attempt execAs %s", username);
        msg_sendmsg(msg, privmand_fd, "execAsProcess(sendmsg)");
        return;
    }

    /* Check things out first.
     * get "/" fd,
     * chroot root
     * stat program
     * sanity check.
     * fchdir rootfd
     * chroot "."
     */

#if 0
    if (rootpath != NULL)
        orig_rootfd = open("/", O_DIRECTORY|O_RDONLY);
#endif

    /* XXX Return error. */
    if (rootpath[0] != '\0' && chroot(rootpath) < 0) {
        boom("execAsProcess(chroot)");
    }

    if (stat(program, &buf) < 0) {
        boom("execAsProcess(stat)");
    }

    if (!S_ISREG(buf.st_mode) || (
                ! (buf.st_mode & S_IXOTH|S_IXUSR|S_IXGRP))) {
        boom("execAsProcess(perms)");
    }

#if 0
    fchdir(orig_rootfd);
#endif
    if (username[0] == '\0')
        memcpy(username, "nobody", sizeof("nobody"));

    pw = getpwnam(username);

    if (pw == NULL)
        boom("execAsProcess(getpwnam)");

    setuid(pw->pw_uid);
    setgid(pw->pw_gid);
    /* bring in argc, envp, and exec */

    argc = msg_getInt(msg);
    argv = (char **) malloc(argc + 1 * sizeof(char*));
    for ( i = 0; i < argc; ++i) {
        char b[1024];
        msg_getString(msg, b, sizeof(b)-1);
        argv[i] = strdup(b);
    }
    argv[i] = NULL;

    envc = msg_getInt(msg);
    envp = (char **) malloc(envc + 1 * sizeof(char*));
    for ( i = 0; i < envc; ++i) {
        char b[1024];
        msg_getString(msg, b, sizeof(b)-1);
        envp[i] = strdup(b);
    }
    envp[i] = NULL;

    msg_initResponce(msg, 0);
    msg_sendmsg(msg, privmand_fd, "execAsProcess(sendmsg)");

    execve(program, argv, envp);

    boom("execAsProcess(execve)");
}


/* Is this client allowed to make this TYPE of request.  For
 * things with finer grained permissions, do that kind of check
 * in the request handler.
 */
static bool validRequest(enum commands c) {
    if (config == NULL)
        return false;
    switch (c) {
    case CMD_OPEN:
    case CMD_EXEC_AS:
        return true; /* just deal with it later */
    case CMD_RERUN_AS:
        return config->rerunas;
    case CMD_BIND:
        return !config->bind_port.empty();
    case CMD_PAM_START:
    case CMD_PAM_AUTHENTICATE:
    case CMD_PAM_ACCT_MGMT:
    case CMD_PAM_END:
    case CMD_PAM_SETCRED:
    case CMD_PAM_OPEN_SESSION:
    case CMD_PAM_CLOSE_SESSION:
    case CMD_PAM_GET_ITEM:
    case CMD_PAM_SET_ITEM:
    case CMD_PAM_GETENV:
    case CMD_PAM_PUTENV:
    case CMD_PAM_CHAUTHTOK:
        return config->auth;
    case CMD_SETUID:
        return false;
    case CMD_FORK:
    case CMD_EXIT:
    case CMD_DAEMON:
        return config->pfork;
    }
    return false;
}

/* Go to the server.  This function is the root of the server.  It
 * initializes things, then goes into a loop listening for client
 * requests.
 */
static void control_loop(void) {
    message_t  *msg = msg_new();
    int         readlen = 0;

    /* Loop on incoming messages
     * child_pid test is for "RERUN_AS"
     */
    while (child_pid != 0 && (readlen = msg_recvmsg(msg, privmand_fd)) > 0) {
        enum commands c;
        c = (enum commands)msg_getInt(msg);

        if (!validRequest(c)) {
            msg_initResponce(msg, -EPERM);
            syslog(LOG_NOTICE, "Unknown request '%c'", (char)c);
            msg_sendmsg(msg, privmand_fd, "control_loop(validRequest)");
            continue;
        }
        /* The handler function for a given request does the responce. */
        void (*fnptr)(message_t*) = function_map[c];
        if (fnptr == NULL) {
            fprintf(stderr, "libprivman: bad command (c = %c)\n", c);
            boom("control_loop(unknown command)");
        }
        /* Call the handler */
        fnptr(msg);

        msg_clear(msg);
    }
    msg_delete(msg);

    if (readlen < 0)
        boom("recvmsg");
}

#define PAM_SIMPLE_HANDLER(name, method)                                \
static void name(message_t *msg) {                                      \
    pamSimpleFunc(msg, method);                                         \
}

PAM_SIMPLE_HANDLER(pamAuthenticate,     pam_authenticate)
PAM_SIMPLE_HANDLER(pamAcctMgmt,         pam_acct_mgmt)
PAM_SIMPLE_HANDLER(pamEnd,              pam_end)
PAM_SIMPLE_HANDLER(pamSetcred,          pam_setcred)
PAM_SIMPLE_HANDLER(pamOpenSession,      pam_open_session)
PAM_SIMPLE_HANDLER(pamCloseSession,     pam_close_session)
PAM_SIMPLE_HANDLER(pamChauthtok,        pam_chauthtok)
PAM_SIMPLE_HANDLER(pamFailDelay,     (int(*)(pam_handle_t*,int))pam_fail_delay)

void privman_serv_init(void)
{
    function_map[ CMD_OPEN              ] = openFile;
    function_map[ CMD_BIND              ] = bindPort;

    function_map[ CMD_PAM_START         ] = pamStart;
    function_map[ CMD_PAM_GET_ITEM      ] = pamGetItem;
    function_map[ CMD_PAM_SET_ITEM      ] = pamSetItem;
    function_map[ CMD_PAM_AUTHENTICATE  ] = pamAuthenticate;
    function_map[ CMD_PAM_ACCT_MGMT     ] = pamAcctMgmt;
    function_map[ CMD_PAM_END           ] = pamEnd;
    function_map[ CMD_PAM_SETCRED       ] = pamSetcred;
    function_map[ CMD_PAM_SETCRED       ] = pamSetcred;
    function_map[ CMD_PAM_OPEN_SESSION  ] = pamOpenSession;
    function_map[ CMD_PAM_CLOSE_SESSION ] = pamCloseSession;
    function_map[ CMD_PAM_CHAUTHTOK     ] = pamChauthtok;
    function_map[ CMD_PAM_FAIL_DELAY    ] = pamFailDelay;

    function_map[ CMD_FORK              ] = forkProcess;
    function_map[ CMD_EXIT              ] = exitServer;
    function_map[ CMD_DAEMON            ] = daemonProcess;

    function_map[ CMD_RERUN_AS          ] = rerunAsProcess;
    function_map[ CMD_EXEC_AS           ] = execAsProcess;


    /* Syslog init. */
    openlog("privman", LOG_PID, LOG_AUTHPRIV);


    /* And the server spends most of its time in here. */
    control_loop();

    /* Two ways we get here
     * 1) CMD_EXIT
     * 2) CMD_RERUN_AS
     *
     * RERUN_AS requires magic.  This is a second child of the
     * privmand server, so we want to call back out of priv_init()
     */
    /* If we are not the root parent, exit here so that only
     * one privmand process returns back to priv_init() in
     * priv_client.cc to wait on the main child.
     *
     * This is either CMD_FORK or CMD_DAEMON.  Either
     */
    if (child_pid != 0) { 
        /* CMD_EXIT */
        if (p_wait_on_child == false)
            _exit(0);
        else {
            int status;
            waitpid(child_pid, &status, 0);
            if (WIFEXITED(status))
                _exit(WEXITSTATUS(status));
            else
                _exit(EXIT_FAILURE);
        }
    }
}
