/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1997-1999 Yutaka Sato
Copyright (c) 1997-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
//////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	windows.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	970202	created
//////////////////////////////////////////////////////////////////////#*/

#include <stdio.h>
#include "ystring.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>

int curLogFd();
int openNull(int);
void strfree(void*);
char *stralloc(const char*);
typedef struct sockaddr *SAP0;
void socklog(PCStr(where),int fd,const SAP0 addr,int rcode,int csock);

void env2arg(PCStr(prefix));
void DO_INITIALIZE(int ac,const char *av[]);
void DO_FINALIZE(int code);
void DO_STARTUP(int ac,const char *av[]);
int isWindows95();
int porting_dbg(PCStr(fmt),...);

int WIN_MTU = 1024;
static int tmpLogFd = -1;
int LastCpid;
int errorECONNRESET;
const char *BINSHELL = "/bin/sh";

#include "config.h"
#include "log.h"
#define LE (LOGLEVEL<0) ? 0:porting_dbg
#define LT (LOGLEVEL<1) ? 0:porting_dbg
#define LV (LOGLEVEL<2) ? 0:porting_dbg
#define LW (LOGLEVEL<4) ? 0:porting_dbg
#define DBGWRITE	write
#define SYST		"UNIX"

#ifdef _MSC_VER
#include "vsignal.h"
#undef DBGWRITE
#define DBGWRITE	__write
#undef SYST
#define SYST		"WIN"

#ifdef __EMX__
#undef SYST
#define SYST		"OS/2"
#include <os2emx.h>
#endif

#define WINSOCKVER	"2.2"
const char *WinSockVer = WINSOCKVER;
#include "ywinsock.h"
#include <sys/timeb.h> /* for ftime() */
#include <share.h> /* _SH_DENYNO */

#include <shlobj.h>
#include <stdio.h>
/*
these are included in "ystring.h"
#include <io.h>
#include <process.h>
#include <direct.h>
*/

int getppid();
int pipe(int*);
int SocketOf(int fd);

int getserversock();
int getclientsock();
int Lseek(int fd,int off,int wh);
int wait3(int *statusp,int options,void *rusage);
static int sessionfd();
static void deltmpfiles();
static int acceptsocket(int asock);
static int bindsocket(int sock,int *portp);
static int connectsocket(int sock,int port);

static int sends(int fd,PCStr(buf),int len,int flags)
{	int rem,wcc,mtu,wcc1;
	int xerr;

	rem = len;
	wcc = 0;
	while( 0 < rem ){
		mtu = rem;
		if( WIN_MTU && WIN_MTU < mtu )
			mtu = WIN_MTU;

		wcc1 = send(fd,buf+wcc,mtu,flags);
		if( wcc1 <= 0 ){
			xerr = WSAGetLastError();
			if( 1 < LOGLEVEL || xerr != WSAEWOULDBLOCK )
			LE("send(%d) = %d+%d errno=%d",len,wcc1,wcc,xerr);
			if( wcc == 0 )
				wcc = -1;
			break;
		}
		wcc += wcc1;
		rem -= wcc1;
	}
	return wcc;
}

typedef union {
 struct sockaddr_in	_sin;
	char		_sab[64]; /**/
} VSAddr;
typedef struct sockaddr *SAP;
/* vsocket.h */
int VSA_atosa(VSAddr *sa,int port,PCStr(addr));
int VSA_port(VSAddr *sap);
const char *VSA_ntoa(VSAddr *sap);
char *VSA_xtoap(VSAddr *sa,PVStr(buf),int siz);
int start_service(int ac,const char *av[]);
void WINthread();
void setBinaryIO();

#define MAXHANDLE	4096
#define MIS		8 /* the max. number of inherited sockets */

static int __write(int fd,PCStr(buf),int size)
{	int oh;
	int ok;
	unsigned long wcc;

	oh = _get_osfhandle(fd);
	ok = WriteFile((HANDLE)oh,buf,size,&wcc,NULL);
	return wcc;
}
static int __read(int fd,char *buf,int size)
{	int oh;
	int ok;
	unsigned long rcc;
	
	oh = _get_osfhandle(fd);
	ok = ReadFile((HANDLE)oh,buf,size,&rcc,NULL);
	return rcc;
}

/*######## CLOCK ########*/
int gettimeofday(struct timeval *tp, struct timezone *tzp)
{	struct timeb timeb;

	ftime(&timeb);
	tp->tv_sec = timeb.time;
	tp->tv_usec = timeb.millitm * 1000;
	LW("gettimeofday(%x,%x) = [%d,%d]",tp,tzp,tp->tv_sec,tp->tv_usec);
	return 0;
}

/*######## SIGNAL ########*/
int sigsetmask(int mask){
	LV("sigsetmask() not supported");
	return -1;
}
int sigblock(int mask){
	LV("sigblock() not supported");
	return -1;
}
int kill(int pid,int sig){
	HANDLE ph;
	unsigned long int xcode;

	if( sig == 0 ){
		errno = 0;
		ph = OpenProcess(PROCESS_QUERY_INFORMATION,0,pid);
		if( ph == NULL ){
			errno = ESRCH;
			return -1;
		}
		CloseHandle(ph);
		return 0;
	}

	ph = (HANDLE)pid;
	/* it should be ph = OpenProcess(pid), but because the pid
	 * argument given is a process handle which is got at spawnvp()...
	 */

	if( !GetExitCodeProcess(ph,&xcode) )
	{	LE("kill(%d,%d) = -1, failed GetExitCodeProcess()",pid,sig);
		return -1;
	}
	if( xcode != STILL_ACTIVE )
	{	LE("kill(%d,%d) = -1, not active (xcode=%d)",pid,sig,xcode);
		return -1;
	}
	if( sig == SIGTERM ){
		if( TerminateProcess(ph,0) )
			return 0;
		else{
			LE("kill(%d,%d) = -1, failed TerminateProcess()",pid,sig);
			return -1;
		}
	}
	LE("kill(%d,%d) not supported",pid,sig);
	return -1;
}

unsigned int alarm(unsigned int sec){
	LE("alarm(%d) not supported",sec);
	return 0;
}
void sleep(unsigned int sec){
	LW("sleep(%d)",sec);
	_sleep(sec*1000);
}

/*######## USER and GROUP ########*/
#include "passwd.h"
static struct passwd passwd;
void setpwent(){
	LV("setpwent()");
}
void endpwent(){
	LV("endpwent()");
}
struct passwd *getpwuid(int uid){
	LV("getpwuid(%d)",uid);
	passwd.pw_name = "windows";
	passwd.pw_passwd = "";
	passwd.pw_uid = 0;
	passwd.pw_gid = 0;
	passwd.pw_quota = 0;
	passwd.pw_comment = "";
	passwd.pw_gecos = "Windows";
	passwd.pw_dir = "/users/default";
	passwd.pw_shell = "";
	return &passwd;
}
int getgrnam(PCStr(name))
{
	LV("getgrnam(%s)",name);
	return 0;
}
int getpwnam(PCStr(name))
{
	LV("getpwnam(%s)",name);
	return 0;
}
int getgrgid(int gid){
	LV("getgrgid(%d)",gid);
	return 0;
}

/* ######## PROCESS OWNER ######## */
int getegid(){
	LW("getegid() = 0");
	return 0;
}
int geteuid(){
	LW("geteuid() = 0");
	return 0;
}
int getuid(){
	LW("getuid() = 0");
	return 0;
}
int getgid(){
	LW("getgid() = 0");
	return 0;
}
int setuid(int uid){
	int rcode;
	if( uid == 0 )
		rcode = 0;
	else	rcode = -1;
	LW("setuid(%d) = %d",uid,rcode);
	return rcode;
}
int setgid(int gid){
	int rcode;
	if( gid == 0 )
		rcode = 0;
	else	rcode = -1;
	LW("setgid(%d) = %d",gid,rcode);
	return rcode;
}
int initgroups(const char *user,int group){
	LW("initgroups() = -1");
	return -1;
}

/*######## FILE ########*/

#include <WINDOWS.H>
#include <WINBASE.H>

static HANDLE _SELF;
static HANDLE SELF()
{
	if( _SELF == 0 ){
		_SELF = OpenProcess(PROCESS_ALL_ACCESS,0,getpid());
		LV("PID=%d SELF_HANDLE=%d",getpid(),_SELF);
	}
	return _SELF;
}

int duphandle(HANDLE sproc,HANDLE shandle,HANDLE dproc,int closesrc,int noinherit)
{	HANDLE dhandle;
	DWORD access;
	BOOL inherit;
	DWORD options;

	access = 0;
	inherit = !noinherit; /* FALSE may be good for some sockets... */
	options = DUPLICATE_SAME_ACCESS;
	if( closesrc )
		options |= DUPLICATE_CLOSE_SOURCE;

	if( sproc == NULL ) sproc = SELF();
	if( dproc == NULL ) dproc = SELF();

	dhandle = NULL;
	DuplicateHandle(sproc,shandle,dproc,&dhandle,
		access,inherit,options);

	if( dhandle == NULL )
	LT("DUPHANDLE: %d,%d => %d,%d",sproc,shandle,dproc,dhandle);
	return (int)dhandle;
}

WITH_symlink(){ return 1; }
int symlink(PCStr(dst),PCStr(src))
{	HRESULT Rc;
	WCHAR wsrc[MAX_PATH]; /**/
	IShellLink* Sl;
	IPersistFile* pf;
	int rcode = -1;
	CStr(dstx,MAX_PATH);

	CoInitialize(NULL);
	MultiByteToWideChar(CP_ACP,0,src,-1,wsrc,MAX_PATH);

	Rc = CoCreateInstance(CLSID_ShellLink,NULL,
		CLSCTX_INPROC_SERVER,IID_IShellLink,(LPVOID *)&Sl);
	if( FAILED(Rc) )
		goto EXIT;

	if( strchr(dst,':') == 0 ){
		if( getcwd(dstx,sizeof(dstx)) == dstx )
		if( dstx[0] != 0 && dstx[1] == ':' )
		{
			Xstrcpy(DVStr(dstx,2),dst);
			dst = dstx;
		}
	}

	Rc = Sl->SetPath(dst);

	Sl->SetDescription("");
	Rc = Sl->QueryInterface(IID_IPersistFile,(LPVOID*)&pf);
	if( SUCCEEDED(Rc) ){
		Rc = pf->Save(wsrc,FALSE);
		pf->Release();
		rcode = 0;
	}
	Sl->Release();

EXIT:	CoUninitialize();
	return rcode;
}

int readlink(PCStr(src),char dst[],unsigned int siz)
{	HRESULT Rc;
	WCHAR wsrc[MAX_PATH]; /**/
	IShellLink* Sl;
	IPersistFile* Pf;
	WIN32_FIND_DATA Fd;
	int rcode = -1;

	CoInitialize(NULL);
	MultiByteToWideChar(CP_ACP,0,src,-1,wsrc,MAX_PATH);
	*dst = 0;
	Rc = CoCreateInstance(CLSID_ShellLink,NULL,
		CLSCTX_INPROC_SERVER,IID_IShellLink,(LPVOID *)&Sl);
	if( FAILED(Rc) )
		goto EXIT;

	Rc = Sl->QueryInterface(IID_IPersistFile,(void**)&Pf);
	if( SUCCEEDED(Rc) ){
		Rc = Pf->Load(wsrc,STGM_READ);
		if( SUCCEEDED(Rc) ){
			Rc = Sl->GetPath(dst,siz,
				(WIN32_FIND_DATA *)&Fd,SLGP_SHORTPATH);
			if( SUCCEEDED(Rc) )
				rcode = 0;
		}
		Pf->Release();
	}
	Sl->Release();

EXIT:	CoUninitialize();
	return rcode;
}

/*######## SOCKET ########*/
static WORD wVersionRequested;
static WSADATA wsaData;
static struct {
	int	qver;
	int	code;
	int	errno;
} wsaStat;
void socket_init(){
	int err;
	int qvmaj,qvmin;

	if( wsaData.szDescription[0] != 0 )
		return;
	qvmaj = 1;
	qvmin = 0;
	sscanf(WinSockVer,"%d.%d",&qvmaj,&qvmin);
	wVersionRequested = MAKEWORD(qvmaj,qvmin);
	/*
	wVersionRequested = MAKEWORD(1,1);
	*/
	err = WSAStartup(wVersionRequested,&wsaData);
	wsaStat.qver = wVersionRequested;
	wsaStat.code = err;
	wsaStat.errno = WSAGetLastError();
}
static void dumpWSAstat()
{	int hi,nh,nhi,nsh,nshi,len,type;
	CStr(sockets,MAXHANDLE*8);
	refQStr(sp,sockets); /**/
	unsigned long int flags;

	if( LOGLEVEL < 4 )
		return;

	LV("WinSock option \"-WSAV:%s\" ReqVer:%x Result:%x Errno:%d",
		WinSockVer,wsaStat.qver,wsaStat.code,wsaStat.errno);
	LV("Version:%x HighVersion:%x Description:%s",
		wsaData.wVersion,wsaData.wHighVersion,wsaData.szDescription);
	LV("SystemStatus:%s MaxSockets:%d MaxUdpDg:%d",
		wsaData.szSystemStatus,wsaData.iMaxSockets,wsaData.iMaxUdpDg);
	if( wsaData.lpVendorInfo )
	LV("VenderInfo:%s",wsaData.lpVendorInfo);

	nh = nsh = nhi = nshi = 0;
	for( hi = 0; hi < MAXHANDLE; hi++ ){
		flags = 0;
		if( GetHandleInformation((HANDLE)hi,&flags) == 0 )
			continue; /* not available on Win95 */
		nh++;
		if( flags & HANDLE_FLAG_INHERIT )
			nhi++;
		if( hi < 128 ){
			/* getsockopt() blocks for lower handle ??? */
			continue;
		}

		len = sizeof(type);
		if( getsockopt(hi,SOL_SOCKET,SO_TYPE,(char*)&type,&len) == 0 ){
			nsh++;
			if( flags & HANDLE_FLAG_INHERIT )
				nshi++;
			sprintf(sp," %d",hi);
			sp += strlen(sp);
			if( type != 1 ){
				sprintf(sp,"/%d",type);
				sp += strlen(sp);
			}
			if( (flags & HANDLE_FLAG_INHERIT) == 0 ){
				sprintf(sp,"#");
				sp += strlen(sp);
			}
		}
	}
	setVStrEnd(sp,0);
	if( *sockets )
	LV("Sockets(%d/%d/%d:%d/%d):%s",nshi,nsh,FD_SETSIZE,nhi,nh,sockets);
}
static void sockInfo()
{
	if( getppid() == 0 ){
		LT("WINSOCK INFO.");
		LT("Desc: %s",wsaData.szDescription);
		LT("Stat: %s",wsaData.szSystemStatus);
		LT("Maxs: %d, Vend: %s",wsaData.iMaxSockets,wsaData.lpVendorInfo);
	}
}
static int ISSOCK(int sock)
{	int len,type;

	len = sizeof(type);
	if( getsockopt(sock,SOL_SOCKET,SO_TYPE,(char*)&type,&len) == 0 )
		return 1 <= type;
	return 0;
}
int Inet_pton(int af,const char *src,void *dst);
int inet_pton(int af,const char *src,void *dst){ return Inet_pton(af,src,dst); }
struct hostent *gethostbyname2(const char *name,int af){
	return 0;
}

static int NO_SOCKOPEN = 1;
/* bad if this value is zero in CHARCODE filter from FTP/HTTP on WindowsNT
 * where the client SOCKET is inherited twice...
 */

static short open_osfhandle[MAXHANDLE]; /* get_osfhandle(fd) emulation */
static short isdup_handles[MAXHANDLE];
static short opened_sockcnt[MAXHANDLE];

static int coes[64];
static int coen;
int setCloseOnExecSocket(int fd)
{	int sock;
	int ci;

	if( 0 <= fd )
	if( sock = SocketOf(fd) ){
		for( ci = 0; ci < coen; ci++ )
			if( coes[ci] == sock )
				return 0;

		LV("set CloseOnExecSocket[%d] = %d/%d",coen,sock,fd);
		coes[coen++] = sock;
		return 0;
	}
	return -1;
}
int clearCloseOnExecSocket(int fd)
{	int sock;
	int ci,cj;

	if( 0 <= fd )
	if( sock = SocketOf(fd) ){
		for( ci = 0; ci < coen; ci++ ){
			if( coes[ci] == sock ){
				for( cj = ci; cj < coen; cj++ )
					coes[cj] = coes[cj+1];
				LV("clr CloseOnExec[%d] = %d/%d",ci,sock,fd);
				coen--;
				return 0;
			}
		}
	}
	return -1;
}
int setInheritance(int ifd,int inherit)
{	HANDLE ih,nh;
	int nfd;

	ih = (HANDLE)_get_osfhandle(ifd);
	nh = (HANDLE)duphandle(0,ih,0,0,!inherit);
	close(ifd);
	nfd = _open_osfhandle((int)nh,0);
	if( nfd != ifd ){
		dup2(nfd,ifd);
		close(nfd);
	}
	LE("setInferitance(%d,%d) %d %d %d",ifd,inherit,ih,nh,nfd);
	return ifd;
}

static int issock(int fd)
{	struct stat st0,st1;
	int bi,diff;

	if( fstat(fd,&st1) != 0 ){
		LV("issock(%d) CANT OPEN1",fd);
		return 0;
	}
	if( fstat(sessionfd(),&st0) != 0 ){
		LE("######## issock(%d) CANT OPEN2",fd);
		exit(1);
	}
	if( st0.st_mode != st1.st_mode ){
		LV("DIFF %d:%d %d:%d",fd,st1.st_mode,sessionfd(),st0.st_mode);
		return 0;
	}
	return 1;
}

static int closeSocket(int fd,int sock)
{	int rcode1,rcode2,isdup;

	clearCloseOnExecSocket(fd);

	if( fd < 0 ){
		rcode1 = -1;
		isdup = 0;
	}else{
		rcode1 = _close(fd);
		isdup = isdup_handles[fd];
		isdup_handles[fd] = 0;
		open_osfhandle[fd] = 0;
	}

	if( 0 < sock )
	if( 0 < opened_sockcnt[sock] )
		opened_sockcnt[sock] -= 1;

	if( sock < 0 )
		rcode2 = -1;
	else
	if( 0 < opened_sockcnt[sock] )
		rcode2 = 0;
	else{
		rcode2 = closesocket(sock);
	}

	LV("-- SOCKET %d*(%3d/%2d) = %d,%d close() dup=%d",
		opened_sockcnt[sock],sock,fd,rcode1,rcode2,isdup);

	return rcode1;

}
static int openSocket(int tfd,int sock,int isdup,PCStr(wfmt),...)
{	int fd;
	CStr(msg,256);
	VARGS(2,wfmt);

	sprintf(msg,wfmt,va[0],va[1]);
	if( sock <= 0 ){
		LE("%s -- openSocket(%d) bad SOCKET handle",msg,sock);
		return -1;
	}
	fd = -1;
	if( !NO_SOCKOPEN ){
		/*
		 * how should it be treated when 0 <= tfd ... dup2() case ?
		 */
		fd = _open_osfhandle(sock,0);
		if( fd < 0 ){
			LE("NO OPEN_OSFHANDLE(SOCKET)");
			NO_SOCKOPEN = 1;
		}
	}
	if( NO_SOCKOPEN ){
		if( 0 <= tfd ){
			if( _dup2(sessionfd(),tfd) == 0 )
				fd = tfd;
			else{
				LE("%s -- failed dup2(x,%d)\n",tfd);
				return -1;
			}
		}else	fd = _dup(sessionfd());
		open_osfhandle[fd] = sock;
	}
	if( isdup ){
		isdup_handles[fd] = sock;
		opened_sockcnt[sock] += 1;
	}else{
		isdup_handles[fd] = 0;
		opened_sockcnt[sock]  = 1;
	}

	LV("-- SOCKET %d*(%3d/%2d) opened %s",
		opened_sockcnt[sock],sock,fd,msg);
	return fd;
}
int SocketOf(int fd)
{	int sock,oh;

	if( fd < 0 || MAXHANDLE <= fd ){
		LW("SocketOf(%d)?",fd);
		return 0;
	}

	oh = _get_osfhandle(fd);

	if( NO_SOCKOPEN ){
		if( 0 < (sock = open_osfhandle[fd]) )
		if( opened_sockcnt[sock] <= 0 || !issock(fd) ){
			open_osfhandle[fd] = 0;
			isdup_handles[fd] = 0;
			if( 0 < opened_sockcnt[sock] ){
				opened_sockcnt[sock] -= 1;
				if( opened_sockcnt[sock] <= 0 )
					closesocket(sock);
			}
			LV("-- SOCKET %d*(%3d/%2d) closed (detected)",
				opened_sockcnt[sock],sock,fd);
			sock = 0;
		}
	}else	sock = oh;

	LW("SocketOf(%d/%d) ISSOCK=%d",sock,fd,ISSOCK(sock));

	if( sock && !ISSOCK(sock) )
		sock = 0;

	return sock;
}
int getsockHandle(int fd){ return (0 < SocketOf(fd)) ? SocketOf(fd) : -1; }

int _SOCKET(int dom,int type,int proto)
{	int sock,fd;
	int wsaerrno;

	socket_init();
	sock = socket(dom,type,proto);
	if( sock < 0 ){
		if( dom == AF_UNIX )
			LV("socket(AF_UNIX) = -1, NOT SUPPORTED");
		wsaerrno = WSAGetLastError();
		if( wsaerrno == WSAEAFNOSUPPORT && dom == AF_INET6 ){
		LE("#### IPv6 is disabled or unsupported on this host ####");
		}
		return sock;
	}
	for( fd = 0; fd < MAXHANDLE; fd++ ){
		if( open_osfhandle[fd] == sock ){
			LE("dangling socket descriptor: %d/%d",sock,fd);
			open_osfhandle[fd] = -1;
		}
	}
	fd = openSocket(-1,sock,0,"socket()");
	LV("socket() = %d/%d",sock,fd);
	return fd;
}
int _BIND(int fd,SAP addr,int len)
{	int sock = SocketOf(fd);
	int rcode;
	int wserrno;

	rcode = bind(sock,addr,len);
	wserrno = WSAGetLastError();
	if( wserrno ){
		LV("bind(%d/%d) = %d errno=(%d <- %d)",sock,fd,rcode,
			errno,wserrno);
		errno = wserrno;
	}else
	LV("bind(%d/%d) = %d",sock,fd,rcode);
	socklog("bind",fd,addr,rcode,sock);
	return rcode;
}
int _LISTEN(int fd,int nblog)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = listen(sock,nblog);
	LV("listen(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
int _ACCEPT(int fd,SAP addr,int *len)
{	int sock = SocketOf(fd);
	int csock,nfd;

	csock = accept(sock,addr,len);
	if( 0 <= csock )
		nfd = openSocket(-1,csock,0,"accept(%d)",fd);
	else	nfd = -1;
	LV("accept(%d/%d) = %d/%d",sock,fd,csock,nfd);
	socklog("accept",fd,addr,nfd,sock);
	return nfd;
}
int _CONNECT(int fd,const SAP addr,int len)
{	int sock = SocketOf(fd);
	int rcode;
	int wserrno;

	rcode = connect(sock,addr,len);
	wserrno = WSAGetLastError();
	if( wserrno ){
		LV("connect(%d/%d) = %d errno=(%d <- %d)",sock,fd,rcode,
			errno,wserrno);
		errno = wserrno;
	}else	LV("connect(%d/%d) = %d",sock,fd,rcode);
	socklog("connect",fd,addr,rcode,sock);
	return rcode;
}
int _SHUTDOWN(int fd,int how)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = shutdown(sock,how);
	LV("shutdown(%d/%d,%d) = %d",sock,fd,how,rcode);
	return rcode;
}
int _SELECT(int sz,struct fd_set *x,struct fd_set *y,struct fd_set *z,struct timeval *tv)
{	int rcode;
	int wserrno;

	LV("select(%d,%d) start",sz,tv?tv->tv_sec:0);
	rcode = select(sz,x,y,z,tv);
	wserrno = WSAGetLastError();
	if( rcode < 0 )
	{
	LE("select() = %d [errno=(%d / %d)]",rcode,errno,wserrno);
		if( errno == EBADF || wserrno == WSAENOTSOCK ){
			LE("#### ERROR: select() applied to non-SOCKET handle");
			_sleep(200);
		}
	}
	else
	LV("select() = %d errno=(%d / %d)",rcode,errno,wserrno);
	return rcode;
}
int _SEND(int fd,const void *buf,unsigned int len,int flags)
{	int sock = SocketOf(fd);
	int wcc;

	wcc = sends(sock,(char*)buf,len,flags);
	LW("send(%d/%d) = %d",sock,fd,wcc);
	return wcc;
}
int _RECV(int fd,void *buf,unsigned int len,int flags)
{	int sock = SocketOf(fd);
	int rcc;

	rcc = recv(sock,(char*)buf,len,flags);
	LW("recv(%d/%d) = %d",sock,fd,rcc);
	return rcc;
}
int _SENDTO(int fd,const void *buf,unsigned int len,int flags,SAP to,unsigned int tolen)
{	int sock = SocketOf(fd);
	int wcc;

	wcc = sendto(sock,(char*)buf,len,flags,to,tolen);
	LW("sendto(%d/%d) = %d",sock,fd,wcc);
	return wcc;
}
int _RECVFROM(int fd,void *buf,unsigned int len,int flags,SAP from,int *fromlen)
{	int sock = SocketOf(fd);
	int rcc;
	int xerr;

	errorECONNRESET = 0;
	rcc = recvfrom(sock,(char*)buf,len,flags,from,fromlen);
	LW("recvfrom(%d/%d) = %d",sock,fd,rcc);
	if( rcc < 0 ){
		xerr = WSAGetLastError();
		if( xerr == WSAECONNRESET )
			errorECONNRESET = 1;
	}
	return rcc;
}
int _GETSOCKNAME(int fd,SAP addr,int *len)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = getsockname(sock,addr,len);
	LW("getsockname(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
int _GETPEERNAME(int fd,SAP addr,int *len)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = getpeername(sock,addr,len);
	LW("getpeername(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
int _SETSOCKOPT(int fd,int lev,int op,const void *val,int len)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = setsockopt(sock,lev,op,(char*)val,len);
	LW("setsockopt(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
int _GETSOCKOPT(int fd,int lev,int op,void *val,int *len)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = getsockopt(sock,lev,op,(char*)val,len);
	LW("getsockopt(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
int _GETHOSTNAME(char *name,unsigned int size)
{
	socket_init();
	return gethostname(name,size);
}

int socketpair(int d,int type,int protocol,int fsv[2])
{	VSAddr ina;
	int inalen,port;
	int sv[2];
	int asock,len;
	int rcode = 0;

	sv[0] = sv[1] = -1;
	asock = socket(d,type,protocol);
	if( asock < 0 )
		return -1;

	inalen = VSA_atosa(&ina,0,"127.0.0.1");

	rcode |= bind(asock,(SAP)&ina,inalen);
	rcode |= listen(asock,2);

	len = sizeof(VSAddr);
	getsockname(asock,(SAP)&ina,&len);
	port = VSA_port(&ina);

	sv[1] = socket(d,type,protocol);
	inalen = VSA_atosa(&ina,port,"127.0.0.1");
	rcode |= connect(sv[1],(SAP)&ina,inalen);

	len = sizeof(VSAddr);
	sv[0] = accept(asock,(SAP)&ina,&len);
	closesocket(asock);

	fsv[0] = openSocket(-1,sv[0],0,"socketpair(0)");
	fsv[1] = openSocket(-1,sv[1],0,"socketpair(1)");

	LV("socketpair() = %d [%d/%d,%d/%d]",rcode,sv[0],fsv[0],sv[1],fsv[1]);
	return rcode;
}

int setNonblockingSocket(int fd,unsigned long int on)
{	int sock;
	int rcode;

	if( sock = SocketOf(fd) ){
		rcode = ioctlsocket((SOCKET)sock,FIONBIO,&on);
		LE("setNonblockingSocket(%d,%d)=%d",fd,on,rcode);
		return rcode;
	}
	return -1;
}
void setNonblockingPipe(int pv[])
{	HANDLE phandle;
	unsigned long int mode;
	int ok;

	mode = PIPE_READMODE_BYTE | PIPE_NOWAIT;
	phandle = (HANDLE)_get_osfhandle(pv[0]);
	ok = SetNamedPipeHandleState(phandle,&mode,NULL,NULL);
	LE("setNonblockingPipe(%d/%d) ok=%d",phandle,pv[0],ok);
}
static int ISPIPE(int pfd)
{	int ok;
	unsigned long int ninst;
	HANDLE phandle;

	phandle = (HANDLE)_get_osfhandle(pfd);
	if( GetFileType(phandle) == FILE_TYPE_PIPE )
		return 1;

	ninst = -1;
	ok = GetNamedPipeHandleState(phandle,NULL,&ninst,NULL,NULL,NULL,0);
	if( ok == TRUE && 0 < ninst )
		return 1;
	else	return 0;
}
int pollPipe(int pfd,int slpmsec)
{	HANDLE phandle;
	int rem,slept,sleep1,s1;
	unsigned long int nready;

	if( SocketOf(pfd) )
		return -1;
	if( !ISPIPE(pfd) )
		return -1;

	phandle = (HANDLE)_get_osfhandle(pfd);
	if( isWindows95() ){
		/* PeekNamedPipe of Win95/98 seems never fail on EOF ... */
		if( 15000 < slpmsec ){
			LT("pollPipe(%d,%d->15000) for Win95/98...",slpmsec);
			slpmsec = 15000;
		}
	}
	for( slept = 0; slpmsec == 0 || slept <= slpmsec; slept += s1 ){
		nready = -1;
		if( PeekNamedPipe(phandle,NULL,0,NULL,&nready,NULL) == FALSE )
			return -1;

		if( nready != 0 )
			return nready;
		if( slpmsec != 0 && slpmsec <= slept )
			break;

		rem = slpmsec - slept;
		if( slept <  500 ) sleep1 =   5; else
		if( slept < 5000 ) sleep1 =  50; else
				   sleep1 = 500;

		if( slpmsec == 0 || sleep1 < rem )
			s1 = sleep1;
		else	s1 = rem;
		/*
		usleep(s1*1000);
		*/
		_sleep(s1);
	}
	return 0;
}

/* ######## NIS ######## */
int yp_get_default_domain(char **domp)
{
	LV("yp_get_default_domain() = -1");
	*domp = NULL;
	return -1;
}
int yp_match(PCStr(dom),PCStr(map),PCStr(name),int len,char **vp,int *vlen)
{
	LV("yp_match() = -1");
	return -1;
}

/* ######## STDIO ######## */

int dup2(int fd1,int fd2)
{	int rcode;
	int sock,nsock,nfd;

	LT("dup2(%d/%d/%d,%d/%d/%d)",
		_get_osfhandle(fd1),SocketOf(fd1),fd1,
		_get_osfhandle(fd2),SocketOf(fd2),fd2);
	if( fd1 == fd2
	 || SocketOf(fd2) && SocketOf(fd1) == SocketOf(fd2)
	){
		return fd2;
	}
	/*
	if( close(dup2) == 0 )
		LE("closed(%d) before dup2(%d,%d)",fd2,fd1,fd2);
	*/
	if( close(fd2) == 0 )
		LT("closed(%d) before dup2(%d,%d)",fd2,fd1,fd2);

	if( sock = SocketOf(fd1) ){
		if( NO_SOCKOPEN )
			nsock = sock;
		else	nsock = duphandle(0,(HANDLE)sock,0,0,0);
		nfd = openSocket(fd2,nsock,1,"dup2(%d/%d)",sock,fd2);
	}else{
		rcode = _dup2(fd1,fd2);
		if( rcode == 0 )
			nfd = fd2;
		else	nfd = -1;
	}
	return nfd;
}
int dup(int fd)
{	int sock,nsock,nfd;

	if( sock = SocketOf(fd) ){
		if( NO_SOCKOPEN )
			nsock = sock;
		else	nsock = duphandle(0,(HANDLE)sock,0,0,0);
		nfd = openSocket(-1,nsock,1,"dup(%d/%d)",sock,fd);
	}else	nfd = _dup(fd);
	return nfd;
}
int close(int fd)
{	int sock,rcode;

	if( sock = SocketOf(fd) )
		rcode = closeSocket(fd,sock);
	else	rcode = _close(fd);
	return rcode;
}
int open(PCStr(path),int flag,int mode)
{	int fd;

	if( strcmp(path,"/dev/null") == 0 )
		path = "nul";
	else
	if( strcmp(path,"/dev/tty") == 0 )
		path = "con";
	fd = _open(path,flag,mode);
	LT("open(%s,%d) = %d",path,flag,fd);
	return fd;
}
#define F_SETFD 1
int fcntl(int fd,int cmd,void *arg){
	int fh;
	int flag,ok,rcode;
	int sock;

	fh = _get_osfhandle(fd);
	switch( cmd ){
	    case F_SETFD:
		if( sock = SocketOf(fd) ){
			LT("fcntl(fh=%d/fd=%d/sock=%d) -> CloseOnExecSocket()",
				fh,fd,sock);
			if( arg )
				return setCloseOnExecSocket(sock);
			else	return clearCloseOnExecSocket(sock);
		}
		flag = HANDLE_FLAG_INHERIT;
		ok = SetHandleInformation((HANDLE)fh,flag,arg?0:flag);
		rcode = ok ? 0 : -1;
		LV("fcntl(%d,F_SETFD,%d) = %d",fd,arg,rcode);
		return rcode;
	}
	LE("fcntl(%d) NOT SUPPORTED",fd);
	return -1;
}

int _read(int fd,char *buf,unsigned int size)
{	int rcc;
	int sock;
	HANDLE oh;
	int xerr;

	if( sock = SocketOf(fd) ){
		rcc = recv(sock,buf,size,0);
		if( rcc < 0 )
		LV("-- SOCKET recv(%d/%d,%x,%d,0) = %d",sock,fd,buf,size,rcc);
		if( rcc <= 0 ){
			xerr = WSAGetLastError();
			LE("-- SOCKET recv(%d)=%d error=%d",fd,rcc,xerr);
			if( xerr == WSAENETRESET
			 || xerr == WSAECONNRESET || xerr == WSAECONNABORTED ){
				errno = xerr;
			}
		}
	}else{
		rcc = __read(fd,buf,size);
		if( rcc < 0 )
		LW("-- read(%d,%x,%d) = %d",fd,buf,size,rcc);
	}
	return rcc;
}
int _write(int fd,PCStr(buf),unsigned int size)
{	int wcc;
	int sock;
	int xerr;

	if( sock = SocketOf(fd) ){
		wcc = sends(sock,buf,size,0);
		if( xerr = WSAGetLastError() ){
			if( 1 < LOGLEVEL || xerr != WSAEWOULDBLOCK )
			LE("-- SOCKET send(%d/%d,%x,%d,0) = %d %d",
				sock,fd,buf,size,wcc,xerr);

			if( getenv("SIGPIPE_TERM") )
			if( xerr == WSAENETRESET
			 || xerr == WSAECONNRESET || xerr == WSAECONNABORTED ){
				LE("send() -- cause SIGTERM for SIGPIPE");
				raise(SIGTERM);
			}
		}else
		LV("-- SOCKET send(%d/%d,%x,%d,0) = %d",sock,fd,buf,size,wcc);
	}else{
		wcc = __write(fd,buf,size);
		LW("-- write(%d,%x,%d) = %d",fd,buf,size,wcc);
	}
	return wcc;
}
int (*win_read)(int,char*,unsigned int) = _read;
int (*win_write)(int,const char*,unsigned int) = _write;

static int sleepsock = -1;
int usleep(unsigned int timeout)
{ 	struct timeval tvbuf,*tv;
	struct fd_set mask;
	int port;

	if( timeout ){
		tv = &tvbuf;
		tv->tv_sec  = timeout / 1000000;
		tv->tv_usec = timeout % 1000000;
	}else	tv = NULL;

	if( sleepsock < 0 ){
		sleepsock = bindsocket(-1,&port);
		if( sleepsock < 0 )
			return -1;
	}
	FD_ZERO(&mask);
	FD_SET(sleepsock,&mask);
	return select(128,&mask,NULL,NULL,tv);
}
void Usleep(int usec){
	int msec;
	msec = (usec + 999) / 1000;
	_sleep(msec);
}

static int pollin(int sock,int msec)
{	struct timeval tv;
	struct fd_set mask;

	tv.tv_sec = msec / 1000;
	tv.tv_usec = (msec % 1000) * 1000;
	FD_ZERO(&mask);
	FD_SET(sock,&mask);
	return select(128,&mask,NULL,NULL,&tv);
}

int uname(void *name)
{
	return -1;
}
int Uname(PVStr(name))
{
	if( isWindows95() )
		strcpy(name,"Windows95");
	else	strcpy(name,"WindowsNT");
	return 0;
}
int isWindows95(){ return getpid() < 0; }
int isWindows(){ return 1; }

#include <time.h>
static void utime2ftime(struct timeval *tv,FILETIME *ftime)
{	int year,mon,day,hour,min,sec;
	struct tm *tm;
	WORD ddate,dtime;

	tm = gmtime(&tv->tv_sec);
	year = tm->tm_year + 1900 - 1980;
	mon  = tm->tm_mon  +  1;
	day  = tm->tm_mday;
	hour = tm->tm_hour;
	min  = tm->tm_min;
	sec  = tm->tm_sec / 2;
	ddate = (year <<  8) | (mon << 5) | day;
	dtime = (hour << 11) | (min << 5) | sec;
	DosDateTimeToFileTime(ddate,dtime,ftime);
}
int utimes(PCStr(path),struct timeval *tvp)
{	HANDLE fh;
	FILETIME atime,mtime;
	BOOL ok;
	struct tm *tm;

	fh = CreateFile(path,GENERIC_READ|GENERIC_WRITE,
		0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
	if( fh == NULL )
		return -1;

	utime2ftime(&tvp[0],&atime);
	utime2ftime(&tvp[1],&mtime);

	ok = SetFileTime(fh,NULL,&atime,&mtime);
	CloseHandle(fh);
	if( ok )
		return 0;
	else	return -1;
}

int newtmp(PCStr(path))
{	HANDLE fh;
	int fd;
	SECURITY_ATTRIBUTES secattr;

	/* setting bInheritHandle seems neccessary on Win9X */
	secattr.nLength = sizeof(SECURITY_ATTRIBUTES);
	secattr.lpSecurityDescriptor = NULL;
	secattr.bInheritHandle = 1;

	fh = CreateFile(path,GENERIC_READ|GENERIC_WRITE,
		0,&secattr,CREATE_NEW,
		FILE_ATTRIBUTE_TEMPORARY|FILE_FLAG_DELETE_ON_CLOSE,
		NULL);

	fd = _open_osfhandle((int)fh,0);
	LV("#### newtmp(%s) = %d/%d",path,fd,fh);
	return fd;
}

/* ################### */
void writev(int fd, struct iovec *iov, int iovcnt)
{
	LE("writev() NOT SUPPORTED");
	exit(1);
}
void sigpause(){
	LE("sigpause() NOT SUPPORTED");
	exit(1);
}

/*
 *	parent-child relationship is not implemented in Windows
 */

/* old struction until DeleGate/7.9.3 */
typedef struct {
	int	p_ppid;		/* parent's process handle */
	int	p_pmode;	/* spawn mode from the parent {P_NOWAIT} */
	int	p_inhfd[4];
	int	p_inhsock[4];	/* socket handle connected to the client */
	int	p_pinhsock[4];	/* parent's clsock inherited if in WindowsNT */
	int	p_toparent;	/* socket for sync and exist status code */
	int	p_nosockopen;
	struct {
	int	p_sessionfd[2];
	} p_common;
	int	p_closeOnExecN;
	int	p_closeOnExec[8];
} OldProcEnv;

/* new structure after DeleGate/7.9.5 */
typedef struct {
	int	p_ppid;		/* parent's process handle */
	int	p_pmode;	/* spawn mode from the parent {P_NOWAIT} */
	int	p_toparent;	/* socket for sync and exist status code */
	int	p_nosockopen;
	int	p_inhfd[MIS];
	int	p_inhsock[MIS];	/* socket handle connected to the client */
	int	p_pinhsock[MIS];/* parent's clsock inherited if in WindowsNT */
	struct {
	int	p_sessionfd[2];
	} p_common;
	int	p_closeOnExecN;
	int	p_closeOnExec[MIS];
} ProcEnv;
static void inheritsock(int si,HANDLE cphandle,ProcEnv *cpe);

typedef struct {
	OldProcEnv	p_ope;
	int		p_envlen;
	MStr(		p_envstr,256);
} TransProcEnv;

void setiv(int di[],int ni,int iv);
void copyiv(int di[],int si[],int ni);
char *htoniv(xPVStr(bp),int bsiz,PCStr(what),int ic,int iv[]);
char *ntohiv(PCStr(buf),PCStr(bp),int blen,PCStr(what),int ic,int iv[],int *np);
static ProcEnv MyProcEnv;
static ProcEnv ProcEnv0;

static void ntoo(OldProcEnv *os,ProcEnv *ns)
{
	os->p_ppid = ns->p_ppid;
	os->p_pmode = ns->p_pmode;
	os->p_toparent = ns->p_toparent;
	os->p_nosockopen = ns->p_nosockopen;
	copyiv(os->p_inhfd,ns->p_inhfd,4);
	copyiv(os->p_inhsock,ns->p_inhsock,4);
	copyiv(os->p_pinhsock,ns->p_pinhsock,4);
	copyiv(os->p_common.p_sessionfd,ns->p_common.p_sessionfd,2);
	os->p_closeOnExecN = ns->p_closeOnExecN;
	copyiv(os->p_closeOnExec,ns->p_closeOnExec,8);
}
static void oton(ProcEnv *ns,OldProcEnv *os)
{
	setiv(ns->p_inhsock,MIS,-1);
	setiv(ns->p_pinhsock,MIS,-1);

	ns->p_ppid = os->p_ppid;
	ns->p_pmode = os->p_pmode;
	ns->p_toparent = os->p_toparent;
	ns->p_nosockopen = os->p_nosockopen;
	copyiv(ns->p_inhfd,os->p_inhfd,3);
	copyiv(ns->p_inhsock,os->p_inhsock,3);
	copyiv(ns->p_pinhsock,os->p_pinhsock,3);
	copyiv(ns->p_common.p_sessionfd,os->p_common.p_sessionfd,2);
	ns->p_closeOnExecN = os->p_closeOnExecN;
	copyiv(ns->p_closeOnExec,os->p_closeOnExec,8);
}
static int htonx(ProcEnv *pe,PVStr(buf),int bsiz)
{	refQStr(bp,buf); /**/
	const char *bx;
	int si;

	bx = buf+bsiz;
	sprintf(bp,"(env 4 %d %d %d %d)",
		pe->p_ppid,pe->p_pmode,pe->p_toparent,pe->p_nosockopen);
	bp += strlen(bp);
	bp = htoniv(QVStr(bp,buf),bx-bp,"inf",MIS,pe->p_inhfd);
	bp = htoniv(QVStr(bp,buf),bx-bp,"ins",MIS,pe->p_inhsock);
	bp = htoniv(QVStr(bp,buf),bx-bp,"pin",MIS,pe->p_pinhsock);
	bp = htoniv(QVStr(bp,buf),bx-bp,"ses",2,pe->p_common.p_sessionfd);
	bp = htoniv(QVStr(bp,buf),bx-bp,"cox",pe->p_closeOnExecN,pe->p_closeOnExec);
	return bp - buf;
}
static void ntohx(ProcEnv *pe,PVStr(buf),int blen)
{	char *bp = (char*)buf; /**/
	const char *bx;
	int si,ne;
	int iv[8]; /**/

	setiv(pe->p_inhsock,MIS,-1);
	setiv(pe->p_pinhsock,MIS,-1);

	bx = buf+blen;
	bp = ntohiv(buf,bp,bx-bp,"env",8,iv,&ne);
	pe->p_ppid = iv[0];
	pe->p_pmode = iv[1];
	pe->p_toparent = iv[2];
	pe->p_nosockopen = iv[3];
	bp = ntohiv(buf,bp,bx-bp,"inf",MIS,pe->p_inhfd,&ne);
	bp = ntohiv(buf,bp,bx-bp,"ins",MIS,pe->p_inhsock,&ne);
	bp = ntohiv(buf,bp,bx-bp,"pin",MIS,pe->p_pinhsock,&ne);
	bp = ntohiv(buf,bp,bx-bp,"ses",2,pe->p_common.p_sessionfd,&ne);
	bp = ntohiv(buf,bp,bx-bp,"cox",MIS,pe->p_closeOnExec,&ne);
	pe->p_closeOnExecN = ne;
}

typedef struct {
	int	c_pid;
	int	c_inherr;
} ChildEnv;

static int argc;
static const char **argv;
extern char **environ;

#define SPAWNENV	"SPAWN_ENVIRON"

extern int _fileinfo;

static int children[MAXHANDLE];
static int child_pid[MAXHANDLE];
static int nalive;
static int ntotal;

static int dupsock(int si,int pid,WSAPROTOCOL_INFO *pip);
int spawnvpe(int pmode,PCStr(path),const char *const *av,const char *const environ[])
{	const char *nenviron[256]; /**/
	CStr(spenv,128);
	int ei;
	int ci,cj,dsock;
	int isock[MIS],si;
	HANDLE cphandle;
	int asock,port,csock,wcc,rcc;
	ProcEnv cpe;
	TransProcEnv tpe;
	CStr(ststat,1);
	const char *env;
	int len,nei;
	BOOL ok;
	DWORD xcode1,xcode2;
	int ai,slogtype;

	_fileinfo = -1;

	asock = bindsocket(-1,&port);
	sprintf(spenv,"%s=%d",SPAWNENV,port);
	len = strlen(SPAWNENV) + 1;

	nei = 0;
	for( ei = 0; env = environ[ei]; ei++ ){
		if( elnumof(nenviron)-2 <= ei )
			break;
		if( strncmp(env,spenv,len) != 0 )
			nenviron[nei++] = env;
	}
	nenviron[nei++] = spenv;
	nenviron[nei] = NULL;

	cphandle = (HANDLE)_spawnvpe(pmode,path,av,nenviron);
	if( (int)cphandle == -1 ){
		CStr(cwd,1024);
		getcwd(cwd,sizeof(cwd));
		LE("failed spawn(%s) at %s",path,cwd);
		fprintf(stderr,"failed spawn(%s) at %s\n",path,cwd);
		return (int)cphandle;
	}

	xcode1 = -1;
	ok = GetExitCodeProcess(cphandle,&xcode1);
	if( xcode1 != STILL_ACTIVE || pollin(asock,10*1000) <= 0 ){
		if( xcode1 == STILL_ACTIVE ){
			LE("failed spawn(), terminate frozen child: %d",
				TerminateProcess((HANDLE)cphandle,0));
			_sleep(10);
		}
		xcode2 = -1;
		ok = GetExitCodeProcess((HANDLE)cphandle,&xcode2);
		CloseHandle(cphandle);
		LE("spawn(%s) = %d, no response from child, %d,%d/%d%s",
			path,cphandle,xcode1,
			ok,xcode2,xcode2==STILL_ACTIVE?"(STILL_ACTIVE)":"");
		closesocket(asock);
		return -1;
	}
	slogtype = LOG_type;
	for( ai = 0; av[ai]; ai++ ){
		if( strncmp(av[ai],"-WSAV:",6) == 0 ){
		}else
		if( strncmp(av[ai],"-W",2) == 0 ){
			LOG_type = (LOG_type & ~0xF) | atoi(av[ai]+2);
			dumpWSAstat();
			break;
		}
	}
	csock = acceptsocket(asock);
	closesocket(asock);
	SetHandleInformation((HANDLE)csock,HANDLE_FLAG_INHERIT,0);

	cpe = ProcEnv0;
	cpe.p_pmode = pmode;
	cpe.p_ppid = getpid();
	cpe.p_nosockopen = NO_SOCKOPEN;
	cpe.p_common = MyProcEnv.p_common;

	for( si = 0; si < MIS; si++ )
		inheritsock(si,cphandle,&cpe);

	cj = 0;
	for( si = 0; si < MIS; si++ )
		isock[si] = cpe.p_inhsock[si];

	for( ci = 0; ci < coen; ci++ ){
		dsock = coes[ci];
		if( ISSOCK(dsock) ){
			int match = 0;
			for( si = 0; si < MIS; si++ ){
				if( isock[si] == dsock ){
					LE("don't closeOnExec (%d) (inherited)",
						dsock);
					match = 1;
					break;
				}
			}
			if( !match ){
				cpe.p_closeOnExec[cj++] = dsock;
			}
		}
	}
	cpe.p_closeOnExecN = cj;

	ntoo(&tpe.p_ope,&cpe);
	tpe.p_envlen = htonx(&cpe,AVStr(tpe.p_envstr),sizeof(tpe.p_envstr));
	wcc = send(csock,(char*)&tpe,sizeof(tpe),0);

	ntotal++;
	nalive++;
	children[csock] = (int)cphandle;

	LastCpid = 0;
	if( 0 < pollin(csock,100) ){
		ChildEnv sce;
		int rcc;
		bzero(&sce,sizeof(sce));
		rcc = recv(csock,(char*)&sce,sizeof(sce),0);
		/*
		if( rcc == sizeof(ChildEnv) ){
		*/
		if( sizeof(sce.c_pid) <= rcc ){
			LastCpid = sce.c_pid;
			if( sce.c_inherr ){
			    for( si = 0; si < MIS; si++ ){
				WSAPROTOCOL_INFO pi;
				if( dupsock(si,sce.c_pid,&pi) ){
					send(csock,(char*)&pi,sizeof(pi),0);
					LV(">>>> sent[%d]",si);
				}
			    }
			}
		}
	}
	else{
		LE("No response from child process");
	}
	child_pid[csock] = LastCpid;

/*
	LE("spawn() = %d, children(alive=%d,total=%d)",
		cphandle,nalive,ntotal);
*/
	LE("spawn() = %d [%d], children(alive=%d,total=%d)",
		cphandle,LastCpid,nalive,ntotal);

	LOG_type = slogtype;
	return (int)cphandle;
}
static int sessionfd(){
	int fd;

	fd = MyProcEnv.p_common.p_sessionfd[0];
	return fd;
}
SessionFd(){ return sessionfd(); }

/* Set it on to emulate systems without SOCKET inheritance (Windows95) */
static int forceDUP = 1;

int CFI_init(int ac,const char *av[]){
	int clsock,svsock;
	int sfd,efd,fd;

	DO_STARTUP(ac,av);
	DO_INITIALIZE(ac,av);
	env2arg("CFI_");

	clsock = getclientsock();
	svsock = getserversock();

	if( clsock < 0 || svsock < 0 )
		LE("CFI_init: clsock=%d svsock=%d\n",clsock,svsock);

	if( clsock != 0 ) dup2(clsock,0);
	if( svsock != 1 ) dup2(svsock,1);

	sfd = sessionfd();
	efd = fileno(stderr);
	for( fd = 3; fd < 32; fd++ ){
		if( fd != sfd && fd != efd )
			close(fd);
	}
	return ac;
}
static const char *tmplog;
static void set_childlog(int ac,const char *av[])
{	int ai;
	const char *arg;
	int fgexe = 0;
	int loglev = 0;
	const char *logdir = 0;
	CStr(lpath,256);
	int now = time(0L);

	for( ai = 0; ai < ac; ai++ ){
		arg = av[ai];
		if( streq(arg,"-v") || streq(arg,"-vv") || streq(arg,"-f") )
			fgexe = 1;
		else
		if( strncmp(arg,"-WSAV:",6) == 0 ){
		}else
		if( strncmp(arg,"-W",2) == 0 ){
			loglev = atoi(arg+2);
			if( logdir = strchr(arg,',') )
				logdir++;
		}
	}
	if( loglev ){
		LOG_type = (LOG_type & ~0xF) | loglev;
		if( !fgexe || logdir ){
			if( logdir == 0 )
				logdir = "/delegate-tmplog";
			sprintf(lpath,"%s/%02d%02d-%04d",logdir,
				(now%3600)/60,now%60,getpid());
			tmpLogFd = creat(lpath,0666);
			if( 0 <= tmpLogFd )
				tmplog = strdup(lpath);
		}
		for( ai = 0; ai < ac; ai++)
			LV("arg[%d] %s",ai,av[ai]);
	}
}
void DO_INITIALIZE(int ac,const char *av[])
{
	if( MyProcEnv.p_ppid == 0 ){
		if( start_service(ac,av) )
			DO_FINALIZE(0);
	}
}
void inheritSOCK(int ppid,HANDLE pphandle,int pport,int si);
static void dumpsock(int csock,const char *fmt,...){
	CStr(buf,128);
	int len,type;
	VSAddr sina,pina;
	VARGS(8,fmt);

	len = sizeof(type);
	type = -1;
	getsockopt(csock,SOL_SOCKET,SO_TYPE,(char*)&type,&len);
	len = sizeof(sina);
	bzero(&sina,sizeof(sina));
	getsockname(csock,(SAP)&sina,&len);
	len = sizeof(pina);
	bzero(&pina,sizeof(pina));
	getpeername(csock,(SAP)&pina,&len);

	sprintf(buf,fmt,VA8);
	LV("%s%d %d [%s:%d]<-[%s:%d]%d",
		buf,csock,type,
		VSA_ntoa(&sina),VSA_port(&sina),
		VSA_ntoa(&pina),VSA_port(&pina),
		WSAGetLastError());
}
static int check_sockets(){
	int si,csock,psock;

	for( si = 0; si < MIS; si++ ){
		csock = MyProcEnv.p_inhsock[si];
		psock = MyProcEnv.p_pinhsock[si];
		if( csock <= 0 )
			continue;
		dumpsock(csock,">>>>  inherited[%d] ",si,csock);
		if( 0 < psock )
		dumpsock(psock,">>>> duplicated[%d] ",si,psock);

		if( !ISSOCK(csock) )
		if( ISSOCK(psock) ){
			MyProcEnv.p_inhsock[si] = psock;
LE(">>>> [%d] %d is not socket, substitute inherited (%d)",si,csock,psock);
		}else{
LE(">>>> [%d] %d is not socket, retrying WSADuplicateSocket ...",si,csock);
			return -1;
		}
	}
	return 0;
}

void DO_STARTUP(int ac,const char *av[])
{	int pin;
	int rcc;
	int pport,psock;
	int ei,nei;
	int ci;
	const char *env;
	int len;
	int ppid;
	int wcc;
	TransProcEnv tpe;
	int ai;
	ChildEnv sce;

	setBinaryIO();
	WINthread();

	argc = ac;
	argv = av;
	for( ai = 0; ai < ac; ai++ ){
		const char *arg = av[ai];
		if( strncmp(arg,"-WSAV:",6) == 0 )
			WinSockVer = stralloc(arg+6);
		else
		if( strncmp(arg,"-w",2) == 0 ){
			int lev = atoi(arg+2);
			if( lev == 0 )
				lev = 1;
			LOG_type = (LOG_type & ~0xF) | lev;
		}
	}

	socket_init();
	dumpWSAstat();

	psock = -1;
	nei = 0;
	len = strlen(SPAWNENV);

	for( ei = 0; env = environ[ei]; ei++ ){
		if( strncmp(env,SPAWNENV,len) == 0 && env[len] == '=' ){
			unsigned long int on = 1;
			unsigned long int off = 0;
			set_childlog(ac,av);
			dumpWSAstat();
			pport = atoi(&env[len+1]);
			ioctlsocket((SOCKET)psock,FIONBIO,&on);
			psock = connectsocket(-1,pport);
			if( psock < 0 ){
				LE("CANNOT CONNECT TO PARENT");
				exit(-1);
			}
			ioctlsocket((SOCKET)psock,FIONBIO,&off);
			if( pollin(psock,10*1000) <= 0 ){
				LE("NO DATA FROM PARENT");
				exit(-1);
			}
			rcc = recv(psock,(char*)&tpe,sizeof(tpe),0);
			LV("inheritance recv() = %d",rcc);

			bzero(&sce,sizeof(sce));
			sce.c_pid = getpid();
			/*
			wcc = send(psock,(char*)&sce,sizeof(ChildEnv),0);
			*/

			if( rcc == sizeof(OldProcEnv) ){
				oton(&MyProcEnv,&tpe.p_ope);
			}else
			if( rcc == sizeof(tpe) ){
				ntohx(&MyProcEnv,AVStr(tpe.p_envstr),tpe.p_envlen);
				sce.c_inherr = check_sockets();
			}else{
				LT("CANNOT READ ARGUMENTS [%d/%d]: %d/%d",
					pport,psock,rcc,sizeof(ProcEnv));
				closesocket(psock);
				exit(-1);
			}
			wcc = send(psock,(char*)&sce,sizeof(ChildEnv),0);
			MyProcEnv.p_toparent = psock;
			LV("STARTUP: %s, got[%d], ppid=%d,pmode=%d",
				env,rcc, MyProcEnv.p_ppid, MyProcEnv.p_pmode);
		}
	}

	ppid = MyProcEnv.p_ppid;

	if( ppid == 0 ){
		pipe(MyProcEnv.p_common.p_sessionfd);
	}else{
		NO_SOCKOPEN = MyProcEnv.p_nosockopen;
	}

	for( ci = 0; ci < MyProcEnv.p_closeOnExecN; ci++ ){
		int sock,rcode;
		sock = MyProcEnv.p_closeOnExec[ci];
		rcode = closesocket(sock);
		LV("do CloseOnExec[%d]=%d, do close=%d",ci,sock,rcode);
	}

	if( 0 <= psock ){
		HANDLE pphandle;
		int si;
		pphandle = OpenProcess(PROCESS_ALL_ACCESS,0,ppid);
		for( si = 0; si < MIS; si++ )
		{	int dclsock;
			WSAPROTOCOL_INFO pi;

			if( sce.c_inherr )
			if( 0 <= MyProcEnv.p_inhsock[si] )
			if( recv(psock,(char*)&pi,sizeof(pi),0) == sizeof(pi) ){
				dclsock = WSASocket(
					FROM_PROTOCOL_INFO,FROM_PROTOCOL_INFO,
					FROM_PROTOCOL_INFO,&pi,0,0);
				LV(">>>> recv[%d] clsock=%d nc=%d %d %d %d",
					si,MyProcEnv.p_inhsock[si],dclsock,
					pi.iAddressFamily,
					pi.iSocketType,
					pi.iProtocol);
				if( 0 < dclsock ){ 
			dumpsock(dclsock,">>>> Duplicated[%d] ",si,dclsock);
					MyProcEnv.p_inhsock[si] = dclsock;
				}
			}
			inheritSOCK(ppid,pphandle,pport,si);
		}

		if( tmplog ){
			close(tmpLogFd);
			tmpLogFd = -1;
			LOG_type = (LOG_type & ~0xF);
			if( LOGLEVEL < 6 ){
				unlink(tmplog);
				tmplog = 0;
			}
		}
	}else{
	}
}
void inheritSOCK(int ppid,HANDLE pphandle,int pport,int si){
	int pclsock,nclsock,fd;
	VSAddr ina;
	int len;

	pclsock = MyProcEnv.p_pinhsock[si];
	nclsock = MyProcEnv.p_inhsock[si];

/*
	LV("PPID=%d/%d CLSOCK=[%d->%d]",ppid,pphandle,pclsock,nclsock);
*/
	LV("[%d] PPID=%d/%d CLSOCK=[%d->%d]",si,ppid,pphandle,pclsock,nclsock);

	/* CLSOCK possibly inherited (WindowsNT) */
	if( !isWindows95() )
	if( 0 <= pclsock && pclsock != nclsock )
	{
		int sx,inherited;
		inherited = 0;
		for( sx = 0; sx < MIS; sx++ )
			if( MyProcEnv.p_inhsock[sx] == pclsock )
				inherited = 1;
		if( inherited == 0 ){
		LV("inheritSOCK -- closesocket(%d)",pclsock);
		closesocket(pclsock);
		}
	}

	if( 0 <= nclsock ){
		if( !ISSOCK(nclsock) ){
			LE("FATAL: inherited handle[%d] %d is not socket",si,nclsock);
			exit(-1);
		}
		MyProcEnv.p_inhfd[si] = fd = openSocket(-1,nclsock,1,"fork()");
		if( fd < 0 )
			exit(-1);

		len = sizeof(VSAddr);
		bzero(&ina,len);
		getpeername(nclsock,(SAP)&ina,&len);

		LV("PARENT=%d, PORT=%d, CLSOCK[%d]=%d/%d:%d [%s:%d]",
			ppid,pport,si,
			nclsock,fd,ISSOCK(nclsock),
			VSA_ntoa(&ina),VSA_port(&ina));
	}
}

static int bindsocket(int sock,int *portp)
{	int port;
	int rcode;
	VSAddr ina;
	int inalen,len;

	if( sock < 0 )
		sock = socket(AF_INET,SOCK_STREAM,0);

	inalen = VSA_atosa(&ina,0,"127.0.0.1");
	rcode = bind(sock,(SAP)&ina,inalen);
	rcode = listen(sock,1);

	if( portp ){
		len = sizeof(VSAddr);
		getsockname(sock,(SAP)&ina,&len);
		*portp = VSA_port(&ina);
	}
	return sock;
}
static int connectsocket(int sock,int port)
{	VSAddr ina;
	int inalen,rcode;

	if( sock < 0 )
		sock = socket(AF_INET,SOCK_STREAM,0);
	if( sock < 0 ){
		LE("connectsocket(%d) failed to create socket",port);
		return -1;
	}

	inalen = VSA_atosa(&ina,port,"127.0.0.1");
	rcode = connect(sock,(SAP)&ina,inalen);

	LV("connect(%d,%d) = %d",sock,port,rcode);
	return sock;
}
static int acceptsocket(int asock)
{	int csock;
	VSAddr ina;
	int len;
	int myport;

	len = sizeof(VSAddr);
	getsockname(asock,(SAP)&ina,&len);
	myport = VSA_port(&ina);

	len = sizeof(VSAddr);
	csock = accept(asock,(SAP)&ina,&len);
	LV("accept(%d,%d) = %d [%s:%d]",asock,myport,csock,
		VSA_ntoa(&ina),VSA_port(&ina));
	return csock;
}

static struct InheritSock {
	int	is_fd;
	int	is_sock;
	int	is_set;
	int	is_close_src;
} INHERIT_SOCK[MIS];
void setclosesource(int si){
	if( isWindows95() )
		INHERIT_SOCK[si].is_close_src = 1;
}

static int dummy_in = -1;
static int dummy_input()
{	int io[2];
	int in2;

	if( dummy_in < 0 ){
		pipe(io);
		dummy_in = io[0];
		if( dummy_in == 0 ){
			in2 = _dup(dummy_in);
			_close(dummy_in);
			dummy_in = in2;
		}
		close(io[1]);
	}
	return dummy_in;
}

static int dupsock(int si,int pid,WSAPROTOCOL_INFO *pip){
	int sock;
	int rcode;

	if( 0 <= (sock = INHERIT_SOCK[si].is_sock) ){
		rcode = WSADuplicateSocket((SOCKET)sock,pid,pip);
		if( rcode != 0 )
			return 0;

		LV(">>>> dupsock(%d,%d)=%d [%d %d %d]",sock,pid,rcode,
			pip->iAddressFamily,pip->iSocketType,pip->iProtocol);
		return 1;
	}else	return 0;
}
static void inheritsock(int si,HANDLE cphandle,ProcEnv *cpe)
{	struct InheritSock *ihs;
	int cs,rcode,dummy;

	ihs = &INHERIT_SOCK[si];
	if( ihs->is_set ){
		ihs->is_set = 0;
		cs = ihs->is_close_src;
		ihs->is_close_src = 0;
		cpe->p_pinhsock[si] = ihs->is_sock;
		cpe->p_inhsock[si] = duphandle(0,(HANDLE)ihs->is_sock,cphandle,cs,0);
		if( cs ){
			dummy = dummy_input();
			rcode = closeSocket(ihs->is_fd,ihs->is_sock);
			_dup2(dummy,ihs->is_fd);
		}
		LV("INHERIT: %d->%d (close_source=%d,%d/%d)",
		ihs->is_sock,cpe->p_inhsock[si],cs,ihs->is_sock,ihs->is_fd);
	}else{
		cpe->p_pinhsock[si] = -1;
		cpe->p_inhsock[si] = -1;
	}
}
int passsock(int si,int sock)
{
	if( sock < 0 )
		return sock;

	LV("set INHERIT_SOCK=%d/%d",SocketOf(sock),sock);
	INHERIT_SOCK[si].is_fd = sock;
	INHERIT_SOCK[si].is_sock = SocketOf(sock);
	INHERIT_SOCK[si].is_set = 1;
	return sock;
}
int recvsock(int si)
{	int fd;

	fd = MyProcEnv.p_inhfd[si];
	LV("get INHERIT_SOCK=%d/%d",SocketOf(fd),fd);
	return fd;
}
int closedups(int si){
	int fd,sock;
	int closed;

	closed = 0;
	sock = MyProcEnv.p_inhsock[si];
	for( fd = 0; fd < MAXHANDLE; fd++ )
		if( isdup_handles[fd] == sock ){
			LE("closedups(%d/%d)",sock,fd);
			closeSocket(fd,sock);
			closed++;
		}
	return closed;
}
int setclientsock(int sock){ return passsock(0,sock); }
int getclientsock(){ return recvsock(0); }
int setserversock(int sock){ return passsock(1,sock); }
int getserversock(){ return recvsock(1); }
int setcontrolsock(int sock){ return passsock(2,sock); }
int getcontrolsock(){ return recvsock(2); }
int setrsvdsock(int si,int sock){
	if( 3+si < MIS )
		return passsock(3+si,sock);
	else	return -1;
}
int getrsvdsock(int si){
	if( 3+si < MIS )
		return recvsock(3+si);
	else	return -1;
}
int send_sock(int dpid,int fd,int closesrc)
{	HANDLE dph;
	int ssock,sock;

	ssock = SocketOf(fd);
	dph = OpenProcess(PROCESS_ALL_ACCESS,0,dpid);
	sock = duphandle(NULL,(HANDLE)ssock,dph,0,0);
	CloseHandle(dph);
	return sock;
}
int recv_sock(int spid,int ssock,int closesrc)
{	HANDLE sph;
	int sock;
	int options;
	int fd;

	sph = OpenProcess(PROCESS_ALL_ACCESS,0,spid);
	sock = duphandle(sph,(HANDLE)ssock,NULL,0,0);
	CloseHandle(sph);
	if( 0 < sock )
		fd = openSocket(-1,sock,0,"recv_sock()");
	else	fd = -1;
	LT("recv_sock(%d/%d,%d) sock=%d/%d,issock=%d err=%d",
		sph,spid,ssock,fd,sock,ISSOCK(sock),WSAGetLastError());
	return fd;
}

WithSocketFile(){ return 0; }
int file_isselectable(int fd){
	if( SocketOf(fd) )
		return 1;
	if( ISPIPE(fd) )
		return 1;
	return 0;
}

extern int WAIT_WNOHANG;

int killchildren()
{	int px,pi,pid,ok,nkill;
	int ph;

	nkill = 0;
	for( px = 0; px <  MAXHANDLE; px++ ){
		ph = children[px];
		if( 0 < ph ){
			pid = child_pid[px];
			ok = TerminateProcess((HANDLE)ph,0);
			nkill++;
			LE("killchildren#%d Terminate %d [%d] = %d",
				nkill,ph,pid,ok);
		}
	}

	if(nkill ){
		_sleep(1);
		for( pi = 0; pi < nkill; pi++ ){
			if( wait3(NULL,WAIT_WNOHANG,NULL) < 0 )
				break;
		}
	}
	return nkill;
}

int waitpid(int pid,int *statusp,int options)
{
	LE("waitpid() = -1 (not supported)");
	return -1;
}
int wait3(int *statusp,int options,void *rusage)
{	int chsock,pid;
	int nready;
	int xpid,xstatus;
	int cpid;
	struct fd_set chset;
	struct timeval tvb,*tv = &tvb;

	xpid = -1;
	xstatus = -1;

	FD_ZERO(&chset);
	for( chsock = 0; chsock < MAXHANDLE; chsock++ )
		if( children[chsock] )
			FD_SET(chsock,&chset);

	if( options == WAIT_WNOHANG ){
		tv->tv_sec = 0;
		tv->tv_usec = 0;
	}else	tv = NULL;

	nready = select(MAXHANDLE,&chset,NULL,NULL,tv);

	if( nready )
	for( chsock = 0; chsock < MAXHANDLE; chsock++ ){
		if( FD_ISSET(chsock,&chset) ){
			cpid = child_pid[chsock];
			pid = children[chsock];
			children[chsock] = 0;
			closesocket(chsock);
			xpid = _cwait(&xstatus,pid,0);
			nalive--;
			if( 0 < xpid )
				break;
		}
	}

	if( 0 <= xpid )
	LE("wait3() = %d [%d], status=%d, children(alive=%d,total=%d)",
		xpid,cpid,xstatus,nalive,ntotal);
/*
	LE("wait3() = %d, status=%d, children(alive=%d,total=%d)",
		xpid,xstatus,nalive,ntotal);
*/

	if( statusp )
		*statusp = xstatus;
	return xpid;
}

static void checkarg(PCStr(path),const char *const av[],int mac,const char *nav[],PVStr(nab))
{	const char *arg;
	refQStr(ap,nab); /**/
	int ai;

	for( ai = 0; arg = av[ai]; ai++ ){
		if( mac-1 <= ai ){
			break;
		}
		if( strpbrk(arg," \t") && *arg != '"' ){
			nav[ai] = ap;
			sprintf(ap,"\"%s\"",arg);
			ap += strlen(ap) + 1;
		}else	nav[ai] = arg;
	}
	nav[ai] = NULL;
}
int spawnvp(int pmode,PCStr(path),const char *const av[])
{	const char *nav[MAX_ARGC]; /**/
	CStr(nab,MAX_ARGB);

	checkarg(path,av,elnumof(nav),nav,AVStr(nab));
	return spawnvpe(pmode,path,nav,environ);
}
int execvp(PCStr(path),const char *const av[])
{	const char *nav[MAX_ARGC]; /**/
	CStr(nab,MAX_ARGB);

	checkarg(path,av,elnumof(nav),nav,AVStr(nab));
	return _execvp(path,nav);
}

int wait(int *statusp)
{	int pid;

	LE("wait(%x) = ...",statusp);
	pid = wait3(NULL,0,NULL);
	LE("wait(%x) = %d",statusp,pid);
	return pid;
}
int getppid(){
	int ppid;
	HANDLE ph;

	ppid = MyProcEnv.p_ppid;
	if( ppid == 0 )
		return ppid;
	ph = OpenProcess(PROCESS_QUERY_INFORMATION,0,ppid);
	if( ph == NULL )
		return 0;
	CloseHandle(ph);
	LV("getppid() = %d",ppid);
	return ppid;
}
int setsid(){
	LT("setsid() = -1 (not supported)");
	return -1;
}

void DO_FINALIZE(int code){
	int fd,sock;

	LV("PID=%d exit(%d) ...",getpid(),code);

	fcloseall();
	/*
	killchildren();
	can be harmful?
	*/

	if( NO_SOCKOPEN )
	for( fd = 0; fd < MAXHANDLE; fd++ )
		if( sock = open_osfhandle[fd] )
			closeSocket(fd,sock);

	deltmpfiles();
	_rmtmp();

	LV("PID=%d exit(%d)",getpid(),code);
}

#else

#define _YSOCKET_H   /* don't include ysocket.h in vsocket.h */
#include "vsocket.h" /* definition of VSAddr */
#include "yarg.h"
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
void setBinaryIO();
int setCloseOnExec(int fd);
int setNonblockingIO(int fd,int on);
int *StructAlloc(int size);
int Lseek(int fd,int off,int wh);
int syslog_DEBUG(PCStr(fmt),...);
int syslog_ERROR(PCStr(fmt),...);

int SocketOf(int sock);
int _BIND(int fd,struct sockaddr *addr,int len)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = bind(sock,addr,len);
	socklog("bind",fd,addr,rcode,fd);
	return rcode;
}
int _ACCEPT(int fd,struct sockaddr *addr,int *len)
{	int csock;

	csock = Xaccept(fd,addr,len);
	socklog("accept",fd,addr,csock,fd);
	return csock;
}
int _CONNECT(int fd,struct sockaddr *addr,int len)
{	int rcode;

	rcode = connect(fd,addr,len);
	socklog("connect",fd,addr,rcode,fd);
	return rcode;
}

void setclosesource(int si){ }
void setclientsock(int sock){ }
int getclientsock(){ return -1; }
int setserversock(int sock){ return -1; }
int getserversock(){ return -1; }
int setcontrolsock(int sock){ return -1; }
int getcontrolsock(){ return -1; }
int setrsvdsock(int si,int sock){ return -1; }
int getrsvdsock(int si){ return -1; }
int send_sock(int dpid,int fd,int closesrc){ return -1; }
int recv_sock(int spid,int ssock,int closesrc){ return -1; }

int SocketOf(int sock){ return sock; }
int closedups(int si){ return 0; }
void setNonblockingPipe(int pv[]){ setNonblockingIO(pv[0],1); }
int setNonblockingSocket(int fd,int on){ return setNonblockingIO(fd,on); }
int file_isselectable(int fd){ return 1; }
int pollPipe(int pfd,int msec){ return -1; }

int create_service(int ac,const char *av[],PCStr(port))
{
	return 0;
}
int delete_service(int ac,const char *av[],PCStr(port),PCStr(arg))
{
	return 0;
}
int killchildren()
{
	return 0;
}

int system(PCStr(com))
{	int pid,xpid;

	if( com == 0 )
		return 1;
	pid = fork();
	if( pid < 0 )
		return -1;
	if( pid == 0 ){
		execlp(BINSHELL,BINSHELL,"-c",com,NULL);
		exit(-1);
	}
	xpid = waitpid(pid,NULL,0);
	return 0; /* should return the status code */
}
int isWindows(){ return 0; }
void DO_INITIALIZE(int ac,const char *av[]);
int CFI_init(int ac,const char *av[]){
	DO_INITIALIZE(ac,av);
	env2arg("CFI_");
	return ac;
}
int SessionFd(){ return -1; }
int setCloseOnExecSocket(int fd){ return setCloseOnExec(fd); }
int clearCloseOnExecSocket(int fd){ return -1; }
int setInheritance(int ifd,int inherit){ return -1; }

#ifdef __EMX__
int putenv_sockhandle(int fd,PVStr(env))
{	int shandle;

	shandle = _getsockhandle(fd);
	if( shandle != -1 ){
		sprintf(env,"EMX_SOCKHANDLE%d=%d",fd,shandle);
		putenv(env);
	}
	return shandle;
}

int getenv_sockhandle(int fd){
	CStr(name,32);
	const char *env;
	int shandle,fdx;

	sprintf(name,"EMX_SOCKHANDLE%d",fd);
	if( (env = getenv(name)) && *env ){
		shandle = atoi(env);
		if( 0 <= shandle ){
			for( fdx = 0; fdx < 32; fdx++ ){
				if( _getsockhandle(fdx) == shandle ){
					dup2(fdx,fd);
					return fd;
				}
			}
			fdx = _impsockhandle(shandle,0);
			if( fdx != fd ){
				dup2(fdx,fd);
				close(fdx);
			}
			return fd;
		}
	}
	return -1;
}

void DO_FINALIZE(int code){
	fcloseall();
	deltmpfiles();
	_rmtmp();
}

void DO_STARTUP(int ac,const char *av[])
{
}
void DO_INITIALIZE(int ac,const char *av[])
{	int fd;
	unsigned long rel = 0, cur;
 
	DosSetRelMaxFH(&rel, &cur);
	if( cur < 48 ){
		LV("increase MaxFH: %d -> %d",cur,48);
		DosSetMaxFH(48);
	}

	for( fd = 0; fd < 32; fd++ )
		getenv_sockhandle(fd);

	setBinaryIO();
}
extern int SPAWN_P_WAIT;
int execvp(PCStr(path),char *const argv[])
{	int stat;
	int fd;
	char envs[32][32]; /**/

	for( fd = 0; fd < 32; fd++ )
		putenv_sockhandle(fd,envs[fd]);

	stat = spawnvp(SPAWN_P_WAIT,path,argv);
	if( stat == -1 )
		return -1;
	else	exit(stat);
}
int WithSocketFile(){ return 0; }
int getsockHandle(int fd){ return _getsockhandle(fd); }
#else
void DO_STARTUP(int ac,const char *av[])
{
}
void DO_INITIALIZE(int ac,const char *av[])
{
	setBinaryIO();
}
void DO_FINALIZE(int code){ }
int WithSocketFile(){ return 1; }
int getsockHandle(int fd){ return -1; }
#endif /* !__EMX__ */

#endif

char *VSA_xtoap(VSAddr *sa,PVStr(buf),int siz);
void socklog(PCStr(where),int fd,const SAP0 addr,int rcode,int csock)
{	CStr(ina,64);
	CStr(arg,64);
	CStr(self,64);
	int len;

	if( (LOG_type & L_SOCKET) == 0 )
		return;

	VSA_xtoap((VSAddr*)addr,AVStr(arg),sizeof(arg));

	len = sizeof(ina);
	bzero(&ina,len);
	getsockname(csock,(SAP)&ina,&len);
	VSA_xtoap((VSAddr*)&ina,AVStr(self),sizeof(self));

	LE("{s} %7s(%2d,%-20s)=%2d %-20s",where,fd,arg,rcode,self);
}

#include "yselect.h" /* FD_SETSIZE */
static int fd_stack[FD_SETSIZE][2];
void push_fd(int fd,int fd2,int rw)
{
	if( fd < 0 || FD_SETSIZE <= fd )
		return;

	fd_stack[fd][rw] = fd2 + 1;
}
int top_fd(int fd,int rw)
{
	if( fd < 0 || FD_SETSIZE <= fd )
		return -1;

	return fd_stack[fd][rw] - 1;
}
int pop_fd(int fd,int rw)
{	int nfd,xfd,xrw;

	if( fd < 0 || FD_SETSIZE <= fd )
		return -1;

	nfd = fd_stack[fd][rw] - 1;
	if( 0 <= nfd ){
		fd_stack[fd][rw] = 0;
		dup2(nfd,fd);

		xrw = (rw+1) % 2;
		xfd = fd_stack[fd][xrw] - 1;
		if( 0 <= xfd ){
			fd_stack[fd][xrw] = 0;
		}
	}
	return nfd;
}
int get_writelimit(int fd,int siz){
	return siz;
}

int porting_dbg(PCStr(fmt),...)
{	CStr(buf,4096);
	int now = time(0L);
	int logfd;
	VARGS(12,fmt);

	sprintf(buf,"(%s) %02d:%02d [%3d] ",SYST,(now%3600)/60,now%60,getpid());
	Xsprintf(TVStr(buf),fmt,VA14);
	strcat(buf,"\n");
	if( 0 <= tmpLogFd )
		logfd = tmpLogFd;
	else
	if( curLogFd() < 0 )
		logfd = fileno(stderr);
	else	logfd = curLogFd();
	Lseek(logfd,0,2);
	DBGWRITE(logfd,buf,strlen(buf));

	if( lFG() && logfd != fileno(stderr) ){
		logfd = fileno(stderr);
		Lseek(logfd,0,2);
		DBGWRITE(logfd,buf,strlen(buf));
	}
	return 0;
}

/*
arg2env(prefix)
*/
void arg2env(PCStr(prefix),int logfd)
{	CStr(env,1024);
	CStr(tmp,1024);

	/*
	sprintf(env,"%sLOGFD=%d",prefix,curLogFd());
	*/
	sprintf(env,"%sLOGFD=%d",prefix,logfd);
	putenv(stralloc(env));
}
void env2arg(PCStr(prefix))
{	CStr(name,1024);
	const char *env;

	sprintf(name,"%sLOGFD",prefix);
	if( env = getenv(name) )
	{	int lfd;

		if( *env == '-' ){
			lfd = openNull(1);
			dup2(lfd,fileno(stderr));
			close(lfd);
		}else
		dup2(atoi(env),fileno(stderr));
	}
}


#include "windows0.c"

/*////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	fcntl.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	961231	extracted from file.c
	990309	merged to windows0.c
//////////////////////////////////////////////////////////////////////#*/

int setNonblockingIO(int fd,int on)
{	int flags;
	int rcode;

	if( F_SETFL == -1 || O_NDELAY == -1 ){
		rcode = setNonblockingSocket(fd,on);
		if( rcode < 0 ){
			syslog_ERROR("Non-Blocking I/O not supported\n");
			return -1;
		}else	return rcode;
	}

	flags = fcntl(fd,F_GETFL,0);
	if( on )
		flags |=  O_NDELAY;
	else	flags &= ~O_NDELAY;
	return fcntl(fd,F_SETFL,(void*)flags);
}

/*
#include <sys/syscall.h>
open(PCStr(path),int flags)
{	int rcode;

	rcode = syscall(SYS_open,path,flags);
	fprintf(stderr,"#### open(%s,%x) = %d\n",path,flags,rcode);
	return rcode;
}
socketpair(in dom,in type,in proto,int sv[])
{	int rcode;

	rcode = syscall(SYS_socketpair,dom,type,proto,sv);
	return rcode;
}
connect(int sock,void *addr,int len)
{	int rcode;

	rcode = syscall(SYS_connect,sock,addr,len);
	return rcode;
}
*/
