/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1998-2000 Electrotechnical Laboratry (ETL), AIST, MITI
Copyright (c) 1998-2000 Yutaka Sato
Copyright (c) 2001-2002 National Institute of Advanced Industrial Science and Technology (AIST)

Permission to use, copy, modify, and distribute this material for any
purpose and without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.
AIST MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	sslway.c (SSL encoder/decoder with SSLeay/openSSL)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

  Given environment:
    file descriptor 0 is a socket connected to a client
    file descriptor 1 is a socket connected to a server

  Commandline argument:
    -cert file -- certificate (possibly with private key) of this SSLway
                  to be shown to a peer
    -key file -- private key file (if not included in the -cert file)
    -pass arg -- the source of passphrase, pass:string or file:path

    -CAfile file -- the name of file contains a CA's certificate
    -CApath dir -- directory contains CA's certificate files each named with
                   `X509 -hash -noout < certificate.pem`

    -Vrfy -- peer's certificate must be shown and must be authorized
    -vrfy -- peer's certificate, if shown, must be authorized
    -Auth -- peer must show its certificate, but it can be unauthorized
    -auth -- just record the peer's certificate, if exists, into log
             -- equals to obsoleted "-client_auth".

    -vd  -- detailed logging
    -vu  -- logging with trace (former default)
    -vt  -- terse logging (current default)
    -vs  -- disalbe any logging

    -ht  through pass if the request is in bare HTTP protocol (GET,HEAD,POST)

    Following options can be omitted when the sslway is called from DeleGate
    with FCL, FSV or FMD since it will be detected automatically.

    -co  apply SSL for the connection to the server [default for FSV]
    -ac  aplly SSL for the accepted connection from the client [default for FCL]
    -ad  accept either SSL or through by auto-detection of SSL-ClientHello
    -ss  negotiate by AUTH SSL for FTP (implicit SSL for data-connection)
    -st  accept STARTTLS (protocol is auto-detect) and SSL tunneling
    -St  require STARTTLS first (protocol is auto-detect) and PBSZ+PROT for FTP
    -{ss|st|St}/protocol enable STARTTLS for the protocol {SMTP,POP,IMAP,FTP}
    -bugs

  Usage:
    delegated FSV=sslway
    delegated FCL=sslway ...

  How to make:
    - do make at .. or ../src directory
    - edit Makefile.go to let SSLEAY points to the directory of libssl.a
    - make -f Makefile.go sslway

History:
	980412	created
	980428	renamed from "sslrelay" to "sslway"
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include <stdlib.h>
#include "ystring.h"
#include "randtext.c"

int (*DELEGATE_MAIN)(int ac,const char *av[]);
void (*DELEGATE_TERMINATE)();

int randstack_call(int strg,iFUNCP func, ...);
char **move_envarg(int ac,const char *av[],const char **areap,int *lengp,int *sizep);
long Gettimeofday(int *usec);
int CFI_init(int ac,const char *av[]);
int PollIns(int timeout,int size,int *mask,int *rmask);
int setNonblockingIO(int fd,int on);
void set_nodelay(int sock,int onoff);
int RecvPeek(int sock,void *buf,int len);
int isWindows();
int SocketOf(int sock);
int LIBFILE_IS(PCStr(file),PVStr(xfile));

#define LSILENT	-1
#define LERROR	0
#define LTRACE	1
#define LDEBUG	2
static int loglevel = LERROR;
#define ERROR	(loglevel < LERROR)?0:DOLOG
#define TRACE	(loglevel < LTRACE)?0:DOLOG
#define DEBUG	(loglevel < LDEBUG)?0:DOLOG

static FILE *stdctl;
static const char *client_host;
static int PID;
static int DOLOG(PCStr(fmt),...)
{	CStr(xfmt,256);
	CStr(head,256);
	VARGS(8,fmt);

	sprintf(head,"## SSLway[%d](%s)",PID,client_host);
	sprintf(xfmt,"%%s %s\n",fmt);
	syslog_ERROR(xfmt,head,VA8);
	return 0;
}

#ifdef __cplusplus
extern "C" {
#endif
/*
#include "ssl.h"
*/
#define SSL_FILETYPE_PEM 1
#define SSL_VERIFY_NONE			0x00
#define SSL_VERIFY_PEER			0x01
#define SSL_VERIFY_FAIL_IF_NO_PEER_CERT	0x02
#define SSL_VERIFY_CLIENT_ONCE		0x04

#define X509_V_FLAG_CRL_CHECK		0x04
#define X509_V_FLAG_CRL_CHECK_ALL	0x08

typedef void SSL_CTX;
typedef void SSL_METHOD;
typedef void SSL;
typedef void X509;
typedef void X509_NAME;
typedef void X509_STORE;
typedef void X509_STORE_CTX;
typedef void BIO;
typedef void RSA;
#define BIO_NOCLOSE 0

unsigned long ERR_get_error(void);
char *ERR_error_string_n(int,char*,int);
void ERR_print_errors_fp(FILE *fp);
void RAND_seed(const void *buf,int num);

BIO *BIO_new_fp(FILE *stream, int close_flag);
int BIO_free(BIO *a);
X509 *PEM_read_bio_X509(BIO*,...);

SSL_CTX *SSL_CTX_new(SSL_METHOD *method);
int  SSL_library_init(void);
SSL *SSL_new(SSL_CTX *ctx);
int  SSL_set_fd(SSL *ssl, int fd);
int  SSL_connect(SSL *ssl);
int  SSL_accept(SSL *ssl);
int  SSL_write(SSL *ssl, const void *buf, int num);
int  SSL_read(SSL *ssl,void *buf,int num);
int  SSL_pending(SSL *s);
void SSL_set_connect_state(SSL *s);
void SSL_set_accept_state(SSL *s);
void SSL_load_error_strings(void );
int  SSL_get_error(SSL *s,int ret_code);
X509 *SSL_get_peer_certificate(SSL *ssl);

long SSL_CTX_ctrl(SSL_CTX *ctx,int cmd, long larg, void *parg);
int  SSL_CTX_check_private_key(SSL_CTX *ctx);
X509_STORE *SSL_CTX_get_cert_store(SSL_CTX *);
int SSL_CTX_load_verify_locations(SSL_CTX *ctx,PCStr(CAfile),PCStr(CApath));
int  SSL_CTX_set_cipher_list(SSL_CTX *,PCStr(str));
typedef int pem_password_cb(char buf[], int size, int rwflag, void *userdata);
void SSL_CTX_set_default_passwd_cb(SSL_CTX *ctx, pem_password_cb *cb);
int  SSL_CTX_set_default_verify_paths(SSL_CTX *ctx);
void SSL_CTX_set_tmp_rsa_callback(SSL_CTX *ctx, RSA *(*cb)(SSL *ssl,int is_export, int keylength));                         
void SSL_CTX_set_verify(SSL_CTX *ctx,int mode, int (*callback)(int, X509_STORE_CTX *));
int  SSL_CTX_use_RSAPrivateKey_file(SSL_CTX *ctx,PCStr(file), int type);
int  SSL_CTX_use_certificate_file(SSL_CTX *ctx,PCStr(file), int type);

SSL_METHOD *SSLv2_method();
SSL_METHOD *SSLv3_method();
SSL_METHOD *SSLv3_server_method();
SSL_METHOD *SSLv3_client_method();
SSL_METHOD *SSLv23_server_method();
SSL_METHOD *SSLv23_client_method();
SSL_METHOD *TLSv1_server_method();
SSL_METHOD *TLSv1_client_method();

X509_NAME *X509_get_issuer_name(X509 *a);
int i2d_X509_bio(BIO *bp,X509 *x509);
char *X509_NAME_oneline(X509_NAME *a,char buf[],int size);
const char *X509_verify_cert_error_string(long n);
X509 *X509_STORE_CTX_get_current_cert(X509_STORE_CTX *ctx);
int X509_STORE_CTX_get_error(X509_STORE_CTX *ctx);
int X509_STORE_CTX_get_error_depth(X509_STORE_CTX *ctx);
void X509_STORE_set_flags(X509_STORE *ctx, long flags);
X509_NAME *X509_get_subject_name(X509 *a);
void X509_free(X509 *a);

RSA *RSA_generate_key(int bits, unsigned long e,void (*callback)(int,int,void *),void *cb_arg);

#ifdef __cplusplus
}
#endif

static SSL_CTX *ssl_new(int serv)
{	SSL_CTX *ctx;
	SSL_METHOD *meth;

	SSL_library_init();
	SSL_load_error_strings();
	if( serv )
		meth = SSLv23_server_method();
	else	meth = SSLv23_client_method();
	ctx = SSL_CTX_new(meth);
	return ctx;
}
static void ssl_printf(SSL *ssl,int fd,PCStr(fmt),...)
{	CStr(buf,0x4000);
	VARGS(8,fmt);

	sprintf(buf,fmt,VA8);
	if( ssl )
		SSL_write(ssl,buf,strlen(buf));
	else	write(fd,buf,strlen(buf));
}
static void ssl_prcert(SSL *ssl,int show,SSL *outssl,int outfd,PCStr(what))
{	X509 *peer;
	CStr(subjb,256);
	const char *sb;
	CStr(issrb,256);
	const char *is;
	const char *dp;
	CStr(ident,256);

	ident[0] = 0;
	if( peer = SSL_get_peer_certificate(ssl) ){
		sb = X509_NAME_oneline(X509_get_subject_name(peer),subjb,sizeof(subjb));
		is = X509_NAME_oneline(X509_get_issuer_name(peer),issrb,sizeof(issrb));
		if( show ){
			ssl_printf(outssl,outfd,
				"##[SSLway: %s's certificate]\r\n",what);
			ssl_printf(outssl,outfd,"## Subject: %s\r\n",sb);
			ssl_printf(outssl,outfd,"## Issuer: %s\r\n",is);
		}
		ERROR("%s's cert. = **subject<<%s>> **issuer<<%s>>",what,sb,is);
		if( dp = (char*)strcasestr(sb,"/emailAddress=") )
			wordscanY(dp+14,AVStr(ident),sizeof(ident),"^/");
		else
		if( dp = (char*)strcasestr(sb,"/email=") )
			wordscanY(dp+7,AVStr(ident),sizeof(ident),"^/");
		else	strcpy(ident,"someone");
		X509_free(peer);
	}else{
		TRACE("%s's cert. = NONE",what);
		strcpy(ident,"anonymous");
		sb = "";
		is = "";
	}
	if( stdctl ){
		fprintf(stdctl,"CFI/1.0 200- Ident:%s\r\n",ident);
		fprintf(stdctl,"CFI/1.0 200 Certificate:%s//%s\r\n",sb,is);
		fflush(stdctl);
	}
}
static void eRR_print_errors_fp(FILE *fp)
{	int code;
	const char *strp;
	CStr(str,1024);
	const char *file;
	const char *line;
	const char *opttxt;

	if( isWindows() ){
		/* OpenSSL-0.9.7c on Win32 aborts in ERR_print_errors_fp() */
		while( code = ERR_get_error() ){
			strp = ERR_error_string_n(code,str,sizeof(str));
			file = "";
			line = "";
			opttxt = "";
			ERROR("SSL-ERRCODE: %X\r\n%d:%s:%s:%s:%s",code,getpid(),
				str,file,line,opttxt);
		}
	}else{
		ERR_print_errors_fp(fp);
	}
}
#define	ERR_print_errors_fp	eRR_print_errors_fp

static SSL *ssl_conn(SSL_CTX *ctx,int confd)
{	SSL *conSSL;

	conSSL = SSL_new(ctx);
	SSL_set_connect_state(conSSL);
	SSL_set_fd(conSSL,SocketOf(confd));
	if( SSL_connect(conSSL) < 0 ){
		ERROR("connect failed");
		ERR_print_errors_fp(stderr);
		return NULL;
	}else{
		TRACE("connected");
		return conSSL;
	}
}
static SSL *ssl_acc(SSL_CTX *ctx,int accfd)
{	SSL *accSSL;

	accSSL = SSL_new(ctx);
	SSL_set_accept_state(accSSL);
	SSL_set_fd(accSSL,SocketOf(accfd));
	if( SSL_accept(accSSL) < 0 ){
		ERROR("accept failed");
		ERR_print_errors_fp(stderr);
		ssl_printf(accSSL,0,"SSLway: accept failed\n");
		return NULL;
	}else{
		TRACE("accepted");
		return accSSL;
	}
}
static void ssl_setCAs(SSL_CTX *ctx,PCStr(file),PCStr(dir))
{	CStr(xfile,1024);
	CStr(xdir,1024);

	if( LIBFILE_IS(file,AVStr(xfile)) ) file = xfile;
	if( LIBFILE_IS(dir, AVStr(xdir))  ) dir = xdir;

	if( !SSL_CTX_load_verify_locations(ctx,file,dir)
	 || !SSL_CTX_set_default_verify_paths(ctx) ){
		ERROR("CAs not found or wrong: [%s][%s]",file,dir);
	}else{
		TRACE("CAs = [%s][%s]",file,dir);
	}
}

typedef struct {
  const char	*c_cert;	/* cetificate file */
  const char	*c_key;		/* private key */
} CertKey1;
typedef struct {
	CertKey1 v_ck[8]; /**/
	int	 v_Ncert;
	int	 v_Nkey;
} CertKeyV;
typedef struct {
	CertKeyV x_certkey;
  const char	*x_pass;	/* to decrypt the cert */
  const char	*x_CApath;	/* CA's certificates */
  const char	*x_CAfile;	/* A CA's certificate */
	int	 x_do_SSL;	/* use SSL */
	int	 x_do_STLS;	/* enable STARTTLS */
	int	 x_nego_FTPDATA;
	int	 x_verify;
} SSLContext;

static const char sv_cert_default[] = "server-cert.pem";
static const char sv_key_default[] = "server-key.pem";
static const char sv_certkey_default[] = "server.pem";
static const char cl_cert_default[] = "client-cert.pem";
static const char cl_key_default[] = "client-key.pem";
static const char cl_certkey_default[] = "client.pem";
static const char *stls_proto;

static SSLContext sslctx[2] = {
	{ { {sv_cert_default, sv_key_default} } },
	{ { {cl_cert_default, cl_key_default} } },
};
static int   acc_bareHTTP = 0;
static int   verify_depth = -1;
static int   do_showCERT = 0;
static const char *cipher_list = NULL;

#define XACC	0
#define XCON	1

#define sv_Cert		sslctx[XACC].x_certkey
#define sv_Ncert	sslctx[XACC].x_certkey.v_Ncert
#define sv_Nkey		sslctx[XACC].x_certkey.v_Nkey
#define sv_cert		sslctx[XACC].x_certkey.v_ck[sv_Ncert].c_cert
#define sv_key		sslctx[XACC].x_certkey.v_ck[sv_Nkey].c_key
#define sv_pass		sslctx[XACC].x_pass
#define cl_CApath	sslctx[XACC].x_CApath
#define cl_CAfile	sslctx[XACC].x_CAfile
#define do_accSSL	sslctx[XACC].x_do_SSL
#define do_accSTLS	sslctx[XACC].x_do_STLS
#define cl_vrfy		sslctx[XACC].x_verify
#define cl_nego_FTPDATA	sslctx[XACC].x_nego_FTPDATA

#define cl_Cert		sslctx[XCON].x_certkey
#define cl_Ncert	sslctx[XCON].x_certkey.v_Ncert
#define cl_Nkey		sslctx[XCON].x_certkey.v_Nkey
#define cl_cert		sslctx[XCON].x_certkey.v_ck[cl_Ncert].c_cert
#define cl_key		sslctx[XCON].x_certkey.v_ck[cl_Nkey].c_key
#define cl_pass		sslctx[XCON].x_pass
#define sv_CApath	sslctx[XCON].x_CApath
#define sv_CAfile	sslctx[XCON].x_CAfile
#define do_conSSL	sslctx[XCON].x_do_SSL
#define do_conSTLS	sslctx[XCON].x_do_STLS
#define sv_vrfy		sslctx[XCON].x_verify
#define sv_nego_FTPDATA	sslctx[XCON].x_nego_FTPDATA

#define ST_OPT		1
#define ST_FORCE	2
#define ST_AUTO		4 /* auto-detection of SSL by Client_Hello */
#define ST_SSL		8 /* AUTH SSL for FTP */

static void passfilename(PCStr(keyfile),PVStr(passfile))
{	refQStr(dp,passfile); /**/

	strcpy(passfile,keyfile);
	dp = strrchr(passfile,'.');
	strcpy(dp,".pas");
}
static int Freadline(PCStr(path),char line[],int size)
{	FILE *fp;
	int rcc;
	const char *dp;

	fp = fopen(path,"r");
	if( fp == NULL )
		return -1;

	if( 0 < (rcc = fread(line,1,size,fp)) )
		line[rcc] = 0;
	else	line[0] = 0;
	fclose(fp);
	if( dp = strpbrk(line,"\r\n") )
		truncVStr(dp);
	return strlen(line);
}
static void scanpass(PCStr(arg))
{	const char *file;
	CStr(path,1024);
	const char *pass;
	CStr(passb,128);

	if( strncmp(arg,"file:",5) == 0 ){
		file = arg+5;
		if( LIBFILE_IS(file,AVStr(path)) )
			file = path;
		passb[0] = 0;
		Freadline(file,passb,sizeof(passb));
		pass = passb;
	}else
	if( strncmp(arg,"pass:",5) == 0 ){
		pass = arg + 5;
	}else{
		ERROR("Usage: -pass { file:path | pass:string }");
		return;
	}
	if( pass[0] ){
		sv_pass = cl_pass = strdup(pass);
	}
}

static int setcert1(SSL_CTX *ctx,PCStr(certfile),PCStr(keyfile),int clnt)
{	int code = 0;
	CStr(cwd,1024);
	CStr(xkeyfile,1024);
	CStr(xcertfile,1024);
	const char *dp;
	CStr(passfile,1024);
	CStr(pass,128);

	if( LIBFILE_IS(keyfile,AVStr(xkeyfile)) )
		keyfile = xkeyfile;
	if( LIBFILE_IS(certfile,AVStr(xcertfile)) )
		certfile = xcertfile;

	if( dp = strrchr(keyfile,'.') ){
		passfilename(keyfile,AVStr(passfile));
		if( 0 <= Freadline(passfile,pass,sizeof(pass)) ){
			if( clnt )
				cl_pass = strdup(pass);
			else	sv_pass = strdup(pass);
		}
	}

	getcwd(cwd,sizeof(cwd));
	if( SSL_CTX_use_certificate_file(ctx,certfile,SSL_FILETYPE_PEM) ){
		DEBUG("certfile loaded: %s",certfile);
	}else{
		ERROR("certfile not found or wrong: %s [at %s]",certfile,cwd);
		code = -1;
	}
	if( SSL_CTX_use_RSAPrivateKey_file(ctx,keyfile,SSL_FILETYPE_PEM) ){
		DEBUG("keyfile loaded: %s",keyfile);
	}else{
		ERROR("keyfile not found or wrong: %s [at %s]",keyfile,cwd);
		code = -1;
	}
	if( !SSL_CTX_check_private_key(ctx) ){
		ERROR("key does not match cert: %s %s",keyfile,certfile);
		code = -1;
	}
	return code;
}
static int setcerts(SSL_CTX *ctx,CertKeyV *certv,int clnt)
{	int certx;
	int code;
	CertKey1 *cert1;

	for( certx = 0; certx <= certv->v_Ncert; certx++ ){
		cert1 = &certv->v_ck[certx];
		code = setcert1(ctx,cert1->c_cert,cert1->c_key,clnt);
		if( code != 0 )
			return code;
	}
	return 0;
}

static void *tmprsa_key;
static void *tmprsa_callback(void *ctx,int exp,int bits)
{
	if( bits != 512 && bits != 1024 ){
		bits = 512;
	}
	if( tmprsa_key == NULL ){
		tmprsa_key = RSA_generate_key(bits,0x10001,NULL,NULL);
	}
	return tmprsa_key;
}
static int verify_callback(int ok,void *ctx)
{	int err,depth;
	const char *errsym;
	void *cert;
	CStr(subjb,256);

	cert =   X509_STORE_CTX_get_current_cert(ctx);
	err =    X509_STORE_CTX_get_error(ctx);
	depth =  X509_STORE_CTX_get_error_depth(ctx);
	X509_NAME_oneline(X509_get_subject_name(cert),subjb,sizeof(subjb));
	errsym = X509_verify_cert_error_string(err);
	ERROR("depth=%d/%d ok=%d %d:\"%s\" %s",
		depth,verify_depth,ok,err,errsym,subjb);

	if( !ok ){
		if( depth <= verify_depth )
			ok = 1;
	}
	return ok;
}

#define SSL_ERROR_WANT_READ        2
#define SSL_ERROR_WANT_WRITE       3
#define SSL_ERROR_WANT_X509_LOOKUP 4

static int SSL_rdwr(int wr,SSL *ssl,void *buf,int siz)
{	int i,xcc,err;

	if( wr )
		xcc = SSL_write(ssl,buf,siz);
	else	xcc = SSL_read(ssl,buf,siz);
	if( xcc < 0 ){
		for( i = 0; i < 8; i++ ){
			err = SSL_get_error(ssl,xcc);
			DEBUG("SSL_%s()=%d ERR=%d",wr?"write":"read",xcc,err);

			if( err != SSL_ERROR_WANT_READ
			 && err != SSL_ERROR_WANT_WRITE
			 && err != SSL_ERROR_WANT_X509_LOOKUP
			)
				break;

			if( wr )
				xcc = SSL_write(ssl,buf,siz);
			else	xcc = SSL_read(ssl,buf,siz);
			if( 0 <= xcc )
				break;
		}
	}
	return xcc;
}
#define SSL_read(ss,bf,sz)	SSL_rdwr(0,ss,bf,sz)
#define SSL_write(ss,bf,sz)	SSL_rdwr(1,ss,(char*)bf,sz)

static void writes(PCStr(what),SSL *ssl,int confd,void *buf,int rcc)
{	int wcc,rem;

	rem = rcc;
	while( 0 < rem ){
		if( ssl )
			wcc = SSL_write(ssl,buf,rem);
		else	wcc = write(confd,buf,rem);
		if( wcc == rem )
			DEBUG("%s: %d/%d -> %d%s",what,rem,rcc,wcc,ssl?"/SSL":"");
		else	ERROR("%s? %d/%d -> %d%s",what,rem,rcc,wcc,ssl?"/SSL":"");
		if( wcc <= 0 )
			break;
		rem -= wcc;
	}
}

static int nego_FTPDATAsv(SSL *accSSL,char buf[],int len);
static void nego_FTPDATAcl(SSL *conSSL,char sbuf[],int len);

static void ssl_relay(SSL *accSSL,int accfd,SSL *conSSL,int confd)
{	int fdv[2],rfdv[2],nready,rcc,wcc;
	CStr(buf,8*1024);

	fdv[0] = accfd;
	fdv[1] = confd;

	if( cl_nego_FTPDATA )
		nego_FTPDATAcl(conSSL,"",0);

	for(;;){
		nready = 0;
		rfdv[0] = rfdv[1] = 0;
		if( accSSL && SSL_pending(accSSL) ){
			rfdv[0] = 1;
			nready++;
		}
		if( conSSL && SSL_pending(conSSL) ){
			rfdv[1] = 1;
			nready++;
		}
		if( nready == 0 ){
			nready = PollIns(0,2,fdv,rfdv);
			if( nready <= 0 )
				break;
		}

		if( rfdv[0] ){
			if( accSSL )
				rcc = SSL_read(accSSL,buf,sizeof(buf));
			else	rcc = read(accfd,buf,sizeof(buf));
			if( rcc <= 0 )
				break;
			if( sv_nego_FTPDATA )
				rcc = nego_FTPDATAsv(accSSL,buf,rcc);
			writes("C-S",conSSL,confd,buf,rcc);
		}
		if( rfdv[1] ){
			if( conSSL )
				rcc = SSL_read(conSSL,buf,sizeof(buf));
			else	rcc = read(confd,buf,sizeof(buf));
			if( rcc <= 0 )
				break;
			if( cl_nego_FTPDATA )
				nego_FTPDATAcl(conSSL,buf,rcc);
			writes("S-C",accSSL,accfd,buf,rcc);
		}
	}
}

/*
 * STARTTLS:
 *   RFC2487 SMTP
 *   RFC2595 IMAP and POP3
 *   RFC2228,draft-murray-auth-ftp-ssl-07 FTP
 */
static char *relay_opening(PCStr(proto),FILE *fs,FILE *tc,PVStr(buf),int bsize)
{	CStr(msgb,1024);

	for(;;){
		if( fgets(buf,bsize,fs) == NULL )
			return NULL;
		fputs(buf,tc);
		fflush(tc);

		if( proto != NULL )
			break;
		if( strncmp(buf,"220",3) == 0 ){
			if( buf[3] == '-' ){
				do {
					fgets(msgb,sizeof(msgb),fs);
					fputs(msgb,tc);
				} while( msgb[3] == '-' );
				fflush(tc);
			}
			if( strstr(buf,"FTP") )
				proto = "FTP";
			else
			proto = "SMTP";
			break;
		}else
		if( strncasecmp(buf,"+OK",3) == 0 ){
			proto = "POP3";
			break;
		}else
		if( strncasecmp(buf,"* OK",4) == 0 ){
			proto = "IMAP";
			break;
		}else{
			return NULL;
		}
	}
	return (char*)proto;
}
static int isinSSL(int fd)
{	unsigned char buf[6]; /**/
	int rcc,leng,type,vmaj,vmin;

	buf[0] = 0x7F;
	RecvPeek(fd,buf,1);
	if( (buf[0] & 0x80) || buf[0] < 0x20 ){
		ERROR("STARTTLS got binary [%X] from client",0xFF&buf[0]);
		if( buf[0] == 0x80 ){
			rcc = RecvPeek(fd,buf,5);
			ERROR("SSL Hello?%d [%X %d %d %d %d]",rcc,buf[0],
				buf[1],buf[2],buf[3],buf[4]);
			leng = (0x7F&buf[0]) << 8 | buf[1];
			type = buf[2];
			if( type == 1 ){ /* SSLv3 ClientHello */
				vmaj = buf[3];
				vmin = buf[4];
				return 1;
			}
		}
		else
		if( buf[0] == 22 ){ /* ConentType:handshake */
			rcc = RecvPeek(fd,buf,sizeof(buf));
			ERROR("SSL Hello?%d [%X %d %d %d %d]",rcc,buf[0],
				buf[1],buf[2],buf[3]<<8|buf[4],buf[5]);
			if( buf[5] == 1 ){
				return 1;
			}
		}
	}
	return 0;
}
static int starttls(int accfd,int confd)
{	FILE *fc,*tc,*fs,*ts;
	int fdv[2],rfdv[2];
	CStr(buf,1024);
	CStr(com,32);
	CStr(arg,32);
	const char *msg;
	CStr(msgb,1024);
	const char *dp;
	const char *proto;
	int xi;

	fdv[0] = accfd;
	fdv[1] = confd;
	fc = fdopen(fdv[0],"r"); setbuf(fc,NULL);
	tc = fdopen(fdv[0],"w");
	fs = fdopen(fdv[1],"r"); setbuf(fs,NULL);
	ts = fdopen(fdv[1],"w");

	proto = stls_proto;
	if( do_conSSL && do_conSTLS ){
		proto = relay_opening(proto,fs,tc,AVStr(buf),sizeof(buf));
		if( proto == NULL )
			return -1;

		ERROR("STARTTLS to server -- %s",proto);
		if( strcasecmp(proto,"FTP") == 0 ){
			if( do_conSTLS & ST_SSL )
				fputs("AUTH SSL\r\n",ts);
			else{
				fputs("AUTH TLS\r\n",ts);
				if( do_conSTLS & ST_FORCE )
					cl_nego_FTPDATA = 1;
			}
		}else
		if( strcasecmp(proto,"SMTP") == 0 ){
			fputs("STARTTLS\r\n",ts);
		}else
		if( strncasecmp(proto,"POP",3) == 0 ){
			fputs("STLS\r\n",ts);
		}else
		if( strcasecmp(proto,"IMAP") == 0 ){
			fputs("stls0 STARTTLS\r\n",ts);
		}
		fflush(ts);
		if( fgets(buf,sizeof(buf),fs) == NULL )
			return -1;
		if( dp = strpbrk(buf,"\r\n") )
			truncVStr(dp);
		ERROR("STARTTLS to server -- %s",buf);
	}
	if( do_accSSL && do_accSTLS ){
	  for( xi = 0; ; xi++ ){
	    PollIns(0,2,fdv,rfdv);
	    if( rfdv[0] ){
		if( xi == 0 /* && accept implicit SSL too */ ){
			if( isinSSL(fdv[0]) )
				return 0;
			if( do_accSTLS == ST_AUTO ){
				ERROR("SSL-autodetect C-S: not in SSL");
				do_accSSL = 0;
				return 0;
			}
		}
		if( fgets(buf,sizeof(buf),fc) == NULL )
			return -1;
		dp = wordscanX(buf,AVStr(com),sizeof(com));
		wordscanX(dp,AVStr(arg),sizeof(arg));
		ERROR("STARTTLS prologue: C-S: [%s][%s]",com,arg);

		/* SMTP */
		if( strcasecmp(com,"EHLO") == 0 ){
			write(accfd,"250 STARTTLS\r\n",14);
			continue;
		}
		if( strcasecmp(com,"STARTTLS") == 0 ){
			msg = "220 Ready to start TLS\r\n";
			write(accfd,msg,strlen(msg));
			ERROR("STARTTLS from SMTP client -- OK");
			break;
		}

		/* POP3 */
		if( strcasecmp(com,"STLS") == 0 ){
			msg = "+OK Begin TLS negotiation\r\n";
			write(accfd,msg,strlen(msg));
			ERROR("STARTTLS from POP client -- OK");
			break;
		}

		/* IMAP */
		if( strcasecmp(arg,"CAPABILITY") == 0 ){
			msg = "* CAPABILITY STARTTLS\r\n";
			write(accfd,msg,strlen(msg));
			sprintf(msgb,"%s OK CAPABILITY\r\n",com);
			write(accfd,msgb,strlen(msgb));
			continue;
		}
		if( strcasecmp(arg,"STARTTLS") == 0 ){
			sprintf(msgb,"%s OK Begin TLS negotiation\r\n",com);
			write(accfd,msgb,strlen(msgb));
			ERROR("STARTTLS from IMAP client -- OK");
			break;
		}

		/* FTP */
		if( strcasecmp(com,"AUTH") == 0 )
		if( strcasecmp(arg,"TLS") == 0 || strcasecmp(arg,"SSL") == 0 ){
			msg = "234 OK\r\n";
			write(accfd,msg,strlen(msg));
			ERROR("AUTH %s from FTP client -- 234 OK",arg);
			if( strcasecmp(arg,"TLS") == 0 && do_accSTLS == ST_FORCE )
				sv_nego_FTPDATA = 1;
			break;
		}

		/* HTTP */
		if( strcasecmp(com,"CONNECT") == 0 ){
			if( proto == 0 ){
				proto = "http";
			}
		}

		if( do_accSTLS == 2 ){
			ERROR("STARTTLS required");
			if( proto != 0 && strcasecmp(proto,"IMAP") == 0 )
				fprintf(tc,"%s BAD do STARTTLS first.\r\n",com);
			else
			if( proto != 0 && strcasecmp(proto,"POP") == 0 )
				fprintf(tc,"+ERR do STLS first.\r\n");
			else
			fprintf(tc,"530 do STARTTLS first.\r\n");
			fflush(tc);
			return -1;
		}
		fputs(buf,ts);
		fflush(ts);
	    }
	    if( rfdv[1] ){
		if( xi == 0 ){
			if( isinSSL(fdv[1]) ) /* will not match */
				return 0;
			if( do_accSTLS == ST_AUTO ){
				ERROR("SSL-autodetect S-C: not in SSL");
				do_accSSL = 0;
				return 0;
			}
		}
		if( proto == NULL ){
			proto = relay_opening(proto,fs,tc,AVStr(buf),sizeof(buf));
			if( proto == NULL )
				return -1;
			ERROR("STARTTLS to client -- %s",proto);
		}else{
		if( fgets(buf,sizeof(buf),fs) == NULL )
			return -1;
		fputs(buf,tc);
		}
		/* HTTP */
		if( proto != NULL && streq(proto,"http") ){
			if( buf[0] == '\r' || buf[1] == '\n' ){
				ERROR("STARTTLS prologue: S-C HTTP-CONNECT DONE");
				fflush(tc);
				break;
			}
		}
		if( dp = strpbrk(buf,"\r\n") )
			truncVStr(dp);
		ERROR("STARTTLS prologue: S-C: %s",buf);
		fflush(tc);
	    }
	  }
	}
	return 0;
}
static int nego_FTPDATAsv(SSL *accSSL,char buf[],int len)
{	CStr(com,32);
	CStr(arg,32);
	const char *dp;
	const char *msg;

	buf[len] = 0;
	dp = wordscanX(buf,AVStr(com),sizeof(com));
	wordscanX(dp,AVStr(arg),sizeof(arg));
	if( strcasecmp(com,"PBSZ") == 0 ){
		msg = "200 OK\r\n";
		SSL_write(accSSL,msg,strlen(msg));
		ERROR("PBSZ %s from FTP client -- 200 OK",arg);
		len = 0;
	}
	else
	if( strcasecmp(com,"PROT") == 0 ){
		msg = "200 OK\r\n";
		SSL_write(accSSL,msg,strlen(msg));
		ERROR("PROT %s from FTP client -- 200 OK",arg);
		len = 0;
		sv_nego_FTPDATA = 0;
	}
	return len;
}
#define FTP_LOGIN_OK	"230"
static void nego_FTPDATAcl(SSL *conSSL,char sbuf[],int len)
{	const char *msg;
	CStr(buf,64);
	CStr(resp,64);
	int rcc;

	if( len != 0 )
	if( strncmp(sbuf,FTP_LOGIN_OK,strlen(FTP_LOGIN_OK)) != 0 )
		return;

	msg = "PBSZ 0\r\n";
	SSL_write(conSSL,msg,strlen(msg));
	rcc = SSL_read(conSSL,buf,sizeof(buf));
	buf[rcc] = 0;
	linescanX(buf,AVStr(resp),sizeof(resp));
	ERROR("STARTTLS/FTP PBSZ 0 -> %s",resp);
	if( atoi(resp) != 200 )
		return;

	msg = "PROT P\r\n";
	SSL_write(conSSL,msg,strlen(msg));
	rcc = SSL_read(conSSL,buf,sizeof(buf));
	buf[rcc] = 0;
	linescanX(buf,AVStr(resp),sizeof(resp));
	ERROR("STARTTLS/FTP PROT P -> %s",resp);
	if( atoi(resp) == 200 )
		cl_nego_FTPDATA = 0;
}

static int HTTP_CAresp(int fd,PCStr(certfile))
{	FILE *tc,*cfp;
	X509 *cert;
	BIO *in,*out;

	tc = fdopen(fd,"w");
	cfp = fopen(certfile,"r");
	if( cfp == NULL )
		return -1;

	fprintf(tc,"HTTP/1.0 200 ok\r\n");
	fprintf(tc,"MIME-Version: 1.0\r\n");
	fprintf(tc,"Content-Type: application/x-x509-ca-cert\r\n");
	fprintf(tc,"\r\n");

	in = BIO_new_fp(cfp,BIO_NOCLOSE);
	cert = PEM_read_bio_X509(in,NULL,NULL);
	out = BIO_new_fp(tc,BIO_NOCLOSE);
	i2d_X509_bio(out,cert);

	BIO_free(in);
	BIO_free(out);
	fclose(tc);
	return 0;
}
static int CArequest(int accfd,int *isHTTP,PCStr(certfile))
{	CStr(method,8);
	CStr(line,1024);
	CStr(url,1024);
	int rcc;

	setNonblockingIO(accfd,1);
	rcc = RecvPeek(accfd,method,6);
	setNonblockingIO(accfd,0);

	if( rcc <= 0 )
		return 0;

	method[rcc] = 0;

	if( strncmp(method,"GET ",4) == 0 ){
		setNonblockingIO(accfd,1);
		rcc = RecvPeek(accfd,line,16);
		setNonblockingIO(accfd,0);
		line[rcc] = 0;

		wordscanX(line+4,AVStr(url),sizeof(url));
		if( strcmp(url,"/-/ca.der") == 0 ){
			HTTP_CAresp(0,certfile);
			TRACE("sent cert");
			return 1;
		}
		*isHTTP = 1;
	}else
	if( strncmp(method,"HEAD ",5) == 0 || strncmp(method,"POST ",5) == 0 )
		*isHTTP = 1;
	return 0;
}
static void rand_seed()
{	int seed[8],si;

	seed[0] = Gettimeofday(&seed[1]);
	seed[2] = getpid();
	seed[3] = getuid();
	seed[4] = (int)seed;
	seed[5] = (int)rand_seed;
	RAND_seed(seed,sizeof(seed));
	for( si = 0; si < 8; si++ )
		seed[si] = 0;
}
static int _passwd(PCStr(what),PCStr(pass),PCStr(keyfile),char buf[],int siz,int vrfy)
{	CStr(passfile,1024);

	if( pass ){
		TRACE("passphrase for %s -- OK",keyfile);
		Xstrcpy(ZVStr(buf,siz),pass);
		return strlen(pass);
	}else{
		passfilename(keyfile,AVStr(passfile));
		ERROR("passphrase for %s -- ERROR: '%s' file not found and SSL_%s_KEY_PASSWD undefined",
			keyfile,passfile,what);
		return -1;
	}
}
static int sv_passwd(char buf[],int siz,int vrfy)
{
	return _passwd("SERVER",sv_pass,sv_key,buf,siz,vrfy);
}
static int cl_passwd(char buf[],int siz,int vrfy)
{
	return _passwd("CLIENT",cl_pass,cl_key,buf,siz,vrfy);
}

static void put_help()
{
	syslog_ERROR("SSLway 2004-12-5 <ysato@delegate.org>\r\n");
}
extern int mainX(int ac,char*av[]);
extern int RANDSTACK_RANGE;

int main(int ac,char *av[])
{
	randtext(-1);
	RANDSTACK_RANGE = 256;
	return randstack_call(1,(iFUNCP)mainX,ac,av);
}
int mainX(int ac,char *av[])
{	int ai;
	const char *arg;
	int accfd,confd;
	SSL_CTX *ctx;
	SSL *accSSL,*conSSL;
	const char *env;
	int fdv[2],rfdv[2];
	int vflag;
	int ctrlopt = 0;
	int vflags = 0;
	void *store;
	int nodelay = 1;

	CFI_init(ac,(const char**)av);
	av = move_envarg(ac,(const char**)av,NULL,NULL,NULL);

	if( env = getenv("CFI_TYPE") ){
		if( strcmp(env,"FCL") == 0 ){
			DEBUG("CFI_TYPE=%s: -ac is assumed",env);
			do_accSSL = 1;
		}else
		if( strcmp(env,"FSV") == 0 || strcmp(env,"FMD") == 0 ){
			DEBUG("CFI_TYPE=%s: -co is assumed",env);
			do_conSSL = 1;
		}
	}
	if( env = getenv("CFI_STAT") ){
		int fd;
		fd = atoi(env);
		stdctl = fdopen(fd,"w");
		fprintf(stdctl,"CFI/1.0 100 start\r\n");
		fflush(stdctl);
	}
	if( env = getenv("SSL_KEY_PASSWD") )
		sv_pass = cl_pass = env;
	if( env = getenv("SSL_CLIENT_KEY_PASSWD") )
		cl_pass = env;
	if( env = getenv("SSL_SERVER_KEY_PASSWD") )
		sv_pass = env;

	PID = getpid();
	if( (client_host = getenv("REMOTE_HOST")) == 0 )
		client_host = "?";

	accfd = dup(0);
	confd = dup(1);

	if( env = getenv("SSL_CIPHER") ) cipher_list = env;

	if( env = getenv("SSL_CERT_FILE") )
		sv_cert = sv_key = cl_cert = cl_key = env;

	if( env = getenv("SSL_SERVER_KEY_FILE" ) ) sv_key  = env;
	if( env = getenv("SSL_SERVER_CERT_FILE") ) sv_cert = env;

	if( env = getenv("SSL_CLIENT_KEY_FILE" ) ) cl_key  = env;
	if( env = getenv("SSL_CLIENT_CERT_FILE") ) cl_cert = env;

	for( ai = 1; ai < ac; ai++ ){
		arg = av[ai];
		if( strcmp(arg,"-help") == 0 || strcmp(arg,"-v") == 0 ){
			put_help();
			exit(0);
		}else 
		if( strncmp(arg,"-vv",3) == 0 || strncmp(arg,"-vd",3) == 0 ){
			loglevel = LDEBUG;
		}else
		if( strncmp(arg,"-vu",3) == 0 ){
			loglevel = LTRACE;
		}else
		if( strncmp(arg,"-vt",3) == 0 ){
			loglevel = LERROR;
		}else
		if( strncmp(arg,"-vs",3) == 0 ){
			loglevel = LSILENT;
		}else
		if( strncasecmp(arg,"-ss",3) == 0 ){
			do_accSTLS = do_conSTLS = ST_SSL;
			if( arg[3] == '/' && arg[4] != 0 )
				stls_proto = strdup(arg+4);
		}else
		if( strncasecmp(arg,"-st",3) == 0 ){
			do_accSTLS = do_conSTLS = arg[1]=='s'?1:2;
			if( arg[3] == '/' && arg[4] != 0 )
				stls_proto = strdup(arg+4);
		}else
		if( strncasecmp(arg,"-ad",3) == 0 ){
			do_accSTLS = do_conSTLS = ST_AUTO;
		}else
		if( strncmp(arg,"-ac",3) == 0 ){
			do_accSSL = 1;
			if( strncmp(arg+3,"/st",3) == 0 ) do_accSTLS = 1;
		}else
		if( strncmp(arg,"-co",3) == 0 ){
			do_conSSL = 1;
			if( strncmp(arg+3,"/st",3) == 0 ) do_conSTLS = 1;
		}else
		if( strncmp(arg,"-ht",3) == 0 ){
			acc_bareHTTP = 1;
		}else
		if( strncmp(arg,"-show",3) == 0 ){
			do_showCERT = 1;
		}else
		if( strcmp(arg,"-CApath") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s directory-name",arg);
				return -1;
			}
			cl_CApath = sv_CApath = av[++ai];
		}else
		if( strcmp(arg,"-CAfile") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s file-name",arg);
				return -1;
			}
			cl_CAfile = sv_CAfile = av[++ai];
		}else
		if( strcasecmp(arg,"-vrfy")==0 || strcasecmp(arg,"-auth")==0 ){
			vflag = SSL_VERIFY_PEER
				| SSL_VERIFY_CLIENT_ONCE;
			if( arg[1] == 'V' || arg[1] == 'A' )
				vflag |= SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			if( arg[1] == 'V' || arg[1] == 'v' )
				verify_depth = -1;
			else	verify_depth = 10;
			cl_vrfy = vflag;
			sv_vrfy = vflag;
		}else
		if( strcasecmp(arg,"-verify") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s max-depth",arg);
				return -1;
			}
			verify_depth = atoi(av[++ai]);
			vflag = SSL_VERIFY_PEER | SSL_VERIFY_CLIENT_ONCE;
			if( arg[1] == 'V' )
				vflag |= SSL_VERIFY_FAIL_IF_NO_PEER_CERT;
			cl_vrfy = vflag;
			sv_vrfy = vflag;
		}else
		if( strcmp(arg,"-client_auth") == 0 ){
			verify_depth = 10;
			cl_vrfy = SSL_VERIFY_PEER
				| SSL_VERIFY_CLIENT_ONCE;
		}else
		if( strcmp(arg,"-cipher") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s cipher-list",arg);
				return -1;
			}
			cipher_list = av[++ai];
		}else
		if( strcmp(arg,"-certkey") == 0
		 || strcmp(arg,"-cert") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s cert-key-file-name",arg);
				return -1;
			}
			if( sv_cert != sv_cert_default ){
				if( elnumof(sv_Cert.v_ck) <= sv_Ncert+1 ){
				}else{
					sv_Ncert++;
					sv_Nkey++;
				}
			}
			if( cl_cert != cl_cert_default ){
				if( elnumof(cl_Cert.v_ck) <= cl_Ncert+1 ){
				}else{
					cl_Ncert++;
					cl_Nkey++;
				}
			}
			sv_cert = sv_key = cl_cert = cl_key = av[++ai];
		}
		else
		if( strcmp(arg,"-key") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s key-file-name",arg);
				return -1;
			}
			sv_key = cl_key = av[++ai];
		}
		else
		if( strcmp(arg,"-pass") == 0 ){
			if( ac <= ai + 1 ){
				ERROR("Usage: %s {pass:str|file:path}");
				return -1;
			}
			scanpass(av[++ai]);
		}
		else
		if( strcmp(arg,"-bugs") == 0 ){
			ctrlopt = 0x000FFFFFL; /* SSL_OP_ALL */
		}
		else
		if( strcmp(arg,"-delay") == 0 ){
			nodelay = 0;
		}
		else
		if( strcmp(arg,"-crl_check") == 0 ){
			vflags |= X509_V_FLAG_CRL_CHECK;
		}
		else
		if( strcmp(arg,"-crl_check_all") == 0 ){
			vflags |= X509_V_FLAG_CRL_CHECK
				| X509_V_FLAG_CRL_CHECK_ALL;
		}
	}

	accSSL = NULL;
	conSSL = NULL;

	if( do_conSSL || do_accSSL )
	{
		rand_seed();
		TRACE("start");
	}

	if( nodelay ){
		set_nodelay(accfd,1);
		set_nodelay(confd,1);
	}

	fdv[0] = accfd;
	fdv[1] = confd;

	if( acc_bareHTTP ){
		int isHTTP;
		if( 0 < PollIns(100,2,fdv,rfdv) && 0<rfdv[0] && rfdv[1] <= 0 ){
			isHTTP = 0;
			if( CArequest(accfd,&isHTTP,sv_cert) )
				return 0;
			if( isHTTP ){
				 /* ... through raw HTTP request ...
				do_accSSL = 0;
				 */
			}
		}
	}

	if( do_conSSL && do_conSTLS || do_accSSL && do_accSTLS ){
		if( starttls(accfd,confd) < 0 )
			return -1;
	}

	if( do_conSSL ){
		ctx = ssl_new(0);
		SSL_CTX_set_default_passwd_cb(ctx,(pem_password_cb*)cl_passwd);
		if( cipher_list )
			SSL_CTX_set_cipher_list(ctx,cipher_list);
		if( cl_cert != cl_cert_default || LIBFILE_IS(cl_cert,VStrNULL) )
			setcerts(ctx,&cl_Cert,1);

		if( sv_CAfile || sv_CApath )
			ssl_setCAs(ctx,sv_CAfile,sv_CApath);
		if( sv_vrfy )
			SSL_CTX_set_verify(ctx,sv_vrfy,verify_callback);
		if( vflags ){
			store = SSL_CTX_get_cert_store(ctx);
			X509_STORE_set_flags(store, vflags);
		}

		conSSL = ssl_conn(ctx,confd);
		if( conSSL == NULL )
			return -1;
	}

	if( do_accSSL ){
		ctx = ssl_new(1);
		if( ctrlopt )
			SSL_CTX_ctrl(ctx,32/*SSL_CTRL_OPTIONS*/,ctrlopt,NULL);
		SSL_CTX_set_default_passwd_cb(ctx,(pem_password_cb*)sv_passwd);
		if( cipher_list )
			SSL_CTX_set_cipher_list(ctx,cipher_list);
		if( setcerts(ctx,&sv_Cert,0) < 0 )
			return -1;
		SSL_CTX_set_tmp_rsa_callback(ctx,tmprsa_callback);

		if( cl_CAfile || cl_CApath )
			ssl_setCAs(ctx,cl_CAfile,cl_CApath);
		if( cl_vrfy )
			SSL_CTX_set_verify(ctx,cl_vrfy,verify_callback);
		if( vflags ){
			store = SSL_CTX_get_cert_store(ctx);
			X509_STORE_set_flags(store, vflags);
		}

		accSSL = ssl_acc(ctx,accfd);
		if( accSSL == NULL )
			return -1;
	}

	if( conSSL )
		ssl_prcert(conSSL,do_showCERT,NULL,  accfd,"server");
	if( accSSL )
		ssl_prcert(accSSL,do_showCERT,accSSL,accfd,"client");

	ssl_relay(accSSL,accfd,conSSL,confd);

	if( do_conSSL || do_accSSL )
		TRACE("done");
	return 0;
}
