/*///////////////////////////////////////////////////////////////////////
Copyright (c) 2002 National Institute of Advanced Industrial Science and Technology (AIST)

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
AIST MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	htaccept.c (ACCEPT method for circuit level proxing)
Author:		Yutaka Sato <ysato@delegate.org>
Description:
	A tentative extension method "ACCEPT" of HTTP, which could
	be used for replacement of BIND command of SOCKS.

	Request:

	   ACCEPT localhost:localport HTTP/1.1
	  +Host: remotehost
	  +Pragma: timeout=seconds
	  +Authorization: ...

	      - localhost and localport can be "*" representing wild-card
	      - all header fields are optional
	      - remotehost can be used to determine
		- forwarding to upstream proxy
		- interface when localhost is "*"
		- access limitation for accept

	Response:

	   HTTP/1.1 100 bound
	   Host: localhost:localport
           Pragma: timeout=seconds

	   HTTP/1.1 200 accepted
	   Host: remotehost:remoteport

	   [bidirectional communication]

	Error:
	   HTTP/1.1 500 could not bound
	   HTTP/1.1 500 accept timedout

	This is enabled with DeleGate paramegers:
		HTTPCONF=methods:+,ACCEPT
		REMIKTTALBE=+,tcprelay
		PORT=8000-8010  // localports to be shared

History:
	020308	created
TODO:
	ACCEPT name
	CONNECT name
	ACCEPT -vhost:port
	CONNECT -vhost:port
//////////////////////////////////////////////////////////////////////#*/
#include "vsocket.h"
#include "delegate.h"
#include "fpoll.h"
#include "http.h"
extern int S_ADDRNOTAVAIL;

int HTTP_ACCEPT(Connection *Conn,PCStr(req),PCStr(head),FILE *fc,FILE *tc)
{	CStr(local,256);
	CStr(remote,256);
	int bsock,asock,timeout,lockfd,rcc;
	int nready,fdn,fdv[2],rdv[2];
	FILE *fs,*ts;
	CStr(ver,32);
	CStr(present,0x1000);
	int pcc;

	if( getFieldValue2(head,"Host",AVStr(remote),sizeof(remote)) ){
	}else	strcpy(remote,"*:*");
	Xsscanf(req,"%*s %s",AVStr(local));
	timeout = 10*1000;
	sprintf(ver,"HTTP/%s",MY_HTTPVER);

	bsock = VSocket(Conn,"BIND/HTTP",-1,AVStr(local),AVStr(remote),"listen=1");
	if( bsock < 0 ){
		if( S_ADDRNOTAVAIL ){
			sv1log("%s: should be forwarded to upstream\n",
				local);
		}
		fprintf(tc,"%s 500 could not bound\r\n\r\n",ver);
		return -2;
	}
	fprintf(tc,"%s 100 bound\r\n",ver);
	fprintf(tc,"Host: %s\r\n",local);
	fprintf(tc,"\r\n");
	fflush(tc);

	fdv[0] = bsock;
	fdv[1] = fileno(fc);
	fdn = 2;
	pcc = 0;
	for(;;){
		nready = PollIns(timeout,fdn,fdv,rdv);
		if( nready <= 0 ){
			closeNonReservedPortSock(bsock);
			fprintf(tc,"%s 500 accept timedout\r\n\r\n",ver);
			return -3;
		}
		if( rdv[0] != 0 )
			break;
		if( rdv[1] != 0 ){
			if( sizeof(present) == pcc ){
				/* stop buffering */
				fdn = 1;
			}
			rcc = read(fdv[1],present+pcc,sizeof(present)-pcc);
			if( 0 < rcc )
				pcc += rcc;
			if( rcc <= 0 ){
				/* disconnection from client */
				closeNonReservedPortSock(bsock);
				return -4;
			}
		}
	}
	lockfd = -1;
	asock = ACCEPT(bsock,0,lockfd,timeout);
	closeNonReservedPortSock(bsock);
	if( asock < 0 ){
		fprintf(tc,"%s 500 accept failed\r\n\r\n",ver);
		return -5;
	}
	getpeerName(asock,AVStr(remote),"%A:%P");
	fprintf(tc,"%s 200 accepted\r\n",ver);
	fprintf(tc,"Host: %s\r\n",remote);
	fprintf(tc,"\r\n");
	fflush(tc);

	fs = fdopen(asock,"r");
	ts = fdopen(asock,"w");
	if( fs == NULL || ts == NULL ){
		fprintf(tc,"%s 500 internal error\r\n\r\n",ver);
		return -6;
	}

	if( 0 < pcc )
		write(asock,present,pcc);
	rcc = relayf_svcl(Conn,fc,tc,fs,ts);
	close(asock);
	return rcc;
}
