/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: BackgroundOpWidget.cc,v 1.34 2002/09/08 19:28:37 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include <sstream>

#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include <assert.h>

#include <gtk--/packer.h>

#include "BackgroundOpWidget.hh"
#include "StockButton.hh"

#include <list>

using namespace libfwbuilder;
using namespace SigC;

BackgroundOpWidget::BackgroundOpWidget() : Gtk::Packer()
{
    init();
    bop=NULL;
}

BackgroundOpWidget::BackgroundOpWidget(BackgroundOp *b) : Gtk::Packer()
{
    init();
    bop=b;
}

void BackgroundOpWidget::init()
{
    Gtk::ScrolledWindow *sw;

    set_border_width(10);
    set_default_pad(0, 10);

    text = manage(new Gtk::CList(1));
    sw   = manage(new Gtk::ScrolledWindow());

    text->column_titles_hide();
    text->column(0).set_auto_resize(true);

    sw->set_policy(GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

//    Gtk::Viewport *vp  = manage( new Gtk::Viewport() );
//    vp->add(*text);
//    sw->add(*vp);
    sw->add(*text);

    add(*sw,GTK_SIDE_TOP, GTK_ANCHOR_CENTER, GTK_EXPAND|GTK_FILL_X|GTK_FILL_Y);

    show_all();

    can_focus();

    logger=NULL;

    hide.connect(SigC::slot(this,&BackgroundOpWidget::on_hide));
}

BackgroundOpWidget::~BackgroundOpWidget()
{
/* this is last resort - if background op. is still there, disconnect it so
 * the widget can die piecfully 
 */
    if (bop!=NULL)  bop->disconnect();
}

void BackgroundOpWidget::connect(libfwbuilder::BackgroundOp *b)
{
    bop=b;
}

void BackgroundOpWidget::disconnect()
{
    if (bop!=NULL)  bop->disconnect();
    bop=NULL;
}

void BackgroundOpWidget::on_hide()
{
    bop->disconnect();
}


BackgroundOpWidget& BackgroundOpWidget::operator<< (const string &str)
{
    return this->operator<<(str.c_str());
}

BackgroundOpWidget& BackgroundOpWidget::operator<< (int    i)
{
    char  str[64];
    sprintf(str,"%d",i);
    return this->operator<<(str);
}


BackgroundOpWidget& BackgroundOpWidget::operator<< (const char *str)
{
    char *cptr;
    char *buf=cxx_strdup(str);
    bool  lf=false;
    if ( (cptr=strchr(buf,'\n'))!=NULL) 
    {
        *cptr='\0';
        lf=true;
    }

    int n=text->rows().size() -1;
    if (n==-1)
    {
        char *sa[1]={ "" };
        text->rows().push_back(sa);
        n=0;
    }
    string last=text->cell(n,0).get_text();
    last=last+buf;
    text->cell(n,0).set_text(last);
    if (lf) {
        char *sa[1]={ "" };
        text->rows().push_back(sa);
        text->moveto(n+1, 0, 1.0, 0);
    }
    delete[] buf;
    return (*this);
}

void  BackgroundOpWidget::clear()
{
    text->rows().clear();
}

/*
 *  Explanation about signal "completed"
 *
 *  Class BackgroundOp clears its flag "running" when background
 *  operation finishes. BackgroundOpWidget connects callback to idle,
 *  so that it can keep track of flag "running" and react when it
 *  changes. BackgroundOpWidget changes button in this callback and
 *  sends signal "completed" to whatever widget connected its callback
 *  to it, so higher level widget can also do something when
 *  operation finishes. If BackgroundOpWidget is used as a part of a
 *  dialog then most probably there will be no use for this signal as
 *  modal dialog will simply stay there waiting for user to click the
 *  "OK" button to close it. On the other hand, if BackgroundOpWidget
 *  is used in non-modal mode then higher level widget will probably
 *  connect its callback to "completed" signal to react. This is used
 *  in Druid running DNS queries.
 *
 *  This callback function gets an argument, which is a flag indicating
 *  whether background operation encountered an error. Detailed error
 *  explanation can be obtained using a call to bop->get_latest_error()
 *  BackgroundOpWidget passes this code up to the higher level widget
 *  the same way - as an argument to the "completed" signal
 *
 *
 *  (vk)
 */
gint BackgroundOpWidget::monitor_operation()
{
    if( logger->ready() ) {
        *this << logger->getLine();
        return 1;
    }

    if (bop==NULL)         return 0; // BackgroundOp has been disconnected

    if (bop->isRunning()) 
    {
        struct timespec ts,rem;
        ts.tv_sec=0;
        ts.tv_nsec=200000000;
        nanosleep(&ts,&rem);
        return 1;
    }
    // send signal "completed", argument is 0 if ok and -1 if error

    FWException *ex=bop->get_latest_error();
    if (ex) {
        *this << ex->toString();
        completed(-1);   // this sends signal to another widget
    } else
        completed(0);   // this sends signal to another widget

    return 0;
}


void BackgroundOpWidget::stop()
{
    if( bop->isRunning()) {

        *this << _("Terminating task. Please wait...\n");


        bop->stop_operation();
    }
}


void BackgroundOpWidget::execute()  throw(FWException)
{
    assert(bop!=NULL);
    
    show();

    try 
    {
	logger=bop->start_operation();

        Gtk::Main::idle.connect(slot(this,&BackgroundOpWidget::monitor_operation));

    } catch(FWException &ex)
    {
	*this << "\n* " << ex.toString();
        throw;
    }
    
    return;
}

string  BackgroundOpWidget::getLogText() 
{
    std::ostringstream str;
    if (text)
    {
        Gtk::CList::RowList &rows=text->rows();
        for (Gtk::CList::RowIterator i=rows.begin(); i!=rows.end(); ++i)
        {
            str << (*i)[0].get_text() << endl;
        }
        return str.str();
    }
    else
        return string("");
}
