/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: FWObjectDatabaseGUI.cc,v 1.48 2003/06/02 07:41:46 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>
#include <assert.h>
#include <unistd.h>

#include <iostream>
#include <algorithm>
#include <functional>

#include "config.h"

#include "fwbuilder/FWObject.hh"
#include "FWObjectDatabaseGUI.hh"
#include "TMPGroupObject.hh"

#include "InterfaceData.hh"

#include "fwbuilder/FWOptions.hh"

#include "fwbuilder/Host.hh"
#include "fwbuilder/Interface.hh"
#include "fwbuilder/IPv4.hh"
#include "fwbuilder/physAddress.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/AddressRange.hh"
#include "fwbuilder/Interval.hh"
#include "fwbuilder/IPService.hh"
#include "fwbuilder/ICMPService.hh"
#include "fwbuilder/TCPService.hh"
#include "fwbuilder/UDPService.hh"
#include "fwbuilder/CustomService.hh"
#include "fwbuilder/Management.hh"

#include "fwbuilder/Rule.hh"
#include "fwbuilder/RuleElement.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/InterfacePolicy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/ObjectGroup.hh"
#include "fwbuilder/ServiceGroup.hh"
#include "fwbuilder/IntervalGroup.hh"
#include "fwbuilder/Resources.hh"

#include "fwbuilder/snmp.hh"
#include "fwbuilder/dns.hh"

#include "FileSel.hh"
#include "MessageDialog.hh"
#include "Preferences.hh"
#include "main_window.hh"
#include "GenericBackgroundOpDialog.hh"

using namespace libfwbuilder;

/****************************************************************/

const string FWObjectDatabaseGUI::DB_TEMPLATE_FILE_NAME = "objects_init.xml" ;

FWObjectDatabaseGUI::FWObjectDatabaseGUI() : FWObjectDatabase()
{
    setDirty(false,true);
}

FWObjectDatabaseGUI::FWObjectDatabaseGUI(const string &filename) throw(FWException)
:FWObjectDatabase()
{
    load(filename,true);
}

FWObjectDatabaseGUI::~FWObjectDatabaseGUI()
{
}

bool FWObjectDatabaseGUI::saveIfModified() throw(FWException)
{
    MessageDialog::DlgReturnValue action;
    
    action=MessageDialog::Question(
        _("Some objects have been modified but not saved.\n\
Do you want to save changes now ?"), main_window::windows.back() );
    
    switch (action) 
    {
    case MessageDialog::OK_YES: save(); break; 
    case MessageDialog::NO:             break;         
    case MessageDialog::CANCEL:         return(false);
    }
    return(true);
}

void FWObjectDatabaseGUI::load(const string &f,bool check_perm) throw(FWException)
{
    if(isDirty(true)) 
    {   // checking all the object recursively
	if (!saveIfModified()) 
            return;  // "Cancel"
    }
    
    data_file = f;
    
    if(data_file=="") 
        return;

    if ( data_file.find("/")!=0 )
    {
        char buf[1024];
        getcwd(buf, sizeof(buf)-1 );
        data_file= string(buf) + "/" + data_file;
    }

    if (check_perm && access(data_file.c_str(), W_OK)!=0 && errno==EACCES)
    {
        char errmsg[1024];
        sprintf(errmsg,
                _("Data file %s  is read-only. You won't be able to save your changes."),
                data_file.c_str());
        MessageDialog::Warning(errmsg);
    }
    
    MessageBoxUpgradePredicate upgrade_predicate;
    FWObjectDatabase::load(data_file, &upgrade_predicate);

/*
 * check if we have support for all platforms used in this file
 */
    FWObject *fwgrp=db->getById( FWObjectDatabaseGUI::getFirewallsId() , true );
    assert(fwgrp!=NULL);

    FWObjectTypedChildIterator j=fwgrp->findByType(Firewall::TYPENAME);
    for ( ; j!=j.end(); ++j ) 
    {
        FWObject *object= *j;

        string pl=object->getStr("platform");
        if (Resources::platform_res.count(pl)==0)
        {
            char errmsg[1024];
            sprintf(errmsg,_("Object %s uses unsupported firewall platform %s"), 
                             object->getName().c_str() , pl.c_str());
            MessageDialog::Error(errmsg);
            object->setStr("platform","unknown");
        }

        string ho=object->getStr("host_OS");
        if (Resources::os_res.count(ho)==0)
        {
            char errmsg[1024];
            sprintf(errmsg,_("Object %s uses unsupported host OS %s"), 
                             object->getName().c_str() , ho.c_str());
            MessageDialog::Error(errmsg);
            object->setStr("host_OS","unknown_os");
        }
    }

    setDirty(false,true);
}

void FWObjectDatabaseGUI::save() throw(FWException)
{
    if (data_file=="") 
    {
	saveAs();
	return;
    }

    saveFile(data_file);
}

void FWObjectDatabaseGUI::saveAs() throw(FWException)
{
    string df;

    while (true)
    {
        FileSel *fs=new FileSel(_("Select file"),
                                Preferences::global_prefs->getWdir(),
                                "*.xml");
        df=fs->run();
        delete fs;
        if (df=="") throw FWException(_("Save cancelled."));
        data_file=df;

        if ( access(data_file.c_str(), F_OK)==0 )
        {
            char errmsg[1024];
            sprintf(errmsg,_("The file %s already exists. Replace it?"),data_file.c_str());

            MessageDialog::DlgReturnValue action;
            action= MessageDialog::Question(errmsg);
/*
 *  yes    - save the file and return
 *  no     - do not save, show file selection dialog again
 *  cancel - abort
 */
            switch (action) 
            {
            case MessageDialog::OK_YES: saveFile(data_file); return; 
            case MessageDialog::NO:     data_file=""; break;
            case MessageDialog::CANCEL: data_file=""; throw FWException(_("Save cancelled."));
            }
        } else
        {
            saveFile(data_file); 
            return;
        }
    }
}

void FWObjectDatabaseGUI::saveFile(const string &filename) throw(FWException)
{
    FWObjectDatabase::saveFile(filename);
}


FWObject* FWObjectDatabaseGUI::create(const string& type_name, bool create_with_root)
{
    FWObject   *nobj;
    if (type_name==TMPGroupObject::TYPENAME)  nobj=new TMPGroupObject();
    else     nobj=FWObjectDatabase::create(type_name,create_with_root);

    assert(nobj!=NULL);
    
    return(nobj);
}


bool FWObjectDatabaseGUI::validateObjectForPositionInTree(FWObject* target,
							  FWObject* obj)
{
    if ( ! target->validateChild(obj) ) return false;

/* few  more strict checks for system groups */
    string where_id=target->getId();

    if (where_id==FWObjectDatabaseGUI::getFirewallsId() &&
	obj->getTypeName()!=Firewall::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getHostsId() &&
	obj->getTypeName()!=Host::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getNetworksId() &&
	obj->getTypeName()!=Network::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getAddressRangesId() &&
	obj->getTypeName()!=AddressRange::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getIPServicesId() &&
	obj->getTypeName()!=IPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getICMPServicesId() &&
	obj->getTypeName()!=ICMPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getTCPServicesId() &&
	obj->getTypeName()!=TCPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getUDPServicesId() &&
	obj->getTypeName()!=UDPService::TYPENAME) return false;
	
    if (where_id==FWObjectDatabaseGUI::getCustomServicesId() &&
	obj->getTypeName()!=CustomService::TYPENAME) return false;

    if (where_id==FWObjectDatabaseGUI::getTimeId() &&
	obj->getTypeName()!=Interval::TYPENAME) return false;

    return true;
}

void FWObjectDatabaseGUI::defaultNamingScheme(FWObject *obj)
{
    if (IPv4::isA(obj) || physAddress::isA(obj))
    {
        FWObject *p =obj;
        do {  p=p->getParent(); } while ( p!=NULL && Interface::cast(p)==NULL );
        assert(p!=NULL);   // p is a pointer to the interface

        Interface *iface=Interface::cast(p);
        assert(iface!=NULL);

        do {  p=p->getParent(); } while ( p!=NULL && Host::cast(p)==NULL );
        assert(p!=NULL);   // p is a pointer to the Host or Firewall

        FWObject *host=p;

        string iface_name=iface->getName();
        string iface_label=iface->getLabel();
        string prefix;
        if (iface_label!="") prefix=host->getName()+":"+iface_label;
        else                 prefix=host->getName()+":"+iface_name;

        if (physAddress::isA(obj))   obj->setName( prefix+"(MAC)" );
        else                         obj->setName( prefix+"(ip)" );

//        obj->setName( host->getName() );
    }
}

list<InterfaceData> FWObjectDatabaseGUI::newInterfaceDataViaSNMP( libfwbuilder::Host *obj , Gtk::Widget *wnd)
{
    list<InterfaceData> res;

#ifdef HAVE_LIBSNMP

    Management *mgmt=obj->getManagementObject();
    assert(mgmt!=NULL);

    string rcomm=mgmt->getSNMPManagement()->getReadCommunity();

    if ( rcomm.empty() ) 
    {
	MessageDialog::Error(_("missing SNMP community string"));
	return res;
    }

// takes address from management interface and copies it into Management object
    IPAddress addr;
    try {
        addr=obj->getManagementAddress(); 

        if (addr==IPAddress("0.0.0.0"))
        {
            vector<IPAddress> addrs=DNS::getHostByName( obj->getName() );
            addr= addrs.front();
        }

    } catch (FWException &ex)
    {
        MessageDialog::Error( obj->getName()+_(" does not have management interface and its address could not be resolved via DNS"),
                              (wnd!=NULL)?wnd:main_window::windows.back() );
        return res;
    }

    int t=Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPTimeout", -1);

    SNMP_interface_query *q=new SNMP_interface_query();
    q->init(addr.toString(),  rcomm,
	    Preferences::global_prefs->getOptInt("/FWBuilderPreferences/Network/SNMPRetries",  SNMP_DEFAULT_RETRIES),
	    t==-1?SNMP_DEFAULT_TIMEOUT:(1000000L*t)
    );

    GenericBackgroundOpDialog bdisp(q);
    
    try
    {
        bdisp.execute();
        
        const map<int, Interface> &intf = q->getInterfaces();
        for(map<int, Interface>::const_iterator i=intf.begin();i!=intf.end(); ++i)
        {
            if ( i->second.isUp() ) 
            {
                InterfaceData idata( i->second );
                res.push_back(idata);
            }
        }

    } catch(const FWException &ex)
    {
        //do nothing
    }

    bdisp.disconnect();
    delete q;

#endif
    return res;
}


list<FWObject*> FWObjectDatabaseGUI::newInterfacesViaSNMP( Host *object , Gtk::Widget *wnd)
{
    list<InterfaceData> ifaces;
    list<FWObject*>     res;

    ifaces=newInterfaceDataViaSNMP(object,wnd);

    if (!ifaces.empty())
    {
        FWObject *o;
        list<FWObject*>    interfaces= object->getByType(Interface::TYPENAME);
        list<InterfaceData> new_ifaces;

        for (list<InterfaceData>::iterator i=ifaces.begin(); i!=ifaces.end(); i++)
        {
            iterator j=std::find_if(interfaces.begin(), interfaces.end(), 
                                    FWObjectNameEQPredicate( i->name ) );
            if (j!=interfaces.end()) interfaces.erase(j);
            else                     new_ifaces.push_back( *i );
        }
/* at this point list interfaces contains only interfaces that weren't
 * found in SNMP poll and new_ifaces contains interfaces that weren't
 * present in the object but were found in the query.
 */
        for (list<FWObject*>::iterator i=interfaces.begin(); i!=interfaces.end(); ++i)
        {
            o=(*i);
            main_window::removeFromAllObjectBooks(o);
            FWObjectDatabaseGUI::db->removeAllInstances(o);
        }

        for (list<InterfaceData>::iterator i=new_ifaces.begin(); i!=new_ifaces.end(); i++)
        {
            i->guessLabel( object->getStr("platform") );

            InterfaceData idata= *i;

            Interface *iface= Interface::cast(FWObjectDatabaseGUI::newInterface(object->getId()));
            iface->setName( idata.name );
            iface->setLabel( idata.label );
            iface->setDyn(idata.isDyn);
            iface->setUnnumbered(idata.isUnnumbered);

            IPv4        *addr=NULL;
            physAddress *pa=NULL;
            if ( ! idata.isDyn && ! idata.isUnnumbered)
            {
                addr=IPv4::cast( FWObjectDatabaseGUI::newIPv4(iface->getId()) );
                addr->setAddress( idata.address );
                addr->setNetmask( idata.netmask );

                if (idata.physicalAddress!="")
                {
                    pa=physAddress::cast( FWObjectDatabaseGUI::newPhysAddress(iface->getId()) );
                    pa->setPhysAddress( idata.physicalAddress );
                }
            }
            iface->setSecurityLevel( idata.securityLevel );

            main_window::insertInAllObjectBooks(iface);
/*
            object->add( iface );
            if (Firewall::isA(object)) ifs->add(new InterfacePolicy());

            FWObjectTypedChildIterator j=(*i)->findByType(IPv4::TYPENAME);
            for ( ; j!=j.end();  ++j)
                defaultNamingScheme( *j );

            FWObjectTypedChildIterator k=(*i)->findByType(physAddress::TYPENAME);
            for ( ; k!=k.end();  ++k)
                defaultNamingScheme( *k );
*/
        }
    }
    return res;
}

list<FWObject*> FWObjectDatabaseGUI::newInterfacesViaSNMP( const string &target_id  , Gtk::Widget *wnd)
{
    Host *obj=Host::cast( FWObjectDatabase::db->getById( target_id, true) );
    if (obj==NULL) return list<FWObject*>();

    return newInterfacesViaSNMP(obj,wnd);
}

FWObject*  FWObjectDatabaseGUI::newObject(const string& where_id , FWObject *o,
                                              bool  set_defaults)
{
    if (o->getName()=="") {  // no name, use type description string instead
	o->setName( Resources::global_res->getObjResourceStr(o,"description") );
    }
    if (set_defaults) Resources::setDefaultProperties(o);
    db->addAt(where_id,o);
    defaultNamingScheme( o );

    return o;
}


FWObject*  FWObjectDatabaseGUI::newFirewall(const string &target_id)
{
    Firewall *fw=Firewall::cast(db->create(Firewall::TYPENAME,true));
    newObject( ((!target_id.empty())?target_id:getFirewallsId()) , fw );
    try {
        Resources::setDefaultOptions(fw);
        Resources::setDefaultTargetOptions("unknown",fw);
        Resources::setDefaultTargetOptions("unknown_os",fw);
    } catch (FWException &ex)
    {
        MessageDialog::Error(ex.toString());
    }
    return fw;
}

FWObject*  FWObjectDatabaseGUI::newHost(const string &target_id)
{
    Host *h=Host::cast(db->create(Host::TYPENAME,true));
    newObject( ((!target_id.empty())?target_id:getHostsId()) , h);
    try {
        Resources::setDefaultOptions(h);
    } catch (FWException &ex)
    {
        MessageDialog::Error(ex.toString());
    }
    return h;
}

FWObject*  FWObjectDatabaseGUI::newInterface(const string &target_id)
{
    FWObject *obj= FWObjectDatabase::db->getById( target_id, true);
    FWObject *o  =db->create(Interface::TYPENAME,true);
    newObject( target_id , o);
    if (Firewall::isA(obj)) o->add(new InterfacePolicy());
    return o;
}

FWObject*  FWObjectDatabaseGUI::newIPv4(const string &target_id)
{
    FWObject *o=db->create(IPv4::TYPENAME,true);
    return newObject( target_id , o);
}

FWObject*  FWObjectDatabaseGUI::newPhysAddress(const string &target_id)
{
    FWObject *o=db->create(physAddress::TYPENAME,true);
    return newObject( target_id , o);
}

FWObject*  FWObjectDatabaseGUI::newNetwork(const string &target_id)
{
    FWObject *o=db->create(Network::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getNetworksId()),o );
}

FWObject*  FWObjectDatabaseGUI::newAddressRange(const string &target_id)
{
    FWObject *o=db->create(AddressRange::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getAddressRangesId()),o );
}

FWObject*  FWObjectDatabaseGUI::newObjectGroup(const string &target_id)
{
    FWObject *o=db->create(ObjectGroup::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getObjectGroupsId()),o);
}

FWObject*  FWObjectDatabaseGUI::newIPService(const string &target_id)
{
    FWObject *o=db->create(IPService::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getIPServicesId()), o);
}

FWObject*  FWObjectDatabaseGUI::newICMPService(const string &target_id)
{
    FWObject *o=db->create(ICMPService::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getICMPServicesId()),o);
}

FWObject*  FWObjectDatabaseGUI::newTCPService(const string &target_id)
{
    FWObject *o=db->create(TCPService::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getTCPServicesId()),o);
}

FWObject*  FWObjectDatabaseGUI::newUDPService(const string &target_id)
{
    FWObject *o=db->create(UDPService::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getUDPServicesId()),o);
}

FWObject*  FWObjectDatabaseGUI::newCustomService(const string &target_id)
{
    FWObject *o=db->create(CustomService::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getCustomServicesId()),o);
}

FWObject*  FWObjectDatabaseGUI::newServiceGroup(const string &target_id)
{
    FWObject *o=db->create(ServiceGroup::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getServiceGroupsId()),o);
}

FWObject*  FWObjectDatabaseGUI::newInterval(const string &target_id)
{
    FWObject *o=db->create(Interval::TYPENAME,true);
    return newObject( ((!target_id.empty())?target_id:getTimeId()),o);
}


