/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ObjectTree.cc,v 1.95 2003/11/03 06:52:55 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include "glademm_support.hh"

#include "FWObjectDatabaseGUI.hh"
#include "ObjectTree.hh"
#include "GroupDialog.hh"
#include "FindDialog.hh"
#include "FWObjectBook.hh"
#include "fwbuilder/Tools.hh"

#include "main_window.hh"
#include "gen_popup_menu.hh"
#include "helpers.hh"
#include "SimplePixmap.hh"

#include "fwbuilder/FWObject.hh"
#include "fwbuilder/Group.hh"
#include "fwbuilder/ObjectGroup.hh"
#include "fwbuilder/ServiceGroup.hh"
#include "fwbuilder/IntervalGroup.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/Firewall.hh"
#include "fwbuilder/Interface.hh"
#include "fwbuilder/IPv4.hh"
#include "fwbuilder/FWOptions.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/InterfacePolicy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/IPService.hh"
#include "fwbuilder/ICMPService.hh"
#include "fwbuilder/TCPService.hh"
#include "fwbuilder/UDPService.hh"


#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>



#include <list>
#include <vector>
#include <map>
#include <sstream>
#include <iostream>


#include <stdlib.h>
#include <stdio.h>
#include <algorithm>
#include <functional>

#include <assert.h>

using namespace libfwbuilder;


static char drag_obj_id[64];


class cmp_ctree_row_id {
    string row_id;
    public: 
    cmp_ctree_row_id(const string &_id) : row_id(_id) {}
    bool operator()(Gtk::CTree_Helpers::Row r) {
        ObjectData *od=(ObjectData*)(r.get_data());
 	return ( od && row_id==od->id );
    }
};

class cmp_clist_row_id {
    string row_id;
    public: 
    cmp_clist_row_id(const string &_id) : row_id(_id) {}
    bool operator()(Gtk::CList_Helpers::Row r) {
        ObjectData *od=(ObjectData*)(r.get_data());
 	return ( od && row_id==od->id );
    }
};


const char* titles1[2]={ _("Name") , NULL };

const char* titles2[3]={ _("Name") , _("Properties") , NULL };



static void free_row_data(gpointer data)
{
    ObjectData *od=(ObjectData*)(data);

    if (od->pm!=NULL) free( od->pm );
    if (od->bm!=NULL) free( od->bm );

    delete od;

//    delete[] ((char*)data);
}


ObjectTree::ObjectTree(main_window *mw,const string &_l) : 
    CTree( (show_properties=Preferences::global_prefs->getOptBool(
        "/FWBuilderPreferences/UI/ShowObjectPropertiesInTree")) ? titles2 : titles1 ) 
{
    main_w=mw;

    lib=_l;
    constructor();
}

void ObjectTree::constructor()
{
    show_all=false;
    rebuild_scheduled=false;

    column_titles_passive();


    set_selection_mode( GTK_SELECTION_SINGLE );

    selected_object="";
    last_clicked_object="";
    drag_object="";
    show_object="";

    set_name("ObjectTree");
    glademm_set_Widget("ObjectTree", this);

    //  set_expander_style( GTK_CTREE_EXPANDER_SQUARE );

    set_expander_style( GTK_CTREE_EXPANDER_CIRCULAR );
    set_line_style( GTK_CTREE_LINES_SOLID  );

    column(0).set_auto_resize(true);
    if (show_properties) column(1).set_auto_resize(true);

    button_press_event.connect(
	slot(this, &ObjectTree::on_button_press_event));
    
    button_release_event.connect_after(
	slot(this, &ObjectTree::on_button_release_event));


    /*****************************************************************
     *  Initialize Drag&DRop mechanism 
     */
    drag_source_set ( 
	static_cast<GdkModifierType>(GDK_BUTTON1_MASK) ,
	target_table, n_targets, 
	static_cast<GdkDragAction>(GDK_ACTION_COPY) );

    drag_begin.connect(
	slot(this,&ObjectTree::source_drag_begin));
    drag_data_get.connect(
	slot(this,&ObjectTree::source_drag_data_get));
    drag_data_delete.connect(
	slot(this,&ObjectTree::source_drag_data_delete));

    processing_selection=false;
    tree_select_row.connect( slot(this,&ObjectTree::selection_made));

    key_press_event.connect(SigC::slot((ObjectTree*)this, 
				       &ObjectTree::on_key_press_event));

}

gint ObjectTree::on_key_press_event(GdkEventKey *ev)
{
    FWObject  *o=NULL;
    guint      n;
    gchar      c;
    string     id;

    o=FWObjectDatabaseGUI::db->getById( selected_object, true ) ;
    assert(o!=NULL);

/*
    cerr << "--------------------\n";


    cerr << _("send_event=") << ev->send_event << endl;

    cerr << _("time=") << ev->time << endl;

    cerr << _("state=") << ev->state << endl;

    cerr << _("keyval=") << ev->keyval << endl;

    cerr << _("length=") << ev->length << endl;

    cerr << _("string=") << ev->string << endl;


*/

    if ( ev->keyval == GDK_F1 ) {   
// F1, just print objects address,ID, name etc.
	o->dump(true,true);
	return(0);
    }

    if ( ev->keyval == GDK_F5 ) {   
// F5, rebuild the tree
	scheduleRebuild();
	return(0);
    }

    if ( ev->keyval == GDK_F6 ) {   
// F6, toggle show_all flag and refresh the tree
	show_all= ! show_all;
	scheduleRebuild();
	return(0);
    }

    if ( ev->keyval == GDK_F7 ) {   
// F7, dump selected object
	o->dump(true,false);
	return(0);
    }

    if ( ev->keyval == GDK_F8 ) {   
// F8, dump the whole databse
	std::ofstream out("fwbuilder_db.dump");
	FWObjectDatabaseGUI::db->dump(out,true,false);
	return(0);
    }

    if ( ! main_w->safe_to_close_right_pane() ) {
	/*
	 *  All key presses change selection in the tree. If it is not safe
	 *  to close right pane and show another object there, then igore it.
	 */
	gtk_signal_emit_stop_by_name( GTK_OBJECT( gtkobj() ),
				      "key_press_event" );
 


 
	return(1);
    }


    Gtk::CTree_Helpers::RowList          rl=rows();
    if (rl.empty()) return(0);
  

    if ( ev->keyval == GDK_Return ) {   // ENTER

	RowList::iterator k;
	for (n=0,k=rl.begin(); k!=rl.end(); ++k,++n) {
            ObjectData *od=(ObjectData*)(k->get_data());
            id=od->id;
	    if (id==selected_object) {

		//    n = current_selected_row;
		// subtree ?
		Gtk::CTree_Helpers::Row  row=rows()[n];
		Gtk::CTree_Helpers::RowList srl = row.subtree();
		if (! srl.empty()) {
		    row.expand();
		    return(0);
		}
//		if (main_w->isDialogDocked()) {
//		    main_w->setCurrentLibrary(lib);
		    main_w->schedule_open_object(id,lib);
//		}
		return(0);
	    }
	}
	return(0);
    }

    if ((ev->keyval >= 0x20) && (ev->keyval <= 0xFF) && 
	((ev->state & GDK_CONTROL_MASK) ==0) ) {

	c = ev->keyval;
  
	RowList::iterator k;
	for (n=0,k=rl.begin(); k!=rl.end(); ++k,++n) {
            ObjectData *od=(ObjectData*)(k->get_data());
            id=od->id;
	    if (id==selected_object) break;
	}
	++k; ++n;
	for (; k!=rl.end(); ++k,++n) {
            ObjectData *od=(ObjectData*)(k->get_data());
            id=od->id;
	    o=FWObjectDatabaseGUI::db->getById( id , true );
	    assert(o!=NULL);
	    if (tolower( o->getName()[0] )==c ) {
		selectRow(n);
		selected_object=id;

		gtk_signal_emit_stop_by_name( GTK_OBJECT( gtkobj() ),
					      "key_press_event" );

		return(true);
	    }
	}
    }
    return 0;
}

void ObjectTree::showObject(const string &id)
{
    if (id=="")	clearSelection();
    else 
    {
	if (selected_object!=id) 
        {
	    expand_to_row(id);
	    selectObject(id);
	}
    }
}

void ObjectTree::showObject(FWObject *o)
{
    showObject(o->getId());
}

void ObjectTree::selectObject(const string &id)
{
/*
 *  CTree does not have method to focus row, so we have to work with
 *  CList
 */
    Gtk::CList::RowList::iterator m=find_if( Gtk::CList::rows().begin(),
					     Gtk::CList::rows().end(),
					     cmp_clist_row_id(id) );
    if (m!=Gtk::CList::rows().end()) {
//	selected_object=id;
	m->select();
	m->focus();
    } else {
	clearSelection();
    }
}

void ObjectTree::selectObject(FWObject *obj)
{
    selectObject(obj->getId());
}

void ObjectTree::selection_made(Gtk::CTree::Row r,gint col)
{
    if (!processing_selection) 
    {
        processing_selection=true;
        ObjectData *od=(ObjectData*)(r.get_data());
        string id=od->id;
        if (id!=selected_object) 
        {
            if ( main_w->safe_to_close_right_pane() )
            {
                main_w->schedule_open_object(id,lib);

                selected_object=id;
                selectObject(selected_object);
            } else 
                selectObject(selected_object);
        }
        processing_selection=false;
    }
}

void ObjectTree::clearSelection()
{
    selected_object="";

    Gtk::CTree_Helpers::RowList          rl=rows();
    if (rl.empty()) return;
  
    RowList::iterator k;
    for (k=rl.begin(); k!=rl.end(); ++k) {
	(*k).unselect();
    }
}


void ObjectTree::selectRow(gint row_n)
{
    /*
     * calling "select()" for CTree_Helpers::Row
     * does not move focus (but moves selection). There is no "focus()"
     * call for CTree_Helpers::Row, so we have to dig out its 
     * equivalentfrom CList_Helpers and call focus() for it
     *
     * It does not really matter which one we use here to call select()
     */
    Gtk::CList_Helpers::Row r = Gtk::CList::row(row_n);
    r->focus();
    r->select();
}

void ObjectTree::rebuildTreeAndShow(const string &id)
{
    rebuildTree();
    showObject(id);
//    main_w->setCurrentLibrary(lib);
    main_w->schedule_open_object(id,lib);
}

void ObjectTree::scheduleRebuild()
{
    if ( !rebuild_scheduled ) {
	Gtk::Main::idle.connect(slot(this,&ObjectTree::rebuild_when_idle));
	rebuild_scheduled=true;
    }
}

gint ObjectTree::rebuild_when_idle()
{
    rebuildTree();
    return false;
}

gint ObjectTree::rebuild_and_show_when_idle()
{
    if (show_object=="") return false;
    rebuildTreeAndShow(show_object);
    show_object="";
    return false;
}

string ObjectTree::get_tree_label(const FWObject *obj)
{
    string name;
    if (Interface::isA(obj))
    {
        name=Interface::constcast(obj)->getLabel();
        if (name=="")  name=obj->getName();
    }
    else    name=obj->getName();

    if (name=="") {  // no name, use type description string instead
        name= Resources::global_res->getObjResourceStr(obj,"description");
    }
    return name;
}

/*
 * need to use const here so I could use this method from inside of function
 * adapter used for sorting.
 */
string ObjectTree::get_properties(const FWObject *o)
{
    if ( ! Preferences::global_prefs->getOptBool(
        "/FWBuilderPreferences/UI/ShowObjectPropertiesInTree") ) return "";

    if (Interface::isA(o))
    {
        const Interface *iface=Interface::constcast(o);
            
        const FWObject *f=o;
        while (f!=NULL && ! (Firewall::isA(f) || Host::isA(f) )) f=f->getParent();
        assert(f!=NULL);

        ostringstream prop;

        if (iface->isDyn())         prop << "Dyn ";
        else
            if (iface->isUnnumbered())  prop << "Unnum ";
            else                        prop << iface->getAddress().toString() << " ";

        if ( Firewall::isA(f) )
        {
// f is a pointer at firewall object
            bool supports_security_levels=false;
            try  {
                supports_security_levels=
                    Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                       "security_levels");
            } catch (FWException &ex)  { }

            if (supports_security_levels)
                prop << "sec " << iface->getSecurityLevel() << " ";
            else
                prop << string( (iface->isExt())?"Ext ":"" );

            if (iface->isManagement()) prop << "Mgmt ";
        }
        return string("  ")+prop.str();
    }

    if (Host::isA(o))
        return string("  ")+Address::constcast(o)->getAddress().toString();

    if (Network::isA(o)) {
	const Network *n=Network::constcast(o);
	return string("  ")+n->getAddress().toString()+string("/")+n->getNetmask().toString();
    }

    if (IPv4::isA(o))
        return string("  ")+Address::constcast(o)->getAddress().toString();

    if (physAddress::isA(o))
        return string("  ")+physAddress::constcast(o)->getPhysAddress();

    if (TCPService::isA(o) || UDPService::isA(o)) {
	string dps,dpe;

	dps=o->getStr("dst_range_start");
	dpe=o->getStr("dst_range_end");

	if (dps==dpe)	return string("  ")+dps;
	else	        return string("  ")+dps+string("-")+dpe;
    }

    if (ICMPService::isA(o)) 
	return string("  ")+o->getStr("type")+string("/")+o->getStr("code");

    if (IPService::isA(o)) 
	return string("  ")+o->getStr("protocol_num");

    return string();
}

class firewall_subtree_sort_order
{
    private:
    map< pair<string,string> , int > decision;

    public:
    firewall_subtree_sort_order()
    {
        decision[ pair<string,string>("Interface","Interface") ] = 0;
        decision[ pair<string,string>("Interface","Policy") ]    = -1;
        decision[ pair<string,string>("Interface","NAT") ]       = -1;

        decision[ pair<string,string>("Policy","Interface") ]    = 1;
        decision[ pair<string,string>("Policy","Policy") ]       = 0;
        decision[ pair<string,string>("Policy","NAT") ]          = -1;

        decision[ pair<string,string>("NAT","Interface") ]       = 1;
        decision[ pair<string,string>("NAT","Policy") ]          = 1;
        decision[ pair<string,string>("NAT","NAT") ]             = 0;
    }
    int cmp(const string &a,const string &b)
    {
        return decision[ pair<string,string>(a,b) ];
    }
};

namespace local{
    firewall_subtree_sort_order ftso;
};

/*
 * sorts objects in the list 
 * sort conditions: 
 *
 * if "this" is a Firewall object, then interfaces should appear
 * first, then Policy, then NAT
 *
 *
 * in all other cases sort alphabetically by object names
 *
 * this predicate is suitable for using in std::sort
 */
class tree_sorting_predicate
{
    typedef enum { FIREWALL, OTHER} subtree_types;
    protected:
    FWObject *o;
    subtree_types subtree;
    public:
    explicit tree_sorting_predicate(FWObject *p) {
        o=p; 
        if (Firewall::isA(p)) subtree=FIREWALL;
        else                  subtree=OTHER;
    }

    bool operator()(const FWObject *a, const FWObject *b) 
    { 
        return cmp(a->getName(), a->getTypeName(), b->getName(), b->getTypeName());
    }

    bool cmp(const string &name_a, const string &type_a,
             const string &name_b, const string &type_b)
    {
        if (subtree==FIREWALL) return local::ftso.cmp(type_a, type_b)<1;
        else                   return name_a <= name_b;
    }
};

/*
 * This predicate implements the same sort order as tree_sorting_predicate
 * 
 * it is suitable for using in std::find_if
 */
class tree_insertion_predicate : public tree_sorting_predicate
{
    FWObject *obj;
    public:
    tree_insertion_predicate(FWObject *parent,FWObject *o):tree_sorting_predicate(parent) {obj=o;}

    bool operator()(Gtk::CTree_Helpers::Row &r)
    {
        ObjectData *od=(ObjectData*)(r.get_data());
        return cmp(obj->getName(), obj->getTypeName(),
                   od->name, od->type_name );
    }
};

void ObjectTree::insertSubtree(  Gtk::CTree_Helpers::RowList::iterator &row , 
				  FWObject *obj )
{
    if (obj==NULL) return;

    if (FWObjectReference::isA(obj) && ! show_all) return;

    if (Resources::global_res->getObjResourceBool(obj,"hidden") &&
	! show_all ) return;

    if (lib=="All" || lib==obj->getLibrary() || Resources::isSystem(obj) ) {
	string name=get_tree_label(obj);
	string properties= get_properties(obj);

        Gtk::CTree_Helpers::RowList rl=row->subtree();
        Gtk::CTree_Helpers::RowList::iterator pos;

#if 0
/*
 * this tries to find position in the tree for the new object so that
 *  branch would be automatically sorted
 * 
 * TODO: add GUI option to turn this feaure on/off
 */
        tree_insertion_predicate p(obj->getParent(),obj);
        int z;
        for (z=0,pos=rl.begin(); pos!=rl.end(); ++pos,++z)
        {
            ObjectData *pod=(ObjectData*)(pos->get_parent().get_data());
            if (pod==NULL) continue;
            if (pod->id!=obj->getParent()->getId()) {pos=rl.end(); break;}

            if ( p(*pos) ) break;
        }
#endif

        string icn_filename;
        if (Preferences::global_prefs->getOptBool("/FWBuilderPreferences/UI/ShowIconsInTree"))
            icn_filename=Resources::getTreeIconFileName(obj);

        pos=rl.end();
        Gtk::CTree_Helpers::RowList::iterator  new_row=
            _insert_row( rl,  pos, 
                         name, 
                         obj->getId(), 
                         obj->getTypeName(), 
                         properties,
                         icn_filename);

        if (Firewall::isA(obj))
        {
            FWObjectTypedChildIterator j=obj->findByType(Interface::TYPENAME);
            for ( ; j!=j.end(); ++j ) {
                insertSubtree( new_row , (*j) );
            }
            FWObject *o;
            o=obj->getFirstByType(Policy::TYPENAME);
            insertSubtree( new_row , o );
            o=obj->getFirstByType(NAT::TYPENAME);
            insertSubtree( new_row , o );

        } else
        {
            for(list<FWObject*>::iterator m=obj->begin(); m!=obj->end(); ++m)
                insertSubtree( new_row , (*m) );
        }
    }
}


void ObjectTree::rebuildTree()
{
    string cur=selected_object;

    using namespace Gtk::CTree_Helpers;

    freeze();
    clear();
    
    RowList::iterator j;

    j=--rows().end();

    for (list<FWObject*>::iterator m=FWObjectDatabaseGUI::db->begin(); 
	 m!=FWObjectDatabaseGUI::db->end(); 
	 ++m) 
    {
	insertSubtree( j , (*m) );
    }
    thaw();

    rebuild_scheduled=false;
}

class traverse_tree_and_find_id {
    string row_id;
    Gtk::CTree_Helpers::Row res;
    public:
    traverse_tree_and_find_id(const string &_id):row_id(_id),res() {}
    operator Gtk::CTree_Helpers::Row() const { return res; }
    const traverse_tree_and_find_id& operator()(Gtk::CTree_Helpers::Row r) {
	if (res==NULL) {
            ObjectData *od=(ObjectData*)(r.get_data());
            string id=od->id;
	    if ( row_id==id ) {
		res=r; r.expand(); return *this; 
	    } else {
		res=std::for_each(r.subtree().begin(),r.subtree().end(),
				  traverse_tree_and_find_id(row_id));
		if (res!=NULL) r.expand();
	    }
	} 

	return *this;
    }
};

/*
 *  returns Row. Since Gtk::CTree_Helpers::Row has special
 *  operator==(gpointer) we can check whether search was sucessful by
 *  comparing returned Row with NULL
 */
Gtk::CTree_Helpers::Row ObjectTree::expand_to_row(const string &id)
{
    return std::for_each(rows().begin(), rows().end(), 
			 traverse_tree_and_find_id(id) );
}


Gtk::CTree_Helpers::RowList::iterator 
ObjectTree::_insert_row(
    Gtk::CTree_Helpers::RowList          &rl,
    Gtk::CTree_Helpers::RowList::iterator where,
    const string &lbl, 
    const string &id, 
    const string &type_name, 
    const string &properties,
    const string &icon_filename)
{
    vector<const gchar*>      item;
    item.push_back( lbl.c_str() );
    if ( ! properties.empty() ) item.push_back( properties.c_str() );

#if 0
    cerr << "insert " << lbl << "(id=" << id <<") at ";

    if (where==rl.end())    cerr << "the end" << endl;
    else
    {
        ObjectData *od1=(ObjectData*)(where->get_data());
        cerr << od1->getName() << " (id=" << od1->getId() << ") " << endl;
    }
#endif

    RowList::iterator j;
    GdkPixbuf *pb=NULL;
    GdkPixmap *pm=NULL;
    GdkBitmap *bm=NULL;

    if (Preferences::global_prefs->getOptBool(
            "/FWBuilderPreferences/UI/ShowIconsInTree"))
    {

        pb=gdk_pixbuf_new_from_file( (char*)(icon_filename.c_str() ) );
        assert (pb!=NULL);
        gdk_pixbuf_render_pixmap_and_mask(pb, &pm, &bm, 127);
//    (*j)[0].set_pixmap(pm,bm);

        j=rl.insert(where, Element(item,pm,bm,pm,bm));
    } else
        j=rl.insert(where, Element(item));

    ObjectData *od=new ObjectData(id,lbl,type_name,pm,bm);
    j->set_data( od , &free_row_data);

    return j;
}


void ObjectTree::insertObject( FWObject *obj )
{
    FWObject *par=obj->getParent();

    freeze();

    expand_to_row( par->getId() );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
					      cmp_ctree_row_id(par->getId()) );
    insertSubtree(oi, obj);

    thaw();

//    main_w->setCurrentLibrary(lib);
//    main_w->schedule_open_object(obj->getId());
//    selectObject(obj->getId());
}


void ObjectTree::insertObject( const string &id )
{
    FWObject *obj=FWObjectDatabaseGUI::db->getById(id,true);
    assert (obj!=NULL);

    insertObject(obj);
}

void   ObjectTree::removeObject( const string &id )
{
    FWObject *obj=FWObjectDatabaseGUI::db->getById(id,true);
    assert (obj!=NULL);

    removeObject(obj);
}

void   ObjectTree::removeObject( FWObject *o )
{
    expand_to_row(o->getId());
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
					      cmp_ctree_row_id(o->getId()) );
    Gtk::CTree::RowList::iterator pi=find_if( rows().begin(),
					      rows().end(),
					      cmp_ctree_row_id(o->getParent()->getId()) );
    pi->subtree().erase(oi);

    if (selected_object==o->getId()) selected_object="";
}

class  cmp_rows {
    FWObject *o;
    public:
    cmp_rows(FWObject *p) { o=p; }
    int operator()(const Gtk::CTree_Helpers::Row &r1,
		   const Gtk::CTree_Helpers::Row &r2) {

	return r1[0].get_text() < r2[0].get_text();
    }
};

void   ObjectTree::sortSubtree(const string &id,int col)
{
    FWObject *o=FWObjectDatabaseGUI::db->getById(id,true);
    assert (o!=NULL);

    sortSubtree(o,col);
}

void   ObjectTree::sortSubtree(FWObject *o,int col)
{
    Gtk::CTree_Helpers::Row r=expand_to_row(o->getId());

//    std::sort(rl.begin(),rl.end(),cmp_rows(o) );

    set_sort_column(col);

    r.sort();
}

class find_and_replace_label {

    string id;
    string new_lbl;
    string properties;

    public:
    find_and_replace_label(const string &_id,
			   const string _new_lbl,
			   const string _i )
    { id=_id; new_lbl=_new_lbl; properties=_i; }

    void operator()(Gtk::CTree_Helpers::Row r) {
        ObjectData *od=(ObjectData*)(r.get_data());
	if ( id==od->id ) {
	    r[0].set_text(new_lbl);
	    if (!properties.empty()) 
		r[1].set_text(properties);
            od->name=new_lbl;
	}
//        else {
//	    Gtk::CTree_Helpers::RowList srl = r.subtree();
//	    std::for_each( srl.begin() , srl.end() , 
//			   find_and_replace_label(id,new_lbl,properties) ); 
//	}
    }
};

void ObjectTree::changeTreeLabel(const string &id)
{
    FWObject *obj=FWObjectDatabaseGUI::db->getById(id,true);
    assert (obj!=NULL);

    string new_lbl = get_tree_label(obj);
    if (new_lbl=="") 
        new_lbl= Resources::global_res->getObjResourceStr(obj,"description");

    string properties = get_properties(obj);

    std::for_each( rows().begin() , rows().end() ,
		   find_and_replace_label(id,new_lbl,properties) );

/*
 * the following is a kludge that uses tree auto-sorting mechanism
 * which kicks in when object is inserted
 */
//    removeObject(obj);
//    insertObject(obj);
}


void ObjectTree::source_drag_begin(GdkDragContext     *context)
{
    drag_object=last_clicked_object;

    return;
}


void ObjectTree::source_drag_data_get  ( GdkDragContext     *context,
					  GtkSelectionData   *selection_data,
					  guint               info,
					  guint32             time )
{
  const guchar    *idptr;
  
  strcpy(drag_obj_id , drag_object.c_str() );
  idptr= (const guchar*)drag_obj_id;

  gtk_selection_data_set (selection_data,
			  gdk_atom_intern("ObjectTree",TRUE),
			  8,
			  (const guchar*)drag_obj_id,
			  strlen(drag_obj_id)+1 );
}

void ObjectTree::source_drag_data_delete ( GdkDragContext     *context )
{
}



gint ObjectTree::on_button_press_event(GdkEventButton *ev)
{
    gint              r,c;

    get_selection_info( int(ev->x) , int(ev->y) , &r , &c );

    if (r<0 || r>get_rows()) return(0);

    ObjectData *od=(ObjectData*)(row(r).get_data());

    if (ev->button != 1) selectRow(r);

//    if (ev->type==GDK_2BUTTON_PRESS && ! main_w->isDialogDocked() &&
//	main_w->safe_to_close_right_pane() ) {
//	main_w->setCurrentLibrary(lib);
//	main_w->schedule_open_object(oid);
//    }

    last_clicked_object=od->id;

    return(0);
}

gint ObjectTree::on_button_release_event(GdkEventButton *ev)
{
    gint              r,c;

    get_selection_info( int(ev->x) , int(ev->y) , &r , &c );

    if (r<0 || r>get_rows()) return(0);

    ObjectData *od=(ObjectData*)(row(r).get_data());
    if (od==NULL) return(0);

    if (ev->type == GDK_BUTTON_RELEASE && ev->button==3  ) {
	TreePopupMenu();
    }

    return(0);
}


string ObjectTree::getCurrentSelection()
{
    return( selected_object );
}

string ObjectTree::getCurrentLibrary()
{
    return lib;
}

void ObjectTree::TreePopupMenu()
{
    FWObject *obj=
	FWObjectDatabaseGUI::db->getById( selected_object , true);
    assert(obj!=NULL);

    if (Policy::isA(obj) ||
        InterfacePolicy::isA(obj) ||
        NAT::isA(obj) ) return;

    if ( Resources::isSystem(obj) )
    {
	const char  *menu_items[] ={ _("Sort by name"), _("Sort by property"), _("Paste"), NULL};

	int          menu_choice;
	gen_popup_menu *gpm;

	gpm=new gen_popup_menu( menu_items );

	Gtk::Widget *itm;

        if (!show_properties)
        {
            itm=find_widget(_("Sort by property"),gpm);
            if (itm) itm->set_sensitive(false);
        }

	itm=find_widget(_("Paste"),gpm);   

	if (itm) itm->set_sensitive(
	    (obj->getTypeName()==ObjectGroup::TYPENAME ||
	     obj->getTypeName()==ServiceGroup::TYPENAME ||
	     obj->getTypeName()==IntervalGroup::TYPENAME )  );

	gpm->popup(0,0);
	menu_choice=gpm->run();
	delete gpm;

	switch (menu_choice) {
	case 0: main_w->on_sort_by_name();  break;
	case 1: main_w->on_sort_by_prop();  break;
	case 2:	main_w->on_pasteobj();	    break;
	}

    } else {
	const char  *menu_items[] ={ 
            _("Duplicate"), 
            "", 
            _("Copy"),
            _("Cut"),
            _("Paste"),
            "",
            _("Delete"),
            "",
            _("Where used"),
            NULL,
            NULL,
            NULL,
            NULL};

        if (Firewall::isA(obj) || Host::isA(obj))
        {
            menu_items[9]="";
            menu_items[10]=_("Add Interface");
            menu_items[11]=_("Add Interfaces via SNMP");
        }

        Interface *intf=Interface::cast(obj);
        if (intf!=NULL)
        {
            menu_items[9]="";
            menu_items[10]=_("Add Address");
            menu_items[11]=_("Add Physical Address");
        }

	int          menu_choice;
	gen_popup_menu *gpm;

	gpm=new gen_popup_menu( menu_items );

	Gtk::Widget *itm;

	itm=find_widget(_("Paste"),gpm);   

	if (itm) itm->set_sensitive(
	    (obj->getTypeName()==ObjectGroup::TYPENAME ||
	     obj->getTypeName()==ServiceGroup::TYPENAME ||
	     obj->getTypeName()==IntervalGroup::TYPENAME ||
             Host::isA(obj) ||
             Firewall::isA(obj) ||
             Interface::isA(obj)
            )  );

        if (intf)
        {
            if ( (itm=find_widget(_("Add Address"),gpm))!=NULL) 
                itm->set_sensitive(!intf->isDyn() && !intf->isUnnumbered());  // "Add Address"

            if ( (itm=find_widget(_("Add Physical Address"),gpm))!=NULL )
                itm->set_sensitive(intf->getPhysicalAddress()==NULL);
        }

	gpm->popup(0,0);
	menu_choice=gpm->run();
	delete gpm;

	switch (menu_choice) {
	case 0:  main_w->on_duplicate();  break;  // Duplicate
	case 2:  main_w->on_copyobj();    break;  // Copy
	case 3:  main_w->on_cutobj();	  break;  // Cut
	case 4:  main_w->on_pasteobj();	  break;  // Paste
	case 6:  main_w->on_delobj();     break;  // Delete
	case 8:  main_w->on_where_used(); break;  // Where Used
        case 10:
            if (Firewall::isA(obj) || Host::isA(obj))
                main_w->on_new_interface();
            if (Interface::isA(obj))
                main_w->on_new_ipv4();
            break;
        case 11:
            if (Firewall::isA(obj) || Host::isA(obj))
                main_w->on_new_interface_via_snmp();
            if (Interface::isA(obj))
                main_w->on_new_physaddress();
            break;
	}
    }
}

string   ObjectTree::getNextId()
{
    expand_to_row( selected_object );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
				      cmp_ctree_row_id(selected_object) );
    ++oi;
    if (oi!=rows().end()) {
        ObjectData *od=(ObjectData*)(oi->get_data());
	if (od)
	    return od->id;
    }
    return "";
}

string   ObjectTree::getPrevId()
{
    expand_to_row( selected_object );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
				      cmp_ctree_row_id(selected_object) );
    --oi;
    if (oi!=rows().end()) {
        ObjectData *od=(ObjectData*)(oi->get_data());
	if (od)
	    return od->id;
    }
    return "";
}

string   ObjectTree::getParentId()
{
    expand_to_row( selected_object );
    Gtk::CTree::RowList::iterator oi=find_if( rows().begin(),
					      rows().end(),
				      cmp_ctree_row_id(selected_object) );

    if (oi!=rows().end()) {
	Gtk::CTree::Row r=oi->get_parent();
        ObjectData *od=(ObjectData*)(r.get_data());
	if (od)
	    return od->id;
    }
    return "";
}






