/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyList.cc,v 1.61 2003/03/01 07:45:28 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include <iostream>

#include "glademm_support.hh"
#include "helpers.hh"
#include "BuiltinDialog.hh"
#include "popupMenu.hh"
#include "MessageDialog.hh"

#include <fwbuilder/FWObjectDatabase.hh>
#include "fwbuilder/Resources.hh"
#include "fwbuilder/Policy.hh"
#include "fwbuilder/NAT.hh"
#include "fwbuilder/Rule.hh"
#include "fwbuilder/Firewall.hh"

#include "PolicyListItem.hh"
#include "PolicyListElement.hh"
#include "PolicyList.hh"

#include <assert.h>



using namespace libfwbuilder;

RuleSetList::RuleSetList(RuleSet *p) : Gtk::VBox()
{
    total_no_of_elements=0;
    ruleset=p;
    current_selected_row=0;
    current_selected=NULL;
    rebuild=false;
    width=height=0;
    popup_menu=NULL;

    set_flags( GTK_CAN_FOCUS );

    tbl=new Gtk::Table(1,2);

    swnd=new Gtk::ScrolledWindow();
    swnd->set_policy(GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    swnd->add_with_viewport( *tbl );

    title_row=new Gtk::HBox();

    title_fxd=new Gtk::Fixed();
    title_fxd->put( *title_row, 0,0 );
    title_fxd->set_usize(10,10);

    pack_start( *title_fxd , false,false, 0);
    pack_start( *swnd ,      true,true,   0);

    tbl->realize.connect(SigC::slot(this, &RuleSetList::on_realize_event));

    size_allocate.connect(SigC::slot(this, &RuleSetList::on_size_allocate));
    tbl->size_allocate.connect(SigC::slot(this, &RuleSetList::on_tbl_size_allocate));
    title_row->size_allocate.connect(SigC::slot(this, &RuleSetList::on_title_size_allocate));

    swnd->get_hadjustment()->value_changed.connect(
        SigC::slot(this, &RuleSetList::on_hscrollbar_value_changed));
    
    bottom=NULL;

    FWObject *f=p;
    while (f!=NULL && !Firewall::isA(f)) f=f->getParent();
    assert(f!=NULL);

// f is a pointer at firewall object

    try {
        supports_logging=Resources::getTargetCapabilityBool(f->getStr("platform"), 
                                                            "logging_in_policy");
        supports_rule_options=    Resources::getTargetCapabilityBool(f->getStr("platform"), 
                                                                     "options_in_policy");
        supports_time=Resources::getTargetCapabilityBool(f->getStr("platform"),
                                                         "supports_time");
    } catch (FWException &ex)
    {
        MessageDialog::Error(ex.toString());
    }

    button_release_event.connect_after(
	SigC::slot(this,&RuleSetList::on_button_release_event));

    show_all();
}

RuleSetList::~RuleSetList()
{
    Clear();
    if (popup_menu!=NULL) delete popup_menu;

    tbl->hide();
    swnd->hide();
    title_row->hide();
    title_fxd->hide();

    swnd->remove();
    title_fxd->remove(*title_row);
    remove(*swnd);
    remove(*title_fxd);

    delete tbl;
    delete swnd;
    delete title_row;
    delete title_fxd;
}

bool RuleSetList::supportsLogging() { return supports_logging; }
bool RuleSetList::supportsRuleOptions() { return supports_rule_options; }

/*
 * this is callback for horisontal scrollbar's "value_changed" signal.
 * We get control here when user moves scrollbar. We just move title
 * bar inside its fixed widget for the same number of pixels.
 */
void RuleSetList::on_hscrollbar_value_changed()
{
    Gtk::Adjustment *hadj=swnd->get_hscrollbar()->get_adjustment();
    title_fxd->move( *title_row , -1* int(hadj->get_value()) , 0 );
}

void RuleSetList::on_size_allocate(GtkAllocation *all)
{
    x=all->x;
    y=all->y;
    width=all->width;
    height=all->height;

    int w1,h1;
    title_fxd->get_window().get_size( w1, h1);

    if (w1!=width)
        title_fxd->set_usize(width,h1);
}

void RuleSetList::on_title_size_allocate(GtkAllocation *all)
{
    int w1,h1;
    title_fxd->get_window().get_size( w1, h1);
    if (h1!=all->height)
        title_fxd->set_usize(w1,all->height);
}


/*
 *   We keep track of the physical size of the table and resize title buttons
 */
void RuleSetList::on_tbl_size_allocate(GtkAllocation *all)
{
    list<Gtk::Widget*>::iterator    ti=titles.begin();
    map<int,Gtk::Widget*>::iterator i=table_elements[0].begin();

    for ( ; i!=table_elements[0].end() && ti!=titles.end(); ++i,++ti)
    {
        assert (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL);

        PolicyListElement *pe=(PolicyListElement*)( (*i).second );
        int w=pe->getWidth();

	gint w1,h1;
	(*ti)->get_window().get_size( w1, h1);

        if (w!=w1)
        { 
            if (w>w1) (*ti)->set_usize(w,-1);
            if (w1>w) pe->set_usize(w1,-1);
        }
    }
}

gint RuleSetList::on_button_release_event(GdkEventButton *ev)
{
    if ( ev->type == GDK_BUTTON_RELEASE && ev->button==3 ) {

        show_popup_menu();
    }
    return( true );
}


void RuleSetList::move_focus(GtkDirectionType dir)
{
    gint r,c;

    if (current_selected!=NULL) {

//	current_selected->deselect();

	r=current_selected->get_row();
	c=current_selected->get_col();

	switch (dir ) {
	case GTK_DIR_UP:     r-=1;    break;
	case GTK_DIR_DOWN:   r+=1;    break;
	case GTK_DIR_LEFT:   c-=1;    break;
	case GTK_DIR_RIGHT:  c+=1;    break;
	default:         return;
	}
	select_element(r,c);
    } else
	select_element(0,0);
}

void RuleSetList::select_element(gint r,gint c)
{
    bool going_up=false;

    PolicyListElement *pe=NULL;
    if ( (pe=get_element(r,c))!=NULL ) {

	if ( current_selected!=NULL) {
	    int cr=current_selected->get_row();
	    current_selected->deselect();
	    current_selected=NULL;
	    if (cr!=r) {
		deactivate_row(cr);
		activate_row(r);
		going_up= (r<cr);
	    }
	}

	set_current_selected(pe);
	pe->select(false);
	if (going_up) pe->select_last_child();
	else          pe->select_first_child();
    }
}

void RuleSetList::deselect_current()
{
    if ( current_selected!=NULL) {
	current_selected->deselect();
	deactivate_row(current_selected->get_row());
	current_selected=NULL;
    }
}

void RuleSetList::set_current_selected(PolicyListElement *pe)
{
    current_selected=pe;
    current_selected_row=pe->get_row();
}

void RuleSetList::request_focus()
{
    if (!has_focus()) {
	grab_focus();
	draw_focus();
    }
}


void RuleSetList::activate_row(gint row)
{
    PolicyListElement *pe;

    deactivate_row( current_selected_row );
    current_selected_row=row;

    if ( ! table_elements.empty() ) {
	map<int,Gtk::Widget*>::iterator i;
	for (i=table_elements[row].begin(); i!=table_elements[row].end(); ++i) {
	    if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		pe=(PolicyListElement*)( (*i).second );
		pe->activate();
	    }
	}
    }
}

void RuleSetList::activate_rule(gint r)
{
    activate_row( get_row_by_rule_num(r) );
}

void RuleSetList::deactivate_row(gint row)
{
//    deselect_current();

    PolicyListElement *pe;
    if ( ! table_elements.empty() ) {
	map<int,Gtk::Widget*>::iterator i;
	for (i=table_elements[row].begin(); i!=table_elements[row].end(); ++i) {
	    if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		pe=(PolicyListElement*)( (*i).second );
		pe->deactivate();
	    }
	}
    }
}

void RuleSetList::deactivate_rule(gint r)
{
    deactivate_row( get_row_by_rule_num(r) );
}


void RuleSetList::schedule_rebuild(int r)
{
    activate_r=r;
    Gtk::Main::idle.connect(slot(this,&RuleSetList::rebuild_when_idle));
}

gint RuleSetList::rebuild_when_idle()
{
    build();
    activate_rule(activate_r);
    return false;
}


void RuleSetList::on_realize_event()
{
//    title_vprt->set_hadjustment( *(swnd->get_hadjustment()) );
}


void RuleSetList::set_data_changed_flag(bool flag)
{
    Gtk::Widget *p=this;
    do { p=p->get_parent(); } while (p!=NULL && p->get_name()!="BuiltinDialog");
    BuiltinDialog *b=dynamic_cast<BuiltinDialog*>(p);
    assert (b!=NULL);
    b->data_changed_flag(flag);
}


void RuleSetList::Clear()
{
    if ( ! table_elements.empty() ) 
    {
        int last_row = get_row_by_rule_num( ruleset->size() +1 );
	for (int r=0; r<=last_row; ++r) 
        {
	    map<int,Gtk::Widget*> row_list= table_elements[r];
	    map<int,Gtk::Widget*>::iterator i;
	    for (i=row_list.begin(); i!=row_list.end(); ++i) {
		Gtk::Widget *w=( (*i).second );
		assert( w!=NULL );
		w->hide();
		tbl->remove(*w);
                delete w;
	    }
	}
    }
    table_elements.clear();

    if (bottom!=NULL)
    {
        bottom->hide();
        tbl->remove(*bottom);
        delete bottom;
        bottom=NULL;
    }
    current_selected=NULL;

    clearTitles();
}

Gtk::Widget* RuleSetList::_register_child(int row,int col,Gtk::Widget *c)
{
    table_elements[row][col]=c;

    return c;
}


void RuleSetList::_add_hseparator(gint row,gint col)
{
    Gtk::HSeparator *sep= new Gtk::HSeparator();
    tbl->attach( *sep , 0 , col , row , row+1 , GTK_FILL, GTK_FILL, 0, 0);
    _register_child(row,col,sep);
}

void RuleSetList::_add_vseparator(gint row,gint col)
{
    Gtk::VSeparator *sep= new Gtk::VSeparator();
    tbl->attach( *sep , col , col+1 , row , row+1 , 0, GTK_FILL, 0, 0);
    _register_child(row,col,sep);
}


gint RuleSetList::get_row_by_rule_num(gint rule_n)
{
//    return( rule_n+1 );     // if there are no separators but title is part of the same table widget
    return (rule_n);
}

gint RuleSetList::get_rule_by_row(gint row_n)
{
//    return( row_n-1 );       // if there are no separators but title is part of the same table widget
    return (row_n);
}

int RuleSetList::resizeTable()
{
//    int tbl_size=ruleset->size()*2+1 +1;
//    resize( tbl_size , countElements()*2 );

    int tbl_size=ruleset->size() +1;
    tbl->resize( tbl_size , countElements() );
    return tbl_size;
}

void RuleSetList::removeWidgetsInRow(int row)
{
    if ( ! table_elements.empty() ) {
	map<int,Gtk::Widget*>::iterator i;
	for (i=table_elements[row].begin(); i!=table_elements[row].end(); ++i) {
	    Gtk::Widget *w= (*i).second;
	    tbl->remove(*w);
	}	
	table_elements[row].clear();
    }
}

void RuleSetList::swapWidgetsInRows(int row1, int row2)
{
    if ( ! table_elements.empty() ) {

	map<int,Gtk::Widget*>  rm1=table_elements[row1];
	map<int,Gtk::Widget*>  rm2=table_elements[row2];
	table_elements[row1].clear();
	table_elements[row2].clear();

	map<int,Gtk::Widget*>::iterator i;
	for (i=rm1.begin(); i!=rm1.end(); ++i) {
	    if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		PolicyListElement* pe=dynamic_cast<PolicyListElement*>( (*i).second );
		int c=pe->get_col();

//		pe->ref();
		tbl->remove(*pe);
		pe->set_row(row2);
		attachTableElement(row2,c,pe);
//		pe->unref();
	    }
	}	
	for (i=rm2.begin(); i!=rm2.end(); ++i) {
	    if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		PolicyListElement* pe=dynamic_cast<PolicyListElement*>( (*i).second );
		int c=pe->get_col();

//		pe->ref();
		tbl->remove(*pe);
		pe->set_row(row1);
		attachTableElement(row1,c,pe);
//		pe->unref();
	    }
	}	

    }

}

void RuleSetList::moveWidgetsDown(int first,int last)
{
    PolicyListElement *pe;
    int last_row = (last!=-1) ? last : get_row_by_rule_num( ruleset->size() +1 );

    if ( ! table_elements.empty() ) {

	for (int r=last_row-1; r>=first; --r) {
	    map<int,Gtk::Widget*>::iterator i;
	    for (i=table_elements[r].begin(); i!=table_elements[r].end(); ++i) {
		if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		    pe=dynamic_cast<PolicyListElement*>( (*i).second );
		    int c=pe->get_col();

//		    pe->ref();
		    tbl->remove(*pe);

		    pe->set_row(r+1);
		    attachTableElement(r+1,c,pe);
//		    pe->unref();
		}
	    }
	}
    }
}

void RuleSetList::moveWidgetsUp(int first,int last)
{
    PolicyListElement *pe;
    int last_row = (last!=-1) ? last : get_row_by_rule_num( ruleset->size() +1 );

    if ( ! table_elements.empty() ) {

	for (int r=first; r<=last_row; ++r) {
	    map<int,Gtk::Widget*>::iterator i;
	    for (i=table_elements[r].begin(); i!=table_elements[r].end(); ++i) {
		if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		    pe=dynamic_cast<PolicyListElement*>( (*i).second );
		    int c=pe->get_col();

//		    pe->ref();
		    tbl->remove(*pe);

		    pe->set_row(r-1);
		    attachTableElement(r-1,c,pe);
//		    pe->unref();
		}
	    }
	    table_elements[r].clear();
	}
    }
}

void RuleSetList::copyRuleContent(Rule *dst, Rule *src)
{
    string id=dst->getId();
    int     p=dst->getPosition();

    if ( src->isDisabled() ) dst->disable();
    else                     dst->enable();

    map<string, string>::const_iterator i;
    for(i=dst->dataBegin(); i!=dst->dataEnd(); ++i) {
	string f= (*i).first;
	dst->setStr(f, src->getStr(f) );
    }

    dst->setComment( src->getComment() );

    list<FWObject*>::iterator j;
    for(j=dst->begin(); j!=dst->end(); ++j) {
	string    dtype= (*j)->getTypeName();
	FWObject *selem= src->getFirstByType(dtype);
	if (selem!=NULL) 
	    (*j)->duplicate(selem);
    }

    if (id!="")	dst->setId(id);
    dst->setPosition(p);
}


void RuleSetList::insertRuleAtTop(Rule *r)
{
    insertRuleBefore(0,r);
    buildTableBottom( get_row_by_rule_num( ruleset->getRuleSetSize() ) );
}

void RuleSetList::insertRuleBefore(gint rule_n,Rule *r)
{
    if (r!=NULL && 
	ruleset->getTypeName()==Policy::TYPENAME && 
	r->getTypeName()!=PolicyRule::TYPENAME)  return;
    if (r!=NULL && 
	ruleset->getTypeName()==NAT::TYPENAME    && 
	r->getTypeName()!=NATRule::TYPENAME   )  return;

    if (ruleset->getRuleSetSize()==0)  rule_n=0;

    Rule *newrule;
    int   nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( (newrule=ruleset->insertRuleBefore(nr))!=NULL ) {

        if (Policy::cast(ruleset) || InterfacePolicy::cast(ruleset))
        {
            (PolicyRule::cast(newrule))->setLogging(supportsLogging());
            (PolicyRule::cast(newrule))->setAction(PolicyRule::Deny);
        }

        if (InterfacePolicy::cast(ruleset))
        {
            (PolicyRule::cast(newrule))->setDirection(PolicyRule::Both);
        }

        if (r!=NULL) {
            copyRuleContent(newrule,r);
        }
        resizeTable();

        moveWidgetsDown( get_row_by_rule_num(nr) );
        buildRule( nr , newrule );
        renumberRules();
        activate_rule( nr );
	
        set_data_changed_flag(true);
    }
    buildTableBottom( get_row_by_rule_num( ruleset->getRuleSetSize() ) );
}


void RuleSetList::appendRuleAfter(gint rule_n,Rule *r)
{
    if (r!=NULL && 
	ruleset->getTypeName()==Policy::TYPENAME && 
	r->getTypeName()!=PolicyRule::TYPENAME)  return;
    if (r!=NULL && 
	ruleset->getTypeName()==NAT::TYPENAME    && 
	r->getTypeName()!=NATRule::TYPENAME   )  return;

    if (ruleset->getRuleSetSize()==0)  insertRuleAtTop(r);
    else
    {
        Rule *newrule;
        int nr;
        nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
        if ( (newrule=ruleset->appendRuleAfter( nr ))!=NULL ) 
        {
            if (Policy::cast(ruleset) || InterfacePolicy::cast(ruleset))
            {
                (PolicyRule::cast(newrule))->setLogging(supportsLogging());
                (PolicyRule::cast(newrule))->setAction(PolicyRule::Deny);
            }

            if (InterfacePolicy::cast(ruleset))
            {
                (PolicyRule::cast(newrule))->setDirection(PolicyRule::Both);
            }

            if (r!=NULL) {
                copyRuleContent(newrule,r);
            }
            resizeTable();

            moveWidgetsDown( get_row_by_rule_num(nr+1) );
            buildRule( nr+1 , newrule );
            renumberRules();
            activate_rule( nr );
	
            set_data_changed_flag(true);
        }
        buildTableBottom( get_row_by_rule_num( ruleset->getRuleSetSize() ) );
    }
}

void RuleSetList::appendRuleAtBottom(Rule *r)
{
    if (ruleset->getRuleSetSize()==0)  insertRuleAtTop(r);
    else {
        appendRuleAfter( ruleset->getRuleSetSize()-1 , r );
        buildTableBottom( get_row_by_rule_num( ruleset->getRuleSetSize() ) );
    }
}

void RuleSetList::delRule(gint rule_n)
{
    deselect_current();

    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->deleteRule(nr) ) {

	deactivate_row(current_selected_row);
	current_selected_row=0;
	removeWidgetsInRow( get_row_by_rule_num(nr) );
	moveWidgetsUp( get_row_by_rule_num(nr+1) );
	renumberRules();
	activate_rule( nr );

	set_data_changed_flag(true);
    }
}

void RuleSetList::moveRuleUp(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->moveRuleUp(nr) ) {
	swapWidgetsInRows( get_row_by_rule_num(nr-1),get_row_by_rule_num(nr) );
	renumberRules();
	activate_rule(nr-1);
	set_data_changed_flag(true);
    }
}

void RuleSetList::moveRuleDown(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->moveRuleDown(nr) ) {
	swapWidgetsInRows( get_row_by_rule_num(nr),get_row_by_rule_num(nr+1) );
	renumberRules();
	activate_rule(nr+1);
	set_data_changed_flag(true);
    }
}

void RuleSetList::moveRuleTo(int rule1,int rule2)
{
    map<int,Gtk::Widget*>::iterator i;
    PolicyListElement*  pe;

    if ( rule1==rule2 || rule1+1==rule2) return;

    if ( ruleset->moveRule(rule1,rule2) ) {
	int row1=get_row_by_rule_num(rule1);
	int row2=get_row_by_rule_num(rule2);

	map<int,Gtk::Widget*> rm1=table_elements[row1];

	int dest_row;

	for (i=rm1.begin(); i!=rm1.end(); ++i) {
	    if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		pe=dynamic_cast<PolicyListElement*>( (*i).second );
//		pe->ref();
		tbl->remove(*pe);
	    }
	}
	table_elements[row1].clear();


	if (row1<row2)   {
	    moveWidgetsUp( row1+1,row2-1 );
	    dest_row=row2-1;
	} else {
	    moveWidgetsDown( row2,row1 );
	    dest_row=row2;
	}

	for (i=rm1.begin(); i!=rm1.end(); ++i) {
	    if (dynamic_cast<PolicyListElement*>( (*i).second )!=NULL) {
		pe=dynamic_cast<PolicyListElement*>( (*i).second );
		int c=pe->get_col();

		pe->set_row(dest_row);
		attachTableElement(dest_row,c,pe);
//		pe->unref();
	    }
	}

	renumberRules();
	activate_rule( get_rule_by_row(dest_row) );
    }
	
    set_data_changed_flag(true);
}

void RuleSetList::disableRule(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->disableRule(nr) ) {
	int r=get_row_by_rule_num( nr );
	map<int,Gtk::Widget*>::iterator i;
	for (i=table_elements[r].begin(); i!=table_elements[r].end(); ++i) {
	    if (dynamic_cast<PolicyListRuleNum*>( (*i).second )!=NULL) {
		PolicyListRuleNum* pen=dynamic_cast<PolicyListRuleNum*>( (*i).second );
		pen->setNeg(true);
		break;
	    }
	}

	set_data_changed_flag(true);
    }
}

void RuleSetList::enableRule(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    if ( ruleset->enableRule(nr) ) {
	int r=get_row_by_rule_num( nr );
	map<int,Gtk::Widget*>::iterator i;
	for (i=table_elements[r].begin(); i!=table_elements[r].end(); ++i) {
	    if (dynamic_cast<PolicyListRuleNum*>( (*i).second )!=NULL) {
		PolicyListRuleNum* pen=dynamic_cast<PolicyListRuleNum*>( (*i).second );
		pen->setNeg(false);
		break;
	    }
	}

	set_data_changed_flag(true);
    }
}

bool  RuleSetList::isRuleDisabled(gint rule_n)
{
    int nr;
    nr=(rule_n==-1)?get_rule_by_row(current_selected_row):rule_n;
    return ( ruleset->isRuleDisabled(nr) );
}

PolicyListElement* RuleSetList::get_element(gint r,gint c)
{
    if ( ! table_elements.empty() )
	return dynamic_cast<PolicyListElement*>( table_elements[r][c] );

    return(NULL);
}


void RuleSetList::attachTableElement(int row, int col, Gtk::Widget *pe,guint x_attach,guint y_attach)
{
    pe->set_user_data(this);
    tbl->attach( *pe , col , col+1 , row , row+1 , x_attach, y_attach, 0, 0);
    if (dynamic_cast<PolicyListElement*>(pe)!=NULL)
	dynamic_cast<PolicyListElement*>(pe)->deactivate();
    pe->show();

    _register_child(row,col,pe);
}

void RuleSetList::clearTitles()
{
    for (list<Gtk::Widget*>::iterator i=titles.begin(); i!=titles.end(); ++i)
    {
        (*i)->hide();
        title_row->remove( *(*i) );
        delete (*i);
    }
    titles.clear();
}

void RuleSetList::addTitleButton( const string &ttl )
{
    Gtk::Button *btn = new Gtk::Button(  ttl );
    btn->set_sensitive(false);
//    attachTableElement(row,col,btn,x_attach,y_attach);
    titles.push_back(btn);
    title_row->pack_start( *btn , false, false, 0);
}

int  RuleSetList::countElements() { return 0; }

void RuleSetList::renumberRules()
{
    PolicyListRuleNum *pen;

    if ( ! table_elements.empty() ) {

	for (unsigned r=0,rule_n=0; r<table_elements.size(); ++r,++rule_n) {
	    map<int,Gtk::Widget*>::iterator i;
	    for (i=table_elements[r].begin(); i!=table_elements[r].end(); ++i) {
		if (dynamic_cast<PolicyListRuleNum*>( (*i).second )!=NULL) {
		    pen=dynamic_cast<PolicyListRuleNum*>( (*i).second );
		    pen->setNum(rule_n);
		    break;
		}
	    }
	}
    }
}

void RuleSetList::buildTitles(){}

void RuleSetList::build()
{
    int   rule_n;

    hide();

    Clear();

    resizeTable();

    buildTitles();

    rule_n=0;

    list<FWObject*>::iterator m;
    for (m=ruleset->begin(); m!=ruleset->end(); ++m,++rule_n ) {
	buildRule( rule_n , *m );
    }

    buildTableBottom( get_row_by_rule_num(rule_n)+1 );

    show_all();

    activate_rule(0);
    select_element(0,0);
    request_focus();
}


void RuleSetList::buildRule(int rule_no, libfwbuilder::FWObject* rule) {}

void RuleSetList::buildTableBottom(int row)
{
    if (bottom!=NULL) 
    {
        bottom->hide();
        tbl->remove(*bottom);
        delete bottom;
    }

    bottom = new Gtk::EventBox();

    tbl->attach( *bottom , 0 , countElements()*2 , row , row+1 , 
                 GTK_FILL|GTK_EXPAND, GTK_FILL|GTK_EXPAND, 0, 1);

    bottom->show();
//    _register_child(row,0,bottom);
}

void RuleSetList::create_popup_menu()
{
    if (popup_menu==NULL)
    {
	const char *menu_items0[] = 
	    { "Insert rule at the top",
	      "Append rule at the bottom",
	      NULL,
	      NULL } ;

	if ( ruleset->size()==0 ) {
	    menu_items0[0]=_("Insert New rule");
	    menu_items0[1]="";
	    menu_items0[2]=_("Paste Rule");
	    menu_items0[3]=NULL;
	}

        popup_menu=new popupMenu(menu_items0);
        popup_menu->menu_select.connect(SigC::slot(this,&RuleSetList::popup_menu_callback));
    }
}

void RuleSetList::show_popup_menu()
{
    create_popup_menu();

    popup_menu->popup(0,0);
}

void RuleSetList::popup_menu_callback(int menu_choice)
{
    switch (menu_choice) {
    case 0:   // insert at the top
        insertRuleAtTop();
        break;

    case 1:   // append at the bottom
        appendRuleAtBottom();
        break;

    case 2:   // paste
    {
        FWObject *obj;
        if ( (obj=FWObjectClipboard::obj_clipboard->getObject())!=NULL &&
             Rule::cast(obj)!=NULL ) {
            appendRuleAtBottom( Rule::cast(obj) );
        }
    }
    break;
    }
}


/***************************************************************************/

void PolicyList::buildTitles()
{
    gint                row, col;

    row=0;
    col=0;

    addTitleButton(_("Num"));         col++;
    addTitleButton(_("Source"));      col++;
    addTitleButton(_("Destination")); col++;
    addTitleButton(_("Service"));     col++;
    addTitleButton(_("Action"));      col++;
    if (supports_time)
    {
        addTitleButton(_("Time"));    col++;
    }
    if (supports_logging || supports_rule_options)
    {
        addTitleButton(_("Options")); col++;
    }
    addTitleButton(_("Comment"));     col++;

    total_no_of_elements=col;
}

int  PolicyList::countElements() { return total_no_of_elements; }

void PolicyList::buildRule(int rule_n, FWObject* r)
{
    int row=get_row_by_rule_num( rule_n );
    int col;
    PolicyListElement  *pe;

    if (PolicyRule::cast(r)!=NULL) {
	PolicyRule *rule=PolicyRule::cast(r);

	col=0;

	pe = new PolicyListRuleNum(row,col,rule_n,rule->isDisabled());
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe = new PolicyListElement(row,col,rule->getSrc());
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe = new PolicyListElement(row,col,rule->getDst());
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe = new PolicyListElement(row,col,rule->getSrv());
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe = new PolicyListRuleAction(row,col,rule);
	attachTableElement(row,col,pe);
        pe->init();
	col++;

        if (supports_time)
        {
/* DTD declares element When optional, so it may not be there (e.g. if
 * XMl file was created in a different UI program that followed our
 * DTD)
 */
            if (rule->getWhen()==NULL)
            {
                FWObject *re;
                re=((FWObjectDatabase*)rule->getRoot())->create("When",true); 
                assert(re!=NULL); 
                rule->add(re);
            }
            pe = new PolicyListElement(row,col,rule->getWhen());
            attachTableElement(row,col,pe);
            pe->init();
            col++;
        }

        if (supports_logging || supports_rule_options)
        {
            pe =  new PolicyListRuleOpt(row,col,rule) ;
            attachTableElement(row,col,pe);
            pe->init();
            col++;
        }

	pe =  new PolicyListRuleComment(row,col,rule) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;


	row++;
    }


}



/***************************************************************************/

void InterfacePolicyList::buildTitles()
{
    gint                row, col;

    row=0;
    col=0;

    addTitleButton(_("Num"));         col++;
    addTitleButton(_("Source"));      col++;
    addTitleButton(_("Destination")); col++;
    addTitleButton(_("Service"));     col++;
    addTitleButton(_("Action"));      col++;
    addTitleButton(_("Direction"));   col++;
    if (supports_time)
    {
        addTitleButton(_("Time"));    col++;
    }
    if (supports_logging || supports_rule_options)
    {
        addTitleButton(_("Options")); col++;
    }
    addTitleButton(_("Comment"));     col++;

    total_no_of_elements=col;
}

int  InterfacePolicyList::countElements() { return total_no_of_elements; }

void InterfacePolicyList::buildRule(int rule_n, FWObject* r)
{
    int row=get_row_by_rule_num(rule_n);
    int col;
    PolicyListElement  *pe;

    if (PolicyRule::cast(r)!=NULL) {
	PolicyRule *rule=PolicyRule::cast(r);

	col=0;

	pe=new PolicyListRuleNum(row,col,rule_n,rule->isDisabled());
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getSrc()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getDst()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getSrv()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListRuleAction(row,col,rule) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListRuleDir(row,col,rule) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

        if (supports_time)
        {
            if (rule->getWhen()==NULL)
            {
                FWObject *re;
                re=((FWObjectDatabase*)rule->getRoot())->create("When",true); 
                assert(re!=NULL); 
                rule->add(re);
            }
            pe = new PolicyListElement(row,col,rule->getWhen());
            attachTableElement(row,col,pe);
            pe->init();
            col++;
        }

        if (supports_logging || supports_rule_options)
        {
            pe =  new PolicyListRuleOpt(row,col,rule) ;
            attachTableElement(row,col,pe);
            pe->init();
            col++;
        }

	pe =  new PolicyListRuleComment(row,col,rule) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;


	row++;
    }
}

/***************************************************************************/

void NATList::buildTitles()
{
    gint                row, col;

    row=0;
    col=0;

    addTitleButton(_("Num"));            col++;
    addTitleButton(_("Original Src"));   col++;
    addTitleButton(_("Original Dst"));   col++;
    addTitleButton(_("Original Srv"));   col++;
    addTitleButton(_("Translated Src")); col++;
    addTitleButton(_("Translated Dst")); col++;
    addTitleButton(_("Translated Srv")); col++;
    addTitleButton(_("Comment"));        col++;

    total_no_of_elements=col;
}

int  NATList::countElements() { return total_no_of_elements; }

void NATList::buildRule(int rule_n, FWObject* r)
{
    int row=get_row_by_rule_num(rule_n);
    int col;
    PolicyListElement  *pe;

    if (NATRule::cast(r)!=NULL) {
	NATRule *rule=NATRule::cast(r);

	col=0;

	pe=new PolicyListRuleNum(row,col,rule_n,rule->isDisabled());
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getOSrc()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getODst()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getOSrv()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getTSrc()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getTDst()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListElement(row,col,rule->getTSrv()) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	pe =  new PolicyListRuleComment(row,col,rule) ;
	attachTableElement(row,col,pe);
        pe->init();
	col++;

	row++;
    }
}

