/* 

                          Firewall Builder

                 Copyright (C) 2000 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: PolicyListItem.cc,v 1.69 2003/04/07 01:04:41 vkurland Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "fwbuilder/libfwbuilder-config.h"
#include "fwbuilder/FWObjectDatabase.hh"
#include "fwbuilder/Host.hh"
#include "fwbuilder/Network.hh"
#include "fwbuilder/Firewall.hh"


#include <gdk-pixbuf/gdk-pixbuf.h>

#include "glademm_support.hh"

#include "ObjectTree.hh"
#include "main_window.hh"
#include "ObjectQuickView.hh"
#include "helpers.hh"
#include "BuiltinDialog.hh"
#include "PolicyListItem.hh"
#include "PolicyListElement.hh"
#include "PolicyList.hh"
#include "SimplePixmap.hh"

#include <assert.h>


#define MAX_COMMENT_FIELD_WIDTH   32

using namespace libfwbuilder;


PolicyListItem::PolicyListItem(const string &pixfl, 
			       const string &neg_pixfl, 
			       const string &label_text,
			       bool  neg_flag) 
{
    object=NULL;

    icon_w=0;
    icon_h=0;
    x=y=0;

    neg=neg_flag;
    label=NULL;

    quick_view_txt="";

    overlap=false;

    x=0;
    y=0;
    width=0;
    height=0;


    icon_size=Resources::global_res->getResourceInt("/FWBuilderResources/UI/SmallIconSize");

    set_name("PolicyListItem");
    set_flags( GTK_CAN_FOCUS );

    hbox=new Gtk::HBox();
    add(*hbox);
    hbox->show();

    if (pixfl!="")  addIcon(pixfl,neg_pixfl);

    addLabel(label_text);

    show_all();

    button_press_event.connect(SigC::slot(this,&PolicyListItem::on_button_press_event));
    button_release_event.connect_after(SigC::slot(this,&PolicyListItem::on_button_release_event));

    enter_notify_event.connect(SigC::slot(this,&PolicyListItem::on_enter_notify_event));
    leave_notify_event.connect(SigC::slot(this,&PolicyListItem::on_leave_notify_event));
}

PolicyListItem::~PolicyListItem()
{
    clear();
    remove();
    delete hbox;
    deactivateObjectQuickView();
    return ;


    for (std::list<SimplePixmap*>::iterator i=pixmaps.begin(); i!=pixmaps.end(); ++i)
    {
	SimplePixmap *pm=(*i);
        if (pm) delete pm;
    }

    for (std::list<SimplePixmap*>::iterator i=n_pixmaps.begin(); i!=n_pixmaps.end(); ++i)
    {
	SimplePixmap *pm=(*i);
        if (pm) delete pm;
    }

    pixmaps.clear();
    n_pixmaps.clear();

    deactivateObjectQuickView();
}

void PolicyListItem::drag_data_received_impl(GdkDragContext   *context,
					     gint              x,
					     gint              y,
					     GtkSelectionData *data,
					     guint             info,
					     guint32           time)
{
    getParentPolicyListElement()->drag_data_received_impl(context,x,y,
							  data,info,time);
}

PolicyListElement* PolicyListItem::getParentPolicyListElement()
{
    Gtk::Widget       *p=this;
    do { p=p->get_parent(); } 
    while ( dynamic_cast<PolicyListElement*>(p)==NULL );
    return (PolicyListElement*)p;
}


void PolicyListItem::activateObjectQuickView()
{
    try{
        if ( object!=NULL || quick_view_txt!="" ) {

            PolicyListElement *pl_el=getParentPolicyListElement();
            RuleElement       *rel=pl_el->get_rule_element();
            if (rel!=NULL && rel->isAny()) return;

            ObjectQuickView* ov=ObjectQuickView::getInstance(main_window::getMainWindowForWidget(this));
            if (object) ov->setObject( object );
            else        ov->setText( quick_view_txt );
            ov->attachTo(this);
            ov->activate();
        }
    } catch (FWException &ex)  {
/* exception could have happened inside ObjectQuickView::setObject
 * (e.g. when object has invalid netmask) */
        ;
    }
}

void PolicyListItem::deactivateObjectQuickView()
{
    ObjectQuickView* ov=ObjectQuickView::getInstance(main_window::getMainWindowForWidget(this));
    ov->deactivate();
}

gint PolicyListItem::on_enter_notify_event(GdkEventCrossing* ev)
{
    activateObjectQuickView();
    return(true);
}

gint PolicyListItem::on_leave_notify_event(GdkEventCrossing* ev)
{
    deactivateObjectQuickView();
    return(true);
}

gint PolicyListItem::key_press_event_impl(GdkEventKey* ev)
{
    return (getParentPolicyListElement()->key_press_event_impl(ev));
}

gint PolicyListItem::on_button_press_event(GdkEventButton *ev)
{
    PolicyListElement *pe=getParentPolicyListElement();

    deactivateObjectQuickView();

    if ( ev->type == GDK_BUTTON_PRESS) {
	pe->deselect_all();
	pe->select();
	pe->set_current_selected(this);
	set_state(GTK_STATE_SELECTED);
    }

    return( true );
}

gint PolicyListItem::on_button_release_event(GdkEventButton *ev)
{
    PolicyListElement *pe=getParentPolicyListElement();

    deactivateObjectQuickView();

    if ( ev->type == GDK_BUTTON_RELEASE && ev->button==3 ) {
	if (get_state()!=GTK_STATE_SELECTED) {
	    pe->deselect_all();
	    pe->select();
	    pe->set_current_selected(this);
	    set_state(GTK_STATE_SELECTED);
	}
//	pe->schedule_popup_menu(this);
	pe->show_popup_menu(this);

/*
 * NB: some menu items may lead to this PolicyListItem object destruction
 * (e.g. 'cut'). We should return from this method immediately.
 */
    }

    return( true );
}


void PolicyListItem::setQuickView(const string &txt)
{
    quick_view_txt=txt;
}

void PolicyListItem::state_changed_impl(GtkStateType p0)
{
    deactivateObjectQuickView();
    Gtk::EventBox::state_changed_impl(p0);
}


void PolicyListItem::addLabel(const string &txt)
{
    if (label==NULL) {
	label= new Gtk::Label(txt);
	hbox->pack_start(*label,false,false);
	label->show();
    } else
	label->set_text(txt);
}

string PolicyListItem::getLabel()
{
    return(label->get_text());
}


void  PolicyListItem::addIcon(const string &icn_file,const string &neg_icn_file)
{
    SimplePixmap *pm;

    pm=new SimplePixmap(icn_file);
    pixmaps.push_back(pm);
    if (!neg) {
        hbox->pack_start(*pm,false,false);
        pm->show();
    }

    if (neg_icn_file!="")
    {
        pm =new SimplePixmap(neg_icn_file);
        n_pixmaps.push_back(pm);
        if (neg) {
            hbox->pack_start(*pm,false,false);
            pm->show();
        }
    }
}


void PolicyListItem::clear()
{
    Gtk::Box_Helpers::BoxList   bl= hbox->children();
    bl.clear();

    for (std::list<SimplePixmap*>::iterator i=pixmaps.begin(); i!=pixmaps.end(); ++i)
    {
	SimplePixmap *pm=(*i);
        delete pm;
    }

    for (std::list<SimplePixmap*>::iterator i=n_pixmaps.begin(); i!=n_pixmaps.end(); ++i)
    {
	SimplePixmap *pm=(*i);
        delete pm;
    }

    pixmaps.clear();
    n_pixmaps.clear();

    if (label!=NULL)  delete label;
    label=NULL;
}


void PolicyListItem::setNeg(bool neg_flag)
{
    SimplePixmap *pm;
    if (neg)
    {
        for (std::list<SimplePixmap*>::iterator i=n_pixmaps.begin(); i!=n_pixmaps.end(); ++i)
        {
            pm=(*i);  if (pm) hbox->remove(*pm);
        }
    }
    else
    {
        for (std::list<SimplePixmap*>::iterator i=pixmaps.begin(); i!=pixmaps.end(); ++i)
        {
            pm=(*i);  if (pm) hbox->remove(*pm);
        }
    }

    neg=neg_flag;

    if (neg)
    {
        for (std::list<SimplePixmap*>::iterator i=n_pixmaps.begin(); i!=n_pixmaps.end(); ++i)
        {
            pm=(*i);
            hbox->pack_start(*pm,false,false);
            hbox->reorder_child(*pm,0);
            pm->show();
        }
    }
    else
    {
        for (std::list<SimplePixmap*>::iterator i=pixmaps.begin(); i!=pixmaps.end(); ++i)
        {
            pm=(*i);
            hbox->pack_start(*pm,false,false);
            hbox->reorder_child(*pm,0);
            pm->show();
        }
    }
}


void PolicyListItem::setOverlap(bool ovr)
{
    overlap=ovr;
}


/************************************************************************/

/*
 * I am using ObjectTree::get_tree_label to generate a label for the
 * object's icon in a manner, uniform with that used in the tree.
 */
PolicyListObjectItem::PolicyListObjectItem(FWObject *o,
					   bool  neg_flag,
					   bool  change_any_2_orig) :
    PolicyListItem( Resources::getIconFileName(o), 
		    Resources::getNegIconFileName(o), 
                    ObjectTree::get_tree_label(o), neg_flag )
//		    o->getName(),  neg_flag  )
{
    object=o;

    drag_source_set ( 
	static_cast<GdkModifierType>(GDK_BUTTON1_MASK) ,
	target_table, n_targets, 
	static_cast<GdkDragAction>(GDK_ACTION_COPY) );


    drag_begin.connect(
	slot(this,&PolicyListObjectItem::source_drag_begin));
    drag_data_get.connect(
	slot(this,&PolicyListObjectItem::source_drag_data_get));

    if (change_any_2_orig &&  getLabel()=="Any")	addLabel("Original");

#ifdef __MINGW32__
/*
 *  PolicyListItem is a child of PolicyListElement (in a sence of
 *  widget structure, not as a class). PolicyListElement is d&d
 *  recipient and actually should be processing drop event. However,
 *  it turned out that on win32 PolicyListItem somehow shielded
 *  PolicyListElement so the latter did not recieve event and
 *  therefore did not process drop. User had to aim accurately at the
 *  narrow border around PolicyListItem widgets to activate drop event
 *  for PolicyListElement. 
 *
 *  Making PolicyListItem d&d recipient fixes the problem, but it does
 *  not look good when it highlights under mouse cursor while user is
 *  dragging an object (it should not matter where within the
 *  PolicyListElement user drops the object, yet if different
 *  PolicyListItems highlight separately, it creates an illusion that
 *  the effect might be different if user drops an object over one or
 *  another existing PolicyListItem). To avoid this unpleasant
 *  highlighting, we use a combinations of flags in drag_dest_set to
 *  turn auto-highlight off. We use the same combination in
 *  PolicyListElement, too. This should be considered a workaround
 *  since we want widgets to highlight when mouse cursor flies over
 *  them, but it seems that gtkmm port to win32 needs to be fixed for
 *  that.  (03/12/02 vk)
 */
    drag_dest_set ( GtkDestDefaults(GTK_DEST_DEFAULT_DROP | GTK_DEST_DEFAULT_MOTION),
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
#endif
}

void PolicyListObjectItem::source_drag_begin(GdkDragContext     *context)
{
    return;
}

void PolicyListObjectItem::source_drag_data_get  ( GdkDragContext     *context,
					   GtkSelectionData   *selection_data,
						   guint               info,
						   guint32             time )
{
    PolicyListObjectItem *plitm=this;

    gtk_selection_data_set (selection_data,
			    gdk_atom_intern("PolicyListObjectItem",TRUE),
			    7,
			    (const guchar*)(&plitm),
			    sizeof(PolicyListItem*) );
}

/************************************************************************/

PolicyListRuleNumItem::PolicyListRuleNumItem(const string& label,
					     bool neg_flag) :
    PolicyListItem(Resources::global_res->getIconPath("Blank"),
		   Resources::global_res->getIconPath("neg"),
		   "",neg_flag)
{
    set_name("PolicyListRuleNumItem");

    setOverlap(true);

    addLabel(label);

    drag_source_set ( 
	static_cast<GdkModifierType>(GDK_BUTTON1_MASK) ,
	target_table, n_targets, 
	static_cast<GdkDragAction>(GDK_ACTION_COPY) );
    drag_begin.connect(
	slot(this,&PolicyListRuleNumItem::source_drag_begin));
    drag_data_get.connect(
	slot(this,&PolicyListRuleNumItem::source_drag_data_get));

#ifdef __MINGW32__
    drag_dest_set ( GtkDestDefaults(GTK_DEST_DEFAULT_DROP | GTK_DEST_DEFAULT_MOTION),
		    target_table, n_targets , 
		    static_cast<GdkDragAction>(GDK_ACTION_COPY) );
#endif
}

void PolicyListRuleNumItem::setNum(const string& label)
{
    addLabel(label);
}

void PolicyListRuleNumItem::source_drag_begin(GdkDragContext     *context)
{
    PolicyListElement *pl_el=getParentPolicyListElement();
    Gtk::Widget       *pl=pl_el->get_parent(); // (table inside RuleSetList )

    GdkPixmap     *pmap;
    GdkBitmap     *mask;
    Gdk_Colormap  cmap ( get_colormap () );

/*
 *     Trick: determine physical size of one policy line in pixels,
 *     then grab portion of the screen in that area and use it as dragging
 *     icon. User is going to see that the whole rule is being dragged 
 */

    gint x,y,w,h,d;
/*
    x=pl->getX();
    y=pl_el->getY();
    h=pl_el->getHeight();
    w=pl->getWidth();
*/

    pl->get_window().get_geometry(x,y,w,h,d);
    y=pl_el->getY();
    h=pl_el->getHeight();

/*
 *  PolicyList may be too large and can be clipped. Check the size of its
 *  parent and if it is smaller, use it.
 */

    Gtk::Widget       *par=pl;
    while ( (par=par->get_parent())!=NULL ) {
	gint w1,h1;
	par->get_window().get_size( w1, h1);
	if (w1<w) w=w1;
	if (h1<h) h=h1;
    }

/*
 * x and y are relative coordinates within pl->get_window()
 */

/*
 * scrnx and scrny represent absolute position of pl->get_window() on
 * the desktop. Since pl (which is a widget showing a RuleSet object)
 * can be scrolled, its Y coordinate can be negative sometimes. If it
 * is negative, it means the widget has been scrolled up and its "top"
 * is above screen's top edge.
 */
    int scrnx, scrny;
    pl->get_window().get_deskrelative_origin(scrnx,scrny);


    if ( scrnx<0 )
    {
// left boundary of pl->get_window() is beyond the left edge of the screen
        int shift= -scrnx;
        x=x+shift+1;
        w=w-shift-1;
    }
    if ( y+scrny<0 )
    {
// top edge of ruleelement widget is above the top edge of the screen
        int shift= -y-scrny;
        y=y+shift+1;
        h=h-shift-1;
    }

    if ( scrnx+x+w > gdk_screen_width() )
        w=gdk_screen_width()-x-scrnx-1; // right boundary of pl->get_window() is beyond the right edge of the screen
    if ( scrny+y+h > gdk_screen_height())
        h=gdk_screen_height()-y-scrny-1;

    GdkPixbuf* npb=gdk_pixbuf_get_from_drawable(NULL, pl->get_window(),
                                                NULL,
                                                x,y,
                                                0,0,
                                                w,h );
    assert(npb!=NULL);
/*
 * make it look dark 
 */

    GdkPixbuf *cpb=gdk_pixbuf_composite_color_simple(npb, w,h, 
                                                     GDK_INTERP_NEAREST,
                                                     127,
                                                     8,
                                                     0,0);
    assert(cpb!=NULL);

    gdk_pixbuf_render_pixmap_and_mask(cpb, &pmap, &mask, 127);


    Gdk_Pixmap drag_icon=Gdk_Pixmap(pmap);
    Gdk_Bitmap drag_mask=Gdk_Bitmap(mask);

    drag_source_set_icon(cmap, drag_icon, drag_mask);
    
}

void PolicyListRuleNumItem::source_drag_data_get( GdkDragContext     *context,
					   GtkSelectionData   *selection_data,
						   guint               info,
						   guint32             time )
{
    PolicyListRuleNumItem *plitm=this;

    gtk_selection_data_set (selection_data,
			    gdk_atom_intern("PolicyListRuleNumItem",TRUE),
			    6,
			    (const guchar*)(&plitm),
			    sizeof(PolicyListItem*) );
}

/************************************************************************/




PolicyListCommentItem::PolicyListCommentItem(const string& text) :
    PolicyListItem("","",text)
{
    label->set_alignment(0, 0.5);
    label->set_justify(GTK_JUSTIFY_LEFT);
}

