/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: main.cpp,v 1.89.4.4 2006/03/26 17:36:27 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"

#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#else
#  ifdef _WIN32
#    include <getopt.h>
#  else
#    include <stdlib.h>
#  endif
#endif

#include <qapplication.h>
#include <qtextcodec.h>
#include <qtooltip.h>

/*
#ifdef _WIN32
#  include <windows.h>
#endif
*/

#include "FWBSettings.h"
#include "RCS.h"
#include "FWWindow.h"
#include "ObjectManipulator.h"
#include "ObjectEditor.h"
#include "FWObjectClipboard.h"
#include "FWBTree.h"
#include "StartWizard.h"
#include "findDialog.h"
#include "listOfLibraries.h"

#include "fwbuilder/FWObject.h"
#include "fwbuilder/Tools.h"
#include "fwbuilder/dns.h"
//#include "fwbuilder/crypto.h"
#include "fwbuilder/XMLTools.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/FWException.h"

#ifndef _WIN32
#  include <sys/wait.h>
#  include <unistd.h>
#  include <termios.h>
#  include <time.h>
#  include <sys/select.h>
#  include <utmp.h>
#else
#  include <direct.h>
#endif

#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

#ifdef HAVE_PTY_H
#include <pty.h>
#endif

#ifdef HAVE_LIBUTIL_H
#include <libutil.h>
#endif

#ifdef HAVE_UTIL_H
#include <util.h>
#endif

//#ifdef Q_OS_MACX
//#  include <util.h>
//#endif

#include <errno.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "../common/init.cpp"

#ifdef ELC
extern bool init2(const std::string &a1,
                  const std::string &moduleName,
                  const std::string &rp,
                  const std::string &rp1,
                  bool f1, bool f2);
#endif

using namespace libfwbuilder;
using namespace std;

static QString    filename;
static QString    objid;

QApplication      *app        = NULL;
FWWindow          *mw         = NULL;
ObjectManipulator *om         = NULL;
ObjectEditor      *oe         = NULL;
QTextEdit         *oi         = NULL;
FWBSettings       *st         = NULL;
findDialog        *fd         = NULL;
int                fwbdebug   = 0;
int                safemode   = 0;
bool               registered = false;
    
listOfLibraries  *addOnLibs;

#ifndef _WIN32

#ifndef HAVE_CFMAKERAW
static inline void cfmakeraw(struct termios *termios_p)
{
    termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
    termios_p->c_oflag &= ~OPOST;
    termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
    termios_p->c_cflag &= ~(CSIZE|PARENB);
    termios_p->c_cflag |= CS8;
}
#endif

#ifndef HAVE_FORKPTY

#include <unistd.h> 
#include <stdlib.h> 
#include <sys/ioctl.h> 
#include <sys/stream.h> 
#include <sys/stropts.h> 

/* fork_pty() remplacement for Solaris. 
* ignore the last two arguments 
* for the moment 
*/ 
int forkpty (int *amaster, char *name, void *unused1, void *unused2)
{
    int master, slave; 
    char *slave_name; 
    pid_t pid; 

    master = open("/dev/ptmx", O_RDWR); 
    if (master < 0) 
        return -1; 

    if (grantpt (master) < 0) 
    { 
        close (master); 
        return -1; 
    } 

    if (unlockpt (master) < 0) 
    { 
        close (master); 
        return -1; 
    } 

    slave_name = ptsname (master); 
    if (slave_name == NULL) 
    { 
        close (master); 
        return -1; 
    } 

    slave = open (slave_name, O_RDWR); 
    if (slave < 0) 
    { 
        close (master); 
        return -1; 
    } 

    if (ioctl (slave, I_PUSH, "ptem") < 0 
        || ioctl (slave, I_PUSH, "ldterm") < 0) 
    { 
        close (slave); 
        close (master); 
        return -1; 
    } 

    if (amaster) 
        *amaster = master; 

    if (name) 
        strcpy (name, slave_name); 

    pid = fork (); 
    switch (pid) 
    { 
    case -1: /* Error */ 
        return -1; 
    case 0: /* Child */ 
        close (master); 
        dup2 (slave, STDIN_FILENO); 
        dup2 (slave, STDOUT_FILENO); 
        dup2 (slave, STDERR_FILENO); 
        return 0; 
    default: /* Parent */ 
        close (slave); 
        return pid; 
    } 

    return -1; 
}

#endif

static struct termios save_termios;
static int            ttysavefd = -1;
static enum { RESET, RAW, CBREAK } ttystate = RESET;

int tty_raw(int fd)
{
    struct termios buf;

    if (tcgetattr(fd, &save_termios) < 0)
    {
        qDebug("Can not switch terminal to raw mode, tcgetattr error '%s'",strerror(errno));
        exit(1);
    }

    buf = save_termios;

    cfmakeraw(&buf);

// this used to use TCSAFLUSH, but that caused stall which I did not
// completely understand. Apparently there was some data in the output
// buffer at the moment when we try to switch tty to raw mode, but I
// could not figure out where this data comes from and why it could
// not be written to the tty. Anyway, this caused semi-random stalls
// in the installer because whenever it called fwbuilder -X, the child
// process would block in this place and stall installer. I had to
// switch to TCSANOW to fix.

    if (tcsetattr(fd, TCSANOW, &buf) < 0)
    {
        qDebug("Can not switch terminal to raw mode, tcsetattr error '%s'",strerror(errno));
        exit(1);
    }

    ttystate = RAW;
    ttysavefd = fd;
    return 0;
}

int echo_off(int fd)
{
    struct stat statbuf;
    if (fstat(fd,&statbuf)!=0) return 0;

    struct termios stermios;
    if (tcgetattr(fd, &stermios)<0)
    {
        qDebug("Can not turn terminal echo off, tcgetattr error '%s'",strerror(errno));
        exit(1);
    }

    stermios.c_lflag &= ~(ECHO | ECHOE | ECHOK | ECHONL);
    stermios.c_oflag &= ~(ONLCR);

    if (tcsetattr(fd, TCSANOW, &stermios)<0)
    {
        qDebug("Can not turn terminal echo off, tcsetattr error '%s'",strerror(errno));
        exit(1);
    }

    return 0;
}

ssize_t writen(int fd,const void *vptr, size_t n)
{
    size_t  nleft;
    ssize_t nwritten;
    const char *ptr;

    ptr = (const char*)(vptr);
    nleft = n;
    if (fwbdebug) qDebug("need to write %d bytes",nleft);
    while (nleft > 0)
    {
        if ( (nwritten = write(fd,ptr,nleft )) <= 0)
            return nwritten;

        if (fwbdebug) qDebug("%d bytes written",nwritten);

        nleft -= nwritten;
        ptr += nwritten;
    }
    return n;
}


#ifndef strndup
char* strndup(const char* s,int n)
{
    char *tbuf = (char*)malloc(n);
    if (tbuf) memcpy(tbuf,s,n);
    return tbuf;
}
#endif
#endif

void usage()
{
    cerr << "Usage: fwbuilder [-?hv] [filename]\n";
}



int main( int argc, char ** argv )
{
    bool        ssh_wrapper=false;
    bool        scp_wrapper=false;
    const char *arg[64];
    int         i, j;

    filename="";
    objid="";
    fwbdebug=0;
    safemode=0;

    bool desktopaware = true;

#if defined(Q_OS_MACX)
    desktopaware = false;
#endif

/*
 * I am using njamd a lot, but gtkmm and probably some other libs
 * generate trap in their global static initialization code. Therefore
 * I need to start the program with env. var. NJAMD_PROT set to "none"
 * and then reset it to something useful here.
 */
#ifdef HAVE_SETENV
    setenv("NJAMD_PROT","strict",1);
#else
#  ifdef HAVE_PUTENV
    putenv("NJAMD_PROT=strict");
#  endif
#endif


#ifndef _WIN32

    i=1;
    j=1;

    for ( ; argv[i]!=NULL; i++)
    {
        if (strncmp(argv[i], "-X", 2)==0) { ssh_wrapper=true; continue; }
        else
            if (strncmp(argv[i], "-Y", 2)==0) { scp_wrapper=true; continue; }
            else
                if (strncmp(argv[i], "-d", 2)==0) { fwbdebug++; continue; }
                else
                    arg[j]=strdup(argv[i]);
        j++;
    }
    arg[j]=NULL;

    if (ssh_wrapper || scp_wrapper)
    {

/* need to create and initialize settings to be able to use ssh/scp
 * configuration in the wrapper
 *
 * Note:
 *
 * We need to keep installation data and program settings in registry
 * folders with different names. QSettings always looks into Current
 * User registry first, so if the folders have the same names, then we
 * store evaluation key in Current User, while it is better to put it
 * in the Local Machine branch.
 *
 * So, installation data goes to HKLM Software\NetCitadel\FirewallBuilder
 * and settings to HKCU Software\NetCitadel\FirewallBuilder2
 * 
 * fwbuilder-lm determines folder path for the license file by
 * reading key Install_Dir under HKLM Software\NetCitadel\FirewallBuilder
 */
        st = new FWBSettings();

/* initialize preferences */
        st->init();

        QString sshcmd=st->getSSHPath();
        QString scpcmd=st->getSCPPath();

        if (sshcmd.isEmpty()) sshcmd="ssh";
        if (scpcmd.isEmpty()) scpcmd="scp";
        
        if (ssh_wrapper)  arg[0]=strdup( sshcmd.latin1() );
        else              arg[0]=strdup( scpcmd.latin1() );

        if (fwbdebug)
            qDebug("cmd: %s",arg[0]);


/* forks ssh with a pty and proxies its communication on stdin/stdout
 * to avoid having to deal with pty. This is only needed on Unix.
 */
        pid_t pid;
        int   mfd;
        char  slave_name[64];
//        char  *pgm;

        pid=forkpty(&mfd,slave_name,NULL,NULL);
        if (pid<0)
        {
            qDebug("Fork failed: %s", strerror(errno));
            exit(1);
        }
        if (pid==0)
        { // child

// turn echo off on stdin
            echo_off(STDIN_FILENO);

            tty_raw(STDIN_FILENO);

            signal(1,SIG_IGN);

            execvp(arg[0],(char* const*)arg);

// if we've got here there was an error
            qDebug("Exec error: %s %s",strerror(errno),arg[0]);
            exit(1);
        }

        tty_raw(mfd);

        fd_set  rfds;
        struct timeval tv;
        int     retval;

        #define BUFFSIZE 512

#ifdef DEBUG_INSTALLER
        int debug_file = open("installer.dbg",O_CREAT|O_WRONLY);
#endif
        char    ibuf[BUFFSIZE];
        char    obuf[BUFFSIZE];
        bool    endOfStream = false;

        while (true)
        {
            tv.tv_sec  = 2;
            tv.tv_usec = 0;

            FD_ZERO(&rfds);
            FD_SET(mfd,            &rfds);
            if (!endOfStream)  FD_SET(STDIN_FILENO  , &rfds);

            retval = select( max(STDIN_FILENO,mfd)+1 , &rfds, NULL, NULL, &tv);
            if (retval==0)  // timeout
            {
                if (fwbdebug) qDebug("timeout");
                if (endOfStream)
                {
                    if (fwbdebug) qDebug("Closing mfd");
                    close(mfd);
                    break;
                }
            }
            if (retval)
            {
                if (FD_ISSET(STDIN_FILENO, &rfds))
                {
                    int n=read(0,ibuf,sizeof(ibuf));
                    if (fwbdebug) qDebug("Read %d bytes from stdin",n);
                    if (n<0)
                    {
                        if (fwbdebug) qDebug("Error on stdin");
                        break;
                    }
                    if (n==0)
                    {
// eof on stdin
                        if (fwbdebug) qDebug("EOF on stdin");
                        endOfStream = true;
                    } else
                    {
/* This is a workaround for the problem that exists with utf-8 encoded data
 * passed to stdin of a QProcess (and vice versa, see
 * http://lists.trolltech.com/qt-interest/2004-10/thread00024-0.html
 */
                        char tbuf[BUFFSIZE];
                        char *tptr1 = ibuf;
                        char *tptr2 = tbuf;
                        int no = n;
                        int nr = 0;
                        while (no)
                        {
                            if (*tptr1 != '\0')
                            {
                                *tptr2++ = *tptr1;
                                nr++;
                            }
                            tptr1++;
                            no--;
                        }

                        int r=writen(mfd,tbuf,nr);
                        if (fwbdebug) qDebug("Wrote %d bytes to mfd",r);
#ifdef DEBUG_INSTALLER
                        write(debug_file,tbuf,nr);
#endif
                    }
                }
                if (FD_ISSET(mfd, &rfds))
                {
                    int n;
                    obuf[0]='\0';
                    n=read(mfd,obuf,sizeof(obuf));
                    if (fwbdebug) qDebug("Read %d bytes from mfd",n);
                    if (n<=0)
                    {
/* eof on mfd - this means ssh process has died */
                        if (fwbdebug) qDebug("EOF on mfd");
                        break;
                    } else
                    {
                        obuf[n]='\0';
                        int r=writen(1,obuf,n);
                        if (fwbdebug) qDebug("Wrote %d bytes to stdout",r);
                    }
                }
            }
        }

#ifdef DEBUG_INSTALLER
        close(debug_file);
#endif
        int status;
        waitpid(pid, &status, 0);
        if (WIFEXITED(status)) exit(WEXITSTATUS(status));
        exit(0);
    }
#endif



    int c;
    while ((c = getopt (argc , argv , "hvf:o:p:dxs")) != EOF )
	switch (c) {
	case 'h':
	    usage();
	    exit(0);

	case 'f':
	    filename=optarg;
	    break;

	case 'o':
	    objid=optarg;
	    break;

        case 'd':
            fwbdebug++;
            break;

	case 'v':
	    cout << VERSION << endl;
	    exit(0);

        case 'x':
            desktopaware=!desktopaware;
            break;

        case 's':
            safemode++;
            break;
	}

    if ( (argc-1)==optind)
        filename = strdup( argv[optind++] );


    try
    {

        if (fwbdebug) qDebug("initializing ...");
    
/* need to initialize in order to be able to use FWBSettings */
        init(argv);

        if (fwbdebug) qDebug("creating app ...");
    
        QApplication::setDesktopSettingsAware(desktopaware);
        app = new QApplication( argc, argv );

        if (fwbdebug) qDebug("reading settings ...");
    
        st = new FWBSettings();

/* initialize preferences */
        st->init();

        if (fwbdebug) qDebug("done");
    
#ifdef ELC
        registered=init2(argv0, "Firewall Builder","fwb_gui","FirewallBuilder",true,true);
#endif

        if (fwbdebug) qDebug("reading resources ...");
        new Resources(respath+FS_SEPARATOR+"resources.xml");
        if (fwbdebug) qDebug("done");

#if 0
        QApplication::setDesktopSettingsAware(desktopaware);
        app = new QApplication( argc, argv );
#endif

        vector<std::string> platforms=Resources::getListOfPlatforms();
        if (platforms.empty() || ( platforms.size()==1 && platforms.front()=="unknown" ))
        {
            qDebug("Failed to load list of supported platforms");
            exit(1);
        }

        if (fwbdebug) qDebug("creating widgets ...");

        new FWBTree();
        new FWObjectDatabase();
        new FWObjectClipboard();

//    cerr << "*** Current locale: " << QTextCodec::locale() << endl;

//    app = new QApplication( argc, argv );

        if (fwbdebug) qDebug("loading translation for the current locale ...");

/* this is an ugly hack to work around broken qt_es.qm translation
 * file shipped with QT 3.3 for Fedora-C4 and possibly other
 * distros. Only Spanish translation seems to be affected.  Trolltech
 * support req. N80793 (although I never got any suggestions for the
 * workaround, nor did they clearly admit the problem despite my
 * sending a simple example program to them to illustrate the
 * bug). They did not seem to care since Spanish translation file that
 * support guy had on his computer was fine, but he admitted it was
 * different from the one shipped with QT 3.3.4 for Fedora-C4 and for
 * Mac OS X. All I got from them was a promise to look into packaging
 * process.
 */
        if (QString(QTextCodec::locale()).find("es")!=0)
        {
            QTranslator qt1(0);
            qt1.load( QString( "qt_" ) + QTextCodec::locale(), QTTRANSLATIONSDIR );
            app->installTranslator( &qt1 );
        
            QTranslator qt2(0);
            qt2.load( QString( "qt_" ) + QTextCodec::locale(),  localepath.c_str() );
            app->installTranslator( &qt2 );
        }

        QTranslator translator(0);
        translator.load(QString("fwbuilder_")+QTextCodec::locale(),localepath.c_str());
        app->installTranslator (&translator);

/* must build list of available libraries _after_ creation of
 * FWObjectDatabase and settings */

        if (fwbdebug) qDebug("loading libraries ...");

        addOnLibs = new listOfLibraries();

        mw  = new FWWindow();
        oe  = new ObjectEditor(mw);
        oe->open(FWObjectDatabase::db);
        oe->hide();
        fd  = new findDialog(mw);
        fd->hide();

        if (fwbdebug) qDebug("loading data file ...");

        if (safemode)  mw->load(NULL);
        else
        {
            if (filename!="")
            {
                try
                {
                    RCS *rcs=new RCS(filename);
                    rcs->co();
                    mw->load(NULL,rcs);
                } catch (FWException &ex)
                {
                    qDebug("Exception: %s",ex.toString().c_str());
                    mw->load(NULL);
                }
            } else
            {
                int sa = st->getStartupAction();
                switch (sa) 
                {
                case 0:   // load standard objects
                {
                    mw->load(NULL); // load standard objects
                    break;
                }
                case 1:   // load last edited file
                {
                    filename=st->getLastEdited();
                    if (filename!="")
                    {
                        try
                        {
                            RCS *rcs=new RCS(filename);
                            rcs->co();
                            mw->load(NULL,rcs);
                        } catch (FWException &ex)
                        {
                            qDebug("Exception: %s",ex.toString().c_str());
                            mw->load(NULL);
                        }
                    } else
                        mw->load(NULL);
                    break;
                }
                default:  // ask user
                {
/* no file specified on command line and autoload option is off */
                    StartWizard sw;
                    if ( sw.exec() != QDialog::Accepted )
                        mw->load(NULL); // load standard objects
                }
                }
            }
        }

        mw->showFirewalls();

        if ( st->getStartupAction()==1 && safemode==0 )
        {
            QString id = st->getStr("UI/visibleFirewall");
            FWObject *o=NULL;
            if ( !id.isEmpty() ) o=mw->db()->getById(id.latin1(),true);
            if (o)
            {
                if (fwbdebug)
                    qDebug("open firewall %s",o->getName().c_str());
                mw->showFirewall(o);
            }

            id = st->getStr("UI/visibleObject");
            o=NULL;
            if ( !id.isEmpty() ) o=mw->db()->getById(id.latin1(),true);
            if (o)
            {
                if (fwbdebug)
                    qDebug("open object %s",o->getName().c_str());
                om->openObject(o);
            }
        }

        QToolTip::setWakeUpDelay( st->getTooltipDelay()*1000 );

        mw->show();

        app->connect( app, SIGNAL( lastWindowClosed() ), app, SLOT( quit() ) );
        app->exec();

        oe->hide();
        fd->hide();
        mw->hide();  // must do this before settings object is destroyed

        addOnLibs->save();  // ditto

        if ( st->getStartupAction()==1 )
        {
/* save the state of the GUI (opened firewall, opened object tree page, etc */
            FWObject *o=mw->getVisibleFirewall();
            if (fwbdebug)
                qDebug("Main: closing. VisibleFirewall = %p",o);

            if (o) st->setStr("UI/visibleFirewall", o->getId().c_str() );

            o=om->getOpened();
            if (o) st->setStr("UI/visibleObject",   o->getId().c_str() );
        }


        st->save();
        delete st;
    }
    catch (FWException &ex)
    {
        qDebug("Exception: %s",ex.toString().c_str());
    }
}
