/* 

                          Firewall Builder

                 Copyright (C) 2004 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: LibExportDialog.cpp,v 1.15 2004/12/05 03:11:28 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

#include "FWWindow.h"
#include "LibExportDialog.h"
#include "FWBSettings.h"
#include "longTextDialog.h"

#include "fwbuilder/FWObject.h"
#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/Library.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/Rule.h"
#include "fwbuilder/RuleElement.h"
#include "fwbuilder/RuleSet.h"
#include "fwbuilder/Policy.h"
#include "fwbuilder/InterfacePolicy.h"
#include "fwbuilder/NAT.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Interface.h"

#include <qlistbox.h>
#include <qcheckbox.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qcursor.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#ifndef _WIN32
#  include <unistd.h>     // for access(2)
#endif

#include <iostream>
#include <algorithm>

using namespace std;
using namespace libfwbuilder;

LibExportDialog::LibExportDialog( const QString& dirName, const QString& filter,
                              QWidget* parent, const char* name, bool modal )
    : QFileDialog( dirName, filter, parent, name, modal)
{
    init();
}

LibExportDialog::LibExportDialog( QWidget* parent, const char* name, bool modal )
    : QFileDialog(parent, name, modal)
{
    init();
}

void LibExportDialog::init()
{
    addFilter( "Firewall Builder 2 files (*.fwl)");
    resize( QSize(500, 450) );

    setMode(QFileDialog::AnyFile);

    setDir( st->getWDir() );

    le=new LibExport_q(this);
    addWidgets(0, le, 0);

    le->exportRO->setChecked(true);

    le->libs->clear();
    list<FWObject*> ll = mw->db()->getRoot()->getByType( Library::TYPENAME );
    int n=0;
    string libicn;

    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++,n++)
    {
        if (libicn.empty())
            libicn=Resources::global_res->getObjResourceStr(*i,"icon-tree").c_str();

        mapOfLibs[n]= (*i);
        le->libs->insertItem(
            QPixmap::fromMimeSource( libicn.c_str() ),
            QString::fromUtf8((*i)->getName().c_str()),
            n);
    }

    connect(le->libs, SIGNAL(selectionChanged(QListBoxItem*)),
            this,     SLOT(libSelected(QListBoxItem*)));
}

void LibExportDialog::libSelected(QListBoxItem *itm)
{
    int selitm=le->libs->index(itm);
    if (selitm<0) return;
    FWObject *selLib = mapOfLibs[selitm];
    setSelection( QString(selLib->getName().c_str())+".fwl");
}

void LibExportDialog::findExternalRefs(FWObject *lib,
                                       FWObject *root,
                                       list<FWReference*> &extRefs)
{
    FWReference *ref=FWReference::cast(root);
    if (ref!=NULL)
    {
        FWObject *plib = ref->getPointer()->getLibrary();
        if ( plib->getId()!=STANDARD_LIB &&
             plib->getId()!=DELETED_LIB  &&
             plib!=lib )
            extRefs.push_back(ref);
        return;
    } else
    {
        for (FWObject::iterator i=root->begin(); i!=root->end(); i++)
            findExternalRefs(lib, *i, extRefs);

    }
}

/*
 * originally I allowed the user to export multiple libraries into a
 * file, but then changed it. Now only a single library can be
 * exported to a file
 */
void LibExportDialog::accept()
{
    QString fname=selectedFile();
    if (fname.findRev(".fwl",-1)==-1) fname=fname+".fwl";

    list<FWObject*>              selectedLibs;
    map<int,FWObject*>::iterator i;
    for (i=mapOfLibs.begin(); i!=mapOfLibs.end(); i++)
        if (le->libs->isSelected(i->first)) selectedLibs.push_back(i->second);

    if (selectedLibs.size()==0)
    {
        QMessageBox::critical(
            this,"Firewall Builder", 
            tr("Please select a library you want to export."),
            "&Continue", QString::null,QString::null,
            0, 1 );

        return;
    }

/* VERY IMPORTANT: External library file must be self-contained,
 * otherwise it can not be exported.
 *
 * check if selected libraries have references to objects in other
 * libraries (not exported to the same file). Exporting such libraries
 * pulls in other ones because of these references. This is confusing
 * because it means we end up with multiple copies of such objects (in
 * exported library file and in user's data file). When user imports
 * this library and opens their file, it is impossible to say which
 * library an object belongs to.
 *
 * This is prohibited. We check if exported set of libraries has such
 * references and refuse to export it. The user is supposed to clean
 * it up by either moving objects into the library they are trying to
 * export, or by rearranging objects. The only exception for this is
 * library "Standard", which is assumed to be always present so we can
 * have references to objects in it.
 */

    QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );

    list<FWReference*> externalRefs;
    for (list<FWObject*>::iterator i=selectedLibs.begin(); i!=selectedLibs.end(); ++i)
        findExternalRefs( *i, *i, externalRefs);

    QApplication::restoreOverrideCursor();

    if (fwbdebug) qDebug("LibExportDialog::accept  externalRefs.size()=%d",
                         externalRefs.size() );

/*
 * if externalRefs.size()!=0, then there were some references pointing
 * outside of the libraries we export. Some of these references may
 * point at other libraries we export, lets find these.
 */
    list<FWReference*> externalRefs2;
    for (list<FWReference*>::iterator i=externalRefs.begin(); i!=externalRefs.end(); ++i)
    {
        FWObject *tgt    = (*i)->getPointer();
        FWObject *tgtlib = tgt->getLibrary();

        if (std::find(selectedLibs.begin(),selectedLibs.end(),tgtlib)!=selectedLibs.end()) continue;
        externalRefs2.push_back(*i);
    }

    if (externalRefs2.size()!=0)
    {
        QString objlist = "";
        QString s       = "";

        for (list<FWReference*>::iterator i=externalRefs2.begin(); i!=externalRefs2.end(); ++i)
        {
            FWReference *robj   = *i;
            FWObject *selLib = robj->getLibrary();
            FWObject *pp     = robj->getParent();
            FWObject *tgt    = robj->getPointer();
            FWObject *tgtlib = tgt->getLibrary();

            if (fwbdebug)
            {
                qDebug("LibExportDialog::accept  tgt: %s pp_type: %s lib: %s",
                       tgt->getName().c_str(),
                       pp->getTypeName().c_str(),
                       tgtlib->getName().c_str());
            }

            if (std::find(selectedLibs.begin(),selectedLibs.end(),tgtlib)!=selectedLibs.end()) continue;

            if (RuleElement::cast(pp)!=NULL)
            {
                FWObject *fw       = pp;
                FWObject *rule     = pp;
                FWObject *ruleset  = pp;
                FWObject *iface    = pp;

                while (rule!=NULL && Rule::cast(rule)==NULL)
                    rule=rule->getParent();
                while (ruleset!=NULL && RuleSet::cast(ruleset)==NULL) 
                    ruleset=ruleset->getParent();
                while (iface!=NULL && Interface::cast(iface)==NULL)
                    iface=iface->getParent();
                while (fw!=NULL && Firewall::cast(fw)==NULL)
                    fw=fw->getParent();

                QString rsname;
                if (Policy::cast(ruleset)!=NULL)
                {
                    s = 
   QObject::tr("Library %1: Firewall '%2' (global policy rule #%3) uses object '%4' from library '%5'")
                        .arg(selLib->getName().c_str())
                        .arg(fw->getName().c_str())
                        .arg(Rule::cast(rule)->getPosition())
                        .arg(tgt->getName().c_str())
                        .arg(tgtlib->getName().c_str());
                }
                if (InterfacePolicy::cast(ruleset)!=NULL)
                {
   QObject::tr("Library %1: Firewall '%2' (interface %3 policy rule #%4) uses object '%5' from library '%6'")
                        .arg(selLib->getName().c_str())
                        .arg(fw->getName().c_str())
                        .arg(iface->getName().c_str())
                        .arg(Rule::cast(rule)->getPosition())
                        .arg(tgt->getName().c_str())
                        .arg(tgtlib->getName().c_str());
                }
                if (NAT::cast(ruleset)!=NULL)
                {
                    s = 
   QObject::tr("Library %1: Firewall '%2' (NAT rule #%3) uses object '%4' from library '%5'")
                        .arg(selLib->getName().c_str())
                        .arg(fw->getName().c_str())
                        .arg(Rule::cast(rule)->getPosition())
                        .arg(tgt->getName().c_str())
                        .arg(tgtlib->getName().c_str());
                }
            } else
            {
                    s = 
   QObject::tr("Library %1: Group '%2' uses object '%3' from library '%4'")
                        .arg(selLib->getName().c_str())
                        .arg(pp->getName().c_str())
                        .arg(tgt->getName().c_str())
                        .arg(tgtlib->getName().c_str());
            }
            s = s + "\n";

            if (fwbdebug) qDebug(s.ascii());

            objlist = objlist + s;
        }

        longTextDialog ltd( this,

            tr("A library that you are trying to export contains references\n"
               "to objects in the other libraries and can not be exported.\n"
               "The following objects need to be moved outside of it or\n"
               "objects that they refer to moved in it:"),
                            objlist );

        ltd.exec();
        return;
    }

    QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );

    FWObjectDatabase *ndb = mw->db()->exportSubtree( selectedLibs );

    QApplication::restoreOverrideCursor();

    if (le->exportRO->isChecked())
    {
        for (list<FWObject*>::iterator i=selectedLibs.begin(); i!=selectedLibs.end(); ++i)
        {
            FWObject *nlib= ndb->getById( (*i)->getId() );
            if (nlib && nlib->getId()!=DELETED_LIB)
                nlib->setReadOnly( true );
        }
    }

    try
    {
        ndb->saveFile( fname.latin1() );
    }
    catch (FWException &ex)
    {
/* error saving the file. Since XMLTools does not return any useful
 * error message in the exception, let's check for obvious problems here
 */
        QString err;
        if (access( fname.latin1(), W_OK)!=0 && errno==EACCES)
            err=QObject::tr("File is read-only");

        QMessageBox::warning(
            this,"Firewall Builder", 
            QObject::tr("Error saving file %1: %2")
            .arg(fname).arg(err),
            "&Continue", QString::null, QString::null,
            0, 1 );
    }

    QFileDialog::accept();
}

