/* 

                          Firewall Builder

                 Copyright (C) 2004 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: instDialog.cpp,v 1.108 2007/07/13 05:32:55 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "../../config.h"
#include "global.h"
#include "utils.h"
#include "utils_no_qt.h"

#include "instDialog.h"
#include "FWBSettings.h"
#include "SSHUnx.h"
#include "SSHPIX.h"
#include "SSHIOS.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"
#include "InstallFirewallViewItem.h"
#include "instOptionsDialog.h"
#include "instBatchOptionsDialog.h"

#include <qcheckbox.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qtimer.h>
#include <qfiledialog.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qprogressbar.h>
#include <qprocess.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qfile.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qspinbox.h>
#include <qgroupbox.h>
#include <qcolor.h>
#include <qtable.h>
#include <qtextcodec.h>
#include <qfileinfo.h>

#include "fwbuilder/Resources.h"
#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/XMLTools.h"
#include "fwbuilder/Interface.h"
#include "fwbuilder/Management.h"

#ifndef _WIN32
#  include <unistd.h>     // for access(2) and getdomainname
#endif

#include <errno.h>
#include <iostream>

using namespace std;
using namespace libfwbuilder;




instDialog::instDialog(QWidget* p, BatchOperation op) : instDialog_q(p)
{
    if (fwbdebug) qDebug("instDialog::instDialog   WFlagse=%0x",getWFlags());

    clearWFlags( Qt::WType_Dialog | Qt::WStyle_SysMenu  );
    
    session=NULL;
    dlg=NULL;
    reqFirewalls.clear();

    currentLog = NULL;
    currentSaveButton = NULL;    
    currentStopButton = NULL;
    showSelectedFlag=false;
    pendingLogLine = "";

    proc.setCommunication( QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr);

    connect(&proc, SIGNAL(readyReadStdout()), this, SLOT(readFromStdout()) );
    connect(&proc, SIGNAL(processExited()),   this, SLOT(processExited()) );
    
    listView4->setSorting(-1);
     
    setHelpEnabled( page(0), false );
    setHelpEnabled( page(1), false );
    setHelpEnabled( page(2), false );
    
    setFinishEnabled(page(pageCount()-1),true);
    lastPage=-1;
    
    findFirewalls();
    if (firewalls.size()==0)
    {
        setTitle( page(pageCount()-1), tr("There is no firewalls to process.") );
        for (int i=0;i<pageCount()-1;i++)
        {
            setAppropriate(page(i),false);
        }
        showPage(page(pageCount()-1));
        return;
    }
    if (firewalls.size()==1) batchInstall->setEnabled(false);
    // setup wizard appropriate pages
    operation=op;
    switch(op)
    {
        case BATCH_COMPILE:
            { // only compilation's requested
                selectInfoLabel->setText(tr("<p align=\"center\"><b><font size=\"+2\">Select firewalls for compilation.</font></b></p>"));
                batchInstFlagFrame->hide();
                setAppropriate(page(2),false);
                
                
                selectTable->hideColumn(1);
                break;
            }
        case BATCH_INSTALL:
            { // full cycle's requested
                break;
            }
        default :
            {
                setTitle( page(pageCount()-1), tr("Unknown operation.") );
                for (int i=0;i<pageCount()-1;i++)
                {
                    setAppropriate(page(i),false);
                }
                showPage(page(pageCount()-1));
            }
    }
    //hide all details
    bool fs=st->readBoolEntry("/FirewallBuilder2/Installer/ShowDetails"    );
    if (fs)
        detailMCframe->show();
    else
        detailMCframe->hide();

    togleDetailMC();
}

instDialog::~instDialog()
{
    if (dlg) 
    {
        delete dlg;
        dlg=NULL;
    }
}

void instDialog::togleDetailMC()
{
    if (detailMCframe->isVisible())
    {
        detailsButton->setText(tr("Show details"));
        detailMCframe->hide();
        st->writeEntry("/FirewallBuilder2/Installer/ShowDetails",false);
    }
    else
    {
        detailsButton->setText(tr("Hide details"));
        detailMCframe->show();
        st->writeEntry("/FirewallBuilder2/Installer/ShowDetails",true);
    }
}

void instDialog::prepareInstConf(Firewall *fw)
{
    if (fwbdebug) qDebug("instDialog::prepareInstConf");
    
}

void instDialog::prepareInstallerOptions()
{
    if (fwbdebug) qDebug("instDialog::prepareInstallerOptions");
    ready=false;
    activationCommandDone=false;
    FWOptions  *fwopt = cnf.fwobj->getOptionsObject();

    fwb_prompt="--**--**--";
    session = NULL;

    cnf.batchInstall = batchInstall->isChecked();

    cnf.incremental    = st->readBoolEntry("/FirewallBuilder2/Installer/incr"    );
    cnf.save_diff      = st->readBoolEntry("/FirewallBuilder2/Installer/savediff");
    cnf.saveStandby    = st->readBoolEntry("/FirewallBuilder2/Installer/saveStandby");
    cnf.dry_run        = st->readBoolEntry("/FirewallBuilder2/Installer/dryrun"  );
    cnf.quiet          = st->readBoolEntry("/FirewallBuilder2/Installer/quiet"   );
    cnf.verbose        = st->readBoolEntry("/FirewallBuilder2/Installer/verbose" );
    cnf.stripComments  = st->readBoolEntry("/FirewallBuilder2/Installer/stripComments" );
    cnf.compressScript = st->readBoolEntry("/FirewallBuilder2/Installer/compressScript" );
    cnf.copyFWB        = st->readBoolEntry("/FirewallBuilder2/Installer/copyFWB" );
    cnf.testRun        = st->readBoolEntry("/FirewallBuilder2/Installer/testRun" );
    cnf.rollback       = st->readBoolEntry("/FirewallBuilder2/Installer/rollback" );
    cnf.rollbackTime   = st->readNumEntry("/FirewallBuilder2/Installer/rollbackTime");
    cnf.cancelRollbackIfSuccess =
        st->readBoolEntry("/FirewallBuilder2/Installer/canceRollbackIfSuccess" );
/* TODO: set cnf.pgm to ssh path here */

    QString platform = cnf.fwobj->getStr("platform").c_str();
    
    //bool f = dlg->testRun->isChecked();

    //QSize pix_options_frame_size = dlg->PIXgroupBox->sizeHint();

    cnf.rollbackTimeUnit=
        Resources::getTargetOptionStr(cnf.fwobj->getStr("host_OS"),
                                      "activation/timeout_units").c_str();


/* we initialize these in FWBSettings constructor on Unix, but do not
 * do it on Windows since there is no standard ssh package there. User
 * is supposed to fill these in in the Preferences dialog, otherwise
 * they can't use installer
 */

    ssh = st->getSSHPath();

/* as of v2.0.3, build 437, incremental install actually installs only
 * ACL and nat commands on PIX. It does not use fwb_pix_diff so there
 *  is no need to disable it anymore
 *
    if ( access(cnf.diff_pgm.latin1(), F_OK|X_OK)!=0 )
    {
        cerr << "could not access " << cnf.diff_pgm << endl;

        incr->setChecked(false);
        incr->setEnabled(false);
        saveDiff->setChecked(false);
        saveDiff->setEnabled(false);
    }
*/
        
    try
    {
        if (cnf.fwobj!=NULL && ! cnf.fwbfile.isEmpty())
        {
            QString aaddr = fwopt->getStr("altAddress").c_str();
            if (!aaddr.isEmpty())
                cnf.maddr = aaddr;
            else
                cnf.maddr =
                    cnf.fwobj->getManagementAddress().toString().c_str();
        }

        setReady(true);

    } catch(FWException &ex)
    {
        setReady(false);
        //showPage( page(1) );
        currentLog->append( ex.toString().c_str() );
    } catch (std::string s) {
        setReady(false);
        //showPage( page(1) );
        currentLog->append( s.c_str() );
    } catch (std::exception ex) {
        setReady(false);
        //showPage( page(1) );
        currentLog->append( ex.what() );
    } catch (...) {
        setReady(false);
        //showPage( page(1) );
        currentLog->append( QObject::tr("Unsupported exception") );
    }

}


void instDialog::storeInstallerOptions()
{
    st->writeEntry("/FirewallBuilder2/Installer/incr",    cnf.incremental);
    st->writeEntry("/FirewallBuilder2/Installer/savediff",cnf.save_diff  );
    st->writeEntry("/FirewallBuilder2/Installer/saveStandby",cnf.saveStandby);
    st->writeEntry("/FirewallBuilder2/Installer/dryrun"  ,cnf.dry_run    );
    st->writeEntry("/FirewallBuilder2/Installer/quiet",   cnf.quiet      );
    st->writeEntry("/FirewallBuilder2/Installer/verbose", cnf.verbose    );
    st->writeEntry("/FirewallBuilder2/Installer/stripComments", cnf.stripComments);
    st->writeEntry("/FirewallBuilder2/Installer/compressScript", cnf.compressScript);
    st->writeEntry("/FirewallBuilder2/Installer/copyFWB",  cnf.copyFWB);
    st->writeEntry("/FirewallBuilder2/Installer/testRun",  cnf.testRun);
    st->writeEntry("/FirewallBuilder2/Installer/rollback", cnf.rollback);
    st->writeEntry("/FirewallBuilder2/Installer/rollbackTime", cnf.rollbackTime);
    st->writeEntry("/FirewallBuilder2/Installer/canceRollbackIfSuccess", 
      cnf.cancelRollbackIfSuccess);
}

void instDialog::append(const QString &s)
{
//    currentLog->moveCursor( QTextEdit::MoveEnd , false );
//    currentLog->insert( s );

    currentLog->append( s );
}

void instDialog::appendRich(const QString &s)
{
    if (currentLog) currentLog->append(s);
}


void instDialog::summary()
{
    appendRich( "<hr>" + QObject::tr("<b>Summary:</b>") );
    if (!customScriptFlag)
    {
        appendRich( QObject::tr("* firewall name : %1")
                .arg(cnf.fwobj->getName().c_str()) );
        appendRich( QObject::tr("* user name : %1")
                .arg(cnf.user) );
        appendRich( QObject::tr("* management address : %1")
                .arg(cnf.maddr) );
        appendRich( QObject::tr("* platform : %1")
                .arg(cnf.fwobj->getStr("platform").c_str())  );
        appendRich( QObject::tr("* host OS : %1")
                .arg(cnf.fwobj->getStr("host_OS").c_str()) );
        appendRich( QObject::tr("* Loading configuration from file %1")
                .arg(cnf.fwbfile));

        if (cnf.incremental)
        {
            appendRich( QObject::tr("* Incremental install"));
        }
        if (cnf.save_diff && cnf.incremental)
        {
            appendRich(
    QObject::tr("* Configuration diff will be saved in file %1").arg(cnf.diff_file));
        }
        if (cnf.dry_run)
        {
            appendRich(
    QObject::tr("* Commands will not be executed on the firewall"));
        }
    }
    else
    {
        appendRich( QObject::tr("* firewall name : %1")
                .arg( (*opListIterator)->getName().c_str()) );
    }
    appendRich("<hr>\n");
}


void instDialog::fillCompileSelectList()
{
    if (fwbdebug) qDebug("instDialog::fillCompileSelectList");
    
    compileMapping.clear();
    installMapping.clear();
    
    selectTable->setNumRows(firewalls.size()); 
    

    QCheckTableItem * citem; 
        
    Firewall* f;
    QDateTime dt;
    int row=0;
    
    bool show_library=false;
    string tmp_libname="";
    
    for (std::list<libfwbuilder::Firewall *>::iterator i=firewalls.begin();
            i!=firewalls.end(); ++i)
    {
        f=*i;
        
        time_t lm=f->getInt("lastModified");
        time_t lc=f->getInt("lastCompiled");
        time_t li=f->getInt("lastInstalled");
        
        selectTable->setText(row,2,QString::fromUtf8(f->getName().c_str()));
        selectTable->setColumnReadOnly(2,true);

        // in fact, if someone use same names for several libraries,
        // additional collumn with library names doesn't help to
        // identify a firewall
        if (!show_library && tmp_libname != "" && tmp_libname != f->getLibraryName())
            show_library = true;
        tmp_libname = f->getLibraryName();
        
        selectTable->setText(row,3,QString::fromUtf8(tmp_libname.c_str()));
        selectTable->setColumnReadOnly(3,true);
        
        citem=new QCheckTableItem(selectTable,"");
        citem->setChecked(
                (f->needsCompile() && reqFirewalls.empty() && !f->getInactive()) || 
                (!reqFirewalls.empty() && reqFirewalls.find(f)!=reqFirewalls.end()));
        selectTable->setItem(row,0,citem);
        compileMapping[f]=citem;
        
        citem=new QCheckTableItem(selectTable,"");
        citem->setChecked(
                (operation==BATCH_INSTALL) &&
                ((f->needsInstall() && reqFirewalls.empty() && !f->getInactive()) ||
                (!reqFirewalls.empty() && reqFirewalls.find(f)!=reqFirewalls.end())));
        selectTable->setItem(row,1,citem);
        installMapping[f]=citem;
            

        QLabel *l;
        QFont font;
        
        
        dt.setTime_t(lm);
        l=new QLabel(selectTable);
        l->setBackgroundColor(Qt::white);
        if (lm>lc && lm>li) {font=l->font();font.setBold(true);l->setFont(font);}
        l->setText((lm)?dt.toString():QString("---"));
        selectTable->setCellWidget(row,4,l);
        
        dt.setTime_t(lc);
        l=new QLabel(selectTable);
        l->setBackgroundColor(Qt::white);
        if (lc>lm && lc>li) {font=l->font();font.setBold(true);l->setFont(font);}
        l->setText((lm)?dt.toString():QString("---"));
        selectTable->setCellWidget(row,5,l);
        
        dt.setTime_t(li);
        l=new QLabel(selectTable);
        l->setBackgroundColor(Qt::white);
        if (li>lc && li>lm) {font=l->font();font.setBold(true);l->setFont(font);}
        l->setText((lm)?dt.toString():QString("---"));
        selectTable->setCellWidget(row,6,l);
        
        row++;        
    }
    if (show_library)
        selectTable->showColumn(3);
    else
        selectTable->hideColumn(3);
       
    for (int i=0;i<selectTable->numCols();i++)
    {
        if (i<4)            
            selectTable->adjustColumn(i);
        else
            selectTable->setColumnWidth(i,200);
    }
    
    //selectTable->setColumnStretchable(2,true);
    //selectTable->sortColumn(2,true,true);
}

void instDialog::selected(const QString &title)
{
    if (fwbdebug) qDebug("instDialog::selected");
    int p = indexOf( currentPage() );
    if (fwbdebug)
        qDebug(QString("to page: %1  from page: %2").arg(p).arg(lastPage));

    switch (p)
    {
    case 0: // select firewalls for compiling and installing
    {
        if (lastPage<0) fillCompileSelectList();
        setNextEnabled(page(0), isTableHasChecked());
        break;   
    }
    case 1: // compiling (installing) firewalls
    {
        setBackEnabled(page(1),false);
        setNextEnabled(page(1),false);
        if (lastPage<1)
        { // starting process
            fillCompileOpList();
            opListIterator=opList.begin();
            if (opList.size()==0)
            {// there are no firewalls for compilation
                // what about installation?

                compileFlag=false;
                     
                fillInstallOpList();
                opListIterator=opList.begin();
                if(opList.size()==0)
                {//there are no firewalls for installation
                    showPage(page(2));
                }
                else
                {
                    lastPage=p;
                    initInstall();
                    installSelected();
                    if (stopProcessFlag) return;
                }
            }
            else
            {
                compileSelected();
            }
        }
        else
        {
            if (opList.size()==0)
            {
                showPage(page(2));
            }
            else
            {
                installSelected();
                if (stopProcessFlag) return;
            }
        }
                
        break;
    }
    case 2: // fin
    {
        setBackEnabled(page(2),false);
        if (compileFlag && operation==BATCH_INSTALL) 
        {
            fillInstallOpList();
            initInstall();
            compileFlag=false;
            if (opList.size()>0)
            {
                opListIterator=opList.begin();
                showPage(page(1));
            }
            else
            {
                fillLastList();
            }
            break;
        }

        if (
            !compileFlag && 
            opList.end()!=opListIterator && 
            operation==BATCH_INSTALL &&
            !stopProcessFlag)
        {
            showPage(page(1));
            break;
        }
        fillLastList();
        break;
    }
    default: { }
    }

    lastPage = indexOf( currentPage() );
}

QString instDialog::getFullPath(instConf &cnf, const QString &file )
{
    if (QDir::isRelativePath(file)) return cnf.wdir + "/" + file;
    else return file;
}

bool instDialog::doInstallPage()
{
    if (fwbdebug) qDebug("instDialog::doInstallPage");
    
/* change of the page when flag ready is 'true' means we should start
 * operation */

    cnf.incremental = dlg->incr->isChecked();
    cnf.dry_run     = dlg->test->isChecked();
    cnf.backup_file = dlg->backupConfigFile->text();
    cnf.backup      = !cnf.backup_file.isEmpty();
    cnf.save_diff   = dlg->saveDiff->isChecked();
    cnf.saveStandby = dlg->saveStandby->isChecked();
    QString aaddr    = dlg->altAddress->text();
    if (!aaddr.isEmpty())
    {
/* alternative address can also be putty session name. In any case,
 * leave it up to ssh to resolve it and signal an error if it can't be
 * resolved ( Putty session name does not have to be in DNS at all ).
 */
        cnf.maddr = aaddr;
    } else
        cnf.maddr = cnf.fwobj->getManagementAddress().toString().c_str();

//        cnf.maddr         = altAddress->text();
    cnf.user          = dlg->uname->text();
    cnf.pwd           = dlg->pwd->text();
    cnf.epwd          = dlg->epwd->text();
    cnf.quiet         = dlg->quiet->isChecked();
    cnf.verbose       = dlg->verbose->isChecked();
    cnf.stripComments = dlg->stripComments->isChecked();
    cnf.compressScript= dlg->compressScript->isChecked();
    cnf.copyFWB       = dlg->copyFWB->isChecked();
    cnf.testRun       = dlg->testRun->isChecked();
    cnf.rollback      = dlg->rollback->isChecked();
    cnf.rollbackTime  = dlg->rollbackTime->value();
    cnf.cancelRollbackIfSuccess = dlg->cancelRollbackIfSuccess->isChecked();

    storeInstallerOptions();
/* check for a common error when multiple interfaces are marked as
 * 'management'
 */
    int nmi = 0;
    list<FWObject*> ll = cnf.fwobj->getByType(Interface::TYPENAME);
    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        Interface *intf = Interface::cast( *i );
        if (intf->isManagement()) nmi++;
    }
    if (nmi>1)
    {
        addToLog(
            QObject::tr("Only one interface of the firewall '%1' must be marked as management interface.")
            .arg(cnf.fwobj->getName().c_str()).latin1() );
        return false;
    }
    if (nmi==0)
    {
        addToLog(
            QObject::tr("One of the interfaces of the firewall '%1' must be marked as management interface.")
            .arg(cnf.fwobj->getName().c_str()).latin1() );
        return false;
    }
    if ((cnf.maddr == "" || cnf.maddr == "0.0.0.0") &&
        dlg->altAddress->text().isEmpty())
    {
        addToLog(
            QObject::tr("Management interface does not have IP address, can not communicate with the firewall.") );
        return false;
    }

    confScript="";

    confFiles.clear();
    if (cnf.copyFWB)
    {
        QFileInfo fwbfile_base(cnf.fwbfile);
            
        if (fwbdebug)
            qDebug( QString("Will copy data file: %1").arg(fwbfile_base.fileName()));

        confFiles.push_back( fwbfile_base.fileName() );
    }
/* read manifest from the conf file */

    QString conffile_path = getFullPath(cnf,cnf.conffile);
    QFile cf( conffile_path  );
    if (cf.open( IO_ReadOnly ) )
    {
        QTextStream stream(&cf);
        QString line;
        while (!stream.eof())
        {
            line    = stream.readLine();
            int pos = -1;
            if ( (pos=line.find(MANIFEST_MARKER))!=-1 )
            {
                int n = pos + QString(MANIFEST_MARKER).length();
//                if (line[n]=='*')  confScript=line.mid(n+2);
//                else               confFiles.push_back( line.mid(n+2) );

                confFiles.push_back( line.mid(n+2) );

                if (fwbdebug)
                    qDebug("instDialog: adding %c %s",
                           line[n].latin1(),
                           line.mid(n+2).ascii());

            }
            line = "";
        }
        cf.close();
    } else
    {
        QMessageBox::critical(this, "Firewall Builder",
                              tr("File %1 not found.").arg(conffile_path),
                              tr("&Continue") );
        return false;
    }

//    if (confScript.isEmpty()) confScript=cnf.conffile;
    if (confFiles.size()==0) confFiles.push_back(cnf.conffile);

    currentProgressBar->setProgress(0);

    summary();

    QStringList  args;

    if (cnf.fwobj->getStr("platform")=="pix" ||
        cnf.fwobj->getStr("platform")=="fwsm" ||
        cnf.fwobj->getStr("platform")=="iosacl"
    )
    {
#ifdef _WIN32
        args.push_back(ssh);

/*
 * putty ignores protocol and port specified in the session file if
 * command line option -ssh is given.
 *
 * On the other hand,the sign of session usage is an empty user name,
 * so we can check for that. If user name is empty, then putty will
 * use current Windows account name to log in to the firewall and this
 * is unlikely to work anyway. This seems to be a decent workaround.
 */
        if (!cnf.user.isEmpty() && ssh.find("plink.exe")!=-1)
        {
            args.push_back("-ssh");
            args.push_back("-pw");
            args.push_back(cnf.pwd);
        }

        if (!cnf.sshArgs.isEmpty())
            args += QStringList::split(" ",cnf.sshArgs);

        if (cnf.verbose) args.push_back("-v");
        if (!cnf.user.isEmpty())
        {
            args.push_back("-l");
            args.push_back(cnf.user);
            args.push_back(cnf.maddr);
        } else
            args.push_back(cnf.maddr);
#else
        args.push_back(argv0.c_str());
        args.push_back("-X");   // fwbuilder works as ssh wrapper
//            args.push_back("-d");
        args.push_back("-t");
        args.push_back("-t");

        if (!cnf.sshArgs.isEmpty())
            args += QStringList::split(" ",cnf.sshArgs);

        if (cnf.verbose) args.push_back("-v");
        if (!cnf.user.isEmpty())
        {
            args.push_back("-l");
            args.push_back(cnf.user);
            args.push_back(cnf.maddr);
        } else
            args.push_back(cnf.maddr);
#endif

        if (cnf.verbose) displayCommand(args);

        phase=1;

        SSHPIX *ssh_object = NULL;
        if (cnf.fwobj->getStr("platform")=="pix" ||
            cnf.fwobj->getStr("platform")=="fwsm")
        {
            ssh_object = new SSHPIX(this,
                                    cnf.fwobj->getName().c_str(),
                                    args,
                                    cnf.pwd,
                                    cnf.epwd,
                                    list<string>());
        } else   // ios
        {
            ssh_object = new SSHIOS(this,
                                    cnf.fwobj->getName().c_str(),
                                    args,
                                    cnf.pwd,
                                    cnf.epwd,
                                    list<string>());
        }

        /*
         * TODO:
         * the structure of scriptlets (command templates) for PIX and
         * IOS is nice and generic, it uses generalized "pre_config"
         * and "post_config" hooks in SSHPIX / SSHIOS classes. Need to
         * do the same for Unix firewalls.
         */
        
        QString cmd = "";
        QStringList pre_config_commands;
        QStringList post_config_commands;

        cmd = cnf.getCmdFromResource("pre_config_commands");
        pre_config_commands = QStringList::split("\n", replaceMacrosInCommand(cmd) );

        if (cnf.rollback)
        {
            cmd = cnf.getCmdFromResource("schedule_rollback");
            pre_config_commands = pre_config_commands +
                QStringList::split("\n", replaceMacrosInCommand(cmd));
        }

        cmd = cnf.getCmdFromResource("post_config_commands");
        post_config_commands = QStringList::split("\n", replaceMacrosInCommand(cmd) );

        if (cnf.cancelRollbackIfSuccess)
        {
            cmd = cnf.getCmdFromResource("cancel_rollback");
            post_config_commands = post_config_commands +
                QStringList::split("\n", replaceMacrosInCommand(cmd));
        }

        if (cnf.saveStandby)
        {
            cmd = cnf.getCmdFromResource("save_standby");
            post_config_commands = post_config_commands +
                QStringList::split("\n", replaceMacrosInCommand(cmd));
        }

        ssh_object->loadPreConfigCommands( pre_config_commands );
        ssh_object->loadPostConfigCommands( post_config_commands );

        runSSH(ssh_object);
        return true;
    }
    // all other Unix-based platforms

/*
 * if user requested test run, store firewall script in a temp file.
 * Always store it in a temp file on linksys
 */
    QString s;

/* user_can_change_install_dir */
    bool uccid=Resources::getTargetOptionBool(
        cnf.fwobj->getStr("host_OS"),"user_can_change_install_dir");

    if (uccid)
        s=cnf.fwobj->getOptionsObject()->getStr("firewall_dir").c_str();

    if (s.isEmpty()) s=Resources::getTargetOptionStr(
        cnf.fwobj->getStr("host_OS"),
        "activation/fwdir").c_str();

    cnf.fwdir     = s;

    continueRun();

    return true;
}

void instDialog::resetInstallSSHSession()
{
    if (fwbdebug) qDebug("instDialog::resetInstallSSHSession");

    if (session!=NULL)
    {
        disconnect(session,SIGNAL(printStdout_sign(const QString&)),
                   this,SLOT(append(const QString&)));

        disconnect(session,SIGNAL(sessionFinished_sign()),
                   this,SLOT(installerFinished()));

        disconnect(session,SIGNAL(sessionFatalError_sign()),
                   this,SLOT(installerError()));

        disconnect(session,SIGNAL(updateProgressBar_sign(int,bool)),
                   this,SLOT(updateProgressBar(int,bool)));

        session->terminate();

        delete session;
        session=NULL;
    }

    activationCommandDone=false;
}

/*
 * This method builds and returns activation command 
 * This method is used for all firewall platforms but PIX
 */
QString instDialog::getActivationCmd()
{
    if (!cnf.activationCmd.isEmpty())
    {
        return cnf.activationCmd;
    }

    QString cmd="";

    string optpath="activation/";

    if (cnf.user=="root")   optpath += "root/";
    else                     optpath += "reg_user/";

    if (cnf.testRun)
    {       
        optpath += "test/";
        if (cnf.rollback) optpath += "rollback/";
        else               optpath += "no_rollback/";
    } else
    {
        optpath += "run/";
        if (cnf.compressScript)  optpath += "compression/";
        else                     optpath += "no_compression/";
    }

    cmd=Resources::getTargetOptionStr(cnf.fwobj->getStr("host_OS"),
                                      optpath).c_str();
    return replaceMacrosInCommand(cmd);
}

QString instDialog::replaceMacrosInCommand(const QString &ocmd)
{

/* replace macros in activation command:
 *
 * %FWSCRIPT%, %FWDIR%, %FWBPROMPT%, %RBTIMEOUT%
 *
 * check if cnf.conffile is a full path. If it is, strip the path part
 * and use only the file name for %FWSCRIPT%
 */
    QString cmd = ocmd;

    QString clean_conffile = cnf.conffile.section(QDir::separator(),-1);
    if (fwbdebug)
    {
        qDebug("Macro substitutions:");
        qDebug(QString("  cnf.conffile=%1").arg(cnf.conffile));
        qDebug(QString("  %%FWSCRIPT%%=%1").arg(clean_conffile));
        qDebug(QString("  %%FWDIR%%=%1").arg(cnf.fwdir));
    }

    cmd.replace("%FWSCRIPT%",clean_conffile);
    cmd.replace("%FWDIR%",cnf.fwdir);
    cmd.replace("%FWBPROMPT%",fwb_prompt);

    int r = cnf.rollbackTime;
    if (cnf.rollbackTimeUnit=="sec")  r = r*60;

    QString rbt;
    rbt.setNum(r);
    cmd.replace("%RBTIMEOUT%",rbt);
    return cmd;
}

void instDialog::initiateCopy(const QString &file)
{
    QStringList args;
    list<string> allConfig;

    if (fwbdebug)
        qDebug("instDialog::initiateCopy for the file %s",file.ascii());

    QString platform=cnf.fwobj->getStr("platform").c_str();
    //if (platform!="pix" && platform!="fwsm") progressBar->show();

    QTextCodec::setCodecForCStrings(QTextCodec::codecForName("latin1"));
    
    std::ifstream   *wfile;

    QString file_with_path = getFullPath(cnf,file);

    wfile = new ifstream(file_with_path.latin1());
    if ( ! *wfile)
    {
        file_with_path = file;  // .fwb file path already includes wdir
        wfile = new ifstream(file_with_path.latin1());
        if ( ! *wfile)
        {
            addToLog(QObject::tr("Can not open file %1").arg(file_with_path));
            delete wfile;
            return;
        }
    }

    /* need to convert strings of the config file from Utf-8 to
     * internal presentation of QT (unicode) so we can process them
     * using methods of QString; will convert them back to Utf-8
     * before sending to the installer process
     */
    string s0;
    while ( !wfile->eof() )
    {
        getline( *wfile, s0);
//        QString s = QString::fromUtf8(s0.c_str());
//        QString s = s0.c_str();
//        s.stripWhiteSpace();

        if (fwbdebug) qDebug("instDialog::initiateCopy s='%s'",s0.c_str());

        allConfig.push_back(s0);
    }
    wfile->close();
    delete wfile;
    wfile=NULL;

//    allConfig.push_back("\004");

#ifdef _WIN32
    args.push_back(ssh);

    if (!cnf.user.isEmpty() && ssh.find("plink.exe")!=-1)
    {
        args.push_back("-ssh");
        args.push_back("-pw");
        args.push_back(cnf.pwd);
    }

#else
    args.push_back(argv0.c_str());
    args.push_back("-X");   // fwbuilder works as ssh wrapper
//    if (fwbdebug>1) args.push_back("-d");
//    args.push_back("-t");
//    args.push_back("-t");
#endif
/* do not change destination, we do chmod on it later */
//        args.push_back( cnf.wdir+"/"+file);

    if (!cnf.sshArgs.isEmpty())
        args += QStringList::split(" ",cnf.sshArgs);

    if (cnf.verbose) args.push_back("-v");
    if (!cnf.user.isEmpty())
    {
        args.push_back("-l");
        args.push_back(cnf.user);
        args.push_back(cnf.maddr);
    } else
        args.push_back(cnf.maddr);

    string optpath="activation/";

    if (cnf.user=="root")   optpath += "root/";
    else                     optpath += "reg_user/";

    if (cnf.testRun)        optpath += "test/";
    else                     optpath += "run/";

    optpath+="copy";

    QString cmd=Resources::getTargetOptionStr(cnf.fwobj->getStr("host_OS"),
                                              optpath).c_str();

/* replace macros in activation command:
 *
 * %FWSCRIPT%, %FWDIR%, %FWBPROMPT%, %RBTIMEOUT%
 *
 * check if cnf.conffile is a full path. If it is, strip the path part
 * and use only the file name for %FWSCRIPT%
 */
    QString file_name_no_spaces = file;
    if (file_name_no_spaces.find(" ")!=-1)
    {
        file_name_no_spaces.replace(" ","\\ ");
    }

    QString clean_file = file_with_path.section(QDir::separator(),-1);
    if (fwbdebug)
    {
        qDebug("Macro substitutions:");
        qDebug(QString("  cnf.conffile=%1").arg(cnf.conffile));
        qDebug(QString("  %%FWSCRIPT%%=%1").arg(file_name_no_spaces));
        qDebug(QString("  %%FWDIR%%=%1").arg(cnf.fwdir));
    }

    cmd.replace("\n","");
    cmd.replace("%FWSCRIPT%", file_name_no_spaces);
    cmd.replace("%FWDIR%", cnf.fwdir);
    cmd.replace("%FWBPROMPT%", fwb_prompt);

    args.push_back(cmd);
        
    addToLog( tr("\nCopying %1 -> %2:%3\n")
                .arg(file_with_path).arg(cnf.maddr).arg(cnf.fwdir) );

    if (cnf.verbose) displayCommand(args);

    phase=1;

    QTextCodec::setCodecForCStrings(QTextCodec::codecForName("latin1"));

    SSHSession *s= new SSHUnx(this,
                              cnf.fwobj->getName().c_str(),
                              args,
                              cnf.pwd,
                              "",
                              allConfig);
    
    s->setCloseStdin(true);
    runSSH(s);
}

void instDialog::displayCommand(const QStringList &args)
{
    QStringList a1 = args;

    for (QStringList::iterator i=a1.begin(); i!=a1.end(); i++)
    {
        if ( (*i)=="-pw" )
        {
            i++;
            *i = "XXXXXX";
            break;
        }
    }
    QString s=a1.join(" ");
    addToLog( tr("Running command '%1'\n\n").arg(s) );
}

void instDialog::finishInstall(bool success)
{
    if (fwbdebug) qDebug("instDialog::finishInstall");
    
    if (success) 
    {
        om->updateLastInstalledTimestamp(*opListIterator);
        opListMapping[*opListIterator]->setText(1,tr("Success"));
        processedFirewalls[*opListIterator].second=tr("Success");
    }
    else
    {
        opListMapping[*opListIterator]->setText(1,tr("Error"));
        processedFirewalls[*opListIterator].second=tr("Error");
    }
    
    opListIterator++;
    
    if(opListIterator!=opList.end() && batchInstall->isChecked() && !stopProcessFlag) 
    {
        installSelected();
        return;
    }
    setNextEnabled( page(1), true);
}

void instDialog::continueRun()
{
    if (fwbdebug) qDebug("instDialog::continueRun");
    
    if (session)
    {
        if (session->getErrorStatus())
        {
            if (fwbdebug) qDebug("session error");
            addToLog( tr("Fatal error, terminating install sequence\n") );
            finishInstall(false);
            //setFinishEnabled( page(1), true );
            return;
        }

        delete session;
        session=NULL;
    }

    if (activationCommandDone)
    {
        if (fwbdebug) qDebug("activationCommandDone");
        addToLog( tr("Done\n") );
        finishInstall();
        return;
    }

    if (cnf.fwobj->getStr("platform")=="pix" ||
        cnf.fwobj->getStr("platform")=="fwsm" ||
        cnf.fwobj->getStr("platform")=="iosacl"
    )
    {
        finishInstall();
        return;
    } else
    {

        if (!confFiles.empty())
        {
            QString cnffile = confFiles.front();
            confFiles.pop_front();
            initiateCopy( cnffile );
        } else
        {
            QStringList args;

            //progressBar->hide();

#ifdef _WIN32
            args.push_back(ssh);

            if (!cnf.user.isEmpty() &&
                ssh.find("plink.exe")!=-1) args.push_back("-ssh");

            args.push_back("-pw");
            args.push_back(cnf.pwd);
#else
            args.push_back(argv0.c_str());
            args.push_back("-X");   // fwbuilder works as ssh wrapper
            args.push_back("-t");
            args.push_back("-t");
#endif
            if (!cnf.sshArgs.isEmpty())
                args += QStringList::split(" ",cnf.sshArgs);

            if (cnf.verbose) args.push_back("-v");

            if (!cnf.user.isEmpty())
            {
                args.push_back("-l");
                args.push_back(cnf.user);
                args.push_back(cnf.maddr);
            } else
                args.push_back(cnf.maddr);
#if 0
            if (!cnf.user.isEmpty())
                args.push_back(cnf.user + "@" + cnf.maddr);
            else
                args.push_back(cnf.maddr);
#endif

            QString cmd = getActivationCmd();
            args.push_back( cmd );

            addToLog( tr("Activating new policy\n") );
            addToLog( "\n");

            if (cnf.verbose) displayCommand(args);

            activationCommandDone=true;

            runSSH( new SSHUnx(this,
                               cnf.fwobj->getName().c_str(),
                               args,
                               cnf.pwd,
                               "",
                               list<string>()) );
        }

    }
}

void instDialog::runSSH(SSHSession *s)
{
    if (fwbdebug) qDebug("instDialog::runSSH()");

    session = s;

    session->setOptions(&cnf);
    session->setFWBPrompt(fwb_prompt);

    connect(session,SIGNAL(printStdout_sign(const QString&)),
            this,SLOT(append(const QString&)));

    connect(session,SIGNAL(sessionFinished_sign()),
            this,SLOT(installerFinished()));

    connect(session,SIGNAL(sessionFatalError_sign()),
            this,SLOT(installerError()));

    connect(session,SIGNAL(updateProgressBar_sign(int,bool)),
            this,SLOT(updateProgressBar(int,bool)));

    session->startSession();
}

void instDialog::updateProgressBar(int n,bool setsize)
{
    if (fwbdebug)
        qDebug("instDialog::updateProgressBar  n=%d setsize=%d",n,setsize);

    if (setsize) currentProgressBar->setTotalSteps(n);
    else
        currentProgressBar->setProgress(currentProgressBar->totalSteps()-n);
}

/* user clicked 'Cancel' */
void instDialog::reject()
{
    if (fwbdebug) qDebug("instDialog::reject()");
    if (session!=NULL)
    {
        if (fwbdebug)
            qDebug("instDialog::reject()  killing ssh session");

        disconnect(session,SIGNAL(printStdout_sign(const QString&)),
                   this,SLOT(append(const QString&)));

        disconnect(session,SIGNAL(sessionFinished_sign()),
                   this,SLOT(installerFinished()));

        disconnect(session,SIGNAL(sessionFatalError_sign()),
                   this,SLOT(installerError()));

        disconnect(session,SIGNAL(updateProgressBar_sign(int,bool)),
                   this,SLOT(updateProgressBar(int,bool)));

        session->terminate();

        delete session;
        session=NULL;
    }
    QWizard::reject();
}


void instDialog::showEvent( QShowEvent *ev)
{
    st->restoreGeometry(this, QRect(200,100,480,500) );
    QDialog::showEvent(ev);
}

void instDialog::hideEvent( QHideEvent *ev)
{
    st->saveGeometry(this);
    QDialog::hideEvent(ev);
}
    
void instDialog::testRunRequested()
{
#if 0
#endif
}

void instDialog::saveLog()
{
    QString dir;
    if (currentLog==NULL) return;
    dir=st->getWDir();
    if (dir.isEmpty())  dir=st->getOpenFileDir();
    if (dir.isEmpty())  dir="~";

    /*
     * We use QTextEdit::append to add lines to the log buffer, each
     append creates a new paragraph so QTextEdit::text returns only
     contents of the last paragraph. Need to reassemble the whole text
     adding text from each paragraph separately.
     */
    QString logText;
    for (int i=0; i<currentLog->paragraphs(); ++i)
    {
        logText += currentLog->text(i);
        logText += "\n";
    }

    QString s = QFileDialog::getSaveFileName(
                    dir,
                    "Text file (*.txt)",
                    this,
                    "save file dialog",
                    "Choose a file" );
    
    
    if (fwbdebug)
        qDebug( "Saving log to file %s", s.ascii() );

    if (!s.isEmpty())
    {
        if (s.findRev(".txt",-1,false)==-1)
        {
            s+=".txt";
        }

        QFile f(s);
        if (f.open(IO_WriteOnly ))
        {
            QTextStream str( &f );
            str << logText;
            f.close();
        }
    }    
}

void instDialog::findFirewalls()
{
    firewalls.clear();
    om->findAllFirewalls(firewalls);
    firewalls.sort(FWObjectNameCmpPredicate());
    saveMCLogButton->setEnabled(true);
}


bool instDialog::runCompile(Firewall *fw)
{
    if (fwbdebug)
    {
        qDebug("instDialog::runCompile");
        qDebug(QString("Firewall:")+fw->getName().c_str());
    }

   
    addToLog( QString("\n") + 
                QObject::tr("Compiling rule sets for firewall: %1").arg(
                    fw->getName().c_str()
                )
    );
    prepareArgForCompiler(fw);
    
    proc.clearArguments();

    proc.setArguments( args );


    currentLog->append( proc.arguments().join(" ")+"\n" );

    if ( !proc.start() )
    {
        currentLog->append( tr("Error: Failed to start program") );
        return false;
    }
    return true;
}


bool instDialog::testFirewall(Firewall *fw)
{
    if (fwbdebug) qDebug("instDialog::testFirewall");
    FWOptions  *fwopt=fw->getOptionsObject();
    customScriptFlag=false;

    Management *mgmt=fw->getManagementObject();
    assert(mgmt!=NULL);
    PolicyInstallScript *pis   = mgmt->getPolicyInstallScript();

/* we don't care about ssh settings if external installer is to be used */

    if ( (/* ! pis->isEnabled() || */ pis->getCommand()=="") &&
         st->getSSHPath().isEmpty())
    {
        QMessageBox::critical(this, "Firewall Builder",
   tr("Policy installer uses Secure Shell to communicate with the firewall.\n"
      "Please configure directory path to the secure shell utility \n"
      "installed on your machine using Preferences dialog"),
                              tr("&Continue") );
        addToLog("Please configure directory path to the secure \n shell utility installed on your machine using Preferences dialog");
        return false;
    }

/* need to save settings so that if the user just changed ssh/scp, the
 * wrapper will pick changes up
 */
    st->save();

    QString ofname = fwopt->getStr("output_file").c_str();
    if (ofname.isEmpty()) ofname = QString(fw->getName().c_str()) + ".fw";

    QString fwfname = getFileDir( mw->getRCS()->getFileName() ) + "/" + ofname;

/* bug #1617501: "Install fails after compile". Check ofname, just in
 * case user put full path name for the output script name in options
 */
    if ( !QFile::exists(fwfname) && !QFile::exists(ofname))
    {
/* need to recompile */
        addToLog(tr("Firewall isn't compiled."));
        if (fwbdebug) qDebug("Firewall isn't compiled.");
        return false;
    }
    

    args.clear();
    
   
    if ( /*! pis->isEnabled() || */ pis->getCommand()=="" )
    {
        //instConf cnf;

        cnf.user          = fwopt->getStr("admUser").c_str();
        cnf.maddr         = fwopt->getStr("altAddress").c_str();
        cnf.sshArgs       = fwopt->getStr("sshArgs").c_str();
        cnf.activationCmd = fwopt->getStr("activationCmd").c_str();

        cnf.fwobj     = fw;
        cnf.fwbfile   = mw->db()->getFileName().c_str();
        cnf.conffile  = ofname;
        cnf.diff_file = QString(fw->getName().c_str())+".diff";
        cnf.wdir      = getFileDir( mw->getRCS()->getFileName() );
        cnf.diff_pgm  = Resources::platform_res[fw->getStr("platform")]->
            getResourceStr("/FWBuilderResources/Target/diff").c_str();

/* set this in instDialog now

        QString s=fwopt->getStr("firewall_dir").c_str();
        if (s.isEmpty()) s="/etc/fw";
        cnf.fwdir     = s;
*/

        cnf.diff_pgm = QString(appRootDir.c_str()) + cnf.diff_pgm;

#ifdef _WIN32
	cnf.diff_pgm = cnf.diff_pgm + ".exe";
#endif

        //instDialog *id = new instDialog( &cnf );

        //int exec_result=id->exec();

    } else
    {
        customScriptFlag=true;
        string inst_script=pis->getCommand();

        QString wdir = getFileDir( mw->getRCS()->getFileName() );


        args.push_back(inst_script.c_str());

        args += QStringList::split(" ", pis->getArguments().c_str() );

        args.push_back("-f");
        args.push_back(mw->db()->getFileName().c_str());

        if (wdir!="")
        {
            args.push_back("-d");
            args.push_back(wdir);
        }

        args.push_back( QString("/%1/%2")
                        .arg(QString::fromUtf8(fw->getLibrary()->getName().c_str()))
                        .arg(fw->getPath(true).c_str() ) );

        //execDialog dlg(this, args );

        //int exec_result=dlg.run();
        //qDebug(QString("Result: %1").arg(exec_result));
        //if (exec_result==0) om->updateLastInstalledTimestamp(fw);
    }
    return true;
}

bool instDialog::prepareArgForCompiler(Firewall *fw)
{
    FWOptions *fwopt=fw->getOptionsObject();

/*
 * I should be able to specify custom compiler for firewall with
 * no platform (e.g. for experiments)
 */
    string compiler=fwopt->getStr("compiler");
    if (compiler=="") 
    {
        compiler=Resources::platform_res[fw->getStr("platform")]->getCompiler();
    }

    if (compiler=="")
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Firewall platform is not specified in this object.\n\
Can't compile firewall policy."),
            tr("&Continue"), QString::null,QString::null,
            0, 1 );
        return false;
    }

/*
 * On Unix compilers are installed in the standard place and are
 * accessible via PATH. On Windows and Mac they get installed in
 * unpredictable directories and need to be found
 *
 * first, check if user specified an absolute path for the compiler,
 * then check  if compiler is registsred in preferences, and if not,
 * look for it in appRootDir and if it is not there, rely on PATH
 */
#if defined(Q_OS_WIN32) ||  defined(Q_OS_MACX)

    if ( ! QFile::exists( compiler.c_str() ) )
    {
        string ts = string("Compilers/")+compiler;
        QString cmppath = st->getStr( ts.c_str() );
        if (!cmppath.isEmpty()) compiler=cmppath.latin1();
        else
        {
            /* try to find compiler in appRootDir. */
            string ts =  appRootDir + FS_SEPARATOR + compiler;
            if ( QFile::exists( ts.c_str() ) )
                compiler = appRootDir + FS_SEPARATOR + compiler;
        }
    }
#endif

#if 0
// if we use WDir for the "-d" option for compiler
    QString wdir;
    if (st->getWDir().isEmpty())
    {
        QString of = rcs->getFileName();
        wdir = of.left( of.findRev('/',-1) );
    } else
    {
        wdir=st->getWDir();
    }
#endif

    QString wdir = getFileDir(mw->getRCS()->getFileName() );

    args.clear();

    args.push_back(compiler.c_str());

    args += QStringList::split(" ", fwopt->getStr("cmdline").c_str() );

    args.push_back("-f");
    args.push_back(mw->db()->getFileName().c_str());

    if (wdir!="")
    {
        args.push_back("-d");
        args.push_back(wdir);
    }

    QString ofname = QString::fromUtf8(fwopt->getStr("output_file").c_str());
    if (!ofname.isEmpty()) 
    {
        args.push_back("-o");
        args.push_back(ofname);
    }

/* there has been a change in v2.1: now resources are installed in
 * directory /usr/share/fwbuilder-2.1 (it used to be just
 * /usr/share/fwbuilder) Compilers that are packaged separately need
 * to know about this but I do not want to hard-code it. It is easier
 * to pass the path on the command line 
 *
 * Update 01/16/06:
 *
 * We now package a copy of resource files with externally packaged
 * compilers (such as fwbuilder-pix), therefore flag "-r" is not
 * needed anyore
 */

#if 0
    args.push_back("-r");
    args.push_back(respath);
#endif

    args.push_back( QString::fromUtf8(fw->getName().c_str()) );
    return true;
}

/*
 * Adds one line of text to the log
 *
 */
void instDialog::addToLog(const QString &line)
{
    if (fwbdebug)
        qDebug("instDialog::addToLog");

    if (currentLog)
    {
        if (line.isEmpty()) return;

//        currentLog->moveCursor( QTextEdit::MoveEnd , false );

        QStringList words=QStringList::split(" ",line);

#if 0  
// although it is nice to be able to print errors in red, this
// will break because of localization
        QColor oc=currentLog->color();
        if (words.first().find("Error")>=0) currentLog->setColor(Qt::red);
        if (words.first().find("Abnormal")>=0) currentLog->setColor(Qt::red);
        if (words.first().find("Warning")>=0) currentLog->setColor(Qt::blue);
#endif        
//        currentLog->insert( line );
        currentLog->append( line );
        
        if (words.first().find("rule")>=0)
        {
            currentProgressBar->setProgress(++processedRules);
        } else
        {
            if (words.first().find("processing")>=0)
            {
                currentProgressBar->reset();
                totalRules=words[1].toInt();
                currentProgressBar->setTotalSteps(totalRules);
                processedRules=0;
            } else
            {
                if (words.first().find("Compiling")>=0)
                {
                    currentLabel->setText(line.stripWhiteSpace());
                    currentProgressBar->reset();
                } else
                {
                    if (line.find("Compiled successfully")>=0)
                    {
                        currentLabel->setText(line.stripWhiteSpace());
                        currentProgressBar->setProgress(currentProgressBar->totalSteps());
                    }
                }
            }
        }
        QApplication::eventLoop()->processEvents(QEventLoop::ExcludeUserInput,1);

        if (fwbdebug)
            qDebug( QString("instDialog::addToLog Current log buffer contents"
                            ": %1 lines, %2 paragraphs, '%3'").
                    arg(currentLog->lines()).
                    arg(currentLog->paragraphs()).
                    arg(currentLog->text()) );
        
    }
}

void instDialog::readFromStdout()
{
    QString buf = proc.readStdout();
    bool endsWithLF = buf.endsWith("\n");
    QString lastLine = "";

    // split on LF
    QStringList bufLines = QStringList::split("\n",buf,TRUE);

    if (fwbdebug)
    {
        qDebug("buf=%s",buf.ascii());
        qDebug("endsWithLF=%d",endsWithLF);
        qDebug("bufLines.size()=%d",bufLines.size());
    }

    // if buf ends with a LF character, the last element in the list is
    // an empty string
    if (endsWithLF && bufLines.last().isEmpty())  bufLines.pop_back();

    // if buf does not end with LF, last element in the list is
    // incomplete line of text
    if (!endsWithLF)
    {
        lastLine = bufLines.last();
        bufLines.pop_back();
    }

    // elements that are left in the list  are all complete lines of text
    for (QStringList::Iterator i=bufLines.begin(); i!=bufLines.end(); ++i)
    {
        addToLog( pendingLogLine + *i + "\n");
        pendingLogLine = "";
    }

    pendingLogLine += lastLine;
}

void instDialog::installerError()
{
    if (fwbdebug) qDebug("instDialog::installerError");

    addToLog( tr("Error: Terminating install sequence\n") );
    finishInstall(false);

    resetInstallSSHSession();
    //setFinishEnabled( page(1), true );

    if (session) delete session;
    session=NULL;
}

void instDialog::installerFinished()
{
    if( fwbdebug) qDebug("instDialog::installerFinished");
        
    if (session->getErrorStatus())
    {
        installerError();
    }

    if (session) delete session;
    session=NULL;

    QTimer::singleShot( 0, this, SLOT(continueRun()) );
}

void instDialog::processExited()
{
    if( fwbdebug) qDebug("instDialog::processExited  ");
    
    int res=proc.exitStatus();

    if (!compileFlag && customScriptFlag)
    {
        if( fwbdebug) qDebug("Custom script installer");
            
        if(res==0)
        {
            om->updateLastInstalledTimestamp(*opListIterator);
            processedFirewalls[*opListIterator].second="Success";
            opListMapping[*opListIterator]->setText(1,tr("Success"));
        }   
        else
        {
            processedFirewalls[*opListIterator].second="Error";
            opListMapping[*opListIterator]->setText(1,tr("Error"));
        }
        currentProgressBar->setProgress(currentProgressBar->totalSteps () ); 
        qApp->processEvents();
            
        opListIterator++;
        
        if (opListIterator!=opList.end() && batchInstall->isChecked() && !stopProcessFlag)
        {
            installSelected();
        }
        else
        {
            currentSaveButton->setEnabled(true);
            setNextEnabled(page(1), true);
            saveMCLogButton->setEnabled(true);
        }
        return;
    }

        
    if(res==0 && proc.normalExit())   
    {
        om->updateLastCompiledTimestamp(*opListIterator);
        processedFirewalls[*opListIterator].first=tr("Success");
        opListMapping[*opListIterator]->setText(1,tr("Success"));
    }
    else
    {
        addToLog( tr("Abnormal program termination") +"\n");
        // WARNING:
        // If compilation of the firewall failed we should not
        // install it
        //
        installMapping[*opListIterator]->setChecked(false);
        processedFirewalls[*opListIterator].second=tr("Skipped");
        processedFirewalls[*opListIterator].first=tr("Error");
        opListMapping[*opListIterator]->setText(1,tr("Error"));
    }

    if (fwbdebug)
        qDebug( QString(" Current log buffer contents: %1 lines, '%2'").
                arg(currentLog->lines()).arg(currentLog->text()) );



    Firewall *f;
    QListViewItem* item;
    
    opListIterator++;
    
    while (opListIterator!=opList.end() && !stopProcessFlag)
    {
        if (currentFirewallsBar) currentFirewallsBar->setProgress(++progress);
        qApp->processEvents();

        f=*opListIterator;
        item=opListMapping[f];
        currentFWLabel->setText(QString::fromUtf8(f->getName().c_str()));
        

        listView4->ensureItemVisible ( opListMapping[f]  );
        
        if(runCompile(f))
        {
            item->setText(1,tr("Compiling ..."));
            return;
        }
        else
        {
            item->setText(1,tr("Failure"));
        }
        ++opListIterator;   
    }
    if (currentFirewallsBar) currentFirewallsBar->setProgress(currentFirewallsBar->totalSteps());
    
    if (currentStopButton)
    {
        currentStopButton->setText(tr("Recompile"));
        disconnect (currentStopButton , SIGNAL(clicked()), this ,SLOT(stopCompile()));
        connect(currentStopButton,SIGNAL(clicked()),this,SLOT(compileSelected()));
        currentStopButton->setEnabled(true);
        
    }
    currentSaveButton->setEnabled(true);
    if (operation==BATCH_COMPILE)
        setFinishEnabled(page(1), true);
    else
        setNextEnabled(page(1), true);

    currentSaveButton->setEnabled(true);

}

void instDialog::selectAllFirewalls()
{
    if (fwbdebug) qDebug("instDialog::selectAllFirewalls");
    if (operation==BATCH_INSTALL)selectAll(installMapping);
    selectAll(compileMapping);
    tableValueChanged(0,0);
}


void instDialog::deselectAllFirewalls()
{
    if (operation==BATCH_INSTALL)deselectAll(installMapping);
    deselectAll(compileMapping);
    tableValueChanged(0,0);
}

void instDialog::selectAll(t_tableMap &mapping)
{
    if (fwbdebug) qDebug("instDialog::selectAll");
    
    t_tableMap::iterator i;

    QCheckTableItem *item;
    
    for(i=mapping.begin();i!=mapping.end();++i)
    {
        item=(*i).second;
        item->setChecked(true);
    }
}
void instDialog::deselectAll(t_tableMap &mapping)
{
    if (fwbdebug) qDebug("instDialog::deselectAll");
    t_tableMap::iterator i;
    QCheckTableItem *item;
    for(i=mapping.begin();i!=mapping.end();++i)
    {
        item=(*i).second;
        item->setChecked(false);
    }
}
void instDialog::fillCompileOpList()
{
    listView4->clear();
    opList.clear();
    processedFirewalls.clear();
    opListMapping.clear();
    
    Firewall * f;
    InstallFirewallViewItem * item;
    t_fwList::reverse_iterator i;
    for(i=firewalls.rbegin();i!=firewalls.rend();++i)
    {
        if(compileMapping[(*i)]->isChecked())
        {
            f=(*i);
            opList.push_front(f);
            item=new InstallFirewallViewItem(listView4,
                    QString::fromUtf8(f->getName().c_str()),false);
            
            opListMapping[f]=item;

            processedFirewalls[f]=make_pair("","");
        }
    }
    
}
void instDialog::compileSelected()
{
    if (fwbdebug) qDebug("instDialog::compileSelected");
    setTitle(page(1),tr("Batch policy rules compilation"));
    
    currentLog = procLogDisplay;
    currentSaveButton = saveMCLogButton;
    currentSaveButton->setEnabled(true);
    currentStopButton = controlMCButton;
    currentProgressBar = compProgress;
    currentFirewallsBar = compFirewallProgress;
    currentLabel = infoMCLabel;
    currentFWLabel = fwMCLabel;
    currentSearchString="Compiling rule sets for firewall: ";
    setNextEnabled(page(0),false);
    
    mw->fileSave();
    compileFlag=true;

    currentProgressBar->reset();
    currentFirewallsBar->reset();
    currentFirewallsBar->setTotalSteps(opList.size());
    progress=0;
    stopProcessFlag=false;

    
    currentLog->clear();

    if (currentStopButton) 
    {
        disconnect(currentStopButton,SIGNAL(clicked()),this,SLOT(compileSelected()));
        connect(currentStopButton,SIGNAL(clicked()),this,SLOT(stopCompile()));
        currentStopButton->setText(tr("Stop"));
        currentStopButton->setEnabled(true);
        
    }
    
    Firewall *f;
    QListViewItem* item;
    
    opListIterator=opList.begin();
    
    while (opListIterator!=opList.end() && !stopProcessFlag)
    {
        if (currentFirewallsBar) currentFirewallsBar->setProgress(++progress);
        qApp->processEvents();

        f=*opListIterator;
        item=opListMapping[f];
        currentFWLabel->setText(QString::fromUtf8(f->getName().c_str()));
        

        listView4->ensureItemVisible ( opListMapping[f]  );
        
        if(runCompile(f))
        {
            item->setText(1,tr("Compiling ..."));
            return;
        }
        else
        {
            item->setText(1,tr("Failure"));
        }
        ++opListIterator;   
    }
    
}
void instDialog::stopCompile()
{
    if( fwbdebug) qDebug("instDialog::stopCompile");
    stopProcessFlag=true;
    
    currentStopButton->setEnabled(false);
   // disconnect(currentStopButton,SIGNAL(clicked()),this,SLOT(stopCompile()));
    
    proc.tryTerminate();
    QTimer::singleShot( 1000, &proc, SLOT( kill() ) );
    
}

void instDialog::stopInstall()
{
    currentStopButton->setEnabled(false);
    stopProcessFlag=true;
}

void instDialog::fillLastList()
{
    lastListView->clear();
    
    QListViewItem *item;
    Firewall* f;
    t_procMess m;
    
    for (map<libfwbuilder::Firewall *, t_procMess>::iterator i=processedFirewalls.begin();
            i!=processedFirewalls.end(); ++i)
    {
        f=(*i).first;
        m=(*i).second;
        
        item=new QListViewItem(lastListView,
                QString::fromUtf8(f->getName().c_str()));
        
        
        item->setText(1,m.first);
        item->setText(2,m.second);

    }
    
    lastListView->setSorting(0);
}

bool instDialog::runInstall(Firewall * fw)
{
    if (fwbdebug) qDebug("instDialog::runInstall");
    if (customScriptFlag)
    {
        if (fwbdebug) qDebug("custom script");
        summary();
        proc.clearArguments();

        proc.setArguments( args );
        addToLog( proc.arguments().join(" ") );
        addToLog("\n");
        if ( !proc.start() )
        {
            addToLog( tr("Error: Failed to start program") );
            return false;
        }
        
    }
    else
    {
        if (fwbdebug) qDebug("build-in installer");
        return doInstallPage();
    }
    return true;
}

void instDialog::fillInstallOpList()
{
    if (fwbdebug) qDebug("instDialog::fillInstallOpList");
    listView4->clear();
    opListMapping.clear();
    opList.clear();
    
    InstallFirewallViewItem * item;
    Firewall * f;
    
    t_fwList::reverse_iterator i;
    for(i=firewalls.rbegin();i!=firewalls.rend();++i)
    {
        f=(*i);

        if (installMapping[f]->isChecked())
        {
            opList.push_front(f);
            item=new InstallFirewallViewItem(listView4,
                    QString::fromUtf8(f->getName().c_str()),false);
            
            opListMapping[f]=item;
            if (processedFirewalls.find(f)==processedFirewalls.end())
                processedFirewalls[f]=make_pair("","");
        }
    }
}
void instDialog::initInstall()
{
    if (fwbdebug)
        qDebug("instDialog::initInstall()");

    currentFirewallsBar =compFirewallProgress;
    currentFirewallsBar->reset();
    currentFirewallsBar->setTotalSteps(opList.size());
    currentStopButton = controlMCButton;
    currentStopButton->setText(tr("Stop"));
    
    disconnect(currentStopButton,SIGNAL(clicked()),0,0);
    connect(currentStopButton,SIGNAL(clicked()),this,SLOT(stopInstall()));

    currentSaveButton = saveMCLogButton;
    currentLog = procLogDisplay;
    currentLog->setTextFormat(Qt::RichText);
    currentProgressBar = compProgress;
    currentLabel = fwMCLabel;
    currentLog->clear();
    currentSearchString=tr("Install firewall: ");
    infoMCLabel->setText("");
    progress=0;
    stopProcessFlag=false;
}

void instDialog::installSelected()
{
    if (fwbdebug) qDebug("instDialog::installSelected");
    if (fwbdebug) qDebug(QString("firewall:")+((*opListIterator)->getName().c_str()));
    setTitle(page(1),tr("Installing firewalls"));
    setNextEnabled(page(1),false);

    bool fPix=false,fCustInst=true;
    
    if (opListIterator==opList.begin() && batchInstall->isChecked())
    {
        analyseInstallQueue(fPix,fCustInst);
        
    }
    while ( opListIterator!=opList.end())
    {
    
        currentSaveButton->setEnabled(true);
        currentProgressBar->reset();
        currentProgressBar->setTotalSteps(100);
        
        currentLabel->setText(QString::fromUtf8((*opListIterator)->getName().c_str()));
        compileFlag=false;
        
        resetInstallSSHSession();
        currentFirewallsBar->setProgress(++progress);

        appendRich("<hr>");
        appendRich(QString("<b>")+currentSearchString+
                    QString::fromUtf8((*opListIterator)->getName().c_str())+"</b>");
        appendRich("\n");
        //qApp->processEvents();
        
    
        if (testFirewall(*opListIterator))
        {
            opListMapping[*opListIterator]->setText(1,tr("Installing ..."));
            //qApp->processEvents();
            
            if (customScriptFlag && fCustInst)
            { // custom installer
                if (fwbdebug) qDebug("custom install script.");

            }
            else
            { // buildin installer
                if (fwbdebug) qDebug("buildin installer");
                // Show options dialog
                if (!batchInstall->isChecked() ||
                    opListIterator==opList.begin())
                {
                    if (dlg) 
                    {
                        delete dlg;
                        dlg=NULL;
                    }
                    
                    prepareInstallerOptions();
                    
                    if (batchInstall->isChecked())
                    {
                        dlg=new instBatchOptionsDialog(this, &cnf);
                    }
                    else
                    {
                        dlg=new instOptionsDialog(this, &cnf);
                    }
                    
                    if (dlg->exec()==QDialog::Rejected)
                    {
                        
                        delete dlg;
                        dlg=NULL;

                        if (batchInstall->isChecked())
                        {
                            stopProcessFlag=true;
                            showPage(page(0));
                            return;
                        }
                        else
                        {
                            processedFirewalls[*opListIterator].second="Cancelled";
                            opListMapping[*opListIterator]->setText(1,tr("Failure"));
                            opListIterator++;
                            setNextEnabled(page(1),true);
                            saveMCLogButton->setEnabled(true);
                            //if (opList.end()!=opListIterator && batchInstall->isChecked())
                            //    showPage(page(2));
                            return;
                            
                        }
                    }
                    setTitle(page(1),
                       QObject::tr("Installing policy rules on firewall '%1'.").arg(
                                 (*opListIterator)->getName().c_str() ));
                    //qApp->processEvents();
                    if (!runInstall(*opListIterator))
                    {
                        if (fwbdebug) qDebug("start error");
                        processedFirewalls[*opListIterator].second="start error";
                        opListMapping[*opListIterator]->setText(1,tr("Failure"));
                        opListIterator++;
                        setNextEnabled(page(1),true);
                        saveMCLogButton->setEnabled(true);
                        if (opList.end()!=opListIterator && batchInstall->isChecked())
                            showPage(page(2));
                        return;
                        
                    }

                    return;
                }
            }
            if (!runInstall(*opListIterator))
            {
                processedFirewalls[*opListIterator].second="start error";
                opListMapping[*opListIterator]->setText(1,tr("Failure"));
                opListIterator++;
                setNextEnabled(page(1),true);
            }
            else
            {
                return;
            }
        }
        else
        {
            processedFirewalls[*opListIterator].second="init error";
            opListMapping[*opListIterator]->setText(1,tr("Failure"));
            opListIterator++;
            setNextEnabled(page(1),true);
        }
    }
}

void instDialog::findFirewallInCompileLog(QListViewItem* item)
{
    if (fwbdebug) qDebug("instDialog::findFirewallInCompileLog");
    Firewall *fw;
    int p=1,i=0;

    detailMCframe->show();
    qApp->processEvents();
    fw=findFirewallbyListItem(item);
    procLogDisplay->moveCursor( QTextEdit::MoveEnd , false );
    procLogDisplay->find(
        currentSearchString + 
        QString::fromUtf8(fw->getName().c_str()),true,true,true,&p,&i);
}

Firewall * instDialog::findFirewallbyListItem(QListViewItem *item)
{
    Firewall * res=NULL;
    t_listMap::iterator i;
    
    for(i=opListMapping.begin();i!=opListMapping.end();++i)
    {
        if ((*i).second==item) 
        {
            res=(*i).first;
            break;
        }
    }
    return res;
}

Firewall * instDialog::findFirewallbyTableItem(QCheckTableItem *item)
{
    Firewall * res=NULL;
    t_tableMap::iterator i;
    
    for(i=compileMapping.begin();i!=compileMapping.end();++i)
    {
        if ((*i).second==item) 
        {
            res=(*i).first;
            return res;
        }
    }

    for(i=installMapping.begin();i!=installMapping.end();++i)
    {
        if ((*i).second==item) 
        {
            res=(*i).first;
            return res;
        }
    }
    
    return res;
}
void instDialog::showSelected()
{
    
    QCheckTableItem* item;
    Firewall *f;
    
    t_fwList::iterator i;
    bool sel;
    
    for(i=firewalls.begin();i!=firewalls.end();++i)
    {
        sel=false;
        
        f=(*i);
        item=compileMapping[f];
        sel|=item->isChecked();

        item=installMapping[f];
        sel|=item->isChecked();
        
        if(!sel )
        {
            if (showSelectedFlag)
            {
                selectTable->showRow(item->row());
            }
            else
            {
                selectTable->hideRow(item->row());
            }
        }
    }
    if (showSelectedFlag) 
    {
        showSelButton->setText(tr("Show selected"));
        pushButton16->setEnabled(true);
        pushButton17->setEnabled(true);
    }
    else
    {
        showSelButton->setText(tr("Show all"));
        pushButton16->setEnabled(false);
        pushButton17->setEnabled(false);
    }
    showSelectedFlag=!showSelectedFlag;
    
}

void instDialog::tableValueChanged(int row, int col)
{
    if (fwbdebug) qDebug("instDialog::tableValueChanged");
    QCheckTableItem *item;
    Firewall *f;
    
    item=(QCheckTableItem *)selectTable->item(row,col);
    f=findFirewallbyTableItem(item);

    
    if (col==0)
    { // Compilation flag has been changed
        if (
                !item->isChecked() && 
                f->getInt("lastCompiled")==0 && 
                installMapping[f]->isChecked())
        {
            installMapping[f]->setChecked(false);
        }
    }
    else if (col==1)
    { // Installation flag has been changed
        if (
                item->isChecked() && 
                f->getInt("lastCompiled")==0) 
        {
            compileMapping[f]->setChecked(true);
        }
        
    }
    
    setNextEnabled(page(0), isTableHasChecked());
}

bool instDialog::isTableHasChecked()
{
    QCheckTableItem *item;
    Firewall *f;
    
    t_fwList::iterator i;
    
    bool res=false;
    
    for(i=firewalls.begin();i!=firewalls.end();++i)
    {
        f=(*i);
        item=compileMapping[f];
        if(item->isChecked()) res = true;
        
        item=installMapping[f];
        if(item->isChecked()) res = true;
        
    }
    return res;
}

void instDialog::analyseInstallQueue(bool &fPix, bool &fCustInst)
{
    if (fwbdebug) qDebug("instDialog::analyseInstallQueue");
    Firewall *f;
    //FWOptions *fwopt;
    Management *mgmt;
    PolicyInstallScript *pis;
    
    fPix=false;
    fCustInst=true;
    
    t_fwList::iterator i;
    for(i=opList.begin(); i!=opList.end(); ++i)
    {
        f=(*i);
        //fwopt=f->getOptionsObject();
        mgmt=f->getManagementObject();
        pis   = mgmt->getPolicyInstallScript();
        
        fPix = fPix || f->getStr("platform")=="pix" || f->getStr("platform")=="fwsm" || f->getStr("platform")=="iosacl";
        fCustInst =  fCustInst && !( pis->getCommand()=="" );
        
        if (cnf.fwobj==NULL && !fCustInst)
            cnf.fwobj=f;
        
        if (fwbdebug)
        {
            qDebug(QString("f:")+f->getName().c_str());
            qDebug(QString("p:")+f->getStr("platform").c_str());
            qDebug(QString("fPix:")+(fPix?"true":"false"));
        }
        
        if (fPix && !fCustInst) return;// nothing can change if we continue loop
    }
}

void instDialog::clearReqFirewalls()
{
    reqFirewalls.clear();
}

void instDialog::addReqFirewall(Firewall *f)
{
    reqFirewalls.insert(f);
}
