/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: SSHSession.cpp,v 1.20 2006/05/15 03:13:48 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include "config.h"
#include "global.h"
#include "utils.h"

#include "SSHSession.h"
#include "instConf.h"

#include <qobject.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qcstring.h>
#include <qtextcodec.h>

#include <iostream>

#ifdef _WIN32
#  include <windows.h>
#endif


using namespace std;

char  *SSHSession::newKeyOpenSSH  ="Are you sure you want to continue connecting (yes/no)?";
char  *SSHSession::newKeyPlink    ="Store key in cache? (y/n)";
char  *SSHSession::newKeyVsh      ="Accept and save? (y/n)";
char  *SSHSession::fingerprintPrompt="key fingerprint is";



SSHSession::SSHSession(const QString &_h,
                       const QStringList &_args,
                       const QString &_p,
                       const QString &_ep,
                       const list<string> &_in)
{
    host = _h;
    args = _args;
    pwd  = _p;
    epwd = _ep;
    input   = _in;
    quiet   = false;
    verbose = false;
    closeStdin = false;
    error = false;
    endOfCopy = false;

    proc = NULL;
    retcode = 0;
    heartBeatTimer = new QTimer(this);
    connect(heartBeatTimer, SIGNAL(timeout()), this, SLOT(heartBeat()) );

    newKeyMsg = tr("You are connecting to the firewall <b>'%1'</b> for the first time. It has provided you its identification in a form of its host public key. The fingerprint of the host public key is: \"%2\" You can save the host key to the local database by pressing YES, or you can cancel connection by pressing NO. You should press YES only if you are sure you are really connected to the firewall <b>'%3'</b>.");


    fwb_prompt="";
    quiet=false;
    verbose=false;
    backup=false;
    incremental=false;
    dry_run=false;
    testRun=false;
    stripComments=false;
    wdir="";
    conffile="";
    backupFile="";
    save_diff="";
    diff_pgm="";
    diff_file="";

}

void SSHSession::startSession()
{
    proc = new QProcess();
    retcode = -1;
    proc->setCommunication(
        QProcess::Stdin|QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr);

    heartBeatTimer->start(1000);

    if (fwbdebug)
        qDebug("SSHSession::startSession  this=%p  proc=%p   heartBeatTimer=%p",
               this,proc,heartBeatTimer);

    connect(proc,SIGNAL(readyReadStdout()), this,  SLOT(readFromStdout() ) );
    connect(proc,SIGNAL(readyReadStderr()), this,  SLOT(readFromStderr() ) );
    connect(proc,SIGNAL(processExited()),   this,  SLOT(processExited() ) );

    QTextCodec::setCodecForCStrings(QTextCodec::codecForName("latin1"));

    for (QStringList::const_iterator i=args.begin(); i!=args.end(); ++i)
    {
        proc->addArgument( *i );
        cmd += *i;
    }

    QStringList env;

#ifdef _WIN32
    env.push_back( QString("APPDATA=")+getenv("APPDATA") );
    env.push_back( QString("HOMEPATH=")+getenv("HOMEPATH") );
    env.push_back( QString("HOMEDRIVE=")+getenv("HOMEDRIVE") );
    env.push_back( QString("ProgramFiles=")+getenv("ProgramFiles") );
/* NB: putty absolutely needs SystemRoot env. var. */
    env.push_back( QString("SystemRoot=")+getenv("SystemRoot") );
    env.push_back( QString("TEMP=")+getenv("TEMP") );
    env.push_back( QString("USERNAME=")+getenv("USERNAME") );
    env.push_back( QString("USERPROFILE=")+getenv("USERPROFILE") );

    env.push_back( QString("HOME=")+getenv("HOMEPATH") );
    env.push_back( QString("USER=")+getenv("USERNAME") );
#else
    env.push_back( QString("HOME=")+getenv("HOME") );
    env.push_back( QString("USER=")+getenv("USER") );
#endif

    env.push_back( QString("TMP=")+getenv("TMP") );
    env.push_back( QString("PATH=")+getenv("PATH") );
    env.push_back( QString("SSH_AUTH_SOCK=")+getenv("SSH_AUTH_SOCK") );

//    emit printStdout_sign( tr("Running command %1\n").arg(cmd) );

    if ( ! proc->start(&env))
    {
        emit printStdout_sign( tr("Failed to start ssh") );
        return;
    }

    if (fwbdebug)
        qDebug("SSHSession::startSession   started child process pid=%ld",
               proc->processIdentifier());


    logged_in = false;
    enable    = false;
    configure = false;
    state = NONE;
}

SSHSession::~SSHSession()
{
    terminate();
}

/*
 * this is redundant and wrong. Should just copy a pointer to instConf
 * object and use that instead of making local copy of each flag.
 */
void SSHSession::setOptions(instConf *cnf)
{
    setQuiet(cnf->quiet);
    setVerbose(cnf->verbose);
    setBackup(cnf->backup);
    setBackupFile(cnf->backup_file);
    setIncr(cnf->incremental);
    setDryRun(cnf->dry_run);
    setSaveStandby(cnf->saveStandby);
    setTestRun(cnf->testRun);
    setStripComments(cnf->stripComments);
    setWDir(cnf->wdir);
    setConfFile(cnf->conffile);
    setSaveDiff(cnf->save_diff);
    setDiffPgm(cnf->diff_pgm);
    setDiffFile(cnf->diff_file);
}

void SSHSession::terminate()
{
    if (fwbdebug)
        qDebug("SSHSession::terminate     this=%p  proc=%p   heartBeatTimer=%p",
               this,proc,heartBeatTimer);

    heartBeatTimer->stop();

    if (proc!=NULL)
    {
        disconnect(proc,SIGNAL(readyReadStdout()),
                   this,SLOT(readFromStdout() ) );
        disconnect(proc,SIGNAL(readyReadStderr()),
                   this,SLOT(readFromStderr() ) );
        disconnect(proc,SIGNAL(processExited()),
                   this,SLOT(processExited() ) );

        if (fwbdebug)
            qDebug("SSHSession::terminate   terminating child process pid=%ld",
                   proc->processIdentifier());
#ifdef _WIN32
        if (proc->processIdentifier() != NULL)
#else
        if (proc->processIdentifier() != -1)
#endif
        {
            // process is stll alive, killing
            QString s=QString(proc->readStdout());
            if (!quiet)
            {
                s.replace('\r',"");    
                emit printStdout_sign(s);
            }
            proc->kill();
            delete proc;
            proc=NULL;
            retcode=-1;
            processExited();
        };
    }
}

void SSHSession::stateMachine()
{
}

/*
 * signal wroteToStdin is connected to slot readyToSend. Can not send
 * next line in this slot because on win32 it emits the signal and
 * thus calls the same slot recursively, without exiting first. On
 * Linux and Mac it seems to exit and then emit the signal and call
 * slot on the next pass of the even loop. Since on win32 this does
 * not happen, need to schedule sending next line via single shot
 * timer instead of calling it directly.
 */
void SSHSession::readyToSend()
{
    QTimer::singleShot( 0, this, SLOT(sendLine()) );
}

void SSHSession::sendLine()
{
    int n=0;
    while (input.size()!=0 && n<10)
    {
        string s = input.front();
        s = s + "\n";
        if (fwbdebug)
            qDebug("SSHSession::sendLine : %d lines to go -- %s",input.size(),s.c_str());
        input.pop_front();

        stdoutBuffer="";

/* it is important that we use writeToStdin(QByteArray &) rather than
 * writeToStdin(QString &) because the latter performs implicit
 * conversion into local locale assuming the string is in Unicode. The
 * string in our case is actually in whatever encoding the firewall
 * script is written to the local filesystem, which may or may not be
 * UTF-8 but is definitely not Unicode. The conversion not only breaks
 * comments that were entered in UTF-8, it makes QProcess miscalculate
 * number of characters in comment lines using UTF-8 which in turns
 * breaks the script even worse because it glues consequitive lines
 * together. Apparently this has been fixed in latest versions of QT
 * 3.x but this is still broken in QT 3.1 which is shipping with
 * RedHat 9 and some other still popular distributions. Since we need
 * to support old QT 3.x, the code must work around this problem.
 */
        QByteArray buf;
        buf.duplicate( s.c_str(), s.length() );
        proc->writeToStdin(buf);

        n++;
    }
    emit updateProgressBar_sign(input.size(),false);
        
    if (input.size()==0)
    {
        if (fwbdebug) qDebug("SSHUnx::sendLine - entire file sent, closeStdin=%d",
                             closeStdin);
        endOfCopy = true;
    }
}

void SSHSession::allDataSent()
{
    if (fwbdebug)
        qDebug("SSHSession::allDataSent   closing stdin");

    disconnect(proc,SIGNAL(wroteToStdin()),this,SLOT(readyToSend()));

#ifdef _WIN32
    Sleep(2000);
#endif
    proc->closeStdin();
#ifdef _WIN32
    Sleep(1000);
#endif
    readFromStdout();
}

void SSHSession::heartBeat()
{
    if (fwbdebug) qDebug("SSHSession::heartBeat");
    readFromStderr();
    readFromStdout();
    if (endOfCopy && closeStdin) 
    {
        allDataSent();
        endOfCopy = false;
    }
}

void SSHSession::readFromStdout()
{
    if (proc)
    {
        QByteArray ba = proc->readStdout();
        if (ba.size()!=0)
        {
            if (fwbdebug)
                qDebug("SSHSession::readFromStdout read %d bytes from child process",ba.size());
            QString s=QString(ba);
            stdoutBuffer=stdoutBuffer + s;

            if (!quiet)
            {
                s.replace('\r',"");    
                emit printStdout_sign(s);
            }

/* state machine operates on stdoutBuffer directly */
            stateMachine();
        }
    }
}

void SSHSession::readFromStderr()
{
    if (proc)
    {
        QByteArray ba = proc->readStderr();
        if (ba.size()!=0)
        {
            QString s=QString(ba);
            emit printStdout_sign(s);
            stderrBuffer=stderrBuffer + QString(s);
        }
    }
}

void SSHSession::processExited()
{
    if (proc) retcode=proc->exitStatus();

    QString exitStatus = (retcode)?QObject::tr("ERROR"):QObject::tr("OK");
    
    emit printStdout_sign(tr("SSH session terminated, exit status: %1").arg(retcode));
    if (retcode) error=true;
    emit sessionFinished_sign();
}

bool SSHSession::cmpPrompt(const QString &str,const QString &prompt)
{
    if (fwbdebug)
        qDebug("SSHSession::cmpPrompt: str='%s' prompt='%s'",
               str.ascii(),prompt.ascii());

    if (str.isEmpty()) return false;

    bool res=(str.findRev(prompt,-1)!=-1);
    if (!res)
    {
        QString s=str.stripWhiteSpace();
        res=(s.findRev(prompt,-1)!=-1);
    }

    if (fwbdebug)
        qDebug("SSHSession::cmpPrompt: res=%d",res);

    return res;
}

bool SSHSession::cmpPrompt(const QString &str,const QRegExp &prompt)
{
    if (fwbdebug)
        qDebug("SSHSession::cmpPrompt: str='%s' prompt='%s' (regexp)",
               str.ascii(),prompt.pattern().ascii());

    if (str.isEmpty()) return false;

    bool res=(str.findRev(prompt,-1)!=-1);

    if (fwbdebug)
        qDebug("SSHSession::cmpPrompt: res=%d",res);

    return res;
}



