/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: newHostDialog.cpp,v 1.11 2006/08/09 08:21:07 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"
#include "platforms.h"

#include "newHostDialog.h"
#include "InterfaceData.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"
#include "ObjConflictResolutionDialog.h"
#include "upgradePredicate.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Resources.h"
#include "fwbuilder/Policy.h"
#include "fwbuilder/InterfacePolicy.h"
#include "fwbuilder/BackgroundOp.h"

#include <qlineedit.h>
#include <qtextedit.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qlistview.h>
#include <qtextbrowser.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qlistbox.h>
#include <qregexp.h>
#include <qapplication.h>
#include <qcursor.h>

#include <iostream>

// must be the last for win
#include "fwbuilder/snmp.h"

//#undef HAVE_LIBSNMP

using namespace libfwbuilder;
using namespace std;

#define OBJECT_NAME_PAGE 0
#define SNMP_PAGE        1
#define MANUAL_PAGE      2
#define TEMPLATES_PAGE   3

newHostDialog::newHostDialog() : newHostDialog_q()
{
    nhst=NULL; 
    tmpldb = NULL;
    snmpPollCompleted=false;
    q=NULL;
    unloadTemplatesLib = false;
    getInterfacesBusy = false;

    timer = new QTimer(this);
    connect( timer, SIGNAL(timeout()), this, SLOT(monitor()) );

    setNextEnabled( QWizard::page(OBJECT_NAME_PAGE), false );
    for (int i=0; i<pageCount(); ++i)
        setHelpEnabled( QWizard::page(i), false );

    iface_list->setItemMargin( 1 );
    iface_list->setAllColumnsShowFocus( true );

    obj_name->setFocus();
}

newHostDialog::~newHostDialog()
{
    if (timer!=NULL) delete timer;
#ifdef HAVE_LIBSNMP
    if (q!=NULL) delete q;
#endif
}

void newHostDialog::changed()
{
    int p = indexOf( currentPage() );
    if (p==OBJECT_NAME_PAGE)
    {
        setNextEnabled( QWizard::page(p), !obj_name->text().isEmpty() );
    }

    if (p==SNMP_PAGE)
    {

        bool f;

#ifdef HAVE_LIBSNMP
        f = use_snmp->isChecked();
#else
        f = false;
        use_snmp->setEnabled( f );
#endif

        snmp_community->setEnabled( f );
        snmpQuery->setEnabled( f );
        snmpProgress->setEnabled( f );
        if (f) snmp_community->setFocus();

        f = use_manual->isChecked() || snmpPollCompleted;
        setNextEnabled( QWizard::page(SNMP_PAGE), f );
    }

    if (p==MANUAL_PAGE)
    {
        if (iface_dyn->isChecked() || iface_unnum->isChecked())
        {
            iface_addr->clear();
            iface_addr->setEnabled(false);
            iface_netmask->clear();
            iface_netmask->setEnabled(false);
        } else
        {
            iface_addr->setEnabled(true);
            iface_netmask->setEnabled(true);
        }
    }
}

void  newHostDialog::monitor()
{
    if (logger==NULL || q==NULL) return;

#ifdef HAVE_LIBSNMP

    if( logger->ready() )
    {
        QString str = logger->getLine().c_str();
        snmpProgress->moveCursor( QTextEdit::MoveEnd , false );
        snmpProgress->insert( str );
        return;
    }

    if (q->isRunning()) return;

    timer->stop();

    const map<int, Interface> &intf = q->getInterfaces();
    for(map<int, Interface>::const_iterator i=intf.begin();i!=intf.end(); ++i)
    {
        if ( i->second.isUp() ) 
        {
            InterfaceData idata( i->second );

            idata.guessLabel("");

            QString dn;
            if (idata.isDyn)        dn+="dyn";
            if (idata.isUnnumbered) dn+="unn";

            new QListViewItem(iface_list, 
                              idata.name.c_str(),
                              idata.label.c_str(),
                              idata.address.c_str(),
                              idata.netmask.c_str(),
                              dn,
                              idata.physicalAddress.c_str() );

//            cerr << "Added interface " << idata.name << endl;

        }
    }

    delete q;
    q=NULL;

#endif

    snmpPollCompleted=true;
    setNextEnabled( QWizard::page(SNMP_PAGE), true );
}

void newHostDialog::getInterfacesViaSNMP()
{
#ifdef HAVE_LIBSNMP

// need to protect from reentry because getAddrByName processes events
    if (q!=NULL || getInterfacesBusy) return;

    snmpPollCompleted=false;
    iface_list->clear();

    string rcomm=snmp_community->text().latin1();

    if ( rcomm.empty() ) 
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Missing SNMP community string."),
            "&Continue", QString::null, QString::null, 0, 1 );
        return ;
    }

    getInterfacesBusy = true;

    IPAddress addr;
    QString name=obj_name->text().latin1();
    try 
    {
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );
        QString a = getAddrByName(name);
        QApplication::restoreOverrideCursor();
        addr = a.ascii();
    } catch (FWException &ex)
    {
        QMessageBox::warning(
            this,"Firewall Builder", 
            tr("Address of %1 could not be obtained via DNS")
            .arg(obj_name->text()),
            "&Continue", QString::null, QString::null, 0, 1 );
        getInterfacesBusy = false;
        return ;
    }

    logger=NULL;
    snmpProgress->clear();

    if (q!=NULL) delete q;
    q=new SNMP_interface_query();
    q->init(addr.toString(),rcomm,SNMP_DEFAULT_RETRIES,SNMP_DEFAULT_TIMEOUT);

    timer->start(0, false);
    
    try
    {
        logger = q->start_operation();

    } catch(const FWException &ex)
    {
        //do nothing
    }

    getInterfacesBusy = false;

#endif
}

bool newHostDialog::appropriate(QWidget *page) const
{
    int p  = indexOf( page );

    if (fwbdebug)
    {
        qDebug("newHostDialog::appropriate  p=%d",p);
    }

    switch (p)
    {
    case OBJECT_NAME_PAGE:
    case TEMPLATES_PAGE:
        return true;

    case SNMP_PAGE:
    case MANUAL_PAGE:
        return (!useTemplate->isChecked());
    }
    return true;
}


void newHostDialog::selected(const QString &title)
{
    int p = indexOf( currentPage() );

    if (fwbdebug) qDebug("newHostDialog::selected  p=%d",p);

// p is a page number _after_ it changed

    switch (p)
    {
    case SNMP_PAGE:
        changed();  // to properly enable/disable widgets
        break;

    case MANUAL_PAGE:
    {
        iface_name->setFocus();

        setNextEnabled( QWizard::page(MANUAL_PAGE), false );
        setFinishEnabled( QWizard::page(MANUAL_PAGE), true );
        break;
    }

    case TEMPLATES_PAGE:
    {
        setFinishEnabled( QWizard::page(TEMPLATES_PAGE), true );
/* load templates if not loaded */

        if (tmpldb==NULL)
        {

            MessageBoxUpgradePredicate upgrade_predicate(this);

            tmpldb = new FWObjectDatabase();
            tmpldb->setReadOnly( false );
            tmpldb->load( tempfname, &upgrade_predicate, librespath);
        }
        FWObject *tlib = tmpldb->getById(TEMPLATE_LIB);

#if 0
        FWObject *tlib = mw->db()->getById(TEMPLATE_LIB);
        if (tlib==NULL)
        {
            FWObject *cl = om->getCurrentLib();
            mw->loadLibrary(tempfname);
            unloadTemplatesLib = true;
            om->loadObjects();
            tlib = mw->db()->getById(TEMPLATE_LIB);
/* restore library that was opened prior loading templates */
            om->openLib(cl);
        }
#endif

        list<FWObject*> fl;
        findHosts(tlib, fl, false);

        QString icn_filename = QString( Resources::global_res->getObjResourceStr(fl.front(), "icon-tree").c_str() );

        templateList->clear();

        int n=0;
        for (list<FWObject*>::iterator m=fl.begin(); m!=fl.end(); m++,n++)
        {
            FWObject *o=*m;
            templateList->insertItem(QPixmap::fromMimeSource( icn_filename ),
                                     o->getName().c_str() );
            templates[ templateList->item( templateList->count()-1 ) ]=o;
        }
        templateList->setCurrentItem(0);
        templateList->setFocus();
        break;
    }
    }
}

void newHostDialog::templateSelected(QListBoxItem *itm)
{
    if (fwbdebug) qDebug("newHostDialog::templateSelected ");

    FWObject *o=templates[itm];
    assert (o!=NULL);

    Host *fw = Host::cast(o);

    templateComment->clear();
    templateComment->append( fw->getComment().c_str() );
    templateComment->setCursorPosition(0,0);

    bool haveOutside = false;
    bool haveInside  = false;
    bool haveDMZ     = false;
    list<FWObject*> ll = fw->getByType(Interface::TYPENAME);
    for (FWObject::iterator i=ll.begin(); i!=ll.end(); i++)
    {
        Interface *intf = Interface::cast( *i );
        QString     nam = intf->getName().c_str();
        QString     lbl = intf->getLabel().c_str();

        if (lbl=="outside"                ||
            nam.find(QRegExp(".*0$"))!=-1 ||
            nam.find(QRegExp(".*0/0$"))!=-1 )
        {
            haveOutside=true;
            intfOutsideLine->show();
            intfOutsideText->show();
            fillInterfaceData(intf,intfOutsideText);
        }
        if (lbl=="inside"                 ||
            nam.find(QRegExp(".*1$"))!=-1 ||
            nam.find(QRegExp(".*0/1$"))!=-1 )
        {
            haveInside=true;
            intfInsideLine->show();
            intfInsideText->show();
            fillInterfaceData(intf,intfInsideText);
        }
    }

    if (!haveOutside) { intfOutsideLine->hide(); intfOutsideText->hide(); }
    if (!haveInside)  { intfInsideLine->hide();  intfInsideText->hide();  }
    if (!haveDMZ)     { intfDMZLine->hide();     intfDMZText->hide();     }
}

void newHostDialog::fillInterfaceData(Interface *intf, QTextBrowser *qte)
{
    qte->clear();
    QString s;

    s += "<table border='0' cellspacing='0' cellpadding='0'>";

    s += "<tr>";
    s += "<td>";
    s +=  tr("Interface: %1 (%2)")
        .arg(intf->getName().c_str())
        .arg(intf->getLabel().c_str());
    s += "</td>";
    s += "</tr>";

    s += "<tr>";
    s += "<td>";
    if (intf->isDyn()) s +=  tr("Dynamic address");
    else
        if (intf->isUnnumbered()) s +=  tr("Unnumbered interface");
        else
            s += QString("%1/%2")
                .arg(intf->getAddress().toString().c_str())
                .arg( intf->getNetmask().toString().c_str());
    s += "</td>";
    s += "</tr>";
    s += "</table>";
    qte->setText(s);
}

void newHostDialog::addInterface()
{
    QString dn = "";
    if (iface_dyn->isChecked())   dn+="Dyn";
    if (iface_unnum->isChecked()) dn+="Unn";

    QString addr;
    QString netm;

    if (!iface_dyn->isChecked() && !iface_unnum->isChecked())
    {
        addr = iface_addr->text();
        netm = iface_netmask->text();

        if (addr.isEmpty()) addr="0.0.0.0";
        if (netm.isEmpty()) netm="0.0.0.0";

        try
        {
            IPAddress(addr.latin1());
            Netmask(netm.latin1());
        }
        catch (FWException &ex)
        {
            QMessageBox::warning(
                this,"Firewall Builder", 
                tr("Illegal address '%1/%2'").arg(addr).arg(netm),
                "&Continue", QString::null, QString::null, 0, 1 );
            return;
        }
    }
    new QListViewItem(iface_list, 
                      iface_name->text(),
                      iface_label->text(),
                      addr,
                      netm,
                      dn,
                      iface_physaddr->text() );
}

void newHostDialog::selectedInterface(QListViewItem *itm)
{
    iface_name->setText( itm->text(0) );
    iface_label->setText( itm->text(1) );
    iface_addr->setText( itm->text(2) );
    iface_netmask->setText( itm->text(3) );
    iface_dyn->setChecked( itm->text(4).find("Dyn")!=-1 );
    iface_unnum->setChecked( itm->text(4).find("Unn")!=-1 );
    iface_physaddr->setText( itm->text(5) );
}

void newHostDialog::updateInterface()
{
    QString dn = "";
    if (iface_dyn->isChecked())   dn+="Dyn";
    if (iface_unnum->isChecked()) dn+="Unn";

    QListViewItem *itm = iface_list->selectedItem();
    if (itm==NULL) return;

    itm->setText( 0 , iface_name->text() );
    itm->setText( 1 , iface_label->text() );
    itm->setText( 2 , iface_addr->text() );
    itm->setText( 3 , iface_netmask->text() );
    itm->setText( 4 , dn );
    itm->setText( 5 , iface_physaddr->text() );
}

void newHostDialog::deleteInterface()
{
    QListViewItem *itm = iface_list->selectedItem();
    if (itm==NULL) return;
    iface_list->takeItem( itm );
}

void newHostDialog::accept()
{
    int p = indexOf( currentPage() );
    
    if (p==TEMPLATES_PAGE)
    {
        QListBoxItem *itm = templateList->item( templateList->currentItem() );
        FWObject *o=templates[itm];
        assert (o!=NULL);

        FWObject *no = om->duplicateObject(om->getCurrentLib(),
                                           o,
                                           obj_name->text(),
                                           false );  // do not ask to autorename
        if (no==NULL)
        {
          QDialog::accept();
          return;
        }
        nhst=Host::cast(no);
    } else
    {
        FWObject *o;
        o=om->createObject(Host::TYPENAME, obj_name->text() );
        if (o==NULL)
        {
          QDialog::accept();
          return;
        }

        nhst=Host::cast(o);

/* create interfaces */

        QListViewItem *itm = iface_list->firstChild();

        while (itm!=NULL)
        {
            QString name    =  itm->text(0);
            QString label   =  itm->text(1);
            QString addr    =  itm->text(2);
            QString netmask =  itm->text(3);
            bool    dyn     =  itm->text(4).find("Dyn")!=-1;
            bool    unnum   =  itm->text(4).find("Unn")!=-1;
            QString physaddr=  itm->text(5);

            Interface *oi = Interface::cast(
                om->createObject(nhst,Interface::TYPENAME, name)
            );
#ifdef USE_INTERFACE_POLICY
            oi->add(new InterfacePolicy());
#endif
            oi->setLabel( label.latin1() );

            if (dyn)   oi->setDyn(true);
            if (unnum) oi->setUnnumbered(true);
            oi->setSecurityLevel(0);

            if (!dyn && !unnum)
            {
                QString addrname=QString("%1:%2:ip")
                    .arg(obj_name->text()).arg(name);
                IPv4 *oa = IPv4::cast(
                    om->createObject(oi, IPv4::TYPENAME,addrname)
                );
                oa->setAddress( addr.latin1()    );
                oa->setNetmask( netmask.latin1() );
            }

            om->updateObjName(oi,"","",false);

            itm=itm->itemBelow();
        }
    }
    if (unloadTemplatesLib)
    {
#if 0
        FWObject *tlib = mw->db()->getById(TEMPLATE_LIB);
        assert (tlib!=NULL);
        string tlibID = tlib->getId();
        if (fwbdebug) qDebug("  Delete library of templates");
        om->delObj(tlib,false);

/*
 * deleting an object places it in the "Deleted objects" library, so
 * we need to remove "templates" library from there.
 *
 * TODO: need to add flags to the API to be able to delete objects
 * without placing them in "Deleted objects" automatically
 */
        FWObject *delObjLib = mw->db()->getById( DELETED_LIB );
        if (delObjLib!=NULL && delObjLib->getById(tlibID)!=NULL)
        {
            if (fwbdebug) qDebug("  Delete library of templates from 'Deleted objects'");
            om->delObj(tlib,false);  // this time from deleted objects lib
        }
#endif

        delete tmpldb;
        tmpldb = NULL;

        unloadTemplatesLib=false;
    }
    QDialog::accept();
}



