/* Handler.java */

/* 
 * Copyright (C) 1996-97 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.PipedReader;
import java.io.PipedWriter;
import java.io.Reader;
import java.io.Writer;
import java.net.Socket;
import java.net.URL;

/**
 * @author Mark Boyns
 */
class Handler extends Thread
{
    Monitor monitor = null;
    FilterManager manager = null;
    Options options = null;
    Client client = null;
    Socket socket = null;
    Request request = null;
    Reply reply = null;
    HttpRelay http = null;
    int currentLength = 0;
    int contentLength = 0;
    Filter filterList[];

    Handler (ThreadGroup t, Runnable r, Monitor m, FilterManager manager, Options options)
    {
	super (t, r);
	this.monitor = m;
	this.manager = manager;
	this.options = options;
    }

    void doit (Socket s)
    {
	socket = s;
	start ();
    }

    synchronized void close ()
    {
	if (client != null)
	{
	    client.close ();
	}
	if (http != null)
	{
	    http.close ();
	}
    }

    public void run ()
    {
	filterList = manager.createFilters ();
	
	try
	{
	    client = new Client (socket);
	    request = client.read ();
	    filter (request);
	    monitor.register (this);

	    if (options.useProxy ())
	    {
		http = new Proxy (options.getString ("muffin.httpProxyHost"),
				  options.getInteger ("muffin.httpProxyPort"));
	    }
	    else
	    {
		http = new Http (new URL (request.getURL ()));
	    }
				   
	    http.write (request);
	    monitor.update (this);

	    reply = http.read ();
	    filter (reply);
	    monitor.update (this);

	    /* Send the reply headers */
	    client.write (reply);

	    try 
	    {
		contentLength = Integer.parseInt (reply.getHeaderField ("Content-length"));
	    }
	    catch (Exception e)
	    {
		contentLength = 0;
	    }
	    currentLength = 0;

	    filter (http.getReader (), client.getWriter ());
	}
	catch (Exception e)
	{
	    error (client.getWriter (), e, request);
	    //e.printStackTrace ();
	}
	
	close ();
	
	monitor.unregister (this);
    }

    void filter (Reply r)
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ReplyFilter)
	    {
		((ReplyFilter)(filterList[i])).filter (r);
	    }
	}
    }

    void filter (Request r)
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof RequestFilter)
	    {
		((RequestFilter)(filterList[i])).filter (r);
	    }
	}
    }

    void filter (Reader in, Writer out) throws java.io.IOException
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ContentFilter)
	    {
		ContentFilter filter = (ContentFilter) filterList[i];
		if (filter.needsFiltration (request, reply))
		{
		    /* Create the IO pipe */
		    PipedWriter pw = new PipedWriter ();
		    PipedReader pr = new PipedReader (pw);

		    filter.setReader (new BufferedReader (in));
		    filter.setWriter (new BufferedWriter (pw));
		
		    new Thread (filter).start ();

		    in = pr;
		}
	    }
	}

	copy (in, out);
    }

    public int getTotalBytes ()
    {
	return contentLength > 0 ? contentLength : 0;
    }

    public int getCurrentBytes ()
    {
	return currentLength;
    }

    public String toString ()
    {
	StringBuffer str = new StringBuffer ();
	if (contentLength > 0)
	{
	    str.append ((int)(((float)currentLength/contentLength)*100));
	    str.append ("% of ");
	    str.append (contentLength/1024);
	    str.append ("k");
	}
	else
	{
	    str.append (currentLength/1024);
	    str.append ("k");
	}
	str.append (" ");
	str.append (request.getURL ());
	
	return str.toString ();
    }

    void error (BufferedWriter out, Exception e, Request r)
    {
	StringBuffer buf = new StringBuffer ();
	buf.append ("While trying to retrieve the URL: <a href=\""+r.getURL()+"\">"+r.getURL()+"</a>\r\n");
	buf.append ("<p>\r\nThe following error was encountered:\r\n<p>\r\n");
	buf.append ("<ul><li>" + e.toString () + "</ul>\r\n");
	String s = (new HttpError (400, buf.toString ())).toString ();
	try
	{
	    out.write (s, 0, s.length ());
	    out.flush ();
	}
	catch (Exception ex)
	{
	}
    }

    void copy (Reader in, Writer out) throws java.io.IOException
    {
	int n;
	char buffer[] = new char[8192];

	while ((n = in.read (buffer, 0, 8192)) > 0)
	{
	    currentLength += n;
	    out.write (buffer, 0, n);
	    monitor.update (this);
	}
	out.flush ();
    }
}


