/* Main.java */

/* 
 * Copyright (C) 1996-97 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.awt.Button;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Event;
import java.awt.Font;
import java.awt.Frame;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Menu;
import java.awt.MenuBar;
import java.awt.MenuItem;
import java.awt.Panel;
import java.awt.Label;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.WindowListener;
import java.awt.event.WindowEvent;
import java.net.InetAddress;

/**
 * Startup interface to Muffin.  Parses command line options, loads user
 * preferences, and creates the GUI interface if necessary.
 *
 * @author Mark Boyns
 */
public class Main extends Frame implements ActionListener, WindowListener
{
    final static String version = "0.5";
    static Options options;
    static UserPrefs userPrefs;
    String localhost;
    Monitor monitor;
    Server server;
    FilterManager manager;
    String infoString;
    Button suspendButton;

    /**
     * Create Main.
     */
    public Main ()
    {
	super ("Muffin");

	infoString = new String ("Muffin " + options.getString ("muffin.version") +
				 " running on " + options.getString ("muffin.host") +
				 " port " + options.getString ("muffin.port"));
	System.out.println (infoString);
	    
	manager = new FilterManager (userPrefs);

	if (options.getBoolean ("muffin.noWindow"))
	{
	    TextMonitor tm = new TextMonitor (infoString);
	    tm.draw ();
	    monitor = tm;
	}
	else
	{
	    monitor = new CanvasMonitor ();
	    gui ();
	}
	
	server = new Server (options.getInteger ("muffin.port"),
			     monitor, manager, options);
	server.run ();
    }

    /**
     * Initialize the GUI interface.
     */
    void gui ()
    {
	setBackground (Color.lightGray);
	setFont (new Font ("Helvetica", Font.BOLD, 12));

	MenuBar bar = new MenuBar ();
	Menu menu = new Menu ("File");
	menu.setFont (new Font ("Helvetica", Font.BOLD, 12));
	MenuItem item = new MenuItem ("Quit");
	item.setActionCommand ("doQuit");
	item.addActionListener (this);
	item.setFont (new Font ("Helvetica", Font.BOLD, 12));
	menu.add (item);
	bar.add (menu);

	menu = new Menu ("Help");
	menu.setFont (new Font ("Helvetica", Font.BOLD, 12));
	item = new MenuItem ("About Muffin...");
	item.setActionCommand ("doAbout");
	item.addActionListener (this);
	item.setFont (new Font ("Helvetica", Font.BOLD, 12));
	menu.add (item);
	bar.setHelpMenu (menu);
	
	setMenuBar (bar);

	if (monitor instanceof Canvas)
	{
	    Canvas canvas = (Canvas) monitor;
	    add ("Center", canvas);
	}

	GridBagLayout layout = new GridBagLayout ();
	Panel controlPanel = new Panel ();
	controlPanel.setLayout (layout);

	GridBagConstraints c = new GridBagConstraints ();
	c.anchor = GridBagConstraints.NORTHWEST;
	c.insets = new Insets (2, 5, 2, 5);
	c.weightx = 1.0;

	Button b;
	b = new Button ("Filters...");
	b.setActionCommand ("doFilters");
	b.addActionListener (this);
	layout.setConstraints (b, c);
	controlPanel.add (b);
	b = new Button ("Options...");
	b.setActionCommand ("doOptions");
	b.addActionListener (this);
	layout.setConstraints (b, c);
	controlPanel.add (b);
	suspendButton = new Button ("Suspend");
	suspendButton.setActionCommand ("doSuspend");
	suspendButton.addActionListener (this);
	layout.setConstraints (suspendButton, c);
	controlPanel.add (suspendButton);
	b = new Button ("Stop");
	b.setActionCommand ("doStop");
	b.addActionListener (this);
	layout.setConstraints (b, c);
	controlPanel.add (b);
	
	Icon icon = new Icon (options);
	c.anchor = GridBagConstraints.EAST;
	c.gridwidth = GridBagConstraints.REMAINDER;
	layout.setConstraints (icon, c);
	controlPanel.add (icon);
	
	add ("North", controlPanel);

	Label l = new Label (infoString);
	l.setFont (new Font ("Helvetica", Font.PLAIN, 10));
	add ("South", l);

	addWindowListener (this);

	pack ();
	setSize (getPreferredSize ());
	show ();
    }

    /**
     * Handle Button events.
     *
     * @param event some event.
     */
    public void actionPerformed (ActionEvent event)
    {
	String arg = event.getActionCommand ();
	
	if ("doQuit".equals (arg))
	{
	    closeApplication ();
	}
	else if ("doAbout".equals (arg))
	{
	    new About (options);
	}
	else if ("doStop".equals (arg))
	{
	    server.stop ();
	}
	else if ("doSuspend".equals (arg))
	{
 	    suspendButton.setLabel ("Resume");
	    suspendButton.setActionCommand ("doResume");
	    server.suspend ();
	    monitor.suspend ();
	}
	else if ("doResume".equals (arg))
	{
 	    suspendButton.setLabel ("Suspend");
	    suspendButton.setActionCommand ("doSuspend");
	    server.resume ();
	    monitor.resume ();
	}
	else if ("doFilters".equals (arg))
	{
	    manager.createFrame ();
	}
	else if ("doOptions".equals (arg))
	{
	    options.createFrame ();
	}
    }

    public void windowActivated (WindowEvent e)
    {
    }
  
    public void windowDeactivated (WindowEvent e)
    {
    }
  
    public void windowClosing (WindowEvent e)
    {
	closeApplication ();
    }
  
    public void windowClosed (WindowEvent e)
    {
    }
  
    public void windowIconified (WindowEvent e)
    {
    }
  
    public void windowDeiconified (WindowEvent e)
    {
    }
  
    public void windowOpened (WindowEvent e)
    {
    }

    void closeApplication ()
    {
	setVisible (false);
	System.exit (0);
    }

    static void systemInfo ()
    {
	System.out.println ("Muffin:");
	System.out.println ("-------");
	System.out.println ("muffin.version " + version);
	System.out.println ();
	System.out.println ("Java Virtual Machine:");
	System.out.println ("---------------------");
    	System.out.println ("java.version " + System.getProperty ("java.version"));
	System.out.println ("java.class.version " + System.getProperty ("java.class.version"));
	System.out.println ("java.class.path " + System.getProperty ("java.class.path"));
	System.out.println ("java.home " + System.getProperty ("java.home"));
	System.out.println ("java.vendor " + System.getProperty ("java.vendor"));
	System.out.println ();
	System.out.println ("Operating System:");
	System.out.println ("-----------------");
	System.out.println ("os.version " + System.getProperty ("os.version"));
	System.out.println ("os.arch " + System.getProperty ("os.arch"));
	System.out.println ("os.name " + System.getProperty ("os.name"));
	System.out.println ();
	System.out.println ("User:");
	System.out.println ("-----");
	System.out.println ("user.name " + System.getProperty ("user.name"));
	System.out.println ("user.dir " + System.getProperty ("user.dir"));
	System.out.println ("user.home " + System.getProperty ("user.home"));
    }

    public static void main (String argv[])
    {
	Prefs args = new Prefs ();
	for (int i = 0; i < argv.length; i++) 
	{
	    if (argv[i].equalsIgnoreCase ("-port")) 
	    {
		try 
		{
		    args.putInteger ("port", Integer.parseInt (argv[++i]));
		}
		catch (Exception e)
		{
		    System.out.println ("invalid port: " + argv[i]);
		    System.exit (1) ;
		}
	    }
	    else if (argv[i].equalsIgnoreCase ("-dir"))
	    {
		args.putString ("dir", argv[++i]);
	    }
	    else if (argv[i].equalsIgnoreCase ("-httpProxyHost"))
	    {
		args.putString ("httpProxyHost", argv[++i]);
	    }
	    else if (argv[i].equalsIgnoreCase ("-httpProxyPort"))
	    {
		try
		{
		    args.putInteger ("httpProxyPort", Integer.parseInt (argv[++i]));
		}
		catch (Exception e)
		{
		    System.out.println ("invalid httpProxyPort: " + argv[i]);
		    System.exit (1) ;
		}
	    }
	    else if (argv[i].equalsIgnoreCase ("-nw"))
	    {
		args.putBoolean ("noWindow", true);
	    }
	    else if (argv[i].equalsIgnoreCase ("-v")
		     || argv[i].equalsIgnoreCase ("-version")
		     || argv[i].equalsIgnoreCase ("--version"))
	    {
		systemInfo ();
		System.exit (0);
	    }
	    else if (argv[i].equalsIgnoreCase ("-help")
		     || argv[i].equalsIgnoreCase ("--help")
		     || argv[i].equalsIgnoreCase ("-h"))
	    {
		System.out.println ("usage: java muffin.Main [options]\n\n"
				    + "-dir DIR              Preferences directory.\n"
				    + "-help                 This useful message.\n"
				    + "-httpProxyHost HOST   Use HOST as the HTTP proxy.\n"
				    + "-httpProxyPort PORT   Use PORT as the HTTP proxy port.\n"
				    + "-nw                   Don't create any windows.\n"
				    + "-port PORT            Listen on PORT for browser requests.\n"
				    + "-v                    Display muffin version.\n");
		System.exit (1);
	    }
	    else 
	    {
		System.out.println ("Unknown option: " + argv[i]);
		System.exit (1) ;
	    }
	}

	/* Create and load user preferences */
	userPrefs = new UserPrefs ();
	if (args.exists ("dir"))
	{
	    userPrefs.setUserDirectory (args.getString ("dir"));
	}
	userPrefs.load ();

	/* Create muffin run-time options */
	options = new Options (userPrefs);
	options.putString ("muffin.version", version);
	options.putString ("muffin.url", "http://muffin.doit.org/");
	try
	{
	    options.putString ("muffin.host", (InetAddress.getLocalHost ()).getHostName ());
	}
	catch (Exception e)
	{
	    options.putString ("muffin.host", "127.0.0.1");
	}
	if (args.exists ("port"))
	{
	    options.putInteger ("muffin.port", args.getInteger ("port"));
	}
	if (args.exists ("httpProxyHost"))
	{
	    options.putString ("muffin.httpProxyHost", args.getString ("httpProxyHost"));
	}
	if (args.exists ("httpProxyPort"))
	{
	    options.putInteger ("muffin.httpProxyPort", args.getInteger ("httpProxyPort"));
	}
	if (args.exists ("noWindow"))
	{
	    options.putBoolean ("muffin.noWindow", args.getBoolean ("noWindow"));
	}

	options.sync ();

	new Main ();
    }
}
