/* Prefs.java */

/* 
 * Copyright (C) 1997 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.io.File;
import sdsu.compare.StringIgnoreCaseComparer;
import sdsu.util.SortedList;

/**
 * @author Mark Boyns
 */
public class Prefs extends Hashtable
{
    String userDirectory = "Muffin";
    
    boolean allowOverride = true;
    
    public Prefs (int capacity)
    {
	super (capacity);
    }

    public Prefs ()
    {
	super ();
    }

    public void setUserDirectory (String dir)
    {
	userDirectory = dir;
    }

    public String getUserDirectory ()
    {
	if (userDirectory.startsWith (System.getProperty ("file.separator")))
	{
	    return userDirectory;
	}
	else
	{
	    return System.getProperty ("user.home") +
		System.getProperty ("file.separator") +
		userDirectory;
	}
    }

    void checkUserDirectory ()
    {
	File dir = new File (getUserDirectory ());
	if (!dir.exists ())
	{
	    System.out.println ("Creating " + getUserDirectory ());
	    if (!dir.mkdirs ())
	    {
		System.out.println ("Can't create " + getUserDirectory ());
	    }
	}
    }

    public String getUserFile (String file)
    {
	return getUserDirectory () +
	    System.getProperty ("file.separator") + file;
    }

    public boolean exists (Object key)
    {
	return containsKey (key);
    }

    public synchronized Object put (Object key, Object value)
    {
	if (allowOverride || !exists (key))
	{
	    return super.put (key, value);
	}
	return null;
    }

    public void setOverride (boolean o)
    {
	allowOverride = o;
    }

    public boolean getOverride ()
    {
	return allowOverride;
    }
    
    public void putString (String key, String value)
    {
	put (key, value);
    }

    public void putBoolean (String key, boolean value)
    {
	put (key, value ? "true" : "false");
    }

    public void putInteger (String key, int value)
    {
	put (key, Integer.toString (value));
    }

    public String getString (String key)
    {
	return (String) get (key);
    }

    public boolean getBoolean (String key)
    {
	String val = getString (key);
	if (val == null)
	{
	    return false;
	}
	return "true".equalsIgnoreCase (val) ? true : false;
    }

    public int getInteger (String key)
    {
	int val;
	try
	{
	    val = Integer.parseInt (getString (key));
	}
	catch (Exception e)
	{
	    val = -1;
	}
	return val;
    }

    public Prefs extract (String clazz)
    {
	Prefs prefs = new Prefs ();
	Enumeration e = keys ();
	while (e.hasMoreElements ())
	{
	    String key = (String) e.nextElement ();
	    if (key.startsWith (clazz + "."))
	    {
		prefs.put (key, get (key));
	    }
	}
	return prefs;
    }

    public void merge (Prefs prefs)
    {
	Enumeration e = prefs.keys ();
	while (e.hasMoreElements ())
	{
	    String key = (String) e.nextElement ();
	    put (key, prefs.get (key));
	}
    }

    public String[] getStringList (String key)
    {
	String tokens = getString (key);
	if (tokens == null)
	{
	    return new String[0];
	}
	StringTokenizer st = new StringTokenizer (tokens, ",");
	String list[] = new String[st.countTokens ()];
	for (int i = 0; st.hasMoreTokens (); i++)
	{
	    list[i] = new String ((String) st.nextToken ());
	}
	return list;
    }

    public void putStringList (String key, String list[])
    {
	StringBuffer buf = new StringBuffer ();
	for (int i = 0; i < list.length; i++)
	{
	    buf.append (list[i]);
	    if (i != list.length - 1)
	    {
		buf.append (",");
	    }
	}
	putString (key, buf.toString ());
    }

    public void print ()
    {
	Enumeration e = keys ();
	while (e.hasMoreElements ())
	{
	    String key = (String) e.nextElement ();
	    System.out.println (" * " + key + " = " + get (key));
	}
    }

    public synchronized Enumeration sortedKeys ()
    {
	Enumeration e = keys ();
	SortedList sorter = new SortedList (StringIgnoreCaseComparer.getInstance ());
	while (e.hasMoreElements ())
	{
	    sorter.addElement ((String) e.nextElement ());
	}
	return sorter.elements ();
    }
}
