/* DocumentInfo.java */

/* 
 * Copyright (C) 1997 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.filter;

import muffin.*;
import muffin.html.HtmlTokenizer;
import muffin.html.Tag;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.Date;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.text.ParsePosition;

public class DocumentInfo implements FilterFactory, ContentFilter
{
    FilterManager manager;
    Prefs prefs;
    Test factory;
    InputStream in = null;
    OutputStream out = null;
    Reply reply;
    Request request;

    public void setManager (FilterManager manager)
    {
	this.manager = manager;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
    }

    public Prefs getPrefs ()
    {
	return prefs;
    }

    public void viewPrefs ()
    {
	System.out.println ("DocumentInfo: no preferences");
    }
    
    public Filter createFilter ()
    {
	Filter f = new DocumentInfo ();
	f.setPrefs (prefs);
	return f;
    }

    public boolean needsFiltration (Request request, Reply reply)
    {
	this.request = request;
	this.reply = reply;
	
	String s = reply.getHeaderField ("Content-type");
	return s != null && s.startsWith ("text/html");
    }
    
    public void setInputStream (InputStream in)
    {
	this.in = in;
    }

    public void setOutputStream (OutputStream out)
    {
	this.out = out;
    }
    
    public void run ()
    {
	try
	{
	    HtmlTokenizer html = new HtmlTokenizer (in);
	    int token;
	    Tag tag;

            while ((token = html.getToken ()) != HtmlTokenizer.TT_EOF)
            {
		if (token == HtmlTokenizer.TT_TAG)
		{
		    tag = html.getTag ();
		    if (tag.is ("body"))
		    {
			html.writeTag (tag, out);
			
			StringBuffer buf = new StringBuffer ();
			buf.append ("<small>\n");
			buf.append ("URL: ");
			buf.append (request.getURL ());
			buf.append ("<br>\n");

			if (reply.containsHeaderField ("Server"))
			{
			    buf.append ("Server: ");
			    buf.append (reply.getHeaderField ("Server"));
			    buf.append ("<br>\n");
			}

			if (reply.containsHeaderField ("Last-Modified"))
			{
			    buf.append ("Last-Modified: ");
			    String str = reply.getHeaderField ("Last-Modified");
			    /* Try to put date in localtime */
			    try
			    {
				/*
				 * HTTP date formats:
				 * Sun, 06 Nov 1994 08:49:37 GMT  ; RFC 822, updated by RFC 1123
				 * Sunday, 06-Nov-94 08:49:37 GMT ; RFC 850, obsoleted by RFC 1036
				 * Sun Nov  6 08:49:37 1994       ; ANSI C's asctime() format
				 */
				SimpleDateFormat format = (SimpleDateFormat) DateFormat.getDateInstance ();
				switch (str.charAt (3))
				{
				case ',':
				    format.applyPattern ("EEE, dd MMM yyyy HH:mm:ss z");
				    break;

				case ' ':
				    format.applyPattern ("EEE MMM dd HH:mm:ss yyyy");
				    break;

				default:
				    format.applyPattern ("EEEE, dd-MMM-yyyy HH:mm:ss z");
				    break;
				}
				ParsePosition pos = new ParsePosition (0);
				Date date = format.parse (reply.getHeaderField ("Last-Modified"), pos);
				buf.append (format.format (date));
			    }
			    catch (Exception e)
			    {
				buf.append (str);
			    }
			    buf.append ("<br>\n");
			}
			buf.append ("</small>\n");
			out.write (buf.toString ().getBytes (), 0, buf.length ());

			/* Quit parsing when <body> is found. */
			break;
		    }
		    else
		    {
			html.writeTag (tag, out);
		    }
		}
		else
		{
		    html.writeToken (out);
                }
            }

	    byte buffer[] = new byte[8192];
	    int n;
	    while ((n = in.read (buffer, 0, buffer.length)) > 0)
	    {
		out.write (buffer, 0, n);
	    }
	    out.flush ();
	    out.close ();
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }
}
