/* This class is modified from Sun's Vector class, so the following
 * is required.
 */
/*
 * @(#)Vector.java	1.29 95/12/01  
 *
 * Copyright (c) 1994 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies. Please refer to the file "copyright.html"
 * for further important copyright and licensing information.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
 * THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR
 * ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */
 
package sdsu.util;

import java.util.Vector;
import java.util.NoSuchElementException;
import java.util.Enumeration;
import java.util.Random;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.StringBufferInputStream;

/**
 * A List object is a vector that can convert itself to a string and "recreate" itself from 
 * that string. The original List object can contain any objects, but
 * the recreated List object will only contain string representations of the original
 * elements in the list.
 * 
 * In a List object string representation (Losr) the list elements are
 * separated by a separatorChar, which defaults to ',', but can be changed. 
 * If the string representation of a list element contains a special character it is quoted. 
 * Special characters include separatorChar, a comment character, and
 * white space characters. See sdsu.util.TokenCharacters for default values. White space and
 * comments can be added to a Losr for readability. Comments start with a comment character and
 * continue upto and include the next '\n' character.
 *
 * @see		Stringizer
 * @see		SimpleTokenizer
 * @see		TokenCharacters
 * @version 2.0 4 June 1997 
 * @author Roger Whitney (<a href=mailto:whitney@cs.sdsu.edu>whitney@cs.sdsu.edu</a>)
 */

public class List implements OrderedCollection, Stringizable
	{
	/**
	 * The buffer where elements are stored.
	 */
	protected Object elementData[];

	/**
	 * The number of elements in the buffer.
	 */
	protected int elementCount = 0;

	/**
	 * The size of the increment. If it is 0 the size of the
	 * the buffer is doubled everytime it needs to grow.
	 */
	protected int capacityIncrement;


	private char separatorChar = ',';

	private TokenCharacters parseTable = new TokenCharacters(
		String.valueOf( separatorChar) );

	
	/**
	 * Create a new list
	 */
	public List()
		{
		this( 10 );
		}

	/**
	 * Create a list with initial given size
	 */
	public List( int initialSize)
		{
		this( initialSize, 0 );
		}

	/**
	 * Constructs an empty List with the specified storage
	 * capacity and the specified capacityIncrement.
	 * @param initialCapacity the initial storage capacity of the list
	 * @param capacityIncrement how much to increase the element's 
	 * size by.
	 */
	public List(int initialCapacity, int capacityIncrement) 
		{
		this.elementData = new Object[initialCapacity];
		this.capacityIncrement = capacityIncrement;
		}

	/**
	 * Create a list initial items from a list
	 */
	public List( Vector initialElements )
		{
		this( initialElements.size() );
		initialElements.copyInto(elementData);
		}

	/**
	 * Converts a string to a list. Converts strings created by a list or vector
	 * Items in string are separated by separatorChar
	 */
	public void fromString( String listString  ) throws ConversionException
		{
		load( new StringBufferInputStream( listString ) );
		}

	/**
	 * Converts a string to a list. Converts strings created by a list or vector
	 * Items in string are separated by separatorChar
	 */
	public void load( InputStream in  ) throws ConversionException
		{
		try
			{
			SimpleTokenizer parser;
			parser = new SimpleTokenizer( in, parseTable );
			
			while ( parser.hasMoreTokens() )
				addElement( parser.nextToken( ) );
			}
		catch (IOException unlikelyToOcurr )
			{
			throw new ConversionException( "IO Exception thrown in converting object");
			}
		}
			
	/**
	 * Converts the list to a string
	 */
	public String toString( ) 
		{
		return toString( null );
		}

	/**
	 * Converts the list to a string with given header information.
	 */
	public String toString( String header ) 
		{
		Stringizer buffer = new Stringizer( parseTable );
		buffer.setHeader( header );
		
		Enumeration elements = elements();

		while ( elements.hasMoreElements() )
			buffer.appendToken( elements.nextElement(), separatorChar );

		return buffer.toString();
		}

	/**
	 * Converts the list to a vector.
	 */
	public Vector toVector()
		{
		Vector newCopy = new Vector( elementData.length );
		Enumeration elementsToCopy = elements();
		while ( elementsToCopy.hasMoreElements() )
			newCopy.addElement( elementsToCopy.nextElement() );
		return newCopy;
		}

	/**
	 * Converts the list to an array.
	 */
	public Object[] toArray()
		{
		Object[] theArray = new Object[ size() ];
		System.arraycopy( elementData, 0, theArray, 0, size());
		return theArray;
		}

	/**
	 * Writes ascii representation of List to Outputstream 
	 */
	 public void save( OutputStream out, String header )
	 	{
	 	PrintStream printOut = new PrintStream( out );
	 	printOut.println( toString( header ) );
	 	printOut.flush();
	 	}

	/**
	 * Set character used to separate elements of list in String representation of List
	 */
	public void setSeparatorChar( char separatorChar )
		{
		this.separatorChar = separatorChar;
		upDateParseTable();
		}

	/**
	 * Sets TokenCharacters used to convert List from/to strings/streams.
	 * Current values for separatorChar override separator 
	 * settings in TokenCharacters object.
	 */
	public void setTokenCharacters( TokenCharacters newParseTable )
		{
		parseTable = newParseTable;
		upDateParseTable();
		}

	private void upDateParseTable()
		{
		parseTable.setSeparatorChars( String.valueOf( separatorChar ) );
		}

/*----------------------------------------------------------------------*/
	// Vector operations

	/**
	 * Trims the list's capacity down to size. Use this operation to
	 * minimize the storage of a list. Subsequent insertions will
	 * cause reallocation.
	 */
	public final synchronized void trimToSize()  
		{
		int oldCapacity = elementData.length;
		if (elementCount < oldCapacity)  
			{
			Object oldData[] = elementData;
			elementData = new Object[elementCount];
			System.arraycopy(oldData, 0, elementData, 0, elementCount);
			}
		}

	/**
	 * Ensures that the list has at least the specified capacity.
	 * @param minCapacity the desired minimum capacity
	 */
	public final synchronized void ensureCapacity(int minCapacity)  
		{
		int oldCapacity = elementData.length;
		if (minCapacity > oldCapacity)  
			{
			Object oldData[] = elementData;
			int newCapacity = (capacityIncrement > 0) ?
			(oldCapacity + capacityIncrement) : (oldCapacity * 2);
			if (newCapacity < minCapacity)  
				{
				newCapacity = minCapacity;
				}
			elementData = new Object[newCapacity];
			System.arraycopy(oldData, 0, elementData, 0, elementCount);
			}
		}

	/**
	 * Sets the size of the list. If the size shrinks, the extra elements
	 * (at the end of the list) are lost; if the size increases, the
	 * new elements are set to null.
	 * @param newSize the new size of the list
	 */
	public final synchronized void setSize(int newSize)  
		{
		if (newSize > elementCount)  
			{
			ensureCapacity(newSize);
			} 
		else  
			{
			for (int i = newSize ; i < elementCount ; i++)  
				{
				elementData[i] = null;
				}
			}
		elementCount = newSize;
		}

	/**
	 * Returns the current capacity of the list.
	 */
	public final int capacity()  
		{
		return elementData.length;
		}

	/**
	 * Returns the number of elements in the list.
	 * Note that this is not the same as the list's capacity.
	 */
	public final int size()  
		{
		return elementCount;
		}

	/**
	 * Returns true if the collection contains no values.
	 */
	public final boolean isEmpty()  
		{
		return elementCount == 0;
		}

	/**
	 * Returns an enumeration of the elements. Use the Enumeration methods on
	 * the returned object to fetch the elements sequentially.
	 */
	public final synchronized Enumeration elements()  
		{
		return new ListEnumerator(this);
		}

	/**
	 * Returns an enumeration of the elements inrevese order. 
	 * That is starts at the back of the list and goes toward the front
	 * of the list.
	 */
	public final synchronized Enumeration elementsReversed()  
		{
		return new ReverseListEnumerator(this);
		}
	
	/**
	 * Returns true if the specified object is a value of the 
	 * collection.
	 * @param elem the desired element
	 */
	public final boolean contains(Object elem)  
		{
		return indexOf(elem, 0) >= 0;
		}

	/**
	 * Searches for the specified object, starting from the first position
	 * and returns an index to it.
	 * @param elem the desired element
	 * @return the index of the element, or -1 if it was not found.
	 */
	public final int indexOf(Object elem)  
		{
		return indexOf(elem, 0);
		}

	/**
	 * Searches for the specified object, starting at the specified 
	 * position and returns an index to it.
	 * @param elem the desired element
	 * @param index the index where to start searching
	 * @return the index of the element, or -1 if it was not found.
	 */
	public synchronized int indexOf(Object elem, int index)  
		{
		for (int i = index ; i < elementCount ; i++) 
			{
			if (elem.equals(elementData[i])) 
				{
				return i;
				}
			}
		return -1;
		}

	/**
	 * Searches backwards for the specified object, starting from the last
	 * position and returns an index to it. 
	 * @param elem the desired element
	 * @return the index of the element, or -1 if it was not found.
	 */
	public final int lastIndexOf(Object elem)  
		{
		return lastIndexOf(elem, elementCount);
		}

	/**
	 * Searches backwards for the specified object, starting from the specified
	 * position and returns an index to it. 
	 * @param elem the desired element
	 * @param index the index where to start searching
	 * @return the index of the element, or -1 if it was not found.
	 */
	public synchronized int lastIndexOf(Object elem, int index)  
		{
		for (int i = index ; --i >= 0 ; ) 
			{
			if (elem.equals(elementData[i])) 
				{
				return i;
				}
			}
		return -1;
		}

	/**
	 * Returns the element at the specified index.
	 * @param index the index of the desired element
	 * @exception ArrayIndexOutOfBoundsException If an invalid 
	 * index was given.
	 */
	public final synchronized Object elementAt(int index)  
		{
		if (index >= elementCount)  
			{
				throw new ArrayIndexOutOfBoundsException(index + " >= " + elementCount);
			}
		/* Since try/catch is free, except when the exception is thrown,
		   put in this extra try/catch to catch negative indexes and
		   display a more informative error message.  This might not
		   be appropriate, especially if we have a decent debugging
		   environment - JP. */
		try  
			{
				return elementData[index];
			} 
		catch (ArrayIndexOutOfBoundsException e)  
			{
				throw new ArrayIndexOutOfBoundsException(index + " < 0");
			}
		}

	/**
	 * Returns the int at the specified index.
	 * @param index the index of the desired element
	 * @exception ArrayIndexOutOfBoundsException If an invalid 
	 * index was given.
	 * @exception ClassCastException If element at index is not really
	 * a numeric type which can be converted to an int.
	 */
	public final synchronized int intAt(int index)
		{
		Number element = (Number) elementAt( index);
		return element.intValue();
		}  


	/**
	 * Returns the double at the specified index.
	 * @param index the index of the desired element
	 * @exception ArrayIndexOutOfBoundsException If an invalid 
	 * index was given.
	 * @exception ClassCastException If element at index is not really
	 * a numeric type which can be converted to a double.
	 */
	public final synchronized double doubleAt(int index)
		{
		Number element = (Number) elementAt( index);
		return element.doubleValue();
		}  

	/**
	 * Returns the first element of the sequence.
	 * @exception NoSuchElementException If the sequence is empty.
	 */
	public final synchronized Object firstElement()  
		{
		if (elementCount == 0)  
			{
			throw new NoSuchElementException();
			}
		return elementData[0];
		}

	/**
	 * Returns the last element of the sequence.
	 * @exception NoSuchElementException If the sequence is empty.
	 */
	public final synchronized Object lastElement()  
		{
		if (elementCount == 0)  
			{
			throw new NoSuchElementException();
			}
		return elementData[elementCount - 1];
		}

	/**
	 * Sets the element at the specified index to be the specified object.
	 * The previous element at that position is discarded.
	 * @param obj what the element is to be set to
	 * @param index the specified index
	 * @exception ArrayIndexOutOfBoundsException If the index was 
	 * invalid.
	 */
	public final synchronized void setElementAt(Object obj, int index)  
		{
		if (index >= elementCount)  
			{
			throw new ArrayIndexOutOfBoundsException(index + " >= " + 
								 elementCount);
			}
		elementData[index] = obj;
		}

	/**
	 * Deletes the element at the specified index. Elements with an index
	 * greater than the current index are moved down.
	 * @param index the element to remove
	 * @exception ArrayIndexOutOfBoundsException If the index was invalid.
	 */
	public final synchronized void removeElementAt(int index)  
		{
		if (index >= elementCount)  
			{
			throw new ArrayIndexOutOfBoundsException(index + " >= " + 
								 elementCount);
			}
		int j = elementCount - index - 1;
		if (j > 0)  
			{
			System.arraycopy(elementData, index + 1, elementData, index, j);
			}
		elementCount--;
		elementData[elementCount] = null; /* to let gc do its work */
		}

	/**
	 * Inserts the specified object as an element at the specified index.
	 * Elements with an index greater or equal to the current index 
	 * are shifted up.
	 * @param obj the element to insert
	 * @param index where to insert the new element
	 * @exception ArrayIndexOutOfBoundsException If the index was invalid.
	 */
	public final synchronized void insertElementAt(Object obj, int index)  
		{
		if (index >= elementCount + 1)  
			{
			throw new ArrayIndexOutOfBoundsException(index + " >= " + 
								 elementCount + 1);
			}
		ensureCapacity(elementCount + 1);
		System.arraycopy(elementData, index, elementData, index + 1, elementCount - index);
		elementData[index] = obj;
		elementCount++;
		}

	/**
	 * Adds the specified object as the last element of the list.
	 * @param obj the element to be added
	 */
	public synchronized void addElement(Object obj)  
		{
		ensureCapacity(elementCount + 1);
		elementData[elementCount++] = obj;
		}

	/**
	 * Adds the specified int as the last element of the list.
	 * @param anInt the element to be added
	 */
	public final synchronized void addElement(int anInt)  
		{
		addElement( new Integer( anInt ) );
		}

	/**
	 * Adds the specified double as the last element of the list.
	 * @param aFloat the element to be added
	 */
	public final synchronized void addElement(double aDouble)  
		{
		addElement( new Double( aDouble ) );
		}

	/**
	 * Adds the array to the end of the list.
	 */
	public void addElements( Object[] elementsToAdd )
		{
		ensureCapacity( elementsToAdd.length + size() );
		System.arraycopy( elementsToAdd, 0, elementData,  size(), elementsToAdd.length);
		elementCount = elementCount +  elementsToAdd.length;
		}

	/**
	 * Adds the vector to the end of the list.
	 */
	public void addElements( Vector elementsToAdd )
		{
		Object[] rawElements = new Object[ elementsToAdd.size() ];
		elementsToAdd.copyInto( rawElements);
		addElements( rawElements );
		}

	/**
	 * Removes the element from the list. If the object occurs more
	 * than once, only the first is removed. If the object is not an
	 * element, returns false.
	 * @param obj the element to be removed
	 * @return true if the element was actually removed; false otherwise.
	 */
	public final synchronized boolean removeElement(Object obj)  
		{
		int i = indexOf(obj);
		if (i >= 0)  
			{
			removeElementAt(i);
			return true;
			}
		return false;
		}

	/**
	 * Removes all elements of the list. The list becomes empty.
	 */
	public final synchronized void removeAllElements()  
		{
		for (int i = 0; i < elementCount; i++)  
			{
			elementData[i] = null;
			}
		elementCount = 0;
		}

	/**
	 * Clones this list. The elements are <strong>not</strong> cloned.
	 */
	public synchronized Object clone()  
		{
		try  
			{ 
			List v = (List)super.clone();
			v.elementData = new Object[elementCount];
			System.arraycopy(elementData, 0, v.elementData, 0, elementCount);
			return v;
			} 
		catch (CloneNotSupportedException e)  
			{ 
			// this shouldn't happen, since we are Cloneable
			throw new InternalError();
			}
		}

	/**
	 * Returns a list with elements in the reverse order
	 * from present list
	 */
	public synchronized OrderedCollection reversed()  
		{
		List reverse = new List( this.size() );
		Enumeration backwords = elementsReversed();
		while ( backwords.hasMoreElements() )
			reverse.addElement( backwords.nextElement() );
		
		return reverse;
		}

	/**
	 * Returns a list with elements from present list,
	 * but in random order.
	 */
	public synchronized OrderedCollection shuffled()  
		{
		List shuffled = new List( this.size() );
		
		Object[] clone = new Object[ this.size() ];
		System.arraycopy(elementData, 0, clone, 0, this.size() );
		
		Random number = new Random();
		
		for ( int end =  this.size(); end > 0; end-- )
			{
			// value from 0 to end-1, including endpoints
			int randomIndex = Math.abs(number.nextInt()) % end;
			shuffled.addElement( clone[randomIndex] );
			
			// remove element from list
			clone[randomIndex] = clone[end-1];
			}
		return shuffled;
		}

	} //class List


final
class ListEnumerator implements Enumeration 
	{
	List enumerated;
	int count;

	ListEnumerator(List v) 
		{
		enumerated = v;
		count = 0;
		}

	public boolean hasMoreElements() 
		{
		return count < enumerated.elementCount;
		}

	public Object nextElement() 
		{
		synchronized (enumerated) 
			{
			if (count < enumerated.elementCount) 
				{
				return enumerated.elementData[count++];
				}
			}
		throw new NoSuchElementException("ListEnumerator");
		}

	} //ListEnumerator
	
final
class ReverseListEnumerator implements Enumeration 
	{
	List enumerated;
	int count;

	ReverseListEnumerator(List v) 
		{
		enumerated = v;
		count = v.elementCount - 1;
		}

	public boolean hasMoreElements() 
		{
		return count >= 0;
		}

	public Object nextElement() 
		{
		synchronized (enumerated) 
			{
			if (count >= 0) 
				{
				return enumerated.elementData[count--];
				}
			}
		throw new NoSuchElementException("ListEnumerator");
		}

	} //ReverseListEnumeratorListEnumerator