/* FilterManager.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.util.Vector;
import java.util.Hashtable;

/**
 * Class to manage all filters.  This class maintains the list of
 * known filters, enabled filters, and all filter preferences.
 *
 * @see muffin.Filter
 * @see muffin.FilterFactory
 * @author Mark Boyns
 */
public class FilterManager implements ConfigurationListener
{
    final String defaultKnownList[] = 
    { 
	"muffin.filter.AnimationKiller",
	"muffin.filter.CookieMonster",
	"muffin.filter.Decaf",
	"muffin.filter.DocumentInfo",
	"muffin.filter.ForwardedFor",
	"muffin.filter.Glossary",
	"muffin.filter.HostnameExpander",
	"muffin.filter.Logger",
	"muffin.filter.Painter",
	"muffin.filter.Preview",
	"muffin.filter.NoThanks",
	"muffin.filter.Referer",
	"muffin.filter.Secretary",
	"muffin.filter.SecretAgent",
	"muffin.filter.SecretServer",
	"muffin.filter.Snoop",
	"muffin.filter.Stats",
    };

    Options options = null;
    Configuration configs = null;
    FilterManagerFrame frame = null;

    UserPrefs userPrefs = null;
    Vector knownFilters = null;
    Vector enabledFilters = null;

    Hashtable knownFiltersCache = null;
    Hashtable enabledFiltersCache = null;

    /**
     * Create a FilterManager.
     */
    FilterManager (Options options, Configuration configs)
    {
	this.options = options;
	this.configs = configs;

	knownFiltersCache = new Hashtable ();
	enabledFiltersCache = new Hashtable ();

	configs.addConfigurationListener (this);
    }

    public void configurationChanged (String name)
    {
	userPrefs = configs.getUserPrefs ();
	
	/* known filters */
	if (! knownFiltersCache.containsKey (configs.getCurrent ()))
	{
	    knownFilters = new Vector (32);
	    knownFiltersCache.put (configs.getCurrent (), knownFilters);
	    String list[] = userPrefs.getStringList ("muffin.knownFilters");
	    if (list.length == 0)
	    {
		list = defaultKnownList;
	    }
	    for (int i = 0; i < list.length; i++)
	    {
		knownFilters.addElement (list[i]);
	    }
	}
	else
	{
	    knownFilters = (Vector) knownFiltersCache.get (configs.getCurrent ());
	}

	/* enabled filters */
	if (! enabledFiltersCache.containsKey (configs.getCurrent ()))
	{
	    enabledFilters = new Vector (32);
	    enabledFiltersCache.put (configs.getCurrent (), enabledFilters);
	    String list[] = userPrefs.getStringList ("muffin.enabledFilters");
	    for (int i = 0; i < list.length; i++)
	    {
		enable (list[i]);
	    }
	}
	else
	{
	    enabledFilters = (Vector) enabledFiltersCache.get (configs.getCurrent ());
	}
    }

    void checkAutoConfig (String pattern)
    {
	String name = configs.autoConfig (pattern);
	if (!name.equals (configs.getCurrent ()))
	{
	    System.out.println ("Automatic change to " + name);
	    configs.setCurrent (name);
	}
    }

    /**
     * Return a list of filters created by each filter's
     * factory method.
     *
     * @see muffin.Handler
     */
    Filter[] createFilters ()
    {
	FilterFactory ff;
	Filter list[] = new Filter[enabledFilters.size ()];
	for (int i = 0; i < list.length; i++)
	{
	    ff = (FilterFactory) enabledFilters.elementAt (i);
	    list[i] = ff.createFilter ();
	}
	return list;
    }

    void createFrame ()
    {
	if (frame == null)
	{
	    frame = new FilterManagerFrame (this);
	}
	frame.hideshow ();
    }

    void append (String clazz)
    {
	knownFilters.addElement (clazz);
	if (frame != null)
	{
	    frame.updateKnownFiltersList ();
	}
    }

    void remove (String clazz)
    {
	for (int i = 0; i < knownFilters.size (); i++)
	{
	    String name = (String) knownFilters.elementAt (i);
	    if (name.equals (clazz))
	    {
		knownFilters.removeElementAt (i);
	    }
	}
	if (frame != null)
	{
	    frame.updateKnownFiltersList ();
	}
    }

    /**
     * Enable a filter
     *
     * @param clazz java class name
     */
    void enable (String clazz)
    {
	try
	{
	    //System.out.println ("Enabling " + clazz);
	    FilterFactory ff = (FilterFactory) (Class.forName (clazz)).newInstance ();
	    Prefs prefs = userPrefs.extract (clazz.substring (clazz.lastIndexOf ('.') + 1));
	    ff.setPrefs (prefs);
	    ff.setManager (this);
	    enabledFilters.addElement (ff);

	    if (frame != null)
	    {
		frame.updateEnabledFiltersList ();
	    }
	}
	catch (Exception e)
	{
	    System.out.println ("");
	    System.out.println ("WARNING: Can't load " + clazz + ": ");
	    System.out.println ("");
	    System.out.println ("         " + e);
	    System.out.println ("");
	    System.out.println ("         You may need to restart muffin with a different CLASSPATH.");
	    System.out.println ("");
	}
    }

    void disable (int i)
    {
	FilterFactory ff = (FilterFactory) enabledFilters.elementAt (i);
	//System.out.println ("Disabling " + ff.getClass ().getName ());
	ff.shutdown ();
	enabledFilters.removeElementAt (i);
	if (frame != null)
	{
	    frame.updateEnabledFiltersList ();
	}
    }

    void disableAll ()
    {
	for (int i = enabledFilters.size () - 1; i >= 0; i--)
	{
	    disable (i);
	}
    }

    /**
     * Save all filter preferences.
     */
    void save ()
    {
	// Update muffin.enabledFilters
	String list[] = new String[enabledFilters.size ()];
	for (int i = 0; i < list.length; i++)
	{
	    FilterFactory ff = (FilterFactory) enabledFilters.elementAt (i);
	    list[i] = (ff.getClass ()).getName ();
	}
	userPrefs.putStringList ("muffin.enabledFilters", list);
	userPrefs.save ();
    }

    /**
     * Save filter preferences.
     *
     * @param ff filter factory
     */
    public void save (FilterFactory ff)
    {
	userPrefs.merge (ff.getPrefs ());
	userPrefs.save ();
    }
}    
