/* Handler.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.Socket;
import java.net.URL;

import muffin.io.*;

/**
 * HTTP transaction handler.  A handler is created by muffin.Server for
 * each HTTP transaction.  Given a socket, the handler will deal with
 * the request, reply, and invoke request, reply, and content filters.
 *
 * @see muffin.Server
 * @author Mark Boyns
 */
class Handler extends Thread
{
    ThreadGroup filterGroup = null;
    Monitor monitor = null;
    FilterManager manager = null;
    Options options = null;
    Client client = null;
    Socket socket = null;
    Request request = null;
    Reply reply = null;
    HttpRelay http = null;
    int currentLength = 0;
    int contentLength = 0;
    Filter filterList[];
    boolean keepAlive = false;

    /**
     * Create a Handler.
     */
    Handler (ThreadGroup t, Runnable r, Monitor m, FilterManager manager, Options options)
    {
	super (t, r);
	this.monitor = m;
	this.manager = manager;
	this.options = options;
    }

    /**
     * Start the handler.
     *
     * @param s a socket
     */
    void doit (Socket s)
    {
	socket = s;
	start ();
    }

    /**
     * Close all connections associated with this handler.
     */
    synchronized void close ()
    {
	if (client != null)
	{
	    client.close ();
	}
	if (http != null)
	{
	    http.close ();
	}
    }

    /**
     * Flush all data to the client.
     */
    void flush ()
    {
	if (client != null)
	{
	    try
	    {
		client.getOutputStream ().flush ();
	    }
	    catch (Exception e)
	    {
	    }
	}
    }

    void kill ()
    {
	monitor.unregister (this);
	stop ();
    }

    /**
     * Where all the work gets done.
     */
    public void run ()
    {
	Thread.currentThread ().setName ("Handler (" + socket.getInetAddress () + ")");
	filterGroup = new ThreadGroup ("Filters (" + socket.getInetAddress () + ")");

	try
	{
	    client = new Client (socket);
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	    return;
	}
	
	monitor.register (this);
	
	do
	{
	    request = null;
	    reply = null;
	    filterList = null;
	    monitor.update (this);

	    /* Wait for the client to send a request. */
	    try
	    {
		request = client.read ();
	    }
	    catch (Exception e)
	    {
		/* No more requests */
		break;
	    }

	    manager.checkAutoConfig (request.getURL ());

	    /* Obtain a list of filters to use. */
	    filterList = manager.createFilters ();

	    try
	    {
		/* Check for a non-proxy request. */
		if (!request.getURL ().startsWith ("http://"))
		{
		    request.setURL ("http://" + options.getString ("muffin.host")
				    + ":" + options.getString ("muffin.port")
				    + request.getURL ());
		}

		monitor.update (this);

		/* Filter the request. */
		if (!options.getBoolean ("muffin.passthru"))
		{
		    /* Redirect the request if necessary */
		    String location = redirect (request);
		    if (location != null)
		    {
			Reply r = Reply.createRedirect (location);
			client.write (r);
			break; /* XXX */
		    }

		    filter (request);
		}

		/* Create the appropriate http relay to handle the
		   request. */
		if (Httpd.sendme (request))
		{
		    http = new Httpd (socket);
		}
		else if (options.useProxy ())
		{
		    http = new Proxy (options.getString ("muffin.httpProxyHost"),
				      options.getInteger ("muffin.httpProxyPort"));
		}
		else
		{
		    http = new Http (new URL (request.getURL ()));
		}

		keepAlive = doKeepAlive (request);

		/* Send the request to the http relay. */
		http.write (request);

		/* Read the reply. */
		reply = http.read ();
		monitor.update (this);

		if (keepAlive)
		{
		    keepAlive = doKeepAlive (request, reply);
		}

		/* Filter the reply. */
		if (!options.getBoolean ("muffin.passthru"))
		{
		    filter (reply);
		}
		monitor.update (this);

		try 
		{
		    contentLength = Integer.parseInt (reply.getHeaderField ("Content-length"));
		}
		catch (Exception e)
		{
		    contentLength = 0;
		}
		currentLength = 0;
		monitor.update (this);

		if (reply.hasContent ())
		{
		    /* Filter the content. */
		    if (!options.getBoolean ("muffin.passthru"))
		    {
			filter (http.getInputStream (), client.getOutputStream ());
		    }
		    else
		    {
			client.write (reply);
			copy (http.getInputStream (), client.getOutputStream (), true);
		    }
		}
		else
		{
		    client.write (reply);
		}
	    }
	    catch (Exception e)
	    {
		error (client.getOutputStream (), e, request);
		keepAlive = false;
	    }
	}
	while (keepAlive);

	close ();
	monitor.unregister (this);
    }

    /**
     * Determine whether or not to maintain a persistent connection
     * based on a request.
     *
     * @param request a request
     */
    boolean doKeepAlive (Request request)
    {
	boolean alive = false;
	
	/* Check for HTTP/1.1 Connection header. */
	if (request.containsHeaderField ("Connection"))
	{
	    if (request.getHeaderField ("Connection").equals ("open"))
	    {
		alive = true;
	    }
	    else if (request.getHeaderField ("Connection").equals ("close"))
	    {
		alive = false;
	    }
	    request.removeHeaderField ("Connection");
	}
	/* Assume persistent connection with HTTP/1.1. */
	else if (request.getProtocol ().equals ("HTTP/1.1"))
	{
	    alive = true;
	}
	/* Check for HTTP/1.0 Keep-Alive connection. */
	else if (request.containsHeaderField ("Proxy-Connection")
		 && request.getHeaderField ("Proxy-Connection").equals ("Keep-Alive"))
	{
	    alive = true;
	    request.removeHeaderField ("Proxy-Connection");
	}
	return alive;
    }

    /**
     * Determine whether or not to maintain a persistent connection
     * based on a request and reply.
     *
     * @param request a request
     * @param reply a reply
     */
    boolean doKeepAlive (Request request, Reply reply)
    {
	boolean alive = false;
	
	reply.removeHeaderField ("Connection");
	reply.removeHeaderField ("Proxy-Connection");

	/* Must have Content-length */
	if (reply.containsHeaderField ("Content-length"))
	{
	    if (request.getProtocol ().equals ("HTTP/1.1"))
	    {
		reply.setHeaderField ("Connection", "open");
	    }
	    else
	    {
		reply.setHeaderField ("Proxy-Connection", "Keep-Alive");
	    }
	    alive = true;
	}

	return alive;
    }

    /**
     * Pass a request through the redirect filters.
     *
     * @param r a request
     */
    String redirect (Request r)
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof RedirectFilter)
	    {
		RedirectFilter rf = (RedirectFilter) filterList[i];
		if (rf.needsRedirection (r))
		{
		    String location = rf.redirect (r);
		    return location;
		}
	    }
	}
	return null;
    }
	    
    /**
     * Pass a reply through the filters.
     *
     * @param r a reply
     */
    void filter (Reply r) throws FilterException
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ReplyFilter)
	    {
		((ReplyFilter)(filterList[i])).filter (r);
	    }
	}
    }

    /**
     * Pass a request through the filters.
     *
     * @param r a request
     */
    void filter (Request r) throws FilterException
    {
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof RequestFilter)
	    {
		((RequestFilter)(filterList[i])).filter (r);
	    }
	}
    }


    /**
     * Pass the content through the filters using ObjectStreams.
     *
     * @param in InputStream
     * @param out OutputStream
     */
    void filter (InputStream in, OutputStream out) throws java.io.IOException
    {
	InputObjectStream inputObjects = new InputObjectStream ();
	SourceObjectStream srcObjects;
	boolean isFiltered = false;

	if (reply.containsHeaderField ("Content-type") &&
	    reply.getContentType ().equals ("text/html"))
	{
	    srcObjects = new HtmlObjectStream (inputObjects);
	}
	else
	{
	    srcObjects = new SourceObjectStream (inputObjects);
	}
	
	for (int i = 0; i < filterList.length; i++)
	{
	    if (filterList[i] instanceof ContentFilter)
	    {
		ContentFilter filter = (ContentFilter) filterList[i];
		if (filter.needsFiltration (request, reply))
		{
		    OutputObjectStream oo = new OutputObjectStream ();
		    InputObjectStream io = new InputObjectStream (oo);

		    filter.setInputObjectStream (inputObjects);
		    filter.setOutputObjectStream (oo);
		
		    Thread t = new Thread (filterGroup, filter);
		    try
		    {
			t.setPriority (Thread.MIN_PRIORITY);
		    }
		    catch (Exception e)
		    {
			e.printStackTrace ();
		    }
		    t.start ();

		    inputObjects = io;
		    isFiltered = true;
		}
	    }
	}

	if (isFiltered)
	{
	    srcObjects.setSourceInputStream (in);

	    Thread srcThread = new Thread (srcObjects);
	    srcThread.setName ("ObjectStream Source (" + socket.getInetAddress () + ")");
	    srcThread.start ();
	    
	    int bufsize = contentLength > 0 ? contentLength : 8192;
	    ByteArrayOutputStream outbuf = new ByteArrayOutputStream (bufsize);

	    copy (inputObjects, outbuf, true);
	    
	    ByteArrayInputStream inbuf = new ByteArrayInputStream (outbuf.toByteArray ());
	    if (reply.containsHeaderField ("Content-length"))
	    {
		reply.setHeaderField ("Content-length", outbuf.size ());
	    }
	    client.write (reply);
	    copy (inbuf, out, false);
	}
	else
	{
	    client.write (reply);
	    copy (in, out, true);
	}
    }

    /**
     * Return the content length.
     */
    int getTotalBytes ()
    {
	return contentLength > 0 ? contentLength : 0;
    }

    /**
     * Return the number of bytes read so far.
     */
    int getCurrentBytes ()
    {
	return currentLength;
    }

    /**
     * Return a string represenation of the hander's state.
     */
    public String toString ()
    {
	StringBuffer str = new StringBuffer ();
	str.append (socket.getInetAddress ());
	str.append (" - ");
	if (keepAlive && request == null)
	{
	    str.append ("Keep-Alive");
	}
	else if (request != null)
	{
	    if (reply != null)
	    {
		str.append ("(");
		str.append (currentLength);
		if (contentLength > 0)
		{
		    str.append ("/");
		    str.append (contentLength);
		}
		str.append (") ");
	    }
	    str.append (request.getCommand ());
	    str.append (" ");
	    str.append (request.getURL ());
	}
	return str.toString ();
    }

    /**
     * Send a error message to the client.
     *
     * @param out client
     * @param e exception that occurred
     * @param r request
     */
    void error (OutputStream out, Exception e, Request r)
    {
	StringBuffer buf = new StringBuffer ();
	buf.append ("While trying to retrieve the URL: <a href=\""+r.getURL()+"\">"+r.getURL()+"</a>\r\n");
	buf.append ("<p>\r\nThe following error was encountered:\r\n<p>\r\n");
	buf.append ("<ul><li>" + e.toString () + "</ul>\r\n");
	String s = new HttpError (options, 400, buf.toString ()).toString ();
	try
	{
	    out.write (s.getBytes (), 0, s.length ());
	    out.flush ();
	}
	catch (Exception ex)
	{
	}
    }

    /**
     * Copy in to out.
     *
     * @param in InputStream
     * @param out OutputStream
     * @param monitored Update the Monitor
     */
    void copy (InputStream in, OutputStream out, boolean monitored) throws java.io.IOException
    {
	int n;
	byte buffer[] = new byte[8192];

	while ((n = in.read (buffer, 0, buffer.length)) > 0)
	{
	    out.write (buffer, 0, n);
	    if (monitored)
	    {
		currentLength += n;
		monitor.update (this);
	    }
	}
	out.flush ();
    }

    /**
     * Copy in to out.
     *
     * @param in InputObjectStream
     * @param out OutputStream
     * @param monitored Update the Monitor
     */
    void copy (InputObjectStream in, OutputStream out, boolean monitored) throws java.io.IOException
    {
	Object obj;
	while ((obj = in.read ()) != null)
	{
	    if (obj instanceof ByteArray)
	    {
		ByteArray bytes = (ByteArray) obj;
		out.write (bytes.getBytes (), 0, bytes.length ());
		currentLength += bytes.length ();
	    }
	    else if (obj instanceof Byte)
	    {
		Byte b = (Byte) obj;
		out.write (b.byteValue ());
		currentLength++;
	    }
	    else
	    {
		System.out.println ("Unknown object: " + obj.toString ());
	    }

	    if (monitored)
	    {
		monitor.update (this);
	    }
	}
	out.flush ();
    }
}
