/* Reply.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin;

import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.util.StringTokenizer;

/**
 * @author Mark Boyns
 */
public class Reply extends Message
{
    void read (InputStream in) throws IOException
    {
	statusLine = readLine (in);
	readHeaders (in);

	int code = getStatusCode ();

	/* RFC 2068: 204 and 304 MUST NOT contain a message body. */
	switch (code)
	{
	case 204: /* No Content */
	case 304: /* Not Modified */
	    /* Ignore the message body if it exists */
	    if (containsHeaderField ("Content-length"))
	    {
		System.out.println ("RFC 2068: Ignoring message-body from " + code + " response - "
				    + getHeaderField ("Server"));
		int contentLength = 0;
		try 
		{
		    contentLength = Integer.parseInt (getHeaderField ("Content-length"));
		}
		catch (Exception e)
		{
		}
		int n;
		byte buffer[] = new byte[8192];
		while ((n = in.read (buffer, 0, buffer.length)) > 0)
		{
		    /* ignore */
		}
		removeHeaderField ("Content-length");
	    }
	    break;
	}
    }

    void write (OutputStream out) throws IOException
    {
	String s = toString ();
	out.write (s.getBytes (), 0, s.length ());
	out.flush ();
    }

    boolean hasContent ()
    {
	switch (getStatusCode ())
	{
	case 204:
	case 304:
	    return false;

	default:
	    return true;
	}
    }

    public String getProtocol ()
    {
	StringTokenizer st = new StringTokenizer (statusLine);
	String protocol = (String) st.nextToken ();
	return protocol;
    }

    public int getStatusCode ()
    {
	StringTokenizer st = new StringTokenizer (statusLine);
	String protocol = (String) st.nextToken ();
	String status = (String) st.nextToken ();
	int code = 0;
	try
	{
	    code = Integer.parseInt (status);
	}
	catch (Exception e)
	{
	    System.out.println ("Malformed or missing status code");
	}
	return code;
    }

    private Hashtable contentTypeParser ()
    {
	Hashtable table = new Hashtable ();
	String type = getHeaderField ("Content-type");
	if (type == null)
	{
	    return table;
	}

	StringTokenizer st = new StringTokenizer (type, ";");
	int count = 0;
	while (st.hasMoreTokens ())
	{
	    String token = st.nextToken ();
	    token = token.trim ();
	    String name;
	    String value;
	    int i = token.indexOf ('=');
	    if (i != -1)
	    {
		name = token.substring (0, i);
		value = token.substring (i+1);
	    }
	    else
	    {
		name = token;
		value = "";
	    }

	    if (count == 0)
	    {
		table.put ("Content-type", name);
	    }
	    else
	    {
		table.put (name, value);
	    }

	    count++;
	}

	return table;
    }

    public String getContentType ()
    {
	Hashtable table = contentTypeParser ();
	return (String) table.get ("Content-type");
    }

    public String getBoundary ()
    {
	Hashtable table = contentTypeParser ();
	return (String) table.get ("boundary");
    }

    public static Reply createRedirect (String url)
    {
	Reply r = new Reply ();
	r.setStatusLine ("HTTP/1.0 302 Moved Temporarily");
	r.setHeaderField ("Location", url);
	return r;
    }
}
