/* HostnameExpanderFilter.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.filter;

import muffin.*;
import java.net.InetAddress;
import java.net.UnknownHostException;

public class HostnameExpanderFilter implements RedirectFilter
{
    final static String[] expandDomains = { ".com", ".edu", ".net", ".org" };

    Prefs prefs;
    HostnameExpander factory;

    HostnameExpanderFilter (HostnameExpander factory)
    {
	this.factory = factory;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
    }

    public boolean needsRedirection (Request r)
    {
	String host = r.getHost ();
	return host.indexOf (".") < 0; /* doesn't have a dot */
    }
    
    public String redirect (Request r)
    {
	String host = expandHostname (r.getHost ());
	if (host != null)
	{
	    StringBuffer buf = new StringBuffer ();
	    buf.append ("http://");
	    buf.append (host);
	    if (r.getPort () != 80)
	    {
		buf.append (":");
		buf.append (r.getPort ());
	    }
	    buf.append (r.getPath ());
	    factory.process (r.getHost () + " -> " + buf.toString () + "\n");
	    return buf.toString ();
	}
	return null;
    }

    /**
     * Return true iff request is redirected.  A request will be
     * redirected if the requested host is not fully qualified and is
     * not locally resolvable and *is* resovable to
     * www.$host.{com,edu,net,org}.
     */
    String expandHostname (String host)
    {
        String resolve;
        int pos;
        InetAddress address;
                
	// this could be a local address attempt to resolve
	try
	{
	    address = InetAddress.getByName(host);
	}
	catch (UnknownHostException uhe)
	{
	    address = null;
	}

	if (address == null)
	{
	    // it wasn't a local address, try www.$host.com and friends
	    for (int i=0; i<expandDomains.length; i++)
	    {
		resolve = "www." + host + expandDomains[i];
		try
		{
		    address = InetAddress.getByName(resolve);
		}
		catch (UnknownHostException uhe)
		{
		    address = null;
		}
		if (address != null)
		{
		    return resolve;
		}
	    }
	}
	
        return null;
    }
}
