/* NoThanksFilter.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package muffin.filter;

import muffin.*;
import muffin.io.*;
import muffin.html.*;
import JP.ac.osaka_u.ender.util.regex.*;
import java.util.Hashtable;

public class NoThanksFilter implements ContentFilter, RequestFilter, ReplyFilter
{
    NoThanks factory;
    Prefs prefs;
    InputObjectStream in = null;
    OutputObjectStream out = null;
    Request request = null;

    public NoThanksFilter (NoThanks factory)
    {
	this.factory = factory;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
    }

    public void filter (Request request) throws FilterException
    {
	String url = request.getURL ();

	/* Check for redirect */
	String location = factory.redirect (url);
	if (location != null)
	{
	    request.setURL (location);
	    return;
	}

	/* Check for killed URL */
	if (factory.isKilled (url))
	{
	    factory.process ("Killed: " + url + "\n");
	    throw new FilterException ("NoThanks URL " + url + " rejected");
	}
    }

    public void filter (Reply reply) throws FilterException
    {
	String content = reply.getContentType ();
	if (content != null && factory.killContent (content))
	{
	    factory.process ("Killed: " + content + "\n");
	    throw new FilterException ("NoThanks content-type " + content + " rejected");
	}
    }
    
    public boolean needsFiltration (Request request, Reply reply)
    {
	this.request = request;
	String s = reply.getContentType ();
	return s != null && s.startsWith ("text/html");
    }
    
    public void setInputObjectStream (InputObjectStream in)
    {
	this.in = in;
    }

    public void setOutputObjectStream (OutputObjectStream out)
    {
	this.out = out;
    }

    public void run ()
    {	
	Thread.currentThread ().setName ("NoThanks");

	try
	{
	    String endTag = null;
	    boolean killingComment = false;

	    Object obj;
            while ((obj = in.read ()) != null)
            {
		Token token = (Token) obj;
		if (killingComment && token.getType () != Token.TT_COMMENT)
		{
		    continue;
		}

		switch (token.getType ())
		{
		case Token.TT_COMMENT:
		    String s = token.toString ();
		    killingComment = factory.killComment (s);
		    if (killingComment)
		    {
			factory.process ("Comment: " + s + "\n");
		    }
		    else
		    {
			out.write (token);
		    }
		    break;
		    
		case Token.TT_TAG:
		    boolean output = true;
		    Tag tag = token.createTag ();

		    if (endTag != null && tag.is (endTag))
		    {
			endTag = null;
			continue;
		    }

		    if (factory.stripTag (tag.name ()))
		    {
			factory.process ("Stripped: " + tag.name () + "\n");
			endTag = factory.stripUntil (tag.name ());
			output = false;
		    }
		    else if (factory.replaceTag (tag.name ()))
		    {
			factory.process ("Replaced: " + tag.name () + "\n");
			tag = factory.replaceTagWith (tag.name ());
		    }
		    else if (factory.checkTagAttributes (tag))
		    {
			if (factory.processTagAttributes (tag))
			{
			    output = false;
			}
		    }
		    else if (factory.checkTag (tag.name ()))
		    {
			for (int i = 0; i < factory.hyperlist.length; i++)
			{
			    if (tag.has (factory.hyperlist[i]))
			    {
				String link = tag.get (factory.hyperlist[i]);
				if (factory.isKilled (link))
				{
				    factory.process ("Killed: " + link + "\n");
				    output = false;
				    if (tag.is ("a"))
				    {
					endTag = "/a";
				    }
				    else if (tag.is ("form"))
				    {
					endTag = "/form";
				    }
				    else if (tag.is ("iframe"))
				    {
					endTag = "/iframe";
				    }
				    else if (tag.is ("layer"))
				    {
					endTag = "/layer";
				    }
				    break;
				}
			    }
			}
		    }
		    if (output)
		    {
			token.importTag (tag);
			out.write (token);
		    }
		    break;
		    
		default:
		    if (endTag == null)
		    {
			out.write (token);
		    }
		    break;
		}
	    }
	    
	    out.flush ();
	    out.close ();
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }
}

