/* NoThanksFilter.java */

/* 
 * Copyright (C) 1996-98 Mark Boyns <boyns@sdsu.edu>
 *
 * This file is part of Muffin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.doit.muffin.filter;

import org.doit.muffin.*;
import org.doit.io.*;
import org.doit.html.*;
import java.util.Hashtable;
import java.util.Enumeration;

public class NoThanksFilter implements ContentFilter, RequestFilter, ReplyFilter
{
    NoThanks factory;
    Prefs prefs;
    InputObjectStream in = null;
    OutputObjectStream out = null;
    Request request = null;

    public NoThanksFilter (NoThanks factory)
    {
	this.factory = factory;
    }
    
    public void setPrefs (Prefs prefs)
    {
	this.prefs = prefs;
    }

    public void filter (Request request) throws FilterException
    {
	String url = request.getURL ();

	/* Check for redirect */
	String location = factory.redirect (url);
	if (location != null)
	{
	    request.setURL (location);
	    return;
	}

	/* Check for killed URL */
	if (factory.isKilled (url))
	{
	    factory.process ("Killed: " + url + "\n");
	    throw new FilterException ("NoThanks URL " + url + " rejected");
	}
    }

    public void filter (Reply reply) throws FilterException
    {
	String content = reply.getContentType ();
	if (content != null && factory.killContent (content))
	{
	    factory.process ("Killed: " + content + "\n");
	    throw new FilterException ("NoThanks content-type " + content + " rejected");
	}
    }
    
    public boolean needsFiltration (Request request, Reply reply)
    {
	this.request = request;
	String s = reply.getContentType ();
	return s != null && s.startsWith ("text/html");
    }
    
    public void setInputObjectStream (InputObjectStream in)
    {
	this.in = in;
    }

    public void setOutputObjectStream (OutputObjectStream out)
    {
	this.out = out;
    }

    public void run ()
    {	
	Thread.currentThread ().setName ("NoThanks");

	try
	{
	    String endTag = null;
	    boolean killingComment = false;

	    Object obj;
            while ((obj = in.read ()) != null)
            {
		Token token = (Token) obj;
		if (killingComment && token.getType () != Token.TT_COMMENT)
		{
		    continue;
		}

		switch (token.getType ())
		{
		case Token.TT_COMMENT:
		    String s = token.toString ();
		    killingComment = factory.killComment (s);
		    if (killingComment)
		    {
			factory.process ("Comment: " + s + "\n");
		    }
		    // AJP modification: Ignore everything up to endTag, if set
		    else if (endTag == null)
		    {
			out.write (token);
		    }
		    break;
		    
		case Token.TT_TAG:
		    boolean output = true;
		    Tag tag = token.createTag ();

		    // AJP modification: Ignore everything up to endTag, if set
		    if (endTag != null)
		    {
			if (tag.is (endTag))
			{
			    endTag = null;
			}
			continue;
		    }
		    
		    if (factory.stripTag (tag.name ()))
		    {
			factory.process ("Stripped: " + tag.name () + "\n");
			endTag = factory.stripUntil (tag.name ());
			output = false;
		    }
		    if (output && factory.replaceTag (tag.name ()))
		    {
			factory.process ("Replaced: " + tag.name () + "\n");
			tag = factory.replaceTagWith (tag.name ());
		    }
		    if (output && factory.checkTagAttributes (tag))
		    {
			if (factory.processTagAttributes (tag))
			{
			    output = false;
			    if (factory.hasEnd (tag.name ()))
			    {
				endTag = "/" + tag.name ();
			    }
			}
		    }
		    if (output && factory.checkTag (tag.name ()) && tag.attributeCount () > 0)
		    {
			Enumeration e = tag.enumerate ();
			while (e.hasMoreElements ())
			{
			    String attr = (String) e.nextElement ();
			    if (factory.checkAttr (attr))
			    {
				String link = tag.get (attr);
				if (link != null && factory.isKilled (link))
				{
				    /* Can't kill tags like body and head.  Instead
				       remove the offending attribute. */
				    if (factory.isRequired (tag.name ()))
				    {
					factory.process ("Killed: " + attr + " = " + link + "\n");
					tag.remove (attr);
				    }
				    /* Kill the tag completely. */
				    else
				    {
					factory.process ("Killed: " + tag.name () + " with " + attr + " = " + link + "\n");
					output = false;
					if (factory.hasEnd (tag.name ()))
					{
					    endTag = "/" + tag.name ();
					}
					break; /* tag is killed */
				    }
				}
			    }
			}
		    }
		    if (output)
		    {
			token.importTag (tag);
			out.write (token);
		    }
		    break;
		    
		default:
		    if (endTag == null)
		    {
			out.write (token);
		    }
		    break;
		}
	    }
	    
	    out.flush ();
	    out.close ();
	}
	catch (Exception e)
	{
	    e.printStackTrace ();
	}
    }
}

