#!/bin/sh

#-----------------------------------------------------------------------
# README
#-----------------------------------------------------------------------
#
# Installation script for rcf. You can use the --prefix parameter to
# specify an alternate base directory (default is /).
#
# Example:
#   ./install.sh --prefix=/usr/local/rcf
#
# Would create the following directories and/or files:
#   /usr/local/rcf/etc/firewall/...
#   /usr/local/rcf/etc/firewall.conf
#   /usr/local/rcf/etc/rc.d/...
#   /usr/local/rcf/sbin/rcf
#   /usr/local/rcf/usr/doc/...
#   /usr/local/rcf/usr/man/...
#   etc.
#
#-----------------------------------------------------------------------
# CHANGES 
#-----------------------------------------------------------------------
#
# 2001-04-13  Jean-Sebastien Morisset <jsmoriss@mvlan.net>
#             If Red Hat's chkconfig binary is found, it'll be used,
#             otherwise that section is skipped.
# 2001-04-01  Jean-Sebastien Morisset <jsmoriss@mvlan.net>
#             Complete over-haul of the screen output.
#
#-----------------------------------------------------------------------
# FUNCTIONS
#-----------------------------------------------------------------------

Read_Command_Line () {
	# ARG_EXEC variable accumulates unrecognized parameters
	# which can be passed-on to an executable.
	#   
	ARG_ORIG="$*"
	while : 
	do  
		for ARG in $*
		do
			case $ARG in 
				--prefix)
					PREFIX="$2"
					readonly PREFIX
					shift
					;;
				*)	ARG_EXEC="$ARG_EXEC $1"
					;;
			esac    
			shift
			continue 2
		done
		break
	done
	unset ARG
}

Security_Check () {
	
	# Before calling this function, the PATH was prefixed with the 
	# standard binary directories. This minimizes the chance of finding
	# a bad grep or sed which are used here.
	#
	echo -n "Checking PATH for relative directories... "
	if [ "`echo $PATH|grep ':\.\(:\|/\|$\)'`" ]
	then
		failed="path"
		echo "failed"
		echo ""
		echo "  Your PATH variable contains relative directories (doesn't start with '/')."
		echo "  For security reasons, you should remove these directories from your PATH"
		echo "  variable permanently (usually set in /etc/profile or ~/.profile)."
		echo ""
		
		[ "$ORIGINAL_PATH" ] && echo "  Original PATH=${ORIGINAL_PATH}"
		echo "     Fixed PATH=`echo $ORIGINAL_PATH|sed -e 's/:[^\/][^:]*:*/:/g'|sed -e 's/:$//'`"
		echo ""

		# Fix the path temporarily to finish the following checks.
		export PATH="`echo $PATH|sed -e 's/:[^\/][^:]*:*/:/g'|sed -e 's/:$//'`"
	else
		echo "ok"
	fi

	# Now that we've potentially broken the PATH variable <smile>, make
	# sure we can still get at all the binaries we'll need.
	#
	echo -n "Checking for essential binaries... "
	if [ ! "`which which 2>/dev/null`" ]
	then
		failed="bin"
		echo "failed"
		echo ""
		echo "  Unable to locate 'which' in any PATH directory!"
		echo ""
	else
		# If we have a 'which', continue checking...
		#
		for bin in cp egrep install mv tar
		do
			if [ ! "`which $bin 2>/dev/null`" ]
			then
				# only print failed once
				[ "$failed" != "bin" ] && { echo "failed"; echo ""; }
				failed="bin"
				echo "  Unable to locate '$bin' in any PATH directory!"
			fi
		done
		unset bin
		[ "$failed" != "bin" ] && echo "ok" || echo ""
	fi

	# If we've detected any error, exit here.
	if [ "$failed" ]
	then
		[ "$failed" != "files" ] && echo ""	# failure was higher up, so add an extra line
		echo "These problems must be addressed before running `Sed_Basename $0` again."
		echo ""
		exit 1
	fi
}

Sed_Basename () {
	echo "$1" | sed -e 's/^[    ]*//;s/[     ]*$//;tc;:c;s/[     ][      ]*/\;/;ta;s/\/*$//;s/.*\///;b;:a;h;s/.*\n//;x;s/\n.*//;s/\/*$//;s/.*\///;tb;:b;G;s/^\(.*\)\(.*\)\n\2$/\1/;t;P;d;'
}

Backup () {
	for BACKUP in $*
	do
		if [ -e "$BACKUP" ]
		then
			echo "  Moving ${BACKUP} to ${BACKUP}.${DATE}..."
			mv ${BACKUP} ${BACKUP}.${DATE}
		fi
	done
}

Mkdir () {
	for DIR in $*
	do
		if [ ! -d $DIR ]
		then
			echo "  Creating ${DIR}..."
			mkdir -p $DIR
		fi
	done
	unset DIR
}

# ======================================================================
# MAIN
# ======================================================================

VERSION=5.2
readonly VERSION

ORIGINAL_UMASK="`umask`"
ORIGINAL_PATH="$PATH"

umask 077

APPNAME="rcf"
PREFIX=""

# Make sure we have all the standard OS paths available. I also prefer
# using the standard OS utilities instead of what we might find under 
# /usr/local/bin. :-)
#
PATH="/sbin:/usr/sbin:/bin:/usr/bin:$PATH"
export PATH

echo ""
echo "SECURITY CHECK"
echo "================================================================================"

Security_Check

DATE="`date +'%y%m%d%H%M'`"

Read_Command_Line

echo ""
echo "INSTALLATION"
echo "================================================================================"

echo "Backing up previous installation..."
Backup \
	${PREFIX}/etc/rc.d/rc.firewall \
	${PREFIX}/etc/rc.d/init.d/firewall \
	${PREFIX}/etc/firewall-modules \
	${PREFIX}/etc/firewall \
	${PREFIX}/sbin/rcf

if [ -f ${PREFIX}/etc/firewall.conf ]
then
	echo "  Copying ${PREFIX}/etc/firewall.conf to ${PREFIX}/etc/firewall.conf.${DATE}..."
	cp ${PREFIX}/etc/firewall.conf ${PREFIX}/etc/firewall.conf.${DATE}
fi

echo "Checking for missing directories..."
Mkdir \
	${PREFIX} \
	${PREFIX}/usr/man \
	${PREFIX}/etc/firewall \
	${PREFIX}/etc/firewall/functions \
	${PREFIX}/etc/firewall/groups \
	${PREFIX}/etc/firewall/modules \
	${PREFIX}/etc/rc.d/init.d \
	${PREFIX}/sbin \
	${PREFIX}/usr/doc/${APPNAME}-${VERSION}

for num in 0 1 2 3 4 5 6; do Mkdir ${PREFIX}/etc/rc.d/rc${num}.d; done

echo "Installing execution and startup scripts..."
install -o root -g root -m 0700 firewall ${PREFIX}/etc/rc.d/init.d/firewall
install -o root -g root -m 0700 rcf ${PREFIX}/sbin/rcf

ln -s ${PREFIX}/sbin/rcf ${PREFIX}/etc/rc.d/rc.firewall

echo "Installing documentation under ${PREFIX}/usr/doc/${APPNAME}-${VERSION}..."
cp ChangeLog FAQ.html INSTALL LICENSE ${PREFIX}/usr/doc/${APPNAME}-${VERSION}

echo "Extracting functions to ${PREFIX}/etc/firewall/functions..."
tar -xvp -C ${PREFIX}/etc/firewall -f functions.tar >/dev/null

echo "Extracting groups to ${PREFIX}/etc/firewall/groups..."
tar -xvp -C ${PREFIX}/etc/firewall -f groups.tar >/dev/null

echo "Extracting modules to ${PREFIX}/etc/firewall/modules..."
tar -xvp -C ${PREFIX}/etc/firewall -f modules.tar >/dev/null

echo "Extracting man pages to ${PREFIX}/usr/man..."
tar -xvp -C ${PREFIX}/usr/man -f man.tar >/dev/null

if [ -x "`which chkconfig 2>/dev/null`" ]
then
	echo "Adding firewall startup scripts..."
	chkconfig --add firewall
fi

echo ""
echo "CONFIGURATION"
echo "================================================================================"

${PREFIX}/sbin/rcf --prefix ${PREFIX}/etc/firewall --config ${PREFIX}/etc/firewall.conf --update-config

