/* vim:ai ts=4 sw=4
 * tproxy.c:
 *
 * This is a transparent proxy server. It is started from inetd to service
 * HTTP requests are transparently accepted and passed to the WWW proxy
 * for handling.
 */

#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <syslog.h>
#include <signal.h>
#include <wait.h>
#include <pwd.h>
#include <grp.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/*
 * Beta testing a hostname translation file. It might be better to use a DBM file,
 * will have to see how well it works out. I probably will want to add regular
 * expressions as well.
 */
#ifdef TRANSLATE_PATH
# define TRANSLATE_NAME(h)	translate_name(h)
# ifndef MAX_TRANSLATIONS
#  define MAX_TRANSLATIONS	100
# endif
#else
# define TRANSLATE_NAME(h)	h
#endif

/*
 * Typedefs.
 */
typedef unsigned long	ipaddr_t;

/*
 * Macros.
 */
#define FD_MAX(a,b)		((a) > (b) ? (a) : (b))

/*
 * Function prototypes.
 */
static void usage(char *prog, char *opt);
static short getportnum(char *portnum);
static ipaddr_t getipaddress(char *ipaddr);
static uid_t getuserid(char *user);
static uid_t getgroupid(uid_t uid);
static int bind_to_port(ipaddr_t bind_ip, short bind_port);
static int connect_to_proxy(ipaddr_t ip, short port);
static void server_main_loop(int sock, ipaddr_t server_ip, short server_port);
static void dead_server(int sig);
static void trans_proxy(int sock, struct sockaddr_in *addr, int len, ipaddr_t server_ip, short server_port);
#ifdef TRANSLATE_PATH
static const char *translate_name(const char *name);
#endif

/*
 * Command line switches.
 */
static int				fully_transparent = 0;

/*
 * Main loop.
 */
int main(int argc, char **argv)
{
	int					arg;
	int					run_as_server = 0;
	ipaddr_t			bind_ip = INADDR_ANY;
	short				bind_port = -1;
	ipaddr_t			server_ip = INADDR_NONE;
	short				server_port = -1;
	uid_t				run_uid = -1;
	gid_t				run_gid = -1;
	int					sock;
	struct sockaddr_in	addr;
	int					len;

	/*
	 * Parse the command line arguments.
	 */
	while ((arg = getopt(argc, argv, "ts:r:b:")) != EOF)
	{
		switch (arg)
		{
		case 't':
			fully_transparent = 1;
			break;

		case 's':
			run_as_server = 1;
			bind_port = getportnum(optarg);
			break;

		case 'r':
			run_uid = getuserid(optarg);
			run_gid = getgroupid(run_uid);
			break;

		case 'b':
			bind_ip = getipaddress(optarg);
			break;

		case '?':
			usage(argv[0], NULL);
			break;
		}
	}

	/*
	 * Process the remaining command line arguments.
	 */
	for (; optind < argc; ++optind)
	{
		if (server_ip == INADDR_NONE)
		{
			server_ip = getipaddress(argv[optind]);
		}
		else if (server_port == -1)
		{
			server_port = getportnum(argv[optind]);
		}
		else
		{
			usage(argv[0], "Extra arguments were specified.");
		}
	}

	/*
	 * Test to make sure required args were provided and are valid.
	 */
	if (server_ip == INADDR_NONE)
	{
		usage(argv[0], "No proxyhost was specified (or it was invalid).");
	}
	if (server_port == -1)
	{
		usage(argv[0], "No proxyport was specified (or it was invalid).");
	}
	if (run_as_server && (bind_ip == INADDR_NONE))
	{
		usage(argv[0], "The server ipaddr is invalid.");
	}
	if (run_as_server && (bind_port == -1))
	{
		usage(argv[0], "No server port was specified (or it was invalid).");
	}

	/*
	 * See if we should run as a server.
	 */
	if (run_as_server)
	{
		/*
		 * Start by binding to the port, the child inherits this socket.
		 */
		sock = bind_to_port(bind_ip, bind_port);

		/*
		 * Start a server process.
		 */
		switch (fork())
		{
		case -1:
			perror("fork()");
			exit(1);

		case 0:
			/*
			 * Open syslog for logging errors.
			 */
			openlog("tproxy", LOG_PID, LOG_DAEMON);

			/*
			 * Ignore some signals.
			 */
			signal(SIGHUP, SIG_IGN);
			signal(SIGINT, SIG_IGN);
			signal(SIGQUIT, SIG_IGN);
			signal(SIGSTOP, SIG_IGN);
			signal(SIGCONT, SIG_IGN);
			signal(SIGPIPE, SIG_IGN);

			/*
			 * Drop back to an untrusted user.
			 */
			setregid(run_gid, run_gid);
			setreuid(run_uid, run_uid);

			/*
			 * Start a new session and group.
			 */
			setsid();
			setpgrp();

			/*
			 * Handle the server main loop.
			 */
			server_main_loop(sock, server_ip, server_port);

			/*
			 * Should never exit.
			 */
			closelog();
			exit(1);
		}

		/*
		 * Parent exits at this stage.
		 */
		exit(0);
	}

	/*
	 * Open syslog for logging errors.
	 */
	openlog("tproxy", LOG_PID, LOG_DAEMON);

	/*
	 * We are running from inetd so find the peer address.
	 */
	len = sizeof(addr);
	if (getpeername(STDIN_FILENO, (struct sockaddr *)&addr, &len) < 0)
	{
		syslog(LOG_ERR, "getpeername(): %m");
		closelog();
		exit(1);
	}

	/*
	 * We are running from inetd so process stdin.
	 */
	trans_proxy(STDIN_FILENO, &addr, len, server_ip, server_port);
	closelog();

	return (0);
}

/*
 * Print some basic help information.
 */
static void usage(char *prog, char *opt)
{
	if (opt)
	{
		fprintf(stderr, "%s: %s\n", prog, opt);
	}
	fprintf(stderr, "usage: %s [-t] [-s port [-r user] [-b ipaddr]] proxyhost proxyport\n", prog);
	fprintf(stderr, "    -t          Act fully transparently (default is HTTP translate).\n");
	fprintf(stderr, "    -s port     Run as a server bound to the specified port.\n");
	fprintf(stderr, "    -r user     Run as the specified user in server mode.\n");
	fprintf(stderr, "    -b ipaddr   Bind to the specified ipaddr in server mode.\n");
	exit(1);
}

/*
 * Return the port number, in network order, of the specified service.
 */
static short getportnum(char *portnum)
{
	char			*digits = portnum;
	struct servent	*serv;
	short			port;

	for (port = 0; isdigit(*digits); ++digits)
	{
		port = (port * 10) + (*digits - '0');
	}

	if ((*digits != '\0') || (port <= 0))
	{
		if ((serv = getservbyname(portnum, "tcp")) != NULL)
		{
			port = ntohs(serv->s_port);
		}
		else
		{
			port = -1;
		}
		endservent();
	}

#ifdef DEBUG
	fprintf(stderr, "Port lookup %s -> %hd\n", portnum, port);
#endif

	return (port);
}

/*
 * Return the IP address of the specified host.
 */
static ipaddr_t getipaddress(char *ipaddr)
{
	struct hostent	*host;
	ipaddr_t		ip;

	if (((ip = inet_addr(ipaddr)) == INADDR_NONE)
	    &&
		(strcmp(ipaddr, "255.255.255.255") != 0))
	{
		if ((host = gethostbyname(ipaddr)) != NULL)
		{
			memcpy(&ip, host->h_addr, sizeof(ip));
		}
		endhostent();
	}

#ifdef DEBUG
	fprintf(stderr, "IP lookup %s -> 0x%08lx\n", ipaddr, ip);
#endif

	return (ip);
}

/*
 * Find the userid of the specified user.
 */
static uid_t getuserid(char *user)
{
	struct passwd	*pw;
	uid_t			uid;

	if ((pw = getpwnam(user)) != NULL)
	{
		uid = pw->pw_uid;
	}
	else if (*user == '#')
	{
		uid = (uid_t)atoi(&user[1]);
	}
	else
	{
		uid = -1;
	}

#ifdef DEBUG
	fprintf(stderr, "User lookup %s -> %d\n", user, uid);
#endif

	endpwent();

	return (uid);
}

/*
 * Find the groupid of the specified user.
 */
static uid_t getgroupid(uid_t uid)
{
	struct passwd	*pw;
	gid_t			gid;

	if ((pw = getpwuid(uid)) != NULL)
	{
		gid = pw->pw_gid;
	}
	else
	{
		gid = -1;
	}

#ifdef DEBUG
	fprintf(stderr, "Group lookup %d -> %d\n", uid, gid);
#endif

	endpwent();

	return (gid);
}

/*
 * Bind to the specified ip and port.
 */
static int bind_to_port(ipaddr_t bind_ip, short bind_port)
{
	struct sockaddr_in	addr;
	int					sock;

	/*
	 * Allocate a socket.
	 */
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		perror("socket()");
		exit(1);
	}

#ifdef DEBUG
	/*
	 * Set the SO_REUSEADDR option for debugging.
	 */
	{
	 	int	one = 1;

		setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof(one));
	}
#endif

	/*
	 * Set the address to listen to.
	 */
	addr.sin_family = AF_INET;
	addr.sin_port = htons(bind_port);
	addr.sin_addr.s_addr = bind_ip;

	/*
	 * Bind our socket to the above address.
	 */
	if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		perror("bind()");
		exit(1);
	}

	/*
	 * Establish a large listen backlog.
	 */
	if (listen(sock, SOMAXCONN) < 0)
	{
		perror("listen()");
		exit(1);
	}

	return (sock);
}

/*
 * Connect to the proxy server.
 */
static int connect_to_proxy(ipaddr_t ip, short port)
{
	struct sockaddr_in	addr;
	int					sock;

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		syslog(LOG_WARNING, "socket(): %m");
		return (-1);
	}

	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	addr.sin_addr.s_addr = ip;

	if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		syslog(LOG_WARNING, "connect(): %m");
		return (-1);
	}

	return (sock);
}

/*
 * This is the main loop when running as a server.
 */
static void server_main_loop(int sock, ipaddr_t server_ip, short server_port)
{
	int					new_sock;
	struct sockaddr_in	addr;
	int					len;

	/*
	 * Capture singals for dead servers.
	 */
	signal(SIGCHLD, dead_server);

	for (;;)
	{
		/*
		 * Accept an incoming connection.
		 */
		len = sizeof(addr);
		while ((new_sock = accept(sock, (struct sockaddr *)&addr, &len)) < 0)
		{
			if (errno != EINTR)
			{
				syslog(LOG_ERR, "accept(): %m");
				sleep(30);
			}
		}

		/*
		 * Create a new process to handle the connection.
		 */
		switch (fork())
		{
		case -1:
			/*
			 * Under load conditions just ignore new connections.
			 */
			break;

		case 0:
			/*
			 * Start the proxy work in the new socket.
			 */
			trans_proxy(new_sock, &addr, len, server_ip, server_port);
			close(new_sock);
			closelog();
			exit(0);
		}

		/*
		 * Close the socket as the child does the handling.
		 */
		close(new_sock);
	}
}

/*
 * Catch dead servers and put them out of their misery.
 */
static void dead_server(int sig)
{
	int		status;

	/*
	 * Restart the signal handler and wait for the dying server.
	 */
	signal(sig, dead_server);
	wait(&status);
}

/*
 * Perform the transparent proxy activity.
 */
static void trans_proxy(int sock, struct sockaddr_in *addr, int len, ipaddr_t server_ip, short server_port)
{
	struct sockaddr_in	to_addr;
	int					to_len;
	struct hostent		*host;
	char				from_host[64];
	char				to_host[64];
	int					read_len;
	int					split;
	char				*uri_pos;
	char				*nl;
	int					proxy;
	int					max_fd;
	fd_set				read_fd;
	static char			line[256];

	/*
	 * The first thing we do is get the IP address that was trying to be connected to.
	 */
	to_len = sizeof(to_addr);
	if (getsockname(sock, (struct sockaddr *)&to_addr, &to_len) < 0)
	{
		syslog(LOG_WARNING, "getsockname(): %m");
		return;
	}

	/*
	 * Determine the hostnames and log a syslog entry.
	 */
	strncpy(to_host, TRANSLATE_NAME(inet_ntoa(to_addr.sin_addr)), sizeof(to_host));
#ifdef DNS_LOOKUPS
	sethostent(0);
	if ((host = gethostbyaddr((char *)&to_addr.sin_addr,
								sizeof(to_addr.sin_addr), AF_INET)) != NULL)
	{
		strncpy(to_host, TRANSLATE_NAME(host->h_name), sizeof(to_host));
	}
	else
	{
		syslog(LOG_INFO, "DNS lookup for %s failed: %m", to_host);
	}
#endif

	strncpy(from_host, TRANSLATE_NAME(inet_ntoa(addr->sin_addr)), sizeof(from_host));
#ifdef DNS_LOOKUPS
#ifdef USELESS_DNS_LOOKUPS
	if ((host = gethostbyaddr((char *)&addr->sin_addr,
								sizeof(addr->sin_addr), AF_INET)) != NULL)
	{
		strncpy(from_host, TRANSLATE_NAME(host->h_name), sizeof(from_host));
	}
	else
	{
		syslog(LOG_INFO, "DNS lookup for %s failed: %m", to_host);
	}
#endif
	endhostent();
#endif

	syslog(LOG_INFO, "Request from %s -> %s", from_host, to_host);

	/*
	 * Connect to the proxy server.
	 */
	proxy = connect_to_proxy(server_ip, server_port);

	/*
	 * If the daemon is operating in HTTP translation mode, then translate the
	 * request line.
	 */
	if (!fully_transparent)
	{
		/*
		 * Read the HTTP request line.
		 */
		switch (read_len = read(sock, line, sizeof(line)))
		{
		case -1:
			syslog(LOG_WARNING, "read() failed: %m");
			/* Drop through */
		case 0:
			close(proxy);
			return;
		}

#ifdef PAY_THE_PENALTY
		/*
		 * Not that I'm mean or anything, but if they don't configure to use the
		 * proxy directly then they pay a penalty.
		 */
		sleep(5);
#endif

		/*
		 * Parse the request line which is of the form METHOD /URI HTTP/1.0
		 * We split the line between the METHOD and /URI. If the URI starts
		 * with a / then we have transparently trapped a request, if it doesn't
		 * then assume that we have been nominated as a proxy by the browser.
		 */
		for (uri_pos = NULL, split = 0; split < read_len; ++split)
		{
			/*
			 * Stop when we find the space between the METHOD and /URI.
			 */
			if (line[split] == ' ')
			{
				uri_pos = &line[split];
				break;
			}
		}

		/*
		 * Translate the URI if we found one and it's valid.
		 */
		if ((uri_pos != NULL) && (uri_pos[1] == '/'))
		{
			*uri_pos++ = '\0';
			write(proxy, line, strlen(line));
			write(proxy, " http://", 8);
			write(proxy, to_host, strlen(to_host));
			write(proxy, uri_pos, read_len - strlen(line) - 1);

			/*
			 * Dump a nice log message showing what we have done.
			 */
			uri_pos[16] = '\0';
			if (((nl = strchr(uri_pos, '\r')) != NULL)
			    ||
				((nl = strchr(uri_pos, '\n')) != NULL))
			{
				*nl = '\0';
			}
			else
			{
				strcat(uri_pos, "...");
			}
			syslog(LOG_DEBUG, "HTTP translated to '%s http://%s%s'", line, to_host, uri_pos);
		}
		else
		{
			write(proxy, line, read_len);

			/*
			 * Dump a nice log message showing what we have done.
			 */
			line[32] = '\0';
			if (((nl = strchr(line, '\r')) != NULL)
			    ||
				((nl = strchr(line, '\n')) != NULL))
			{
				*nl = '\0';
			}
			else
			{
				strcat(line, "...");
			}
			syslog(LOG_DEBUG, "HTTP untranslated to '%s'", line);
		}
	}

	/*
	 * Continue by passing data back and forth between the client and proxy.
	 */
	for (;;)
	{
		/*
		 * Construct a select read mask from both file descriptors.
		 */
		FD_ZERO(&read_fd);
		FD_SET(sock, &read_fd);
		FD_SET(proxy, &read_fd);
		max_fd = FD_MAX(sock, proxy);

		/*
		 * Wait for some data to be read.
		 */
		if (select(max_fd + 1, &read_fd, NULL, NULL, NULL) < 0)
		{
			syslog(LOG_WARNING, "select(): %m");
			close(proxy);
			return;
		}

		/*
		 * See if any data can be read from the client.
		 */
		if (FD_ISSET(sock, &read_fd))
		{
			switch (read_len = read(sock, line, sizeof(line)))
			{
			case -1:
				syslog(LOG_WARNING, "read(client) failed: %m");
				close(proxy);
				return;

			case 0:
				close(proxy);
				return;

			default:
				write(proxy, line, read_len);
				break;
			}
		}

		/*
		 * See if any data can be read from the proxy.
		 */
		if (FD_ISSET(proxy, &read_fd))
		{
			switch (read_len = read(proxy, line, sizeof(line)))
			{
			case -1:
				syslog(LOG_WARNING, "read(proxy) failed: %m");
				close(proxy);
				return;

			case 0:
				close(proxy);
				return;

			default:
				write(sock, line, read_len);
				break;
			}
		}
	}
}

#ifdef TRANSLATE_PATH
/*
 * Use a table to translate hostname for which the reverse DNS lookup
 * is wrong or fails.
 */
static const char *translate_name(const char *name)
{
	static char	*translate_table[MAX_TRANSLATIONS];
	int			index;
	static FILE	*fp = NULL;
	char		line[128];
	char		*new_name;
	char		*nl;

	/*
	 * Load the translation table if it's not already loaded.
	 */
	if (fp == NULL)
	{
		/*
		 * Check if the file exists.
		 */
		if ((fp = fopen(TRANSLATE_PATH, "r")) != NULL)
		{
			/*
			 * Load each line into the translate table.
			 */
			index = 0;
			while (fgets(line, sizeof(line), fp) != NULL)
			{
				if ((new_name = strchr(line, ':')) != NULL)
				{
					*new_name++ = '\0';
					if ((nl = strchr(new_name, '\n')) != NULL)
					{
						*nl = '\0';
						translate_table[index++] = strdup(line);
						translate_table[index++] = strdup(new_name);
					}
				}
			}
			translate_table[index] = NULL;
			fclose(fp);
		}
	}

	for (index = 0; translate_table[index] != NULL; ++index)
	{
		if ((strcmp(name, translate_table[index]) == 0)
			&&
			(translate_table[index + 1] != NULL))
		{
			return (translate_table[index + 1]);
		}
	}

	return (name);
}
#endif
