/* vim:ai ts=4 sw=4
 * tproxy.c:
 *
 * This is a transparent proxy server. It can be started from inetd to
 * service HTTP requests, or run as a stand-alone daemon. HTTP requests
 * are transparently accepted and passed to the WWW proxy cache for
 * handling. The HTTP request line is modified into a form usable by
 * the WWW proxy cache. The destination WWW site name is extracted from
 * the HTTP headers to avoid DNS lookups.
 */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
#include <syslog.h>
#endif
#include <paths.h>
#include <signal.h>
#include <fcntl.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#ifdef NEED_GETOPT_H
#include <getopt.h>
#endif

#ifdef IPFILTER
#include <fcntl.h>
#include <sys/ioctl.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <net/if.h>
#include <netinet/ip_compat.h>
#include <netinet/ip_fil.h>
#include <netinet/ip_nat.h>
#endif

#if defined(SIGCLD) && !defined(SIGCHLD)
#define SIGCHLD SIGCLD
#endif

#define BUFFER_SIZE			(32769)

#define IN_METHOD			0
#define IN_URL				1
#define IN_SKIP_URL			2
#define IN_VERSION			3
#define IN_ALMOST_HEADER	4
#define IN_HEADER			5
#define IN_SKIP_CONNECTION	6

/*
 * Typedefs.
 */
typedef unsigned long	ipaddr_t;

/*
 * Macros.
 */
#define FD_MAX(a,b)		((a) > (b) ? (a) : (b))
#define get_end_pos(b)	strlen(b)

/*
 * Function prototypes.
 */
static void usage(char *prog, char *opt);
static short getportnum(char *portnum);
static ipaddr_t getipaddress(char *ipaddr);
static uid_t getuserid(char *user);
static uid_t getgroupid(uid_t uid);
static int bind_to_port(ipaddr_t bind_ip, short bind_port);
static int connect_to_proxy(ipaddr_t ip, short port);
static void lookup_hostname(struct sockaddr_in *addr, char *hostname,
							int hostlen, int needed);
static void server_main_loop(int sock, char *server_hostname, short server_port);
static void trans_proxy(int sock, struct sockaddr_in *addr,
						char *server_hostname, short server_port);
static int	get_site_from_headers(int sock, char *hostname, int hostlen,
								  char *headers, int header_max,
								  int *header_len);
static void alarm_signal(int sig);
static void log_time(void);
static void append_string(char *dest, int max_size, char *source);
static void append_char(char *dest, int max_size, char source);

/*
 * Command line switches.
 */
static int				daemonize = 1;
static int				fully_transparent = 0;
static char				*force_url = NULL;
#ifdef LOG_TO_FILE
static FILE				*log_file = NULL;
#endif

#ifdef IPFILTER
/*
 * The /dev/ipnat device node.
 */
static int				natdev = -1;
#endif

/*
 * Main loop.
 */
int main(int argc, char **argv)
{
	int					arg;
	int					run_as_server = 0;
	ipaddr_t			bind_ip = INADDR_ANY;
	short				bind_port = -1;
	char 				*server_hostname = NULL;
	short				server_port = -1;
	uid_t				run_uid = -1;
	gid_t				run_gid = -1;
#ifndef DEBUG
	int					fd;
#endif
	int					sock;
	struct sockaddr_in	addr;
	int					len;

	/*
	 * Parse the command line arguments.
	 */
	while ((arg = getopt(argc, argv, "dts:r:b:f:l:")) != EOF)
	{
		switch (arg)
		{
		case 'd':
			daemonize = 0;
			break;

		case 't':
			fully_transparent = 1;
			break;

		case 's':
			run_as_server = 1;
			bind_port = getportnum(optarg);
			break;

		case 'r':
			run_uid = getuserid(optarg);
			run_gid = getgroupid(run_uid);
			break;

		case 'b':
			bind_ip = getipaddress(optarg);
			break;

		case 'f':
			force_url = optarg;
			break;

		case 'l':
#ifdef LOG_TO_FILE
			log_file = fopen(optarg, "a");
			break;
#endif

		case '?':
			usage(argv[0], NULL);
			break;
		}
	}

	/*
	 * Process the remaining command line arguments.
	 */
	for (; optind < argc; ++optind)
	{
		if (server_hostname == NULL)
		{
			server_hostname = argv[optind];
		}
		else if (server_port == -1)
		{
			server_port = getportnum(argv[optind]);
		}
		else
		{
			usage(argv[0], "Extra arguments were specified.");
		}
	}

	/*
	 * Test to make sure required args were provided and are valid.
	 */
	if (server_hostname == NULL)
	{
		usage(argv[0], "No proxyhost was specified (or it was invalid).");
	}
	if (server_port == -1)
	{
		usage(argv[0], "No proxyport was specified (or it was invalid).");
	}
	if (run_as_server && (bind_ip == INADDR_NONE))
	{
		usage(argv[0], "The server ipaddr is invalid.");
	}
	if (run_as_server && (bind_port == -1))
	{
		usage(argv[0], "No server port was specified (or it was invalid).");
	}

#ifdef PATH_LOG_FILE
	if (log_file == NULL)
	{
		log_file = fopen(PATH_LOG_FILE, "a");
	}
#endif

#ifdef IPFILTER
	/*
	 * Now is a good time to open /dev/ipnat, before giving up our uid/gid.
	 */
	if ((natdev = open(IPL_NAT, O_RDONLY)) < 0)
	{
		perror("open(" IPL_NAT ")");
		exit(1);
	}
#endif

	/*
	 * See if we should run as a server.
	 */
	if (run_as_server)
	{
		/*
		 * Start by binding to the port, the child inherits this socket.
		 */
		sock = bind_to_port(bind_ip, bind_port);

		/*
		 * Start a server process.
		 */
		if (daemonize)
		{
			switch (fork())
			{
			case -1:
				/*
				 * We could not fork a daemon process.
				 */
				perror("fork()");
				exit(1);
			case 0:
				/*
				 * The child continues as the daemon process.
				 */
				break;
			default:
				/*
				 * Parent exits at this stage.
				 */
				_exit(0);
			}

			/*
			 * Start a new session and group.
			 */
			setsid();
		}

		/*
		 * Set the current directory to the root filesystem.
		 */
		chdir("/");

#ifndef DEBUG
		/*
		 * Point file descriptors to /dev/null, unless DEBUG is defined.
		 */
		if ((fd = open(_PATH_DEVNULL, O_RDWR, 0)) >= 0)
		{
			dup2(fd, STDIN_FILENO);
			dup2(fd, STDOUT_FILENO);
			dup2(fd, STDERR_FILENO);
			if (fd > STDERR_FILENO)
				close(fd);
		}
#endif

#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		/*
		 * Open syslog for logging errors.
		 */
		openlog(argv[0], LOG_PID, LOG_DAEMON);
#endif

		/*
		 * Ignore some signals.
		 */
		signal(SIGHUP, SIG_IGN);
		if (daemonize)
			signal(SIGINT, SIG_IGN);
		signal(SIGQUIT, SIG_IGN);
		signal(SIGTSTP, SIG_IGN);
		signal(SIGCONT, SIG_IGN);
		signal(SIGPIPE, SIG_IGN);
		signal(SIGALRM, alarm_signal);

		/*
		 * Drop back to an untrusted user.
		 */
		setgid(run_gid);
		setuid(run_uid);

		/*
		 * Handle the server main loop.
		 */
		server_main_loop(sock, server_hostname, server_port);

		/*
		 * Should never exit.
		 */
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		closelog();
#endif
		exit(1);
	}

#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
	/*
	 * Open syslog for logging errors.
	 */
	openlog(argv[0], LOG_PID, LOG_DAEMON);
#endif

	/*
	 * We are running from inetd so find the peer address.
	 */
	len = sizeof(addr);
	if (getpeername(STDIN_FILENO, (struct sockaddr *)&addr, &len) < 0)
	{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		syslog(LOG_ERR, "getpeername(): %m");
#ifndef DEBUG
		closelog();
#endif
#endif
#ifndef DEBUG
		exit(1);
#endif
	}

	/*
	 * Set the keepalive socket option to on.
	 */
	{
		int		on = 1;
		setsockopt(STDIN_FILENO, SOL_SOCKET, SO_KEEPALIVE, (char *)&on, sizeof(on));
	}

	/*
	 * We are running from inetd so process stdin.
	 */
	trans_proxy(STDIN_FILENO, &addr, server_hostname, server_port);
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
	closelog();
#endif

	return (0);
}

/*
 * Print some basic help information.
 */
static void usage(char *prog, char *opt)
{
	if (opt)
	{
		fprintf(stderr, "%s: %s\n", prog, opt);
	}
#ifdef LOG_TO_FILE
	fprintf(stderr, "usage: %s [-dt] [-f url] [-s port [-r user] [-b ipaddr]] [-l file] proxyhost proxyport\n", prog);
#else
	fprintf(stderr, "usage: %s [-dt] [-f url] [-s port [-r user] [-b ipaddr]] proxyhost proxyport\n", prog);
#endif
	fprintf(stderr, "    -d          When running as a server, do not background the daemon.\n");
	fprintf(stderr, "    -t          Act fully transparently (default is HTTP translate).\n");
	fprintf(stderr, "    -f url      Force access to always go to specified URL.\n");
	fprintf(stderr, "    -s port     Run as a server bound to the specified port.\n");
	fprintf(stderr, "    -r user     Run as the specified user in server mode.\n");
	fprintf(stderr, "    -b ipaddr   Bind to the specified ipaddr in server mode.\n");
#ifdef LOG_TO_FILE
	fprintf(stderr, "    -l file     Log accesses to the specified file.\n");
#endif
	exit(1);
}

/*
 * Return the port number, in network order, of the specified service.
 */
static short getportnum(char *portnum)
{
	char			*digits = portnum;
	struct servent	*serv;
	short			port;

	for (port = 0; isdigit(*digits); ++digits)
	{
		port = (port * 10) + (*digits - '0');
	}

	if ((*digits != '\0') || (port <= 0))
	{
		if ((serv = getservbyname(portnum, "tcp")) != NULL)
		{
			port = ntohs(serv->s_port);
		}
		else
		{
			port = -1;
		}
		endservent();
	}

#ifdef DEBUG
	fprintf(stderr, "Port lookup %s -> %hd\n", portnum, port);
#endif

	return (port);
}

/*
 * Return the IP address of the specified host.
 */
static ipaddr_t getipaddress(char *ipaddr)
{
	struct hostent	*host;
	ipaddr_t		ip;

	if (((ip = inet_addr(ipaddr)) == INADDR_NONE)
	    &&
		(strcmp(ipaddr, "255.255.255.255") != 0))
	{
		if ((host = gethostbyname(ipaddr)) != NULL)
		{
			memcpy(&ip, host->h_addr, sizeof(ip));
		}
		endhostent();
	}

#ifdef DEBUG
	fprintf(stderr, "IP lookup %s -> 0x%08lx\n", ipaddr, ip);
#endif

	return (ip);
}

/*
 * Find the userid of the specified user.
 */
static uid_t getuserid(char *user)
{
	struct passwd	*pw;
	uid_t			uid;

	if ((pw = getpwnam(user)) != NULL)
	{
		uid = pw->pw_uid;
	}
	else if (*user == '#')
	{
		uid = (uid_t)atoi(&user[1]);
	}
	else
	{
		uid = -1;
	}

#ifdef DEBUG
	fprintf(stderr, "User lookup %s -> %d\n", user, uid);
#endif

	endpwent();

	return (uid);
}

/*
 * Find the groupid of the specified user.
 */
static uid_t getgroupid(uid_t uid)
{
	struct passwd	*pw;
	gid_t			gid;

	if ((pw = getpwuid(uid)) != NULL)
	{
		gid = pw->pw_gid;
	}
	else
	{
		gid = -1;
	}

#ifdef DEBUG
	fprintf(stderr, "Group lookup %d -> %d\n", uid, gid);
#endif

	endpwent();

	return (gid);
}

/*
 * Bind to the specified ip and port.
 */
static int bind_to_port(ipaddr_t bind_ip, short bind_port)
{
	struct sockaddr_in	addr;
	int					sock;

	/*
	 * Allocate a socket.
	 */
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		perror("socket()");
		exit(1);
	}

#ifdef DEBUG
	/*
	 * Set the SO_REUSEADDR option for debugging.
	 */
	{
	 	int	on = 1;

		setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));
	}
#endif

	/*
	 * Set the address to listen to.
	 */
	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(bind_port);
	addr.sin_addr.s_addr = bind_ip;

	/*
	 * Bind our socket to the above address.
	 */
	if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
		perror("bind()");
		exit(1);
	}

	/*
	 * Establish a large listen backlog.
	 */
	if (listen(sock, SOMAXCONN) < 0)
	{
		perror("listen()");
		exit(1);
	}

	return (sock);
}

/*
 * Connect to the proxy server.
 */
static int connect_to_proxy(ipaddr_t ip, short port)
{
	struct sockaddr_in	addr;
	int					sock;

	/*
	 * Allocate a socket.
	 */
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		syslog(LOG_WARNING, "socket(): %m");
#endif
		return (-1);
	}

	/*
	 * Set the address to connect to.
	 */
	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);
	addr.sin_addr.s_addr = ip;

	/*
	 * Connect our socket to the above address.
	 */
	if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
	{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		syslog(LOG_WARNING, "connect(): %m");
#endif
		return (-1);
	}

	/*
	 * Set the keepalive socket option to on.
	 */
	{
		int		on = 1;
		setsockopt(STDIN_FILENO, SOL_SOCKET, SO_KEEPALIVE, (char *)&on, sizeof(on));
	}
	return (sock);
}

/*
 * Translate a sockaddr_in structure into a usable ASCII hostname.
 */
static void lookup_hostname(struct sockaddr_in *addr, char *hostname,
							int hostlen, int needed)
{
	struct hostent	*host;

	/*
	 * Get the hostname IP in dotted decimal in case the lookup fails.
	 */
	strncpy(hostname, inet_ntoa(addr->sin_addr), hostlen);
	hostname[hostlen - 1] = '\0';

#ifdef DNS_LOOKUPS
	/*
	 * If the needed flag is set then always do a lookup.
	 */
	if (needed)
	{
		if ((host = gethostbyaddr((char *)&addr->sin_addr,
								  sizeof(addr->sin_addr), AF_INET)) != NULL)
		{
			strncpy(hostname, host->h_name, hostlen);
			hostname[hostlen - 1] = '\0';
		}
		else
		{
#ifdef LOG_TO_SYSLOG
			syslog(LOG_INFO, "DNS lookup for %s failed: %m", hostname);
#endif
		}
	}
#ifdef USELESS_DNS_LOOKUPS
	else
	{
		if ((host = gethostbyaddr((char *)&addr->sin_addr,
								  sizeof(addr->sin_addr), AF_INET)) != NULL)
		{
			strncpy(hostname, host->h_name, hostlen);
			hostname[hostlen - 1] = '\0';
		}
		else
		{
#ifdef LOG_TO_SYSLOG
			syslog(LOG_INFO, "DNS lookup for %s failed: %m", hostname);
#endif
		}
	}
#endif
#endif
}

/*
 * This is the main loop when running as a server.
 */
static void server_main_loop(int sock, char *server_hostname, short server_port)
{
	int					new_sock;
	struct sockaddr_in	addr;
	int					len;

	/*
	 * Ignore dead servers so no zombies should be left hanging.
	 */
#ifdef SA_NOCLDWAIT
	{
		struct sigaction	sa;

		memset(&sa, 0, sizeof(sa));
		sa.sa_handler = SIG_IGN;
		sa.sa_flags = SA_NOCLDWAIT;
		sigaction(SIGCHLD, &sa, NULL);
	}
#else
 	signal(SIGCHLD, SIG_IGN);
#endif

	for (;;)
	{
		/*
		 * Accept an incoming connection.
		 */
		len = sizeof(addr);
		while ((new_sock = accept(sock, (struct sockaddr *)&addr, &len)) < 0)
		{
			/*
			 * Connection resets are common enough to log them as debug only.
			 */
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
			if (errno != ECONNRESET)
				syslog(LOG_ERR, "accept(): %m");
# ifdef LOG_TO_SYSLOG
			else
				syslog(LOG_DEBUG, "accept(): %m");
# endif
#endif
		}

		/*
		 * Create a new process to handle the connection.
		 */
		switch (fork())
		{
		case -1:
			/*
			 * Under load conditions just ignore new connections.
			 */
			break;

		case 0:
			/*
			 * Start the proxy work in the new socket.
			 */
			trans_proxy(new_sock, &addr, server_hostname, server_port);
			close(new_sock);
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
			closelog();
#endif
			exit(0);
		}

		/*
		 * Close the socket as the child does the handling.
		 */
		close(new_sock);
	}
}

/*
 * Perform the transparent proxy activity.
 */
static void trans_proxy(int sock, struct sockaddr_in *from_addr,
						char *server_hostname, short server_port)
{
	ipaddr_t			server_ip;
	struct sockaddr_in	to_addr;
	int					to_len;
	int					proxy;
	char				from_host[64];
	char				to_host[64];
	static char			headers[BUFFER_SIZE];
	int					header_len;
	static char			proxy_request[BUFFER_SIZE];
	int					mode;
	char				*current_pos;
	int					url_start = 0;
	int					url_end = 0;
	int					translate_url = 0;
	int					max_fd;
	fd_set				read_fd;
	int					read_len;
#ifdef IPFILTER
	natlookup_t			natlook;
#endif

	/*
	 * Lookup proxy server IP address. Because we do the lookup for every
	 * request we make use of DNS round-robin for load balancing.
	 */
	server_ip = getipaddress(server_hostname);
	if (server_ip == INADDR_NONE)
	{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		syslog(LOG_ERR, "Could not resolve IP address for proxy: %s", server_hostname);
#endif
		return;
	}

	/*
	 * The first thing we do is get the IP address that the client was
	 * trying to connected to. Here lies part of the magic. Normally
	 * getsockname returns our address, but not with transparent proxying.
	 */
	to_len = sizeof(to_addr);
	if (getsockname(sock, (struct sockaddr *)&to_addr, &to_len) < 0)
	{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		syslog(LOG_ERR, "getsockname(): %m");
#endif
#ifndef DEBUG
		return;
#endif
	}

#ifdef IPFILTER
	/*
	 * However, under IPFILTER, we have to do more work. The "to_addr" has
	 * been changed by the IP filter to trick the IP stack into processing
	 * the packet locally. We need to lookup the real remote address
	 * ourselves using an ioctl().
	 */

	/*
	 * Build up the NAT natlookup structure.
	 */
	memset(&natlook, 0, sizeof(natlook));
	natlook.nl_inip = to_addr.sin_addr;
	natlook.nl_outip = from_addr->sin_addr;
	natlook.nl_flags = IPN_TCP;
	natlook.nl_inport = to_addr.sin_port;
	natlook.nl_outport = from_addr->sin_port;

	/*
	 * Use the NAT device to lookup the mapping pair.
	 */
	if (ioctl(natdev, SIOCGNATL, &natlook) == -1)
	{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		syslog(LOG_ERR, "ioctl(SIOCGNATL): %m");
#endif
		return;
	}

	to_addr.sin_addr = natlook.nl_realip;
	to_addr.sin_port = natlook.nl_realport;
#endif

	/*
	 * Connect to the proxy server.
	 */
	proxy = connect_to_proxy(server_ip, server_port);

	/*
	 * Fully transparent saves a lot of trouble, but then again it doesn't
	 * support the transparent proxy feature. If the daemon is operating in
	 * HTTP translation mode then translate the request line.
	 */
	if (fully_transparent)
	{
		/*
		 * Lookup an ASCII representation of the host's IP address.
		 */
		lookup_hostname(from_addr, from_host, sizeof(from_host), 0);
		lookup_hostname(&to_addr, to_host, sizeof(to_host), 1);

		/*
		 * Log the facts about the connection.
		 */
#ifdef LOG_TO_SYSLOG
		syslog(LOG_INFO, "Transparent %s -> %s", from_host, to_host);
#endif
#ifdef LOG_TO_FILE
		if (log_file)
		{
			log_time();
			fprintf(log_file, "Transparent %s -> %s\n", from_host, to_host);
		}
#endif
	}
	else
	{
		/*
		 * Lookup an ASCII representation of the from host's IP address.
		 */
		lookup_hostname(from_addr, from_host, sizeof(from_host), 0);

		/*
		 * See if the destination site can be found in the HTTP headers.
		 */
		switch (get_site_from_headers(sock, to_host, sizeof(to_host),
								  headers, sizeof(headers) - 1,
								  &header_len))
		{
		case 1:
			/*
			 * Log the facts about the connection.
			 */
#ifdef LOG_TO_SYSLOG
			syslog(LOG_INFO, "Request NoDNS %s -> %s", from_host, to_host);
#endif
#ifdef LOG_TO_FILE
			if (log_file)
			{
				log_time();
				fprintf(log_file, "Request NoDNS %s -> %s\n", from_host, to_host);
			}
#endif
			break;

		case 0:
			/*
			 * Fallback is to use the DNS to find the destination site.
			 */
			lookup_hostname(&to_addr, to_host, sizeof(to_host), 1);

			/*
			 * Log the facts about the connection.
			 */
#ifdef LOG_TO_SYSLOG
			syslog(LOG_INFO, "Request %s -> %s", from_host, to_host);
#endif
#ifdef LOG_TO_FILE
			if (log_file)
			{
				log_time();
				fprintf(log_file, "Request %s -> %s\n", from_host, to_host);
			}
#endif
			break;

		case -1:
			close(proxy);
			return;
		}

#ifdef DEBUG_INPUT_REQUESTS
		/*
		 * Log the request headers.
		 */
#ifdef LOG_TO_SYSLOG
		syslog(LOG_NOTICE, "Request In: '%s'", headers);
#endif
#ifdef LOG_TO_FILE
		if (log_file)
		{
			log_time();
			fprintf(log_file, "Request In: '%s'\n", headers);
		}
#endif

#ifdef DEBUG
		fprintf(stderr, "Request In: '%s'\n", headers);
#endif
#endif

		/*
		 * Parse the request which is of the form...
		 *     METHOD [proto://host]/URI[ HTTP/1.0]\r\n
		 *     Header: data\r\n
		 *     ...
		 *     \r\n
		 *
		 * We use a state machine to parse the request.
		 */
		mode = IN_METHOD;
		for (current_pos = headers; *current_pos; ++current_pos)
		{
			switch (mode)
			{
			case IN_METHOD:
				append_char(proxy_request, BUFFER_SIZE, *current_pos);
				url_start = get_end_pos(proxy_request);
				if (*current_pos == ' ')
				{
					if (force_url)
					{
						append_string(proxy_request, BUFFER_SIZE, force_url);
						url_end = get_end_pos(proxy_request);
						mode = IN_SKIP_URL;
					}
					else
					{
						if (current_pos[1] == '/')
						{
							translate_url = 1;
							append_string(proxy_request, BUFFER_SIZE, "http://");
							append_string(proxy_request, BUFFER_SIZE, to_host);
						}
						mode = IN_URL;
					}
				}
				break;

			case IN_URL:
				if (*current_pos == ' ')
				{
					url_end = get_end_pos(proxy_request);
					mode = IN_VERSION;
				}
				else if (*current_pos == '\r')
				{
					url_end = get_end_pos(proxy_request);
					mode = IN_ALMOST_HEADER;
				}
				append_char(proxy_request, BUFFER_SIZE, *current_pos);
				break;

			case IN_SKIP_URL:
				if (*current_pos == ' ')
				{
					append_char(proxy_request, BUFFER_SIZE, *current_pos);
					mode = IN_VERSION;
				}
				else if (*current_pos == '\r')
				{
					append_char(proxy_request, BUFFER_SIZE, *current_pos);
					mode = IN_ALMOST_HEADER;
				}
				break;

			case IN_VERSION:
				append_char(proxy_request, BUFFER_SIZE, *current_pos);
				if (*current_pos == '\r')
					mode = IN_ALMOST_HEADER;
				break;

			case IN_ALMOST_HEADER:
				append_char(proxy_request, BUFFER_SIZE, *current_pos);
				if (*current_pos == '\n')
					mode = IN_HEADER;
				break;

			case IN_HEADER:
				if (*current_pos == '\r')
				{
					append_char(proxy_request, BUFFER_SIZE, *current_pos);
					mode = IN_ALMOST_HEADER;
				}
				else if (strncasecmp(current_pos, "Connection:", 11) == 0)
					mode = IN_SKIP_CONNECTION;
				else
					append_char(proxy_request, BUFFER_SIZE, *current_pos);
				break;
			
			case IN_SKIP_CONNECTION:
				if (*current_pos == '\n')
					mode = IN_HEADER;
				break;
			}
		}

		if (write(proxy, proxy_request, get_end_pos(proxy_request)) < 0)
		{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
			syslog(LOG_WARNING, "write(proxy) failed: %m");
#endif
			close(proxy);
			return;
		}

#ifdef DEBUG_OUTPUT_REQUESTS
#ifdef LOG_TO_SYSLOG
		syslog(LOG_NOTICE, "Request Out: '%s'", proxy_request);
#endif
#ifdef LOG_TO_FILE
		if (log_file)
		{
			log_time();
			fprintf(log_file, "Request Out: '%s'\n", proxy_request);
		}
#endif

#ifdef DEBUG
		fprintf(stderr, "Request Out: '%s'\n", proxy_request);
#endif
#endif

		/*
		 * Dump a nice log message showing what we have done.
		 */
		proxy_request[url_end] = '\0';
		if (force_url)
		{
#ifdef LOG_TO_SYSLOG
			syslog(LOG_NOTICE, "Forced '%s'", &proxy_request[url_start]);
#endif
#ifdef LOG_TO_FILE
			if (log_file)
			{
				log_time();
				fprintf(log_file, "Forced '%s'\n", &proxy_request[url_start]);
			}
#endif
#ifdef DEBUG
			fprintf(stderr, "Forced '%s'\n", &proxy_request[url_start]);
#endif
		}
		else if (translate_url)
		{
#ifdef LOG_TO_SYSLOG
			syslog(LOG_NOTICE, "Translated '%s'", &proxy_request[url_start]);
#endif
#ifdef LOG_TO_FILE
			if (log_file)
			{
				log_time();
				fprintf(log_file, "Translated '%s'\n", &proxy_request[url_start]);
			}
#endif
#ifdef DEBUG
			fprintf(stderr, "Translated '%s'\n", &proxy_request[url_start]);
#endif
		}
		else
		{
#ifdef LOG_TO_SYSLOG
			syslog(LOG_NOTICE, "Untranslated '%s'", &proxy_request[url_start]);
#endif
#ifdef LOG_TO_FILE
			if (log_file)
			{
				log_time();
				fprintf(log_file, "Untranslated '%s'\n", &proxy_request[url_start]);
			}
#endif
#ifdef DEBUG
			fprintf(stderr, "Untranslated '%s'\n", &proxy_request[url_start]);
#endif
		}
	}

	/*
	 * Continue by passing data back and forth between the client and proxy.
	 */
	for (;;)
	{
		/*
		 * Construct a select read mask from both file descriptors.
		 */
		FD_ZERO(&read_fd);
		FD_SET(sock, &read_fd);
		FD_SET(proxy, &read_fd);
		max_fd = FD_MAX(sock, proxy);

		/*
		 * Allow 2 minutes for any activity.
		 */
		alarm(120);

		/*
		 * Wait for some data to be read.
		 */
		if (select(max_fd + 1, &read_fd, NULL, NULL, NULL) < 0)
		{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
			syslog(LOG_ERR, "select(): %m");
#endif
			close(proxy);
			return;
		}

		/*
		 * We have something so disable the alarm.
		 */
		alarm(0);

		/*
		 * See if any data can be read from the client.
		 */
		if (FD_ISSET(sock, &read_fd))
		{
			switch (read_len = read(sock, headers, sizeof(headers) - 1))
			{
			case -1:
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
				if (errno != ECONNRESET)
					syslog(LOG_WARNING, "read(client) failed: %m");
#ifdef LOG_TO_SYSLOG
				else
					syslog(LOG_DEBUG, "read(client) failed: %m");
#endif
#endif
				close(proxy);
				return;

			case 0:
				close(proxy);
				return;

			default:
				if (write(proxy, headers, read_len) < 0)
				{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
					syslog(LOG_WARNING, "write(proxy) failed: %m");
#endif
					close(proxy);
					return;
				}
				break;
			}
		}

		/*
		 * See if any data can be read from the proxy.
		 */
		if (FD_ISSET(proxy, &read_fd))
		{
			switch (read_len = read(proxy, headers, sizeof(headers) - 1))
			{
			case -1:
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
				if (errno != ECONNRESET)
					syslog(LOG_WARNING, "read(proxy) failed: %m");
				else
					syslog(LOG_DEBUG, "read(proxy) failed: %m");
#endif
				close(proxy);
				return;

			case 0:
				close(proxy);
				return;

			default:
				if (write(sock, headers, read_len) < 0)
				{
#ifdef LOG_TO_SYSLOG
					syslog(LOG_WARNING, "write(client) failed: %m");
#endif
					close(proxy);
					return;
				}
				break;
			}
		}
	}
}

/*
 * Read in the headers for the HTTP reset and store in the headers array.
 * Then search the headers for a Host: hostname[:port] header.
 */
static int	get_site_from_headers(int sock, char *hostname, int hostlen,
								  char *headers, int header_max,
								  int *header_len)
{
	int		read_len;
	char	*start;
	char	*end;

	/*
	 * Allow 2 minutes to read in the headers.
	 */
	alarm(120);

	/*
	 * Read in a buffers worth of data, hopefully all of the headers
	 * will be contained in this initial read. This is tagged with a
	 * big !!FIX ME!!
	 */
	switch (read_len = read(sock, headers, header_max))
	{
	case -1:
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
		if (errno != ECONNRESET)
			syslog(LOG_WARNING, "read(client) failed: %m");
#ifdef LOG_TO_SYSLOG
		else
			syslog(LOG_DEBUG, "read(client) failed: %m");
#endif
#endif
		return (-1);

	case 0:
		return (0);
	}
	*header_len = read_len;

	/*
	 * Terminate the buffer so the string functions don't run
	 * of the end.
	 */
	headers[read_len] = '\0';

	/*
	 * See if the specified string exists in the buffer.
	 */
	if ((start = strstr(headers, "\r\nHost: ")) != NULL)
	{
		/*
		 * We found it, so now skip over it.
		 */
		start += strlen("\r\nHost: ");

		/*
		 * Now see if the field we want is terminated by one
		 * the termination characters.
		 */
		if ((end = strpbrk(start, ":\r\n\t ")) != NULL)
		{
			/*
			 * We found the end, now copy the field to the output.
			 */
			while ((start < end) && (hostlen-- > 1))
			{
				*hostname++ = *start++;
			}
			*hostname = '\0';

			/*
			 * Return success.
			 */
			return (1);
		}
	}

	/*
	 * Return failure.
	 */
	return (0);
}

/*
 * Catch alarm signals and exit.
 */
static void alarm_signal(int sig)
{
#if defined(LOG_TO_SYSLOG) || defined(LOG_FAULTS_TO_SYSLOG)
	syslog(LOG_DEBUG, "Alarm signal caught - connection timeout");
#endif
#ifdef LOG_TO_FILE
	if (log_file)
	{
		log_time();
		fprintf(log_file, "Alarm signal caught - connection timeout\n");
	}
#endif
	exit(1);
}

static void log_time(void)
{
	time_t	now;
	char	date[64];

	if (log_file)
	{
		time(&now);
		strftime(date, sizeof(date), "%a, %d %b %Y %H:%M:%S", localtime(&now));
		fprintf(log_file, "%s  ", date);
	}
}

/*
 * Appends source string to destination string.
 */
static void append_string(char *dest, int max_size, char *source)
{
	size_t	dest_len;
	size_t	source_len;
	size_t	max_add_len;
 
 	/*
	 * Get the length of what is already there, and what we have to add.
	 */
	dest_len = strlen(dest);
	source_len = strlen(source);
 
 	/*
	 * Calculate how many characters we can append.
	 */
 	if ((dest_len + source_len + 1) < max_size)
		max_add_len = source_len;
	else
		max_add_len = max_size - dest_len - 1;

	/*
	 * If any space at all available, then append what we can.
	 */
	if (max_add_len > 0)
	{
		memcpy(&dest[dest_len], source, max_add_len);
		dest[dest_len + max_add_len] = '\0';
	}
}

static void append_char(char *dest, int max_size, char source)
{
	size_t	dest_len;
 
 	/*
	 * Get the length of what is already there.
	 */
	dest_len = strlen(dest);
 
	/*
	 * If any space at all available, then append what we can.
	 */
	if ((dest_len + 1) < max_size)
	{
		dest[dest_len] = source;
		dest[dest_len + 1] = '\0';
	}
}
