/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <pwd.h>
#include <string.h>
#include <sys/wait.h>
#include <utmp.h>
#include "common.h"
#include "hash.h"

static int                      userExcluded(uid_t uid);
static int                      flagsExcluded(word flags);

static int                      nProhibited = 0;
static int                      prohibited[16];
static int                      nRequired = 0;
static int                      required[16];
static HashTable                userRecTable;
static Boolean                  userFileOnly;

int
acua_forEach(int argc, char **argv)
{
  struct passwd           *pw;
  int                      i,
                           nArgs = 0;
  char                     login[MAX_LOGINCHARS + 1],
                          *arg[64];
  UserRec                  ur,
                          *urPtr;

  readConfig();

  for (optind = 1; (optind < argc) && (argv[optind][0] == '-'); ) {
    i = argv[optind++][1];
    switch (i) {
    case 'p':
      prohibited[nProhibited++] = atoi(argv[optind++]); break;
    case 'r':
      required[nRequired++] = atoi(argv[optind++]); break;
    case 'u':
      userFileOnly = TRUE; break;
    }
  }
  
  if (optind >= argc) 
    errQuit("usage: forEach {-p n} {-r n} [-u] <command>");
  
  for (i = optind + 1; i < argc; i++) {
    if (!strcmp(argv[i], "{}"))
      arg[nArgs++] = login;
    else
      arg[nArgs++] = argv[i];
  }
  arg[nArgs] = NULL;

  userFileOpen();
  hashTableInit(&userRecTable, 0x4000, UIDhash, UIDcomp, 0, TRUE);
  while (!userFileRead(&ur)) {
    urPtr = (UserRec*)malloc(sizeof(UserRec));
    *urPtr = ur;
    hashTableAdd(&userRecTable, urPtr);
  }
  userFileClose();

  setpwent();
  while ((pw = getpwent()) != NULL) {
    if (userExcluded(pw->pw_uid))
      continue;
    urPtr = (UserRec*)hashTableSearch(&userRecTable, &pw->pw_uid);
    if (userFileOnly && !urPtr)
      continue;
    if (urPtr && flagsExcluded(urPtr->flags))
      continue;
    strncpy(login, pw->pw_name, MAX_LOGINCHARS);
    login[MAX_LOGINCHARS] = '\0';
    runCommandV(argv[optind], arg);
  }
  endpwent();
  
  hashTableDeinit(&userRecTable);
  return 0;
}

int
userExcluded(uid_t uid)
{
  int                      i;

  for (i = 0; i < nExcluded; i++)
    if (excluded[i] == uid)
      return 1;
  return 0;
}

int
flagsExcluded(word flags) {
  int                      i;
  word                     mask;

  for (i = 0; i < nProhibited; i++) {
    mask = 0x00010000 << prohibited[i];
    if (flags & mask) return 1;
  }
  for (i = 0; i < nRequired; i++) {
    mask = 0x00010000 << required[i];
    if (!(flags & mask)) return 1;
  }
  return 0;
}
