/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <ctype.h>
#include <limits.h>
#include <string.h>
#include <sys/stat.h>
#include <syslog.h>
#include "common.h"

#define   MSG_DEFAULT           0
#define   MSG_LOCK              1
#define   MSG_MAX_LOGINS        2
#define   MSG_BUSY              3
#define   MSG_DATA_PERIOD       4
#define   MSG_TIME_CLASS        5
#define   MSG_TIME_PERIOD       6
#define   MSG_SESSION_WAIT      7
#define   MSG_NO_RECORD         8

void                     printFile(char *path);
word                     nLoginsj(HashTable *loginRec, uid_t uid);

const char              *msgPath[9] = { LIB"/acua_login.default",
                                        LIB"/acua_login.lock",
                                        LIB"/acua_login.max_logins",
                                        LIB"/acua_login.busy",
                                        LIB"/acua_login.data_period",
                                        LIB"/acua_login.time_class",
                                        LIB"/acua_login.time_period",
					LIB"/acua_login.session_wait",
                                        LIB"/acua_login.no_record" };

int                      nLoginRecs;
LoginRec                 loginRec[MAX_USERS];
int                      nProcs;
ProcRec                  procRec[MAX_PROCS];

int
main(int argc, char **argv)
{
  int                      i,
                           msg = -1,
                           curClass = -1;
  dev_t                    tty;
  uid_t                    uid = getuid();
  UserRec                  ur,
                           ur2;
  int                      sLeft,
                           sLeft2;
  uid_t                    guestUID;
  int                      nLower = 0,
                           systemBusy;
  char                     hostname[256],
                           dPath[256];
  Boolean                  loginFound,
                           userRecFound = FALSE;
  HashTable                loginRecTable;
  word                     myHostAddr;
  time_t                   lastOnlineTime;
  int                      hostIdx;

  readConfig();
  gethostname(hostname, 256); myHostAddr = hostAddr(hostname);
  tty = devNumFromName(ttyname(STDIN_FILENO) + 5);
  guestUID = UIDfromLogin("guest");
  userList(&nLoginRecs, loginRec);
  procList(&nProcs, procRec);
  systemBusy = nKick(nProcs, procRec);
  hashTableInit(&loginRecTable, MAX_USERS, UIDhash, UIDcomp, 0, FALSE);
  for (i = 0; i < nLoginRecs; i++)
    hashTableAdd(&loginRecTable, &loginRec[i]);
  userFileOpen();
  if (!uid) {
    if (argc == 1) {
      tty = devNumFromName(ttyname(STDIN_FILENO) + 5);
      for (i = 0; i < nLoginRecs; i++)
        if (loginRec[i].tty == tty) {
          uid = loginRec[i].uid;
          break;
        }
      if (i == nLoginRecs) {
        msg = MSG_DEFAULT;
        goto reject;
      }
    } else {
      uid = UIDfromLogin(argv[1]);
      if (uid == (uid_t)-1) {
        msg = MSG_DEFAULT;
        goto reject;
      }
      loginFound = FALSE;
      for (i = 0; i < nLoginRecs; i++)
        if ((lineNo(loginRec[i].tty) >= 0) && (loginRec[i].uid == uid)) {
          loginFound = TRUE;
          tty = loginRec[i].tty;
        }
      if (loginFound == FALSE) {
        msg = MSG_DEFAULT;
        goto reject;
      }
    }
  }
  /* Check if this is an ACUA line (acua.config, Devices directive) */
  if (lineNo(tty) < 0)
    goto accept;
  if (uid == guestUID) {
    for (i = 0; i < nLoginRecs && loginRec[i].tty != tty; i++);
    if (i == nLoginRecs) {
      msg = MSG_DEFAULT;
      goto reject;
    }
    bzero(&ur, sizeof(UserRec));
    ur.flags = optGuestPriority << 13;
    ur.tLeft = ur.sLeft = optGuestTime - ((time(NULL) - loginRec[i].time) / 60);
    ur.tLimit = optGuestTime;
    ur.sLeft = ur.sLimit = ur.tLimit;
    for (i = 0; i < MAX_TIME_CLASSES; i++) {
      ur.cLeft[i] = 0;
      ur.cLimit[i] = -1;
    }
    goto guest;
  }

  // find their user record
  if (userFileSearch(&ur, uid)) {
      if (optNoRecLogin == 1) {
	  goto accept;
      } else {
          msg = MSG_NO_RECORD;
	  goto reject;
      }
  }
  userRecFound = TRUE;
  curClass = curTimeClass(&ur);

  // are they locked out?
  if (LOCK(ur.flags)) {
    msg = MSG_LOCK;
    goto reject;
  }

  // update nLogins
  addHost(&ur, myHostAddr);
  hostIdx = findHost(&ur, myHostAddr);
  /* Should call nLogins, but overloading wasn't working */
  ur.nLogins[hostIdx] = nLoginsj(&loginRecTable, ur.uid);
  
  // impose limit on max. simultaneous sessions
  if (nLogins(&ur) > ur.maxLogins) {
    msg = MSG_MAX_LOGINS;
    goto reject;
  }

  // don't allow login to be done after 60 seconds
  loginFound = FALSE;
  for (i = 0; i < nLoginRecs; i++)
    if ((loginRec[i].uid == uid) && (loginRec[i].tty == tty)) {
      if (getuid() && (time(NULL) - loginRec[i].time >= 60)) {
        msg = MSG_DEFAULT;
        goto reject;
      }
      loginFound = TRUE;
      break;
    }

  /* if we couldn't find a login record... */
  if (!loginFound) {
      if (optNoRecLogin == 1) {
	  goto accept;
      } else {
	  msg = MSG_NO_RECORD;
	  goto reject;
      }
  }

guest:
  if (ur.tLimit == 0) {
    msg = MSG_DEFAULT;
    goto reject;
  }
  if ((ur.bLimit && ur.bTx + ur.bRx > ur.bLimit) ||
      (ur.bTxLimit && ur.bTx > ur.bTxLimit) ||
      (ur.bRxLimit && ur.bRx > ur.bRxLimit))
  {
    msg = MSG_DATA_PERIOD;
    goto reject;
  }

  // calculate sLeft
  sLeft = INT_MAX;
  if ((!SMARTBOOT(ur.flags) || systemBusy) && ur.tLimit >= 0)
    sLeft = min(sLeft, ur.tLeft + ur.credit);
  if ((!SSMARTBOOT(ur.flags) || systemBusy) && ur.sLimit >= 0)
    sLeft = min(sLeft, ur.sLimit);
  if ((!TCSMARTBOOT(ur.flags) || systemBusy) && curClass >= 0 && ur.cLimit[curClass] >= 0)
    sLeft = min(sLeft, ur.cLeft[curClass]);
  if (!PRIORITY(ur.flags) && systemBusy)
    sLeft = min(sLeft, 0);
  if ((sLeft <= 0) && !optMaxKick)
    goto reject;

  // impose ReturnDelay
  lastOnlineTime = lastOnline(&ur);
  if ((!SSMARTBOOT(ur.flags) || systemBusy) &&
      (ur.sLimit > 0) &&
      (ur.sLeft <= (int)(0.10 * ur.sLimit)) &&
      (lastOnlineTime != 0) &&
      ((time(NULL) - lastOnlineTime) < (optReturnDelay * 60)))
  {
    msg = MSG_SESSION_WAIT;
    goto reject;
  }
  goto accept;

  /* 
   * If we get this far, then:
   * 
   * o The user has no time
   * o Max-Kick is enabled
   * 
   * The decision is non-trivial -- the user could potentially get booted off
   * the system the next time acua_updated does an update.  Now a check is done
   * to see whether or not that would happen.  If it would happen, we will boot
   * the user off now.  There is no point (except perhaps to indulge in a
   * warped sense of humour) in allowing a user on, only to boot them off
   * in < 60 seconds.  Of course, that can still happen despite the most
   * strenuous checking, since users may come and go at any time...
   */
  userFileRewind();
  while (!userFileRead(&ur2)) {
    if (ur2.uid == uid)
      continue;
    for (i = 0; i < nProcs; i++)
      if ((ur2.uid == procRec[i].uid || ur2.uid == procRec[i].euid) &&
          (lineNo(procRec[i].tty) >= 0))
        break;
    if (i == nProcs)
      continue;
    sLeft2 = INT_MAX;
    if ((!SMARTBOOT(ur2.flags) || systemBusy) && ur2.tLimit >= 0)
      sLeft2 = min(sLeft2, ur2.tLeft + ur2.credit);
    if ((!SSMARTBOOT(ur2.flags) || systemBusy) && ur2.sLimit >= 0)
      sLeft2 = min(sLeft2, ur2.sLeft);
    if ((!TCSMARTBOOT(ur2.flags) || systemBusy) && curClass >= 0 && ur2.cLimit[curClass] >= 0)
      sLeft2 = min(sLeft2, ur2.cLeft[curClass]);
    if (!PRIORITY(ur2.flags) && systemBusy)
      sLeft2 = min(sLeft2, -(ur2.sLimit - ur2.sLeft));
    if (PRIORITY(ur2.flags) < PRIORITY(ur.flags) || (PRIORITY(ur2.flags) == PRIORITY(ur.flags) && sLeft2 < sLeft))
      nLower++;
  }
  for (i = 0; i < nLoginRecs; i++) {
    if (loginRec[i].tty == tty)
      continue;
    if (loginRec[i].uid != guestUID)
      continue;
    sLeft2 = (optGuestTime * 60 - (time(NULL) - loginRec[i].time)) / 60 + 1;
    if (!optGuestPriority && systemBusy)
      sLeft2 = min(sLeft2, -(optGuestTime - sLeft2));
    if (optGuestPriority < (int)PRIORITY(ur.flags) || (optGuestPriority == (int)PRIORITY(ur.flags) && sLeft2 < sLeft))
      nLower++;
  }
  if (nLower >= systemBusy)
    goto accept;
reject:
  userFileClose();
  if (msg < 0) {
    if (ur.tLeft <= 0)
      msg = MSG_TIME_PERIOD;
    else if (curClass >= 0 && ur.cLimit[curClass] >= 0 && ur.cLeft[curClass] <= 0)
      msg = MSG_TIME_CLASS;
    else if (!PRIORITY(ur.flags) && systemBusy)
      msg = MSG_BUSY;
    else
      msg = MSG_DEFAULT;
  }

  /* Create tempory file */
  snprintf(dPath, sizeof(dPath), "/tmp/acua_login.XXXXXX");
  if (mkstemp(dPath) == -1) {
      /* couldn't make the tempory file */
      syslog(LOG_ERR, "Could not create tempory file in /tmp.  Login denied.");
      printf("\n\nSorry, you are not able to login at this time due to a\n");
      printf("technical error.  Please contact your service provider.\n\n\n");
      fflush(stdout);
      sleep(5);
      return(1);
  }

  preprocessFile(&ur, (char*)msgPath[msg], dPath);
  printFile(dPath);
  unlink(dPath);
  fflush(stdout);
  sleep(5);
  return(1);
accept:
  if (userRecFound) {
    ur.sLeft = ur.sLimit;
    ur.bStx = ur.bSrx = 0;
    hostIdx = findHost(&ur, myHostAddr);
    if (ur.lastLogin[hostIdx] == 0) {
      ur.lastLogin[hostIdx] = time(NULL);
      ur.lastOnline[hostIdx] = time(NULL);
    }
    bzero(ur.nBytes, 60 * sizeof(word));
    userFileEdit(&ur);
  }
  userFileClose();
  hashTableDeinit(&loginRecTable);
  return(0);
}

void
printFile(char *path)
{
  FILE                    *f = fopen(path, "r");
  int                     s;

  if (!f) perrQuit("fopen");
  printf("\n");
  while ((s = fgetc(f)) != EOF) putchar(s);
  printf("\n");
  fclose(f);
}

word nLoginsj(HashTable *loginRec, uid_t uid)
{
    int numLogins = 0;

    LoginRec *rec = (LoginRec*)hashTableSearch(loginRec, &uid);
    if (rec != NULL) {
	if (lineNo(rec->tty) >= 0)
	    numLogins++;
	/* Modified, to update rec at next search */
	while ((rec = (LoginRec*)hashTableSearchNext(loginRec, &uid)) != NULL)
	    if (lineNo(rec->tty) >= 0)
		numLogins++;
    }

    return(numLogins);
}

