/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <ctype.h>
#include <limits.h>
#include <string.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <utime.h>
#include "common.h"

#define MAX_EXEMPT  MAX_USERS

void                     processCleanTmpConfig();
void                     cleanTmp(char *path);
void                     appendSlash(char *s);
int                      userExempt(uid_t uid);

int                      aTime,
                         cTime,
                         mTime;
int                      nPaths;
int                      nExempt;
uid_t                    exempt[MAX_EXEMPT];
int                      nLoginRecs;
LoginRec                 loginRec[MAX_USERS];

int
main()
{
  userList(&nLoginRecs, loginRec);
  processCleanTmpConfig();
  return 0;
}

void
processCleanTmpConfig()
{
  int                      i,
                           li;
  uid_t                    uid;
  char                     line[256],
                           token[256];
  FILE                    *f;

  if ((f = fopen(LIB"/clean_tmp.config", "rt")) == NULL)
    perrQuit("fopen");
  while (fgets(line, 256, f) != NULL) {
    li = 0;
    if (getToken(token, line, &li))
      continue;
    if (!strcasecmp(token, "path") && nPaths < 16) {
      if (getToken(token, line, &li))
        continue;
      cleanTmp(token);
    } else if (!strcasecmp(token, "atime")) {
      if (getToken(token, line, &li))
        continue;
      aTime = atoi(token) * (24 * 60 * 60);
    } else if (!strcasecmp(token, "ctime")) {
      if (getToken(token, line, &li))
        continue;
      cTime = atoi(token) * (24 * 60 * 60);
    } else if (!strcasecmp(token, "mtime")) {
      if (getToken(token, line, &li))
        continue;
      mTime = atoi(token) * (24 * 60 * 60);
    } else if (!strcasecmp(token, "exempt")) {
      while (!getToken(token, line, &li) && nExempt < MAX_EXEMPT)
        if ((uid = UIDfromLogin(token)) != (uid_t) - 1)
          exempt[nExempt++] = uid;
    } else if (!strcasecmp(token, "unexempt")) {
      while (!getToken(token, line, &li)) {
        if (token[0] == '*') nExempt = 0;
        else if ((uid = UIDfromLogin(token)) != (uid_t)-1)
          for (i = 0; i < nExempt; i++)
            if (exempt[i] == uid) {
              exempt[i] = exempt[nExempt - 1];
              nExempt--;
            }
      }
    }
  }
  fclose(f);
}

void
cleanTmp(char *path)
{
  time_t                   t = time(NULL);
  struct stat              _stat;
  int                      rm;
  char                    *tpath = (char *) malloc(PATH_MAX + 1);
  DIR                     *dir = opendir(path);
  struct dirent           *_dirent;

  if (dir == NULL) {
    perror("opendir");
    return;
  }
  while ((_dirent = readdir(dir)) != NULL) {
    if (strcmp(_dirent->d_name, ".") && strcmp(_dirent->d_name, "..")) {
      strcpy(tpath, path);
      appendSlash(tpath);
      strcat(tpath, _dirent->d_name);
      if (lstat(tpath, &_stat))
        continue;
      if (_stat.st_atime > t || _stat.st_ctime > t || _stat.st_mtime > t) {
        utime(tpath, NULL);
        if (lstat(tpath, &_stat))
          continue;
      }
      rm = (!userExempt(_stat.st_uid) &&
            !userLoggedIn(nLoginRecs, loginRec, _stat.st_uid) &&
            t - _stat.st_atime >= aTime &&
            t - _stat.st_ctime >= cTime &&
            t - _stat.st_mtime >= mTime);
      if (S_ISDIR(_stat.st_mode)) {
        cleanTmp(tpath);
        if (rm)
          rmdir(tpath);
      } else {
        if (rm)
          unlink(tpath);
      }
    }
  }
  closedir(dir);
  free(tpath);
}

void
appendSlash(char *s)
{
  int                      len = strlen(s);

  if (s[len - 1] != '/') {
    s[len] = '/';
    s[len + 1] = '\0';
  }
}

int
userExempt(uid_t uid)
{
  int                      i;

  for (i = 0; i < nExempt; i++)
    if (exempt[i] == uid)
      return 1;
  return 0;
}
