/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <assert.h>
#include <errno.h>
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#include <netdb.h>
#include <netinet/in.h>
#include <paths.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <syslog.h>
#include <sys/dir.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include "common.h"

#define TEMP_BUF 256
#define DEV_NAME 16

void
errQuit(char *format,...)
{
  char                     buf[TEMP_BUF];
  va_list                  ap;

  va_start(ap, format);
  vsnprintf(buf, TEMP_BUF, format, ap);
  fprintf(stderr, "%s\n", buf);
  if (userFile)
    userFileClose();
  if (banFile)
    banFileClose();
  va_end(ap);
  exit(1);
}

// same as errQuit() except logs to syslog also.
void
errlQuit(char *format,...)
{
  // FIXME: fixed length
  char                     buf[256];
  va_list                  ap;

  va_start(ap, format);
  vsprintf(buf, format, ap);
  fprintf(stderr, "%s\n", buf);
  syslog(LOG_ERR, "%s", buf);
  if (userFile)
    userFileClose();
  if (banFile)
    banFileClose();
  va_end(ap);
  closelog();  // close syslog.
  exit(1);
}

void
perrQuit(char *format,...)
{
  char                     buf[TEMP_BUF];
  va_list                  ap;

  va_start(ap, format);
  vsnprintf(buf, TEMP_BUF, format, ap);
  perror(buf);
  if (userFile)
    userFileClose();
  if (banFile)
    banFileClose();
  va_end(ap);
  exit(1);
}

// same as perrQuit() except logs to syslog.
void
perrlQuit(char *format,...)
{
  // FIXME: fixed length
  // FIXME: check for formatting bug problems.
  char                     buf[256];
  va_list                  ap;

  va_start(ap, format);
  vsprintf(buf, format, ap);
  perror(buf);
  syslog(LOG_ERR, "%s", buf);
  if (userFile)
    userFileClose();
  if (banFile)
    banFileClose();
  va_end(ap);
  closelog();   // close syslog.
  exit(1);
}

dev_t
devNumFromName(char *devName)
{
  struct stat              _stat;
  char                     devPath[DEV_NAME];

  snprintf(devPath, DEV_NAME, "/dev/%s", devName);
  if (stat(devPath, &_stat) < 0)
    return (dev_t)-1;
  return _stat.st_rdev;
}

int
lineNo(dev_t dev)
{
  int                      i;
  
  for (i = 0; i < nLines; i++)
    if (lineDev[i] == dev)
      return i;
  return -1;
}

int
runCommand(char *path, ...)
{
  char                    *arg[16];
  va_list                  ap;
  int                      status,
                           i;
  pid_t                    pid;

  va_start(ap, path);
  for (i = strlen(path) - 1; i >= 0 && path[i] != '/'; i--);
  arg[0] = path + i + 1;
  for (i = 1; i < 16; i++) {
    arg[i] = va_arg(ap, char *);
    if (arg[i] == NULL)
      break;
  }
  if ((pid = fork()) < 0)
    perrQuit("fork");
  else if (!pid) {
    execvp(path, arg);
    perrQuit("execv");
  } else
    waitpid(pid, &status, 0);
  va_end(ap);
  return WEXITSTATUS(status);
}

int
runCommandV(char *path, char **_arg)
{
  char                    *arg[16];
  int                      status,
                           i;
  pid_t                    pid;

  for (i = strlen(path) - 1; i >= 0 && path[i] != '/'; i--);
  arg[0] = path + i + 1;
  for (i = 1; _arg[i - 1] != NULL; i++)
    arg[i] = _arg[i - 1];
  arg[i] = NULL;
  if ((pid = fork()) < 0)
    perrQuit("fork");
  else if (!pid) {
    execvp(path, arg);
    perrQuit("execv");
  } else
    waitpid(pid, &status, 0);
  return WEXITSTATUS(status);
}

void
daemonInit()
{
  int                      i;

  signal(SIGHUP, SIG_IGN);
  if (fork())
    exit(0);
  setsid();
  for (i = 0; i <= 2; i++)
    close(i);
  chdir("/");
  umask(0077);
}

time_t
addTime(time_t t, char *s)
{
  time_t                   res,
                           endOfDay,
                           endOfMonth,
                           endOfYear;
  struct tm               *tmP = localtime(&t);
  int                      len = strlen(s);
  char                     str[16];
  char                     lastC = tolower(s[len - 1]);
  int                      i,
                           j,
                           n;

  strcpy(str, s);
  if (!isdigit(lastC))
    str[--len] = '\0';

  for (i = 0; i < len; i++)
    if (str[i] == '/') break;
  if (i != len) {
    struct tm _tm;

    bzero(&_tm, sizeof(_tm));
    sscanf(str, "%d/%d/%d", &_tm.tm_year, &_tm.tm_mon, &_tm.tm_mday);
    if (_tm.tm_year < 98)
      _tm.tm_year += 100;
    _tm.tm_mon -= 1;
    return mktime(&_tm);
  }

  n = atoi(str);
  if (n <= 0) return 0;
  endOfDay = t +
    (60 - tmP->tm_sec) +
    60 * (59 - tmP->tm_min) +
    60 * 60 * (23 - tmP->tm_hour);
  if (isdigit(lastC) || lastC == 'd')
    return endOfDay + (n - 1) * 24 * 60 * 60;
  endOfMonth = endOfDay +
    24 * 60 * 60 * (days[tmP->tm_mon] - 1 - (tmP->tm_mday - 1));
  if (lastC == 'm') {
    for (res = endOfMonth, j = (tmP->tm_mon + 1) % 12, i = 1; i < n; i++) {
      res += days[j] * 24 * 60 * 60;
      j = (j + 1) % 12;
    }
    return res;
  }
  for (endOfYear = endOfMonth, i = tmP->tm_mon + 1; i < 12; i++)
    endOfYear += 24 * 60 * 60 * days[i];
  if (lastC == 'y')
    return endOfYear + 365 * 24 * 60 * 60 * (n - 1);
  return 0;
}

void
procList(int *nProcs, ProcRec * procRec)
{
  int                      i,
                           li;
  FILE                    *f;
  char                     tpath[32],
                           line[256],
                           token[256];
  DIR                     *dir = opendir("/proc");
  struct dirent           *_dirent;

  if (!dir)
    perrQuit("opendir");
  *nProcs = 0;
  while ((_dirent = readdir(dir)) != NULL) {
    if (!isdigit(_dirent->d_name[0]))
      continue;
    procRec[*nProcs].pid = atoi(_dirent->d_name);
    sprintf(tpath, "/proc/%s/stat", _dirent->d_name);
    f = fopen(tpath, "r");
    if (!f)
      continue;
    if (!fgets(line, 256, f))
      goto l0;
    for (li = 0, i = 0; i < 2; i++)
      if (getToken(token, line, &li))
        goto l0;
    token[strlen(token) - 1] = '\0';
    strncpy(procRec[*nProcs].name, token + 1, 15);
    procRec[*nProcs].name[15] = '\0';
    for (i = 2; i < 7; i++)
      if (getToken(token, line, &li))
	  goto l0;
    procRec[*nProcs].tty = atoi(token);
    fclose(f);
    sprintf(tpath, "/proc/%s/status", _dirent->d_name);
    f = fopen(tpath, "r");
    if (!f)
	continue;

    while (!feof(f))
    {
	/* Find the line in the /proc/pid/status file that starts with Uid:
	 * This file got changed between Kernel 2.2 and 2.4, and caused
         * ACUA to not work on 2.4 kernels properley.
	 */
	if (!fgets(line, 256, f))
	    goto l0;
	if (!strncmp(line, "Uid:", 4))
            break;  /* Found the right line */
    }
    // Linux 2.2
    //    for (i = 0; i < 5; i++)
    // Linux 2.4
    //for (i = 0; i < 6; i++)
    //    if (!fgets(line, 256, f))
    //        goto l0;
    li = 0;
    if (getToken(token, line, &li))
      goto l0;
    if (getToken(token, line, &li))
      goto l0;
    procRec[*nProcs].uid = atoi(token);
    if (getToken(token, line, &li))
      goto l0;
    procRec[*nProcs].euid = atoi(token);
    (*nProcs)++;
l0:
    fclose(f);
  }
  closedir(dir);
}

int
nKick(int nProcs, ProcRec * procRec)
{
  int                      i,
                           n;
  byte                     lineBusy[MAX_USERS];

  bzero(lineBusy, MAX_USERS);
  for (i = 0; i < nProcs; i++) {
    n = lineNo(procRec[i].tty);
    if (n >= 0)
      lineBusy[n] = 1;
  }
  for (n = 0, i = 0; i < nLines; i++)
    if (!lineBusy[i])
      n++;
  n = (optBusyThreshold + 1) - n;
  return max(0, n);
}

void
userList(int *nLogins, LoginRec * loginRec)
{
  struct utmp              _utmp;
  uid_t                    uid;
  char                     login[MAX_LOGINCHARS + 1];
  FILE                    *f = fopen(_PATH_UTMP, "rb");

  if (!f)
    perrQuit("fopen");
  *nLogins = 0;
  while (fread(&_utmp, sizeof(struct utmp), 1, f)) {
    if (_utmp.ut_type != USER_PROCESS)
      continue;
    strncpy(login, _utmp.ut_user, MAX_LOGINCHARS);
    login[MAX_LOGINCHARS] = '\0';
    if ((uid = UIDfromLogin(login)) == (uid_t)-1)
      continue;
    loginRec[*nLogins].uid = uid;
    loginRec[*nLogins].tty = devNumFromName(_utmp.ut_line);
    strncpy(loginRec[*nLogins].ttyName, _utmp.ut_line, 12);
    loginRec[*nLogins].ttyName[12] = '\0';
    loginRec[(*nLogins)++].time = _utmp.ut_time;
  }
  fclose(f);
}

time_t
lastLogin(UserRec *ur)
{
  time_t res = 0;
  for (int i = 0; i < MAX_HOSTS; i++)
    res = max(res, ur->lastLogin[i]);
  return res;
}

time_t
lastOnline(UserRec *ur)
{
  time_t res = 0;
  for (int i = 0; i < MAX_HOSTS; i++)
    res = max(res, ur->lastOnline[i]);
  return res;
}

void
addHost(UserRec *ur, word host)
{
  if (ur->nHosts < MAX_HOSTS) {
    int hostIdx = findHost(ur, host);
    if (hostIdx < 0) {
      ur->host[ur->nHosts] = host;
      ur->nLogins[ur->nHosts] = 0;
      ur->lastLogin[ur->nHosts] = 0;
      ur->lastOnline[ur->nHosts++] = 0;
    }
  }
}

int
findHost(UserRec *ur, word host)
{
  int hostIdx = -1;
  for (int i = 0; i < ur->nHosts; i++)
    if (ur->host[i] == host) {
      hostIdx = i;
      break;
    }
  return hostIdx;
}

void
removeHost(UserRec *ur, word host)
{
  int i = findHost(ur, host);
  if (i >= 0) {
    ur->host[i] = ur->host[ur->nHosts - 1];
    ur->lastLogin[i] = ur->lastLogin[ur->nHosts - 1];
    ur->lastOnline[i] = ur->lastOnline[ur->nHosts - 1];
    ur->nHosts--;
  }
}

word
nLogins(UserRec *ur)
{
  int res = 0;

  for (int i = 0; i < ur->nHosts; i++)
    res += ur->nLogins[i];
  return res;
}

word
nLogins(HashTable *loginRec, uid_t uid)
{
  int numLogins = 0;

  LoginRec *rec = (LoginRec*)hashTableSearch(loginRec, &uid);
  if (rec != NULL) {
    if (lineNo(rec->tty) >= 0)
	numLogins++;

    /* Modified, update rec at next search */
    while ((rec = (LoginRec*)hashTableSearchNext(loginRec, &uid)) != NULL)
    //while (hashTableSearchNext(loginRec, &uid) != NULL)
      if (lineNo(rec->tty) >= 0)
        numLogins++;
  }
  return numLogins;
}

int
userLoggedIn(int nLogins, LoginRec *loginRec, uid_t uid)
{
  int                      i;

  for (i = 0; i < nLogins; i++)
    if (loginRec[i].uid == uid)
      return 1;
  return 0;
}

int
getToken(char *d, char *s, int *idx)
{
  int                      si,
                           di,
                           sLen = strlen(s);

  for (si = *idx; si < sLen && isspace(s[si]); si++);
  for (di = 0; isgraph(s[si]); si++)
    d[di++] = s[si];
  d[di] = '\0';
  *idx = si;
  return di == 0;
}

int isNumber (char *test)
{
  /* Description:
   *   To determine if the test string given is all numbers.
   * Returns:
   *   0 = Contains non-numbers
   *   1 = String is numbers only
   */
  unsigned int  j;

  for (j=0;j < strlen(test); j++)
    if (!isdigit(test[j])) {
      return(0);
    }
  return(1);
}

uid_t
UIDfromLogin(char *login)
{
  uid_t                    res = (uid_t)-1;
  struct passwd           *passwdp;

  if (strlen(login) > MAX_LOGINCHARS)
    login[MAX_LOGINCHARS] = '\0';
  if ((passwdp = getpwnam(login)) != NULL)
    res = passwdp->pw_uid;
  return res;
}

gid_t
GIDfromLogin(char *login)
{
  /* Description:
   *   Find the GID of the login given.
   * Returns:
   *    -1 = Failed.
   *    Anything else is the GID.
   */
  gid_t                    res = (gid_t)-1;
  struct passwd           *passwdp;

  if (strlen(login) > MAX_LOGINCHARS)
    login[MAX_LOGINCHARS] = '\0';
  if ((passwdp = getpwnam(login)) != NULL)
    res = passwdp->pw_gid;
  return res;
}

char *
loginFromUID(char *login, uid_t uid)
{
  char                    *res = NULL;
  struct passwd           *passwdp;

  login[0] = '\0';
  if ((passwdp = getpwuid(uid)) != NULL) {
    strncpy(login, passwdp->pw_name, MAX_LOGINCHARS);
    login[MAX_LOGINCHARS] = '\0';
    res = login;
  }
  return res;
}

void
userFileOpen()
{
  Boolean  writeMagic = FALSE;
  char     magic[ACUA_USERS_MAGIC_LEN];

  if (!geteuid()) {
    fLock((char*)userFilePath, LOCK_EXCLUSIVE | LOCK_BLOCK);
    userFileFD = open(userFilePath, O_RDONLY);
    if (userFileFD < 0) writeMagic = TRUE;
    else close(userFileFD);
    userFileFD = open(userFilePath, O_CREAT | O_RDWR, 0644);
    assert(userFileFD >= 0);
    fchmod(userFileFD, 0644);
    flock(userFileFD, LOCK_EX);
    fcntl(userFileFD, F_SETFD, 1);
    userFile = fdopen(userFileFD, "r+b");
    if (writeMagic) {
      fwrite(ACUA_USERS_MAGIC, 1, ACUA_USERS_MAGIC_LEN, userFile);
      rewind(userFile);
    }
  } else {
    userFileFD = open(userFilePath, O_RDONLY);
    assert(userFileFD >= 0);
    flock(userFileFD, LOCK_SH);
    userFile = fdopen(userFileFD, "rb");
  }

  fread(magic, 1, ACUA_USERS_MAGIC_LEN, userFile);
  if (strcmp(magic, ACUA_USERS_MAGIC))
    errQuit("Magic # check failed for %s", LIB"/acua_users");
}

void
userFileClose()
{
  if (flock(userFileFD, LOCK_UN) < 0) {
		  // Error locking
		  perror("libcacua.cc:userFileClose:flock:");
		  exit(1);
  };
  if (!geteuid())
    fLock((char *) userFilePath, LOCK_UNLOCK);
  if (fclose(userFile) < 0) {
		  // error closing userFile
		  perror("libacua.cc:userFileClose:fclose:");
		  exit(1);
  } else {
		  // userFile closed sucessful
		  userFile = NULL;
  };
}

int
userFileRead(UserRec *ur)
{
  return !fread(ur, sizeof(UserRec), 1, userFile);
}

int
userFileWrite(UserRec *ur)
{
  return !fwrite(ur, sizeof(UserRec), 1, userFile);
}

void
userFileEdit(UserRec *ur)
{
  fseek(userFile, -sizeof(UserRec), SEEK_CUR);
  userFileWrite(ur);
}

int
userFileSearch(UserRec *ur, uid_t uid)
{
  userFileRewind();
  while (!userFileRead(ur))
    if (ur->uid == uid)
      return 0;
  return 1;
}

void
userFileRewind()
{
  char  magic[ACUA_USERS_MAGIC_LEN];

  rewind(userFile);
  fread(magic, 1, ACUA_USERS_MAGIC_LEN, userFile);
}

void
banFileOpen()
{
  if (geteuid())
    errQuit("must be root");
  if (!geteuid())
    fLock((char *) banFilePath, LOCK_EXCLUSIVE | LOCK_BLOCK);
  banFileFD = open(banFilePath, O_CREAT | O_RDWR, 0644);
  assert(banFileFD >= 0);
  if (!geteuid())
    flock(banFileFD, LOCK_EX);
  else
    flock(banFileFD, LOCK_SH);
  fcntl(banFileFD, F_SETFD, 1);
  banFile = fdopen(banFileFD, "r+b");
}

void
banFileClose()
{
  if (flock(banFileFD, LOCK_UN) < 0) {
		  // Error locking file
		  perror("libacua.cc:banFileClose:flock:");
		  exit(1);
  };
  if (!geteuid())
    fLock((char *) banFilePath, LOCK_UNLOCK);
  if (fclose(banFile) < 0) {
		  // Error closing ban file
		  perror("libacua.cc:banFileClose:fclose");
		  exit(1);
  } else {
		  // banFile closed sucessful
		  banFile = NULL;
  };
}

int
banFileRead(BanRec *br)
{
  return !fread(br, sizeof(BanRec), 1, banFile);
}

int
banFileWrite(BanRec *br)
{
  return !fwrite(br, sizeof(BanRec), 1, banFile);
}

int
banFileSearch(BanRec *br, word phNoArea, word phNoLocal)
{
  char                     phNoStr[11],
                           phNoAreaStr[11],
                           phNoLocalStr[11];

  sprintf(phNoAreaStr, "%u", phNoArea);
  sprintf(phNoLocalStr, "%u", phNoLocal);
  rewind(banFile);
  while (!banFileRead(br)) {
    if (br->flags) {
      sprintf(phNoStr, "%u", br->phNoArea);
      if ((phNoArea && !strncmp(phNoStr, phNoAreaStr, strlen(phNoStr))) ||
          (phNoLocal && !strncmp(phNoStr, phNoLocalStr, strlen(phNoStr))) ||
          (!phNoArea && !phNoLocal && !br->phNoArea && !br->phNoLocal))
        return 0;
    } else if (br->phNoArea == phNoArea && br->phNoLocal == phNoLocal)
      return 0;
  }
  return 1;
}

int
phNo2Words(word * phNoArea, word * phNoLocal, char *phNo, int phNoDigits)
{
  int                      i,
                           len;
  char                     tPhNo[32],
                           phNoAreaStr[32],
                           phNoLocalStr[32];

  for (len = 0, i = 0; phNo[i] != '\0'; i++)
    if (isdigit(phNo[i]))
      tPhNo[len++] = phNo[i];
  tPhNo[len] = '\0';
  if ((len < phNoDigits) || (len > 18))
    return 1;
  strcpy(phNoLocalStr, tPhNo + len - phNoDigits);
  strcpy(phNoAreaStr, tPhNo);
  phNoAreaStr[len - phNoDigits] = '\0';
  *phNoArea = atol(phNoAreaStr);
  *phNoLocal = atol(phNoLocalStr);
  return 0;
}

char *
words2PhNo(char *phNo, word phNoArea, word phNoLocal)
{
  int                      i = 0,
                           j = 0,
                           k,
                           l,
                           skipChars;
  char                     phNoAreaStr[10],
                           phNoLocalStr[10],
                          *s;

  sprintf(phNoAreaStr, "%u", phNoArea);
  sprintf(phNoLocalStr, "%u", phNoLocal);
  if (!phNoLocal) {
    strcpy(phNo, phNoAreaStr);
    return phNo;
  }
  if (phNoArea) {
    s = optPhNoAreaFormat;
    skipChars = stringSum(optPhNoAreaFormat) - strlen(phNoAreaStr);
    while (*s != '\0') {
      if (isdigit(*s)) {
        l = strtol(s, &s, 10);
        if (skipChars) {
          skipChars -= l;
          if (skipChars == 0)
            while (*s == ')' || *s == ' ')
              s++;
          else if (skipChars < 0)
            goto err;
          continue;
        }
        for (k = 0; k < l; k++)
          phNo[i++] = phNoAreaStr[j++];
      } else
        while (!isdigit(*s) && *s != '\0')
          if (skipChars)
            s++;
          else
            phNo[i++] = *s++;
    }
    if (phNoAreaStr[j] != '\0')
      goto err;
    phNo[i++] = ' ';
  }
  if (phNoLocal) {
    j = 0;
    s = optPhNoLocalFormat;
    while (*s != '\0') {
      if (isdigit(*s)) {
        l = strtol(s, &s, 10);
        for (k = 0; k < l; k++)
          phNo[i++] = phNoLocalStr[j++];
      } else
        while (!isdigit(*s) && *s != '\0')
          phNo[i++] = *s++;
    }
    if (phNoLocalStr[j] != '\0')
      goto err;
  }
  phNo[i] = '\0';
  return phNo;
err:
  phNo[0] = '\0';
  return NULL;
}

int
stringSum(char *s)
{
  int                      sum = 0;

  while (*s != '\0') {
    while (!isdigit(*s) && *s != '\0')
      s++;
    if (*s != '\0')
      sum += strtol(s, &s, 10);
  }
  return sum;
}

void
bytes2ASCII(char *str, dword bytes)
{
  if (bytes >= 1024 * 1024 * 1024)
    sprintf(str, "%.2f GB", (float) bytes / (1024 * 1024 * 1024));
  else if (bytes >= 1024 * 1024)
    sprintf(str, "%.2f MB", (float) bytes / (1024 * 1024));
  else
    sprintf(str, "%.2f KB", (float) bytes / 1024);
}

int
inTimeClass(int tc)
{
  time_t                   t;
  struct tm               *tmP;

  time(&t);
  tmP = localtime(&t);
  if (timeClass[tc].startDay <= timeClass[tc].endDay) {
    if ((tmP->tm_wday < timeClass[tc].startDay) ||
        (tmP->tm_wday > timeClass[tc].endDay))
      return 0;
  } else {
    if ((tmP->tm_wday > timeClass[tc].endDay) &&
        (tmP->tm_wday < timeClass[tc].startDay))
      return 0;
  }
  if (timeClass[tc].startHour <= timeClass[tc].endHour) {
    if ((tmP->tm_hour < timeClass[tc].startHour) ||
        (tmP->tm_hour > timeClass[tc].endHour))
      return 0;
  } else {
    if ((tmP->tm_hour > timeClass[tc].endHour) &&
        (tmP->tm_hour < timeClass[tc].startHour))
      return 0;
  }
  if ((tmP->tm_hour == timeClass[tc].startHour) &&
      (tmP->tm_min < timeClass[tc].startMin))
  {
      return 0;
  }
  if ((tmP->tm_hour == timeClass[tc].endHour) &&
      (tmP->tm_min > timeClass[tc].endMin))
  {
      return 0;
  }
  return 1;
}

int
curTimeClass(UserRec * ur)
{
  int                      i,
                           tc = -1,
                           tcLeft = 0;

  for (i = 0; i < nTimeClasses; i++) {
    if (inTimeClass(i)) {
      if (!ur)
        return i;
      if (tc < 0 ||
          (ur->cLimit[i] >= 0 &&
           (ur->cLimit[tc] < 0 || ur->cLeft[i] < tcLeft)))
      {
        tc = i;
        tcLeft = ur->cLeft[i];
      }
    }
  }
  if (tc >= 0)
    return tc;
  else
    return -1;
}

/*
 * This only does X-locks.  S-lock implementation is left as an exercise
 * for the reader :)
 */
int
fLock(char *path, int operation)
{
  pid_t                    myPID = getpid(),
                           theirPID;
  int                      nProcs;
  ProcRec                  procRec[MAX_PROCS];
  char                    *s,
                           hostname[64],
                           theirHostname[64];
  char                     myName[16] = "",
                           theirName[16];
  char                     buf[PATH_MAX];
  char                     linkPath[PATH_MAX];
  char                     pidPath[PATH_MAX],
                           theirPIDpath[PATH_MAX];
  int                      i,
                           len = strlen(path),
                           dirChars;
  char                    *name;
  FILE                    *f;
  int                      block = operation & LOCK_BLOCK;

  if (block)
    operation &= ~LOCK_BLOCK;
  procList(&nProcs, procRec);
  for (i = 0; i < nProcs; i++)
    if (procRec[i].pid == myPID) {
      strcpy(myName, procRec[i].name);
      break;
    }
  for (i = len - 1; i; i--)
    if (path[i] == '/')
      break;
  if (i < 0)
    name = path;
  else
    name = path + i + 1;
  dirChars = name - path;
  strncpy(linkPath, path, dirChars);
  linkPath[dirChars] = '\0';
  sprintf(linkPath + dirChars, "LOCK_%s", name);
  strncpy(pidPath, path, dirChars);
  pidPath[dirChars] = '\0';
  gethostname(hostname, 64);
  for (s = hostname; *s && *s != '.'; s++);
  if (*s == '.')
    *s = '\0';
  sprintf(pidPath + dirChars, "LOCK_%s.%s.%u", name, hostname, myPID);
  if (operation == LOCK_UNLOCK) {
    unlink(pidPath);
    unlink(linkPath);
    return 0;
  }
  f = fopen(pidPath, "w");
  fprintf(f, "%s %s %u\n", hostname, myName, myPID);
  fclose(f);
l0:
  if (symlink(pidPath + dirChars, linkPath) < 0) {
    i = readlink(linkPath, buf, PATH_MAX);
    if (i < 0)
      goto l0;
    else {
      buf[i] = '\0';
      strncpy(theirPIDpath, path, dirChars);
      theirPIDpath[dirChars] = '\0';
      strcpy(theirPIDpath + dirChars, buf);
    }
    f = fopen(theirPIDpath, "r");
    if (!f) {
      unlink(linkPath);
      goto l0;
    }
    i = fscanf(f, "%s %s %u", theirHostname, theirName, &theirPID);
    fclose(f);
    if (i < 3) {
      unlink(theirPIDpath);
      unlink(linkPath);
      goto l0;
    }
    if (strcmp(hostname, theirHostname))
      goto l1;
    procList(&nProcs, procRec);
    for (i = 0; i < nProcs; i++)
      if (procRec[i].pid == theirPID && !strcmp(procRec[i].name, theirName))
  break;
    if (i == nProcs) {
      unlink(theirPIDpath);
      unlink(linkPath);
      goto l0;
    } else
      goto l1;
  } else
    return 0;
l1:
  if (block) {
    sleep(1);
    goto l0;
  }
  unlink(pidPath);
  return 1;
}

void
preprocessFile(UserRec *ur, char *inPath, char *outPath)
{
  FILE                    *inFile[16],
                          *outFile = fopen(outPath, "w");
  char                     inLine[256], outLine[256], token[256], token2[256];
  int                      curFile = 0, inLen, outLen, tokenLen, token2Len,
                           i, j, k, arg;

  inFile[0] = fopen(inPath, "r");
  if (inFile[0] == NULL)
    errQuit("couldn't open input file: %s", inPath);
  if (outFile == NULL)
    errQuit("couldn't open output file: %s", outPath);
l0:
  while (curFile >= 0) {
    while (fgets(inLine, 256, inFile[curFile])) {
      inLen = strlen(inLine);
      outLen = 0;
      i = 0;
      while (i < inLen) {
        if (inLine[i] == '$') {
          if (inLine[++i] == '$') {
            i++;
            outLine[outLen++] = '$';
          } else {
            tokenLen = 0;
            while (isalpha(inLine[i]))
              token[tokenLen++] = inLine[i++];
            token[tokenLen] = '\0';
            token2Len = 0;
            while (isdigit(inLine[i]))
              token2[token2Len++] = inLine[i++];
            token2[token2Len] = '\0';
            if (token2Len) arg = atoi(token2);
            else arg = -1;
            
            if (!strcasecmp(token, "bRx")) {
              bytes2ASCII(token, ur->bRx);
            } else if (!strcasecmp(token, "bRxLimit")) {
              if (ur->bRxLimit) bytes2ASCII(token, ur->bRxLimit);
              else strcpy(token, "+INF");
            } else if (!strcasecmp(token, "bTx")) {
              bytes2ASCII(token, ur->bTx);
            } else if (!strcasecmp(token, "bTxLimit")) {
              if (ur->bTxLimit) bytes2ASCII(token, ur->bTxLimit);
              else strcpy(token, "+INF");
            } else if (!strcasecmp(token, "bLimit")) {
              if (ur->bLimit) bytes2ASCII(token, ur->bLimit);
              else strcpy(token, "+INF");
            } else if (!strcasecmp(token, "bSlimit")) {
              if (ur->bSlimit) bytes2ASCII(token, ur->bSlimit);
              else strcpy(token, "+INF");
            } else if (!strcasecmp(token, "bSrx")) {
              bytes2ASCII(token, ur->bSrx);
            } else if (!strcasecmp(token, "bSrxLimit")) {
              if (ur->bSrxLimit) bytes2ASCII(token, ur->bSrxLimit);
              else strcpy(token, "+INF");
            } else if (!strcasecmp(token, "bStx")) {
              bytes2ASCII(token, ur->bStx);
            } else if (!strcasecmp(token, "bStxLimit")) {
              if (ur->bStxLimit) bytes2ASCII(token, ur->bStxLimit);
              else strcpy(token, "+INF");
            } else if (!strcasecmp(token, "bSxfer")) {
              bytes2ASCII(token, ur->bStx + ur->bSrx);
            } else if (!strcasecmp(token, "bXfer")) {
              bytes2ASCII(token, ur->bTx + ur->bRx);
            } else if (!strcasecmp(token, "cLeft")) {
              if (arg >= 0 && arg < MAX_TIME_CLASSES)
                hhmm(token, ur->cLeft[arg]);
            } else if (!strcasecmp(token, "cLimit")) {
              if (arg >= 0 && arg < MAX_TIME_CLASSES) {
                if (ur->cLimit[arg] >= 0)
                  hhmm(token, ur->cLimit[arg]);
                else strcpy(token, "+INF");
              }
            } else if (!strcasecmp(token, "creationDate")) {
              strcpy(token, ctime(&ur->creation));
              token[strlen(token) - 1] = '\0';
            } else if (!strcasecmp(token, "credit")) {
              hhmm(token, ur->credit);
            } else if (!strcasecmp(token, "expireAction")) {
              if (!ur->expire)
                strcpy(token, "N/A");
              else if (EXPIRE(ur->flags) == EXPIRE_DELETE) {
                strcpy(token2, ctime(&ur->expire));
                token2[strlen(token2) - 1] = '\0';
                sprintf(token, "DELETE in %.2f days [%s]",
                        (float)(ur->expire - time(NULL)) / (24 * 60 * 60),
                        token2);
              } else {
                strcpy(token2, ctime(&ur->expire));
                token2[strlen(token2) - 1] = '\0';
                sprintf(token, "UNSUBSCRIBE in %.2f days [%s]",
                        (float)(ur->expire - time(NULL)) / (24 * 60 * 60),
                        token2);
              }
            } else if (!strcasecmp(token, "expireDate")) {
              if (!ur->expire)
                strcpy(token, "N/A");
              else {
                strcpy(token, ctime(&ur->expire));
                token[strlen(token) - 1] = '\0';
              }
            } else if (!strcasecmp(token, "expireDays")) {
              sprintf(token, "%.2f",  (float)(ur->expire - time(NULL)) / (24 * 60 * 60));
            } else if (!strcasecmp(token, "expireType")) {
              if (!ur->expire)
                strcpy(token, "N/A");
              if (EXPIRE(ur->flags) == EXPIRE_DELETE)
                strcpy(token, "account");
              else
                strcpy(token, "subscription");
            } else if (!strcasecmp(token, "flags")) {
              word                    j = 0;
              
              token[0] = '\0';
              if (ur->flags & FLG_SMARTTIME) {
                if (j++) strcat(token, " ");
                strcat(token, "SMARTTIME");
              }
              if (ur->flags & FLG_SSMARTTIME) {
                if (j++) strcat(token, " ");
                strcat(token, "SSMARTTIME");
              }
              if (ur->flags & FLG_TCSMARTTIME) {
                if (j++) strcat(token, " ");
                strcat(token, "TCSMARTTIME");
              }
              if (ur->flags & FLG_SMARTBOOT) {
                if (j++) strcat(token, " ");
                strcat(token, "SMARTBOOT");
              }
              if (ur->flags & FLG_SSMARTBOOT) {
                if (j++) strcat(token, " ");
                strcat(token, "SSMARTBOOT");
              }
              if (ur->flags & FLG_TCSMARTBOOT) {
                if (j++) strcat(token, " ");
                strcat(token, "TCSMARTBOOT");
              }
              if (ur->flags & FLG_ISMARTBOOT) {
                if (j++) strcat(token, " ");
                strcat(token, "ISMARTBOOT");
              }
              if (ur->flags & FLG_WARNBOOT) {
                if (j++) strcat(token, " ");
                strcat(token, "WARNBOOT");
              }
              if (ur->flags & FLG_EXPLAINBOOT) {
                if (j++) strcat(token, " ");
                strcat(token, "EXPLAINBOOT");
              }
              if (ur->flags & FLG_NOUNSUB) {
                if (j++) strcat(token, " ");
                strcat(token, "NOUNSUB");
              }
            } else if (!strcasecmp(token, "gecos")) {
              struct passwd           *passwdp;
              int                      li = 0;
              
              if ((passwdp = getpwuid(ur->uid)) == NULL)
                token[0] = '\0';
              else {
                if (arg < 0) {
                  for (j = 0; passwdp->pw_gecos[j] && passwdp->pw_gecos[j] != ','; j++)
                    token[j] = passwdp->pw_gecos[j];
                  token[j] = '\0';
                } else {
                  for (j = 0; j <= arg; j++)
                    if (getToken(token, passwdp->pw_gecos, &li)) break;
                  if (j <= arg) token[0] = '\0';
                }
              }
            } else if (!strcasecmp(token, "idleLimit")) {
              hhmm(token, ur->idleLimit);
            } else if (!strcasecmp(token, "include")) {
              while (isspace(inLine[i])) i++;
              j = 0;
              while (i < inLen && !isspace(inLine[i]))
                token2[j++] = inLine[i++];
              token2[j] = '\0';
              if (token2[0] != '.' && token2[0] != '/')
                sprintf(token, LIB"/%s", token2);
              else
                strcpy(token, token2);
              inFile[++curFile] = fopen(token, "r");
              if (inFile[curFile] == NULL)
                errQuit("couldn't open input file");
              goto l0;
            } else if (!strcasecmp(token, "lastLogin")) {
              time_t lastLoginTime = lastLogin(ur);
              strcpy(token, ctime(&lastLoginTime));
              token[strlen(token) - 1] = '\0';
            } else if (!strcasecmp(token, "lastOnline")) {
              time_t lastOnlineTime = lastOnline(ur);
              strcpy(token, ctime(&lastOnlineTime));
              token[strlen(token) - 1] = '\0';
            } else if (!strcasecmp(token, "lastOnDays")) {
              time_t lastLoginTime = lastOnline(ur);
              sprintf(token, "-%.2f",  (float)(time(NULL) - lastLoginTime) / (24 * 60 * 60));
            } else if (!strcasecmp(token, "lockInfo")) {
              if (ur->lockDate == 0) strcpy(token, "N/A");
              else {
                if (LOCK(ur->flags))
                  sprintf(token, "locked out on %s", ctime(&ur->lockDate));
                else
                  sprintf(token, "unlocked on %s", ctime(&ur->lockDate));
                token[strlen(token) - 1] = '\0';
              }
            } else if (!strcasecmp(token, "login")) {
              loginFromUID(token, ur->uid);
            } else if (!strcasecmp(token, "maxDeduct")) {
              sprintf(token, "%u", ur->maxDeduct);
            } else if (!strcasecmp(token, "maxLogins")) {
              sprintf(token, "%u", ur->maxLogins);
            } else if (!strcasecmp(token, "online")) {
              if (ur->nHosts > 0) {
                sprintf(token, "YES [");
                for (j = 0; j < ur->nHosts; j++) {
                  struct hostent   *hp;
                  hp = gethostbyaddr((char*)&ur->host[j], sizeof(word), AF_INET);
                  if (hp) {
                    if (j) strcat(token, ", ");
                    sprintf(token2, "%s:%d", hp->h_name, ur->nLogins[j]);
                    strcat(token, token2);
                  }
                }
                strcat(token, "]");
              } else strcpy(token, "NO");
            } else if (!strcasecmp(token, "overallUtilization")) {
              sprintf(token, "%.2f%%", (float)ur->tMinutes / (((time(NULL) - ur->creation) / 60) + 1) * 100);
            } else if (!strcasecmp(token, "phNo")) {
              if (!ur->phNoLocal) strcpy(token, "N/A");
              else words2PhNo(token, ur->phNoArea, ur->phNoLocal);
            } else if (!strcasecmp(token, "PPPidleBytes")) {
              sprintf(token, "%u", ur->PPPidleBytes);
            } else if (!strcasecmp(token, "PPPidleMinutes")) {
              hhmm(token, ur->PPPidleMinutes);
            } else if (!strcasecmp(token, "priority")) {
              sprintf(token, "%d", PRIORITY(ur->flags));
            } else if (!strcasecmp(token, "sLeft")) {
              hhmm(token, ur->sLeft);
            } else if (!strcasecmp(token, "sLimit")) {
              hhmm(token, ur->sLimit);
            } else if (!strcasecmp(token, "subscriptionInfo")) {
              if (ur->subscrDate == 0) strcpy(token, "N/A");
              else {
                if (EXPIRE(ur->flags) == EXPIRE_DELETE)
                  sprintf(token, "unsubscribed on %s", ctime(&ur->subscrDate));
                else
                  sprintf(token, "subscribed on %s", ctime(&ur->subscrDate));
                token[strlen(token) - 1] = '\0';
              }
            } if (!strcasecmp(token, "tLeft")) {
              hhmm(token, ur->tLeft);
            } else if (!strcasecmp(token, "tLimit")) {
              hhmm(token, ur->tLimit);
            } else if (!strcasecmp(token, "uflags")) {
              k = 0;
              token[0] = '\0';
              for (j = 0; j < 16; j++) {
                word mask = 0x00010000 << j;
                if (ur->flags & mask) {
                  if (k++) strcat(token, " ");
                  sprintf(token + strlen(token), "%u", j);
                }
              }
            }
            k = strlen(token);
            for (j = 0; j < k; j++)
              outLine[outLen++] = token[j];
          }
        } else outLine[outLen++] = inLine[i++];
      }
      outLine[outLen] = '\0';
      fputs(outLine, outFile);
    }
    fclose(inFile[curFile--]);
  }
  fclose(outFile);
}

void
hhmm(char *s, int nMinutes) {
  word  hh, mm;

  if (nMinutes < 0) {
    nMinutes = -nMinutes;
    strcpy(s++, "-");
  }
  hh = nMinutes / 60;
  mm = nMinutes - hh * 60;
  sprintf(s, "%02u:%02u", hh, mm);
}

void
PPPfindUnits(HashTable *PPPuser, HashTable *loginRecTable, HashTable *procRecTable)
{
  int                      i,
                           len,
                           PPPunit;
  DIR                     *d;
  FILE                    *f;
  pid_t                    pid;
  struct dirent           *de;
  int                      unitStrLen;
  char                     unitStr[8],
                           str[32];
  LoginRec                *loginRec = NULL;
  ProcRec                 *procRec;
  PPPuserRec              *p;
  Boolean                  found;

  hashTableInit(PPPuser, MAX_USERS, UIDhash, UIDcomp, 0, TRUE);
  if ((d = opendir(_PATH_VARRUN)) == NULL)
    perrQuit("opendir failed");
  while ((de = readdir(d)) != NULL) {
    len = strlen(de->d_name);
    if (len < 8 || len > 10)
      continue;
    if (strncmp(de->d_name, "ppp", 3))
      continue;
    if (strcmp(de->d_name + len - 4, ".pid"))
      continue;
    for (unitStrLen = 0, i = 3; isdigit(de->d_name[i]); i++)
      unitStr[unitStrLen++] = de->d_name[i];
    unitStr[unitStrLen] = '\0';
    PPPunit = atoi(unitStr);
    if (PPPunit >= MAX_PPP_UNITS)
      continue;
    sprintf(str, "%s/ppp%d.pid", _PATH_VARRUN, PPPunit);
    if ((f = fopen(str, "r")) == NULL)
      continue;
    fgets(str, 32, f);
    pid = atoi(str);
    fclose(f);
    procRec = (ProcRec*)hashTableSearch(procRecTable, &pid);
    if (!procRec) continue;
    found = FALSE;
    len = strlen(procRec->name);
    if (len >= 3) {
      for (i = 0; i <= len - 3; i++)
        if (!strncmp(procRec->name + i, "ppp", 3)) {
          found = TRUE;
          break;
        }
    }
    if (found) {
      loginRec = (LoginRec*)hashTableSearch(loginRecTable, &procRec->tty);
      if (!loginRec) continue;
      p = (PPPuserRec*)malloc(sizeof(PPPuserRec));
      p->root = (Boolean)(procRec->uid == 0);
      p->uid = loginRec->uid;
      p->pid = pid;
      p->unit = PPPunit;
      p->tty = procRec->tty;
      p->loginTime = loginRec->time;
      hashTableAdd(PPPuser, p);
    }
  }
  closedir(d);
}

word
hostAddr(char *hostname) {
  word res = 0;
  struct hostent *hp = gethostbyname(hostname);

  if (hp)
    memcpy(&res, hp->h_addr, sizeof(word));
  return res;
}

word
UIDhash(word size, void *key)
{
  return (*(uid_t*)key) % size;
}

int
UIDcomp(void *p1, void *p2)
{
  return (int) (*(uid_t*)p1 - *(uid_t*)p2);
}

word
PIDhash(word size, void *key)
{
  return (*(pid_t*)key) % size;
}

int
PIDcomp(void *p1, void *p2)
{
  return (int) (*(pid_t*)p1 - *(pid_t*)p2);
}

word
TTYhash(word size, void *key)
{
  return (*(dev_t*)key) % size;
}

int
TTYcomp(void *p1, void *p2)
{
  return (int) (*(dev_t*)p1 - *(dev_t*)p2);
}

void
readConfig()
{
  int                      i,
                           li;
  char                     line[256],
                           token[256],
                           devPath[16];
  FILE                    *f;
  struct stat              _stat;

  // check for environment variables
  if (configFilePath)
	  free(configFilePath);
  if ((getenv("ACUA_CONFIG")) && (!getuid())) {
	  configFilePath = strdup(getenv("ACUA_CONFIG"));
  } else {
	  configFilePath = strdup(LIB"/acua.config");
  };
  if(!configFilePath)
	  errQuit(strerror(errno));

  if (userFilePath)
	  free(userFilePath);
  if ((getenv("ACUA_USERS")) && (!getuid())) {
	  userFilePath = strdup(getenv("ACUA_USERS"));
  } else {
	  userFilePath = strdup(LIB"/acua_users");
  };
  if(!userFilePath)
	  errQuit(strerror(errno));
  
  if (banFilePath)
	  free(banFilePath);
  if ((getenv("ACUA_BANNED_USERS")) && (!getuid())) {
	  banFilePath = strdup(getenv("ACUA_BANNED_USERS"));
  } else {
	  banFilePath = strdup(LIB"/banned_users");
  };
  if(!banFilePath)
	  errQuit(strerror(errno));

  if ((f = fopen(configFilePath, "rt")) == NULL)
    errQuit("couldn't open config file");

  // initialize data
  nLines = 0;
  optPurgeDays = 0;
  optMinDeduct = 0;
  optBusyThreshold = 0;
  optMaxKick = 0;
  optLowCPUpriority = 0;
  optHighCPUpriority = 0;
  optReturnDelay = 0;
  optGuestTime = 15;
  optGuestPriority = 4;
  optMailHost[0] = '\0';
  strncpy(optMailProg, "/usr/bin/mail", sizeof(optMailProg));
  optMailWait = 10;             // default 10 second wait before killed
  optMailUser = 0;              // UID of the user that will mail out
  optMailGroup = 0;             // GID of the user that will mail out
  optExplainBoot = 0;
  nBootWarnTimes = 0;
  optPPPWarnBoot = 0;
  nExpireWarnTimes = 0;
  optWarnExpireCC[0] = '\0';
  optIdleLimit = 0;
  optPPPidleMinutes = 0;
  optPPPidleBytes = 0;
  optSmartTime = 0;
  optSessionSmartTime = 0;
  optTimeClassSmartTime = 0;
  optSmartBoot = 0;
  optSessionSmartBoot = 0;
  optTimeClassSmartBoot = 0;
  optIdleSmartBoot = 0;
  nTimeClasses = 0;
  nExcluded = 0;
  optModemDial[0] = '\0';
  optPhNoAreaFormat[0] = '\0';
  optPhNoLocalFormat[0] = '\0';
  optNoRecLogin = 0;

  while (fgets(line, 256, f) != NULL) {
    li = 0;
    if (getToken(token, line, &li))
      continue;
    if (!strcasecmp(token, "Devices")) {
      while (!getToken(token, line, &li)) {
        if (isdigit(token[0])) {
          int                      n = atoi(token) - 1;
          
          if (nLines)
            for (i = 0; i < n; i++) {
              lineDev[nLines] = lineDev[nLines - 1] + 1;
              nLines++;
            }
        } else {
	    sprintf(devPath, "/dev/%s", token);
            //if (stat(devPath, &_stat) < 0)
	    if ((stat(devPath, &_stat) < 0) && (strncmp(devPath,"/dev/pts/",9)!=0))
		perrQuit("couldn't stat %s", devPath);
	    else if (strncmp(devPath,"/dev/pts/",9)==0) {
		char *tokenvar=rindex(devPath,'/')+1;
		lineDev[nLines++] = 34816 + atoi(tokenvar);
	    } else lineDev[nLines++] = _stat.st_rdev;
	    //lineDev[nLines++] = _stat.st_rdev;
        }
      }
    } else if (!strcasecmp(token, "PurgeDays")) {
      if (getToken(token, line, &li))
        continue;
      optPurgeDays = atoi(token);
    } else if (!strcasecmp(token, "MinDeduct")) {
      if (getToken(token, line, &li))
        continue;
      optMinDeduct = atoi(token);
    } else if (!strcasecmp(token, "BusyThreshold")) {
      if (getToken(token, line, &li))
        continue;
      optBusyThreshold = atoi(token);
    } else if (!strcasecmp(token, "MaxKick")) {
      optMaxKick = 1;
    } else if (!strcasecmp(token, "CPUpriority")) {
      if (getToken(token, line, &li))
        continue;
      optLowCPUpriority = atoi(token);
      if (getToken(token, line, &li))
        continue;
      optHighCPUpriority = atoi(token);
    } else if (!strcasecmp(token, "ReturnDelay")) {
      if (getToken(token, line, &li))
        continue;
      optReturnDelay = atoi(token);
    } else if (!strcasecmp(token, "GuestTime")) {
      if (getToken(token, line, &li))
        continue;
      optGuestTime = atoi(token);
    } else if (!strcasecmp(token, "GuestPriority")) {
      if (getToken(token, line, &li))
        continue;
      optGuestPriority = atoi(token);
    } else if (!strcasecmp(token, "MailHost")) {
      if (getToken(token, line, &li))
        continue;
      strcpy(optMailHost, token);
    } else if (!strcasecmp(token, "MailProg")) {
      if (getToken(token, line, &li))
        continue;
      strcpy(optMailProg, token);
    } else if (!strcasecmp(token, "MailWait")) {
      if (getToken(token, line, &li))
        continue;
      optMailWait = atoi(token);
    } else if (!strcasecmp(token, "MailUser")) {
      if (getToken(token, line, &li))
        continue;
      if (isNumber(token)) optMailUser = atoi(token); else
        optMailUser = UIDfromLogin(token);
      if (optMailUser == -1) {
        syslog(LOG_ERR, "Could not determine MailUser UID, falling back to root");
        optMailUser = 0;
      } else {
        // try to determine the group to use
        loginFromUID(token, optMailUser);     // warning: token reused here!
        optMailGroup = GIDfromLogin(token);
        if (optMailGroup == -1) {
          syslog(LOG_ERR, "Could not determine MailUser GID, falling back to root");
          optMailGroup = 0;
        }
      }
    } else if (!strcasecmp(token, "ExplainBoot")) {
      optExplainBoot = 1;
    } else if (!strcasecmp(token, "NoRecLogin")) {
      optNoRecLogin = 1;
    } else if (!strcasecmp(token, "WarnBoot")) {
      while (!getToken(token, line, &li) && nBootWarnTimes < MAX_BOOT_WARN_TIMES)
        bootWarnTime[nBootWarnTimes++] = atoi(token);
    } else if (!strcasecmp(token, "PPPWarnBoot")) {
      if (getToken(token, line, &li))
        continue;
      optPPPWarnBoot = atoi(token);
    } else if (!strcasecmp(token, "WarnExpire")) {
      while (!getToken(token, line, &li) && nExpireWarnTimes < MAX_EXPIRE_WARN_TIMES)
        expireWarnTime[nExpireWarnTimes++] = atoi(token);
    } else if (!strcasecmp(token, "WarnExpireCC")) {
      if (getToken(token, line, &li))
        continue;
      strcpy(optWarnExpireCC, token);
    } else if (!strcasecmp(token, "IdleBoot")) {
      if (getToken(token, line, &li))
        continue;
      optIdleLimit = atoi(token);
    } else if (!strcasecmp(token, "PPPIdleBoot")) {
      if (getToken(token, line, &li))
        continue;
      optPPPidleMinutes = atoi(token);
      optPPPidleMinutes = min(60, optPPPidleMinutes);
      if (getToken(token, line, &li))
        continue;
      optPPPidleBytes = atoi(token);
    } else if (!strcasecmp(token, "SmartTime")) {
      optSmartTime = 1;
    } else if (!strcasecmp(token, "SessionSmartTime")) {
      optSessionSmartTime = 1;
    } else if (!strcasecmp(token, "TimeClassSmartTime")) {
      optTimeClassSmartTime = 1;
    } else if (!strcasecmp(token, "SmartBoot")) {
      optSmartBoot = 1;
    } else if (!strcasecmp(token, "SessionSmartBoot")) {
      optSessionSmartBoot = 1;
    } else if (!strcasecmp(token, "TimeClassSmartBoot")) {
      optTimeClassSmartBoot = 1;
    } else if (!strcasecmp(token, "IdleSmartBoot")) {
      optIdleSmartBoot = 1;
    } else if (!strcasecmp(token, "TimeClass")) {
      for (i = li; line[i]; i++)
        if (!isdigit(line[i]))
          line[i] = ' ';
      for (;;) {
        if (getToken(token, line, &li))
          break;
        timeClass[nTimeClasses].startDay = atoi(token);

        if (getToken(token, line, &li))
          break;
        timeClass[nTimeClasses].endDay = atoi(token);

        if (getToken(token, line, &li))
          break;
        timeClass[nTimeClasses].startHour = atoi(token);

        if (getToken(token, line, &li))
          break;
        timeClass[nTimeClasses].startMin = atoi(token);

        if (getToken(token, line, &li))
          break;
        timeClass[nTimeClasses].endHour = atoi(token);

        if (getToken(token, line, &li))
          break;
        timeClass[nTimeClasses].endMin = atoi(token);

        if (timeClass[nTimeClasses].endMin == 0)
        {
            timeClass[nTimeClasses].endHour--;
            timeClass[nTimeClasses].endMin = 59;
        }
        else
        {
            timeClass[nTimeClasses].endMin--;
        }

        nTimeClasses++;
      }
    } else if (!strcasecmp(token, "ForEachExclude")) {
      while (!getToken(token, line, &li)) {
        excluded[nExcluded] = UIDfromLogin(token);
        if (excluded[nExcluded] == (uid_t) - 1)
          continue;
        else
          nExcluded++;
      }
    } else if (!strcasecmp(token, "ModemDial")) {
      if (getToken(token, line, &li))
        continue;
      strcpy(optModemDial, token);
    } else if (!strcasecmp(token, "PhNoAreaFormat")) {
      if (getToken(token, line, &li))
        continue;
      strcpy(optPhNoAreaFormat, token);
      while (!getToken(token, line, &li)) {
        strcat(optPhNoAreaFormat, " ");
        strcat(optPhNoAreaFormat, token);
      }
    } else if (!strcasecmp(token, "PhNoLocalFormat")) {
      if (getToken(token, line, &li))
        continue;
      strcpy(optPhNoLocalFormat, token);
      while (!getToken(token, line, &li)) {
        strcat(optPhNoLocalFormat, " ");
        strcat(optPhNoLocalFormat, token);
      }
      optPhNoDigits = stringSum(optPhNoLocalFormat);
    }
  }
  fclose(f);
}
