/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <ctype.h>
#include <limits.h>
#include <signal.h>
#include <string.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include "common.h"

#define   MAX_EXEMPT  64
#define   MAX_KILL  16

void                     processCleanProcConfig();
int                      cleanProc(int sig);
int                      userExempt(uid_t uid);

int                      nExempt = 0;
uid_t                    exempt[MAX_EXEMPT];
int                      nKillName = 0;
char                     killName[MAX_KILL][64];
int                      nLoginRecs = 0;
LoginRec                 loginRec[MAX_USERS];
int                      nProcs;
ProcRec                  procRec[MAX_PROCS];
uid_t			 optMinUID;

int
main()
{
  processCleanProcConfig();
  userList(&nLoginRecs, loginRec);
  procList(&nProcs, procRec);
  if (cleanProc(SIGTERM)) {
    sleep(5);
    procList(&nProcs, procRec);
    cleanProc(SIGKILL);
  }
  return 0;
}

void
processCleanProcConfig()
{
  int                      li;
  uid_t                    uid;
  char                     line[256],
                           token[256];
  FILE                    *f;

  optMinUID = 500;   /* processes belonging to < uid 500 won't get killed */
                     /* unless this is changed in the config file */

  if ((f = fopen(LIB"/clean_proc.config", "rt")) == NULL)
    perrQuit("fopen");
  while (fgets(line, 256, f) != NULL) {
    li = 0;
    if (getToken(token, line, &li))
      continue;
    if (!strcasecmp(token, "exempt")) {
      while (!getToken(token, line, &li) && nExempt < MAX_EXEMPT)
        if ((uid = UIDfromLogin(token)) != (uid_t) - 1)
          exempt[nExempt++] = uid;
    } else if (!strcasecmp(token, "minuid")) {
      if (!getToken(token, line, &li))
        optMinUID = atoi(token);
    } else if (!strcasecmp(token, "kill")) {
      while (!getToken(token, line, &li) && nKillName < MAX_KILL)
        strcpy(killName[nKillName++], token);
    }
  }
  fclose(f);
}

int
cleanProc(int sig)
{
  int                      found,
                           i,
                           j,
                           nKill = 0;
  pid_t                    killPID[MAX_PROCS];

  for (i = 0; i < nProcs; i++) {
    if (procRec[i].tty)
      continue;
    for (found = 0, j = 0; j < nKillName; j++)
      if (!strcmp(killName[j], procRec[i].name))
        found = 1;
    if (found ||
        (!userExempt(procRec[i].uid) && !userLoggedIn(nLoginRecs, loginRec, procRec[i].uid)))
      killPID[nKill++] = procRec[i].pid;
  }
  for (i = 0; i < nKill; i++)
    kill(killPID[i], sig);
  return nKill;
}

int
userExempt(uid_t uid)
{
  int                      i;

  if (uid < optMinUID)
    return(1);

  for (i = 0; i < nExempt; i++)
    if (exempt[i] == uid)
      return 1;
  return 0;
}
