/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#ifndef COMMON_H
#define COMMON_H

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <utmp.h>
#include <string.h>
#include "types.h"
#include "macros.h"
#include "hash.h"

// magic # for acua_users file
#define ACUA_USERS_MAGIC                "ACUA202"
#define ACUA_USERS_MAGIC_LEN            8

// ACUA Version
#define ACUA_VERSION                    "3.03"
#define ACUA_VERSION_EXTRA		"Designed for Linux Kernel 2.4.x"

// max hosts a user can be logged into
#define MAX_HOSTS                       16
// max users online
#define MAX_USERS                       256
// max processes running
#define MAX_PROCS                       512
// max expire warning times
#define MAX_EXPIRE_WARN_TIMES           16
// max boot warning times
#define MAX_BOOT_WARN_TIMES             16
// max # of time classes
#define MAX_TIME_CLASSES                16
// max chars in a login name (not including NULL terminator)
#define MAX_LOGINCHARS                  16
// max PPP devices
#define MAX_PPP_UNITS                   MAX_USERS
// op codes for fLock
// get an exclusive file lock
#define LOCK_EXCLUSIVE                  0
// unlock the file
#define LOCK_UNLOCK                     1
// block until lock is obtained
#define LOCK_BLOCK                      2
// bits for UserRec.flags field
#define FLG_PRIORITY                    0xe000
#define FLG_SMARTTIME                   0x1000
#define FLG_SSMARTTIME                  0x0800
#define FLG_TCSMARTTIME                 0x0400
#define FLG_SMARTBOOT                   0x0200
#define FLG_SSMARTBOOT                  0x0100
#define FLG_TCSMARTBOOT                 0x0080
#define FLG_ISMARTBOOT                  0x0040
#define FLG_EXPIRE                      0x0020
#define FLG_WARNEXPIRE                  0x0010
#define FLG_WARNBOOT                    0x0008
#define FLG_EXPLAINBOOT                 0x0004
#define FLG_NOUNSUB                     0x0002
#define FLG_LOCK                        0x0001
// macros to access fields in the UserRec.flags bitfield
// get NOUNSUB flag
#define NOUNSUB(f)                      (((f) & FLG_NOUNSUB)
// get priority (0-7)
#define PRIORITY(f)                     (((f) & FLG_PRIORITY) >> 13)
// get SmartTime flag
#define SMARTTIME(f)                    ((f) & FLG_SMARTTIME)
// get Session SmartTime flag
#define SSMARTTIME(f)                   ((f) & FLG_SSMARTTIME)
// get TimeClass SmartTime flag
#define TCSMARTTIME(f)                  ((f) & FLG_TCSMARTTIME)
// get SmartBoot flag
#define SMARTBOOT(f)                    ((f) & FLG_SMARTBOOT)
// get Session SmartBoot flag
#define SSMARTBOOT(f)                   ((f) & FLG_SSMARTBOOT)
// get TimeClass SmartBoot flag
#define TCSMARTBOOT(f)                  ((f) & FLG_TCSMARTBOOT)
// get Idle SmartBoot flag
#define ISMARTBOOT(f)                   ((f) & FLG_ISMARTBOOT)
// get expire flag (see EXPIRE_DELETE and EXPIRE_UNSUBSCRIBE below)
#define EXPIRE(f)                       ((f) & FLG_EXPIRE)
// get WarnExpire flag
#define WARNEXPIRE(f)                   ((f) & FLG_WARNEXPIRE)
// get WarnBoot flag
#define WARNBOOT(f)                     ((f) & FLG_WARNBOOT)
// get ExplainBoot flag
#define EXPLAINBOOT(f)                  ((f) & FLG_EXPLAINBOOT)
// get Lock flag
#define LOCK(f)                         ((f) & FLG_LOCK)
// values for EXPIRE(f)
#define EXPIRE_DELETE                   0x0000
#define EXPIRE_UNSUBSCRIBE              0x0020

typedef struct
{
    // UNIX user id
    uid_t         uid                             __attribute__((packed));
    // flags (see FLG_* and *(f) macros)
    word          flags                           __attribute__((packed));
    // area code
    word          phNoArea                        __attribute__((packed));
    // local phone number
    word          phNoLocal                       __attribute__((packed));
    // number of hosts user is logged into
    byte          nHosts                          __attribute__((packed));
    // hosts user is logged into
    word          host[MAX_HOSTS]                 __attribute__((packed));
    // number of logins on each host
    byte          nLogins[MAX_HOSTS]              __attribute__((packed));
    // last login on each host
    time_t        lastLogin[MAX_HOSTS]            __attribute__((packed));
    // last time online for each host
    time_t        lastOnline[MAX_HOSTS]           __attribute__((packed));
    // max logins permitted
    word          maxLogins                       __attribute__((packed));
    // maximum #/minutes to deduct per minute (e.g. even if a user
    // has logins on 5 hosts, you can only deduct 2 minutes if you want...)
    word          maxDeduct                       __attribute__((packed));
    // idle limit in minutes
    int           idleLimit                       __attribute__((packed));
    // PPP user must xfer at least PPPidleBytes in last PPPidleMinutes
    // or be considered idle
    int           PPPidleMinutes                  __attribute__((packed));
    word          PPPidleBytes                    __attribute__((packed));
    // overall time left
    int           tLeft                           __attribute__((packed));
    // overall time limit
    int           tLimit                          __attribute__((packed));
    // how much credit to extend the user (how far beyond their time
    // limit they can go)
    int           credit                          __attribute__((packed));
    // session time left
    int           sLeft                           __attribute__((packed));
    // session time limit
    int           sLimit                          __attribute__((packed));
    // class time left for each time class
    int           cLeft[MAX_TIME_CLASSES]         __attribute__((packed));
    // class time limit for each time class
    int           cLimit[MAX_TIME_CLASSES]        __attribute__((packed));
    // overall bytes transmitted (e.g. uploaded)
    dword         bTx                             __attribute__((packed));
    // overall bytes received (e.g. downloaded)
    dword         bRx                             __attribute__((packed));
    // bytes transmitted limit
    dword         bTxLimit                        __attribute__((packed));
    // bytes received limit
    dword         bRxLimit                        __attribute__((packed));
    // bytes transferred (xmit + recv) limit
    dword         bLimit                          __attribute__((packed));
    // session bytes transmitted
    dword         bStx                            __attribute__((packed));
    // session bytes received
    dword         bSrx                            __attribute__((packed));
    // session bytes transmitted limit
    dword         bStxLimit                       __attribute__((packed));
    // session bytes received limit
    dword         bSrxLimit                       __attribute__((packed));
    // session bytes transferred (xmit + recv) limit
    dword         bSlimit                         __attribute__((packed));
    // bytes xferred in each of last 60 minutes
    // (used by acua_updated to determine idleness)
    word          nBytes[60]                      __attribute__((packed));
    // when user was locked
    time_t        lockDate                        __attribute__((packed));
    // when user was subscribed
    time_t        subscrDate                      __attribute__((packed));
    // When a user is subscribed, data is copied into these fields so that
    // it can be retrieved when the user is unsubscribed.  This allows you
    // to make modifications to a user that will be undone when they are
    // unsubscribed.  e.g. subFlags is a copy of flags, subMaxLogins is
    // a copy of maxLogins, etc...  see acua_subscribe, acua_unSubscribe
    time_t        subExpire                       __attribute__((packed));
    word          subFlags                        __attribute__((packed));
    word          subMaxLogins                    __attribute__((packed));
    word          subMaxDeduct                    __attribute__((packed));
    int           subIdleLimit                    __attribute__((packed));
    int           subPPPidleMinutes               __attribute__((packed));
    word          subPPPidleBytes                 __attribute__((packed));
    int           subTlimit                       __attribute__((packed));
    int           subCredit                       __attribute__((packed));
    int           subSlimit                       __attribute__((packed));
    int           subClimit[MAX_TIME_CLASSES]     __attribute__((packed));
    dword         subBtxLimit                     __attribute__((packed));
    dword         subBrxLimit                     __attribute__((packed));
    dword         subBlimit                       __attribute__((packed));
    dword         subBStxLimit                    __attribute__((packed));
    dword         subBSrxLimit                    __attribute__((packed));
    dword         subBSlimit                      __attribute__((packed));
    // total minutes online
    word          tMinutes                        __attribute__((packed));
    // when account was created
    time_t        creation                        __attribute__((packed));
    // expiry date
    time_t        expire                          __attribute__((packed));
} UserRec;      /* sizeof(UserRec) = 885 */

typedef struct
{
    time_t        startTime;
    time_t        expire;
    word          nMinutes[24 * 60];
    word          linesBusy[24 * 60];
    word          sessionLength[24 * 60];
} UsageRec;

typedef struct
{ 
    byte          flags                           __attribute__((packed));
    word          phNoArea                        __attribute__((packed));
    word          phNoLocal                       __attribute__((packed));
    time_t        expire                          __attribute__((packed));
} BanRec;

typedef struct
{
    pid_t         pid;
    uid_t         uid;
    uid_t         euid;
    dev_t         tty;
    char          name[16];
} ProcRec;

typedef struct
{
    uid_t         uid;
    dev_t         tty;
    char          ttyName[13];
    time_t        time;
} LoginRec;

typedef struct
{
    int           startDay;
    int           endDay;
    int           startHour;
    int           startMin;
    int           endHour;
    int           endMin;
} TimeClass;

typedef struct PPPuserRec
{
  uid_t         uid;
  Boolean       root;
  pid_t         pid;
  int           unit;
  dev_t         tty;
  time_t        loginTime;
} PPPuserRec;

// these are interfaces for most of the acua commands (see acua.cc)
extern int               acua_addRec(int argc, char **argv);
extern int               acua_ban(int argc, char **argv);
extern int               acua_daysLeft(int argc, char **argv);
extern int               acua_delRec(int argc, char **argv);
extern int               acua_dump(int argc, char **argv);
extern int               acua_dumpStats(int argc, char **argv);
extern int               acua_expire(int argc, char **argv);
extern int               acua_forEach(int argc, char **argv);
extern int               acua_kickUser(int argc, char **argv);
extern int               acua_lock(int argc, char **argv);
extern int               acua_modRec(int argc, char **argv);
extern int               acua_purge(int argc, char **argv);
extern int               acua_renew(int argc, char **argv);
extern int               acua_subscribe(int argc, char **argv);
extern int               acua_sync(int argc, char **argv);
extern int               acua_touch(int argc, char **argv);
extern int               acua_unBan(int argc, char **argv);
extern int               acua_unLock(int argc, char **argv);
extern int               acua_unSubscribe(int argc, char **argv);
extern int               acua_pp(int argc, char **argv);
extern int               acua_timeLeft(int argc, char **argv);

// quit with an error message
extern void              errQuit(char *format,...);
// quit with an error message - with syslog support
extern void              errlQuit(char *format,...);
// quit with an OS error message
extern void              perrQuit(char *format,...);
// quit with an OS error message - with syslog support
extern void              perrlQuit(char *format,...);
// get a device number from a device name
extern dev_t             devNumFromName(char *devName);
// get an acua line# from a device number
extern int               lineNo(dev_t dev);
// run a command
extern int               runCommand(char *path,...);
extern int               runCommandV(char *path, char **arg);
// become a daemon
extern void              daemonInit();
// add time to the given time
extern time_t            addTime(time_t t, char *s);
// get a process list
extern void              procList(int *nProcs, ProcRec *procRec);
// determine the maximum number of users to kick off now
// (in order to return system to non-busy state)
extern int               nKick(int nProcs, ProcRec *procRec);
// get a list of logged-in users
extern void              userList(int *nLogins, LoginRec *loginRec);
// last login on *any* host
extern time_t            lastLogin(UserRec *ur);
// last online time on *any* host
extern time_t            lastOnline(UserRec *ur);
// add a host to the list of hosts user is logged into
extern void              addHost(UserRec *ur, word host);
// find index of host in list of hosts user is logged into
extern int               findHost(UserRec *ur, word host);
// remove a host from the list of hosts user is logged into
extern void              removeHost(UserRec *ur, word host);
// get total number of logins on *all* hosts
extern word              nLogins(UserRec *ur);
// get total number of logins for the given user
extern word              nLogins(HashTable *loginRec, uid_t uid);
// determine whether a user is logged in
extern int               userLoggedIn(int nLogins,
                                      LoginRec *loginRec,
                                      uid_t uid);
// tokenizer
extern int               getToken(char *d, char *s, int *idx);
// determine if the test string given is all numbers or not
extern int               isNumber (char *test);
// convert a login name to a UNIX user id
extern uid_t             UIDfromLogin(char *login);
// find the group the login name belongs to
extern gid_t             GIDfromLogin(char *login);
// convert a UID into a login name
extern char             *loginFromUID(char *login, uid_t uid);
// open the user file
extern void              userFileOpen();
// close the user file
extern void              userFileClose();
// read a record from the user file
extern int               userFileRead(UserRec *ur);
// write a record to the user file (at current file position)
extern int               userFileWrite(UserRec *ur);
// edit a record in the user file (rewind one record, the write a record)
extern void              userFileEdit(UserRec *ur);
// search the user file for the given uid
extern int               userFileSearch(UserRec *ur, uid_t uid);
// rewind the user file
extern void              userFileRewind();
// open the ban file
extern void              banFileOpen();
// close the ban file
extern void              banFileClose();
// read a ban record
extern int               banFileRead(BanRec *br);
// write a ban record
extern int               banFileWrite(BanRec *br);
// search for a ban on an area/local phone number
extern int               banFileSearch(BanRec *br,
                                       word phNoArea,
                                       word phNoLocal);
// convert a phone number to integers
extern int               phNo2Words(word *phNoArea,
                                    word *phNoLocal,
                                    char *phNo,
                                    int phNoDigits);
// convert integers into a phone number
extern char             *words2PhNo(char *phNo,
                                    word phNoArea,
                                    word phNoLocal);
// determine sum of integers in a string
extern int               stringSum(char *s);
// convert byte total to string (e.g. 1048576 ==> "1 MB")
extern void              bytes2ASCII(char *str, dword bytes);
// determine whether current time falls within the given time class
extern int               inTimeClass(int tc);
// determine the current time class
extern int               curTimeClass(UserRec *ur);
// lock a file
extern int               fLock(char *path, int operation);
// do ACUA pre-processing on a file
extern void              preprocessFile(UserRec *ur,
                                        char *inPath,
                                        char *outPath);
// convert minutes to "hh:mm" string
extern void              hhmm(char *s, int nMinutes);
// find currently active PPP units
extern void              PPPfindUnits(HashTable *PPPuser,
                                      HashTable *loginRecTable,
                                      HashTable *procRecTable);
// get an INET host address from a hostname (e.g. resolve the name)
extern word              hostAddr(char *hostname);
// hash a user ID
extern word              UIDhash(word size, void *key);
// compare user IDs
extern int               UIDcomp(void *p1, void *p2);
// hash a process ID
extern word              PIDhash(word size, void *key);
// compare process IDs
extern int               PIDcomp(void *p1, void *p2);
// hash a TTY
extern word              TTYhash(word size, void *key);
// compare TTYs
extern int               TTYcomp(void *p1, void *p2);
// read the ACUA configuration (acua.config) and use it to set the
// opt* variables below
extern void              readConfig();

extern const int         days[12];
extern char             *configFilePath;
extern char             *userFilePath;
extern FILE             *userFile;
extern char             *banFilePath;
extern FILE             *banFile;
extern int               optPurgeDays;
extern int               optMinDeduct;
extern int               optBusyThreshold;
extern int               optMaxKick;
extern int               optLowCPUpriority,
                         optHighCPUpriority;
extern int               optReturnDelay;
extern int               optGuestTime;
extern int               optGuestPriority;
extern char              optMailHost[64];
extern char              optMailProg[64];
extern int               optMailWait;
extern int               optMailUser;
extern int               optMailGroup;
extern int               optExplainBoot;
extern int               nBootWarnTimes;
extern int               bootWarnTime[MAX_BOOT_WARN_TIMES];
extern int               optPPPWarnBoot;
extern int               nExpireWarnTimes;
extern int               expireWarnTime[MAX_EXPIRE_WARN_TIMES];
extern char              optWarnExpireCC[64];
extern int               optIdleLimit;
extern int               optPPPidleMinutes;
extern int               optPPPidleBytes;
extern int               optSmartTime;
extern int               optSessionSmartTime;
extern int               optTimeClassSmartTime;
extern int               optSmartBoot;
extern int               optSessionSmartBoot;
extern int               optTimeClassSmartBoot;
extern int               optIdleSmartBoot;
extern int               nTimeClasses;
extern TimeClass         timeClass[MAX_TIME_CLASSES];
extern int               nExcluded;
extern uid_t             excluded[64];
extern char              optModemDial[64];
extern char              optPhNoAreaFormat[16];
extern char              optPhNoLocalFormat[16];
extern int               optPhNoDigits;
extern int               optNoRecLogin;
extern int               userFileFD,
                         banFileFD;
extern int               nLines;
extern dev_t             lineDev[MAX_USERS];

#endif
