/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2001  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <ctype.h>
#include <limits.h>
#include <string.h>
#include <sys/stat.h>
#include <syslog.h>
#include <time.h>
#include "security.h"
#include "common.h"

#define   MSG_DEFAULT                0
#define   MSG_LOCK                   1
#define   MSG_MAX_LOGINS             2
#define   MSG_BUSY                   3
#define   MSG_DATA_PERIOD            4
#define   MSG_TIME_CLASS             5
#define   MSG_TIME_PERIOD            6
#define   MSG_SESSION_WAIT           7
#define   MSG_NO_RECORD              8
#define   MSG_NO_LOGIN               9
#define   MSG_BAD_LOGIN             10
#define   MSG_NO_TIME_LIMIT         11
#define   MSG_MAX_SUPPORTED_CONN    12
#define   MSG_CONN_TYPE             13

void                     printFile(char *path);

const char              *msgPath[14] = { LIB"/acua_login.default",
					 LIB"/acua_login.lock",
                                         LIB"/acua_login.max_logins",
                                         LIB"/acua_login.busy",
                                         LIB"/acua_login.data_period",
                                         LIB"/acua_login.time_class",
                                         LIB"/acua_login.time_period",
					 LIB"/acua_login.session_wait",
					 LIB"/acua_login.no_record",
					 LIB"/acua_login.no_login",
					 LIB"/acua_login.bad_login",
					 LIB"/acua_login.tlimit_zero",
					 LIB"/acua_login.max_conn",
					 LIB"/acua_login.conn_type" };

const char           *secMsgPath[7] = { LIB"/acua_login.sec_term_owner",
                                        LIB"/acua_login.sec_logname_uid",
                                        LIB"/acua_login.sec_unknown_user",
                                        LIB"/acua_login.sec_ppp_spoofed",
					LIB"/acua_login.sec_unknown_conn",
					LIB"/acua_login.sec_stat_failed",
                                        LIB"/acua_login.sec_cipe_spoofed" };

int
main(int argc, char **argv)
{
    int                      msg = -1,
    			     curClass = -1;
    uid_t                    uid;
    UserRec                  ur;
    int                      sLeft;
    char                     hostname[256],
    			     dPath[256],
                            *login;
    Boolean                  userRecFound = FALSE;
    word                     myHostAddr;
    int                      hostIdx,
			     conninfoIdx;

    if (argc > 1) {
	printf("Usage: acua_login\n\n");
	printf("acua_login determines the type of login (TTY, PPP, PPPoE) by looking\n");
	printf("           at the environment variables.  For more information please\n");
	printf("           see the acua_login documentation which comes with ACUA.\n");
	return(1);
    }

    readConfig();
    login = getLoginName();
    if (!login)
    {
	msg = MSG_NO_LOGIN;
	goto reject;
    }

    /* check that the type of login is supported */
    if (loginType() == CONN_NONE)
    {
	msg = MSG_CONN_TYPE;
	goto reject;
    }
    
    /* Security checks.  We probably can't catch every situation but
     * it's better to try to catch out malicious users than to not try
     * at all.
     */
    msg = securityCheck(loginType(), login);
    if (msg != SECURITY_OK)
    {
	goto reject;
    }

    gethostname(hostname, 256);
    myHostAddr = hostAddr(hostname);
    uid = UIDfromLogin(login);

    if (uid == (uid_t)-1) {
	msg = MSG_BAD_LOGIN;
	goto reject;
    }

    userFileOpen();

    // find their user record
    if (userFileSearch(&ur, uid)) {
	if (optNoRecLogin == 1) {
	    goto accept;
	} else {
	    msg = MSG_NO_RECORD;
	    goto reject;
	}
    }
    userRecFound = TRUE;

    curClass = curTimeClass(&ur);

    // are they locked out?
    if (LOCK(ur.flags)) {
	msg = MSG_LOCK;
	goto reject;
    }

    // deny logins with a time limit of 0
    if (ur.tLimit == 0) {
	msg = MSG_NO_TIME_LIMIT;
	goto reject;
    }

    if ((ur.bLimit && ur.bTx + ur.bRx > ur.bLimit) ||
	(ur.bTxLimit && ur.bTx > ur.bTxLimit) ||
	(ur.bRxLimit && ur.bRx > ur.bRxLimit))
    {
	msg = MSG_DATA_PERIOD;
	goto reject;
    }

    // calculate sLeft
    sLeft = INT_MAX;
    if ((!SMARTBOOT(ur.flags)) && ur.tLimit >= 0)
	sLeft = min(sLeft, ur.tLeft + ur.credit);
    if ((!SSMARTBOOT(ur.flags)) && ur.sLimit >= 0)
	sLeft = min(sLeft, ur.sLimit);
    if ((!TCSMARTBOOT(ur.flags)) && curClass >= 0 && ur.cLimit[curClass] >= 0)
	sLeft = min(sLeft, ur.cLeft[curClass]);
    if ((sLeft <= 0))
	goto reject;

    goto accept;

reject:
    if (msg < 0) {
	if (ur.tLeft <= 0)
	    msg = MSG_TIME_PERIOD;
	else if (curClass >= 0 && ur.cLimit[curClass] >= 0 && ur.cLeft[curClass] <= 0)
	    msg = MSG_TIME_CLASS;
	else
	    msg = MSG_DEFAULT;
    }

    /* Create tempory file */
    snprintf(dPath, sizeof(dPath), "/tmp/acua_login.XXXXXX");
    if (mkstemp(dPath) == -1) {
	/* couldn't make the tempory file */
	syslog(LOG_ERR, "Could not create tempory file in /tmp.  Login denied.");
	printf("\n\nSorry, you are not able to login at this time due to a\n");
	printf("technical error.  Please contact your service provider.\n\n");
	fflush(stdout);
        userFileClose();
	return(1);
    }

    if (msg < SECURITY_OK)
    {
	preprocessFile(&ur, (char*)msgPath[msg], dPath);
	printFile(dPath);
	unlink(dPath);
    } else {
	memset(&dPath, 0, sizeof(dPath));
	snprintf(dPath, sizeof(dPath), "%s", (char*)secMsgPath[msg-50]);
        printFile(dPath);
    }
    fflush(stdout);
    userFileClose();
    sleep(5);
    return(1);

accept:
    if (userRecFound) {

	if ((conninfoIdx = conninfoSearchEmpty(&ur)) == -1)
	{
	    /* no more slots left! */
	    syslog(LOG_ERR, "Maximum supported connections reached for user [%s]", login);
	    msg = MSG_MAX_SUPPORTED_CONN;
	    goto reject;
	}

	// update nLogins, and make sure we're not allowing more than one connection.
	addHost(&ur, myHostAddr);
	hostIdx = findHost(&ur, myHostAddr);
	if (ur.nLogins[hostIdx] >= ur.maxLogins) {
	    removeHost(&ur, myHostAddr);
	    msg = MSG_MAX_LOGINS;
	    goto reject;
	}
	ur.nLogins[hostIdx] += (word) 1;

        /* fill in conninfo information */
	ur.conninfo.type[conninfoIdx] = loginType();
	if (ur.conninfo.type[conninfoIdx] == CONN_NONE)
	{
	    msg = MSG_CONN_TYPE;
	    goto reject;
	}

	if (ur.conninfo.type[conninfoIdx] == CONN_TTY)
	{
	    /* TTY Connection */
	    memcpy(&ur.conninfo.if_name[conninfoIdx],
		   ttyname(STDIN_FILENO) + 5,
		   sizeof(ur.conninfo.if_name[conninfoIdx]));
	    ur.conninfo.pid[conninfoIdx] = getppid();
	} else if ((ur.conninfo.type[conninfoIdx] == CONN_PPP) ||
		   (ur.conninfo.type[conninfoIdx] == CONN_PPPoE))
	{
	    /* PPP / PPPoE Connection */
	    memcpy(&ur.conninfo.if_name[conninfoIdx],
		   getenv("IFNAME"),
		   sizeof(ur.conninfo.if_name[conninfoIdx]));
            ur.conninfo.if_name[conninfoIdx][sizeof(ur.conninfo.if_name[conninfoIdx])] = '\0';
	    ur.conninfo.pid[conninfoIdx] = atoi(getenv("PPPD_PID"));
	} else if (ur.conninfo.type[conninfoIdx] == CONN_CIPE)
	{
            /* CIPE Connection */
	    memcpy(&ur.conninfo.if_name[conninfoIdx],
		    getenv("CIPE_IFNAME"),
		    sizeof(ur.conninfo.if_name[conninfoIdx]));
	    ur.conninfo.if_name[conninfoIdx][sizeof(ur.conninfo.if_name[conninfoIdx])] = '\0';
	    ur.conninfo.pid[conninfoIdx] = atoi(getenv("CIPE_PID"));
	} else {
	    msg = MSG_CONN_TYPE;
	    goto reject;
	}

	/* common for all types of connections */
	ur.conninfo.loginTime[conninfoIdx] = time(NULL);
	ur.conninfo.host[conninfoIdx] = myHostAddr;

	ur.sLeft = ur.sLimit;
	ur.bStx = ur.bSrx = 0;
	ur.loginTime = time(NULL);
	hostIdx = findHost(&ur, myHostAddr);
	if (ur.lastLogin[hostIdx] == 0) {
	    ur.lastLogin[hostIdx] = time(NULL);
	    ur.lastOnline[hostIdx] = time(NULL);
	}
	memset(ur.nBytes, 0, 60 * sizeof(word));
	userFileEdit(&ur);
    }
    userFileClose();
    return(0);
}

void
printFile(char *path)
{
    FILE                    *f = fopen(path, "r");
    int                     s;

    if (!f) perrQuit("fopen");
    printf("\n");
    while ((s = fgetc(f)) != EOF) putchar(s);
    printf("\n");
    fclose(f);
}

