/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2001  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <assert.h>
#include <ctype.h>
#include <string.h>
#include "common.h"

void                     usage();

int
acua_modRec(int argc, char **argv)
{
  int                      i,
                       optStrict = 0,
                           loginIdx;
  uid_t                    uid;
  UserRec                  ur;

  readConfig();

  if (argc < 3)
    usage();
  if (!strcmp(argv[1], "-s")) {
    optStrict = 1;
    loginIdx = 2;
  } else loginIdx = 1;
  argc = argc - loginIdx;
  if (argc < 2)
    usage();
  uid = UIDfromLogin(argv[loginIdx]);
  if (uid == (uid_t)-1)
    errQuit("user does not exist: %s", argv[loginIdx]);
  userFileOpen();
  if (userFileSearch(&ur, uid))
    errQuit("user not found: %s", argv[loginIdx]);
  if (argc == 2 && (argv[loginIdx + 1][0] == '+' || argv[loginIdx + 1][0] == '-')) {
    word                     mask = 0;

    if (isdigit(argv[loginIdx + 1][1])) {
      mask = atoi(argv[loginIdx + 1] + 1);
      if (mask > 15)
        errQuit("valid user flags are in the range [0,15]");
      mask = 0x00010000 << mask;
    } else if (!strcasecmp(argv[loginIdx + 1] + 1, "SMARTTIME"))
      mask = FLG_SMARTTIME;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "SSMARTTIME"))
      mask = FLG_SSMARTTIME;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "TCSMARTTIME"))
      mask = FLG_TCSMARTTIME;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "SMARTBOOT"))
      mask = FLG_SMARTBOOT;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "SSMARTBOOT"))
      mask = FLG_SSMARTBOOT;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "TCSMARTBOOT"))
      mask = FLG_TCSMARTBOOT;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "ISMARTBOOT"))
      mask = FLG_ISMARTBOOT;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "WARNBOOT"))
      mask = FLG_WARNBOOT;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "EXPLAINBOOT"))
      mask = FLG_EXPLAINBOOT;
    else if (!strcasecmp(argv[loginIdx + 1] + 1, "NOUNSUB"))
      mask = FLG_NOUNSUB;
    else
      errQuit("unknown flag: %s", argv[loginIdx + 1] + 1);
    if (argv[loginIdx + 1][0] == '+')
      ur.flags |= mask;
    else
      ur.flags &= ~mask;
  } else if (!isdigit(argv[loginIdx + 1][0])) {
    if (argc == 4) {
      if (!strcasecmp(argv[loginIdx + 1], "priority")) {
        ur.flags &= ~FLG_PRIORITY;
        ur.flags |= atoi(argv[loginIdx + 3]) << 13;
      } else if (!strcasecmp(argv[loginIdx + 1], "phNo")) {
        if (phNo2Words(&ur.phNoArea, &ur.phNoLocal, argv[loginIdx + 3], optPhNoDigits) || (ur.phNoArea && !ur.phNoLocal))
          errQuit("invalid phone number: %s", argv[loginIdx + 3]);
      } else if (!strcasecmp(argv[loginIdx + 1], "expire")) {
        if (!strcmp(argv[loginIdx + 2], "+=")) {
          if (!ur.expire) ur.expire = time(NULL);
          ur.expire = addTime(ur.expire, argv[loginIdx + 3]);
        } else if (!strcmp(argv[loginIdx + 2], "-=")) {
          time_t diff = addTime(ur.expire, argv[loginIdx + 3]) - ur.expire;
          ur.expire -= diff;
        } else if (!strcmp(argv[loginIdx + 2], "=")) {
          ur.expire = addTime(time(NULL), argv[loginIdx + 3]);
        } else
          errQuit("unknown operator: %s", argv[loginIdx + 2]);
        if (EXPIRE(ur.flags) == EXPIRE_UNSUBSCRIBE && !ur.expire)
          errQuit("<expire> = 0 invalid (user %s has a subscription).", argv[loginIdx]);
      } else if (argv[loginIdx + 1][0] == 'b') {
        int                      t = atol(argv[loginIdx + 3]);
        dword                   *p = NULL;

        if (!strcasecmp(argv[loginIdx + 1], "bTx"))
          p = &ur.bTx;
        else if (!strcasecmp(argv[loginIdx + 1], "bRx"))
          p = &ur.bRx;
        else if (!strcasecmp(argv[loginIdx + 1], "bTxLimit"))
          p = &ur.bTxLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "bRxLimit"))
          p = &ur.bRxLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "bLimit"))
          p = &ur.bLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "bStxLimit"))
          p = &ur.bStxLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "bSrxLimit"))
          p = &ur.bSrxLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "bSlimit"))
          p = &ur.bSlimit;
        else
          errQuit("unknown field: %s", argv[loginIdx + 1]);
        if (!strcmp(argv[loginIdx + 2], "+="))
          *p += t;
        else if (!strcmp(argv[loginIdx + 2], "-="))
          *p -= t;
        else if (!strcmp(argv[loginIdx + 2], "="))
          *p = t;
        else
          errQuit("unknown operator: %s", argv[loginIdx + 2]);
      } else {
        int                      t = atoi(argv[loginIdx + 3]);
        int                     *p = NULL;

        if (!strcasecmp(argv[loginIdx + 1], "maxLogins"))
          p = (int*)&ur.maxLogins;
        else if (!strcasecmp(argv[loginIdx + 1], "maxDeduct"))
          p = (int*)&ur.maxDeduct;
        else if (!strcasecmp(argv[loginIdx + 1], "idleLimit"))
          p = (int*)&ur.idleLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "PPPidleBytes"))
          p = (int*)&ur.PPPidleBytes;
        else if (!strcasecmp(argv[loginIdx + 1], "PPPidleMin"))
          p = (int*)&ur.PPPidleMinutes;
        else if (!strcasecmp(argv[loginIdx + 1], "tLeft"))
          p = &ur.tLeft;
        else if (!strcasecmp(argv[loginIdx + 1], "tLimit"))
          p = &ur.tLimit;
        else if (!strcasecmp(argv[loginIdx + 1], "credit"))
          p = &ur.credit;
        else if (!strcasecmp(argv[loginIdx + 1], "sLeft"))
          p = &ur.sLeft;
        else if (!strcasecmp(argv[loginIdx + 1], "sLimit"))
          p = &ur.sLimit;
        else
          errQuit("unknown field: %s", argv[loginIdx + 1]);
        if (!strcmp(argv[loginIdx + 2], "+="))
          *p += t;
        else if (!strcmp(argv[loginIdx + 2], "-="))
          *p -= t;
        else if (!strcmp(argv[loginIdx + 2], "="))
          *p = t;
        else
          errQuit("unknown operator: %s", argv[loginIdx + 2]);
      }
    } else if (argc == 5) {
      int                      c = atoi(argv[loginIdx + 2]);
      int                      t = atoi(argv[loginIdx + 4]);
      int                     *p = NULL;

      if (!strncasecmp(argv[loginIdx + 1], "cLeft", 5)) {
        if (c < 0 || c >= MAX_TIME_CLASSES)
          errQuit("invalid time class: ", argv[loginIdx + 2]);
        p = &ur.cLeft[c];
      } else if (!strncasecmp(argv[loginIdx + 1], "cLimit", 6)) {
        if (c < 0 || c >= MAX_TIME_CLASSES)
          errQuit("invalid time class: ", argv[loginIdx + 2]);
        p = &ur.cLimit[c];
      } else
        errQuit("unknown field: %s", argv[loginIdx + 1]);
      if (!strcmp(argv[loginIdx + 3], "+="))
        *p += t;
      else if (!strcmp(argv[loginIdx + 3], "-="))
        *p -= t;
      else if (!strcmp(argv[loginIdx + 3], "="))
        *p = t;
      else
        errQuit("unknown operator: %s", argv[loginIdx + 3]);
    } else
      usage();
  } else if (argc >= 3) {
    int                      sessionLimit,
                             priority;

    sessionLimit = argc >= 4 ? atoi(argv[loginIdx + 3]) : atoi(argv[loginIdx + 2]);
    if (sessionLimit < 0 || sessionLimit > atoi(argv[loginIdx + 2]))
      errQuit("invalid sessionLimit: %d", sessionLimit);
    priority = argc >= 5 ? atoi(argv[loginIdx + 4]) : 4;
    if (priority < 0 || priority > 7)
      errQuit("invalid priority: %d", priority);
    ur.flags = (ur.flags & ~FLG_PRIORITY) | (priority << 13);
    ur.expire = addTime(time(NULL), argv[loginIdx + 1]);
    if (EXPIRE(ur.flags) == EXPIRE_UNSUBSCRIBE && !ur.expire)
      errQuit("<expire> = 0 invalid (user %s has a subscription).", argv[loginIdx]);
    ur.tLeft = ur.tLimit = atoi(argv[loginIdx + 2]);
    ur.sLeft = ur.sLimit = sessionLimit;
    for (i = 0; i < MAX_TIME_CLASSES; i++)
      ur.cLeft[i] = ur.cLimit[i] = -1;
    for (i = loginIdx + 5; i < min(5 + MAX_TIME_CLASSES, argc); i++)
      ur.cLeft[i - loginIdx] = ur.cLimit[i - loginIdx] = atoi(argv[i]);
  } else
    usage();
  if (optStrict) {
    if (ur.tLeft > ur.tLimit) ur.tLeft = ur.tLimit;
    if (ur.sLeft > ur.sLimit) ur.sLeft = ur.sLimit;
    for (i = 0; i < MAX_TIME_CLASSES; i++)
      if (ur.cLeft[i] > ur.cLimit[i])
        ur.cLeft[i] = ur.cLimit[i];
  }
  userFileEdit(&ur);
  userFileClose();
  printf("user modified: %s\n", argv[loginIdx]);
  return 0;
}

void
usage()
{
  printf("usage: modRec [-s]\n");
  printf("       <login> <expire> <timeLimit> [sessionLimit] [priority] {classLimit}\n");
  printf("     | <login> <+|-><SMARTTIME>|<SSMARTTIME>|<TCSMARTTIME>|\n");
  printf("                    <SMARTBOOT>|<SSMARTBOOT>|<TCSMARTBOOT>|\n");
  printf("                    <ISMARTBOOT>|<WARNBOOT>|<EXPLAINBOOT>|<0-15>\n");
  printf("     | <login> <maxLogins>|<maxDeduct> <+=|-=|=> <n>\n");
  printf("     | <login> <idleLimit>|<PPPidleBytes>|<PPPidleMin> <+=|-=|=> <n>\n");
  printf("     | <login> <tLeft>|<tLimit>|<credit>|<sLeft>|<sLimit> <+=|-=|=> <minutes>\n");
  printf("     | <login> <cLeft>|<cLimit> <class> <+=|-=|=> <minutes>\n");
  printf("     | <login> <bTx>|<bRx>|<bTxLimit>|<bRxLimit>|<bLimit>|\n");
  printf("               <bStxLimit>|<bSrxLimit>|<bSlimit> <+=|-=|=> <bytes>\n");
  printf("     | <login> priority = <priority>\n");
  printf("     | <login> phNo = <phNo>\n");
  errQuit("     | <login> expire <+=|-=|=> <expire>");
}
