/*   ACUA - Access Control and User Administration.
 *   Copyright (C) 2000  Robert Davidson.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   You may not sell ACUA or any part of ACUA for profit.
 */

#include <arpa/inet.h>
#include <assert.h>
#include <ctype.h>
#include <fcntl.h>
#include <limits.h>
#if __GLIBC__ >= 2
#include <asm/types.h>
#include <linux/if.h>
#else
#include <linux/types.h>
#include <linux/if.h>
#endif /* __GLIBC__ >= 2 */
#include <linux/ppp_defs.h>
#include <linux/if_ppp.h>
#include <math.h>
#include <pwd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/dir.h>
#include <sys/ioctl.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <syslog.h>
#include <utime.h>
#include <unistd.h>
#include "common.h"

#define PUNT_BUSY               0
#define PUNT_DATA_PERIOD        1
#define PUNT_DATA_SESSION       2
#define PUNT_TIME_IDLE          3
#define PUNT_TIME_CLASS         4
#define PUNT_TIME_PERIOD        5
#define PUNT_TIME_SESSION       6
#define USAGE_TOTAL             0
#define USAGE_DAY               1
#define USAGE_WEEK              2
#define USAGE_MONTH             3
#define USAGE_YEAR              4


typedef struct VictimRec {
  UserRec               ur;
  dev_t                 tty;
  char                  ttyName[12];
  int                   reason;
  int                   sLeft;
  int                   sendMail;
} VictimRec;

void                     update();
//char                    *userTTY(char *ttyPath, int uid, PPPuserRec *PPPuser);
void                     userMessage(char *ttyName, char *format,...);
int                      sendMail(UserRec *ur, char *login, char *cc,
                                  char *subject, char *sPath, char *dPath);
int                      PPPidleLeft(UserRec *ur);
int                      TTYidleLeft(UserRec *ur);
int     		 PPPupdate(UserRec *ur);
int                      warnExpire(time_t expire);
int                      CPUpriority(int priority);
int                      victimCmp(const VictimRec *v1, const VictimRec *v2);
void                     readUsageRecords();
void                     writeUsageRecords(int curMinute);
void                     unlinkOldLockFiles();
void                     segv_handler(int junk);
void			 killProcess(pid_t pid);

const char              *msgTimeWarning = "You have %d minute(s) remaining.";
const char              *msgIdleWarning = "You will be logged out after 1 more minute of idle time.";
const char              *msgPath[7] = { LIB"/acua_updated.busy",
                                        LIB"/acua_updated.data_period",
                                        LIB"/acua_updated.data_session",
                                        LIB"/acua_updated.idle",
                                        LIB"/acua_updated.time_class",
                                        LIB"/acua_updated.time_period",
                                        LIB"/acua_updated.time_session" };
const char              *puntReason[7] = { "system busy",
                                           "data limit exceeded",
                                           "session data limit exceeded",
                                           "idle time limit exceeded",
                                           "class time limit exceeded",
                                           "time limit exceeded",
                                           "session time limit exceeded" };
const char              *usageRecExpire[5] = { "", "1d", "7d", "1m", "12m" };
const char              *usageRecName[5] = { "overall",
                                             "day",
                                             "week",
                                             "month",
                                             "year" };

word                     myHostAddr;
int                      automatic = 1,
                         runAsDaemon = 1,
                         effect = 1,
                         loggingLevel = 1;
time_t                   configTime;
int                      systemBusy = 0;
int                      nLoginRecs;
LoginRec                 loginRec[MAX_USERS];
int                      nProcs;
uid_t                    guestUID;
UsageRec                 usageRec[5];

int
main(int argc, char **argv)
{
  int                      i;
  struct stat              _stat;
  time_t                   t;
  struct tm               *tmP;
  char                     hostname[256];
  struct in_addr           _in_addr;
  UserRec                  ur;

  signal(SIGSEGV, segv_handler);
  openlog("acua_updated", LOG_PID, LOG_DAEMON);
  readConfig();
  
  /* check if mail program is correct or not */
  if (stat(optMailProg, &_stat))
      errlQuit("refusing to start - MailProg is not a valid program. Check acua.config file.");

  configTime = time(NULL);
  while ((i = getopt(argc, argv, "adel")) != EOF) {
    switch (i) {
    case 'a':
      automatic = 0;
      break;
    case 'd':
      runAsDaemon = 0;
      break;
    case 'e':
      effect = 0;
      break;
    case 'l':
      loggingLevel = atoi(argv[optind++]);
      break;
    }
  }
  
  if (runAsDaemon)
    daemonInit();

  guestUID = UIDfromLogin("guest");
  gethostname(hostname, 256);
  myHostAddr = hostAddr(hostname);
  
  _in_addr.s_addr = myHostAddr;
  if (loggingLevel >= 1)
    syslog(LOG_INFO, "Hostname = %s, IP = %s",
           hostname,
           inet_ntoa(_in_addr));

  readConfig();
  /* Initialize the user database */
  syslog(LOG_INFO, "Initializing...");
  userFileOpen();
  while (!userFileRead(&ur))
  {
      conninfoUpdate(&ur, myHostAddr);
      userFileEdit(&ur);
  }
  userFileClose();
  syslog(LOG_INFO, "ACUA is up and running.");

  for (;;) {
    // re-read the config file if it's been updated
    if (stat(configFilePath, &_stat) < 0)
      perrQuit("couldn't stat %s", configFilePath);
    if (_stat.st_mtime > configTime) {
      readConfig();
      if (configTime && loggingLevel >= 1)
        syslog(LOG_INFO, "re-read %s", configFilePath);
      configTime = _stat.st_mtime;
    }

    /* do sync, purge, expire */
    time(&t);
    tmP = localtime(&t);
    if (tmP->tm_min == 0) {
      unlinkOldLockFiles();
    }
    /* FIXME - fix this crap up.  sync should be done AFTER purge/expire! */
    if (automatic && (tmP->tm_hour == 3)) {
      char *_argv[16] = { "acua", NULL, NULL };
      char arg[16];
      switch (tmP->tm_min) {
      case 0:
        if (acua_sync(1, _argv))
          syslog(LOG_ERR, "acua_sync could not complete.");
        break;
      case 1:
        if (optPurgeDays >= 14) {
          sprintf(arg, "%d", optPurgeDays);
          _argv[1] = arg;
          acua_purge(2, _argv);
          _argv[1] = NULL;
        }
        break;
      case 2:
        acua_expire(1, _argv);
        break;
      }
    }

    // do the update
    update();

    // FIXME - this is flawed!
    time(&t);
    tmP = localtime(&t);
    sleep(60 - tmP->tm_sec);
  }
  return 0;
}

void
update()
{
  char                     login[MAX_LOGINCHARS + 1],
                           ttyPath[32],
                           dPath[256];
  int                      i,
                           x,
                           nOnline = 0,
                           idleLeft,
                           sLeft,
  //                         sessionLength,
                           curClass,
                           deductMins,
                           deductTime,
                           deductSession,
                           puntDataPeriod,
                           puntDataSession,
  //                         nKill = 0,
                           nVictims,
                           curMinute;
//  struct stat              _stat;
  VictimRec                victim[MAX_USERS];
  UserRec                  ur;
//  LoginRec                *curLogin;
//  ProcRec                 *curProc;
//  pid_t                    killPID[MAX_PROCS];
  //PPPuserRec              *PPPuser;
  int                      PPPuser = 0;
  time_t                   t;
  struct tm               *tmP;
  word                     nUserLogins;
  int                      hostIdx;

  time(&t);
  tmP = localtime(&t);
  curMinute = tmP->tm_hour * 60 + tmP->tm_min;

  nVictims = 0;

  userFileOpen();
  readUsageRecords();
  while (!userFileRead(&ur)) {
    // make sure the user has a passwd entry
    if (!loginFromUID(login, ur.uid)) continue;
    
    // warn users when their accounts/subscriptions are almost expired
    if (ur.expire) {
      if (WARNEXPIRE(ur.flags) && !warnExpire(ur.expire)) {
        ur.flags &= ~FLG_WARNEXPIRE;
        userFileEdit(&ur);
      } else if (!WARNEXPIRE(ur.flags) && warnExpire(ur.expire)) {
        char subject[30];
        char *cc;

        // set the subject
        if (EXPIRE(ur.flags) == EXPIRE_DELETE)
          sprintf(subject, "Account Expires Soon");
        else
          sprintf(subject, "Subscription Expires Soon");

        /* Create tempory file */
	snprintf(dPath, sizeof(dPath), "/tmp/acua_updated.XXXXXX");
	if (mkstemp(dPath) == -1) {
            syslog(LOG_ERR, "Could not send mail to [%s]", login);
	    errlQuit("Could not create tempory file in /tmp");
	}
          
        // set the CC string
        if (optWarnExpireCC[0] != '\0') {
          cc = (char *) malloc(strlen(optWarnExpireCC)+1);
          sprintf(cc, "%s", optWarnExpireCC);
        } else cc = NULL;

        // send the mail
        sendMail(&ur, login, cc, subject, LIB"/acua_updated.warn_expire", dPath);
        if (optWarnExpireCC[0] != '\0') free(cc);
        ur.flags |= FLG_WARNEXPIRE;
      }
    }

    // clear out any hosts the user has no session to
    for (i = 0; i < ur.nHosts; i++)
	if ((time(NULL) > ur.lastOnline[i]) &&
	    ((time(NULL) - ur.lastOnline[i])) > 600)
	{
	    removeHost(&ur, ur.host[i]);
	    i = -1;
	}

    // make sure this host is in ur.host array, if not then save any
    // changes we have made and skip to the next record.
    if ((hostIdx = findHost(&ur, myHostAddr)) < 0)
    {
	userFileEdit(&ur);
	continue;
    }

    if ((ur.nLogins[hostIdx] = conninfoUpdate(&ur, myHostAddr)) == 0)
    {
	/* user has logged out */
	removeHost(&ur, myHostAddr);
        userFileEdit(&ur);
	continue;
    }

    userFileEdit(&ur);

    if (PPPUserOnline(&ur))
    {
        PPPuser = 1;
	goto found;
    }

    if (TTYUserOnline(&ur))
    {
	// set the CPU priority, see if they have any processes
	setpriority(PRIO_USER, ur.uid, CPUpriority(PRIORITY(ur.flags)));
        goto found;
    }

	// user is not online anymore -- record stats and remove this host from ur.host[]
/*	sessionLength = (t - ur.lastLogin[hostIdx]) / 60;
	if (sessionLength < optMinDeduct)
	{
	    ur.tLeft -= (optMinDeduct - sessionLength);
	    ur.sLeft = ur.sLimit - optMinDeduct;
	}
	if (sessionLength < 24 * 60)
	    for (i = 0; i < 5; i++)
		usageRec[i].sessionLength[sessionLength]++;
	removeHost(&ur, myHostAddr);
	userFileEdit(&ur);
	continue;
*/

found:
    // increment tMinutes for this user
    ur.tMinutes++;

    //nOnline += ur.nLogins[hostIdx];
    //if (!userTTY(ttyPath, ur.uid, PPPuser))
    //    continue;
    curClass = curTimeClass(&ur);
    PPPupdate(&ur);
    nUserLogins = nLogins(&ur);

    // update their record
    deductMins = min(ur.nLogins[hostIdx], ur.maxDeduct);
    deductTime = deductSession = 1;
    if (SMARTTIME(ur.flags) && !systemBusy)
	deductTime = 0;
    if (SSMARTTIME(ur.flags) && !systemBusy)
	deductSession = 0;
    for (i = 0; i < nTimeClasses; i++) {
	if (inTimeClass(i)) {
	    if (ur.cLimit[i] <= -3)
		deductTime = 0;
	    if (ur.cLimit[i] <= -2)
		deductSession = 0;
	    if (ur.cLimit[i] >= 0 && (!TCSMARTTIME(ur.flags) || systemBusy))
		ur.cLeft[i] -= deductMins;
	}
    }
    if (deductTime)
	ur.tLeft -= deductMins;
    if (deductSession)
	ur.sLeft -= deductMins;
    time(&ur.lastOnline[hostIdx]);
    if (effect)
	userFileEdit(&ur);

    // figure out how much time they have left
    sLeft = INT_MAX;
    if ((!SMARTBOOT(ur.flags) || systemBusy) && ur.tLimit >= 0)
	sLeft = min(sLeft, ur.tLeft + ur.credit);
    if ((!SSMARTBOOT(ur.flags) || systemBusy) && ur.sLimit >= 0)
	sLeft = min(sLeft, ur.sLeft);
    if ((!TCSMARTBOOT(ur.flags) || systemBusy) && curClass >= 0 && ur.cLimit[curClass] >= 0)
	sLeft = min(sLeft, ur.cLeft[curClass]);
    if (!PRIORITY(ur.flags) && systemBusy)
	sLeft = min(sLeft, -(int)(time(NULL) - ur.lastLogin[hostIdx]));

    // figure out how much idle time they have left
    idleLeft = INT_MAX;
    if (PPPuser && ur.PPPidleBytes && ur.PPPidleMinutes) {
	idleLeft = PPPidleLeft(&ur);
    } else if (!PPPuser && ur.idleLimit) {
	idleLeft = TTYidleLeft(&ur);
    }

    // send a warning message if necessary
    if (effect && WARNBOOT(ur.flags) && PPPuser) {
	if ((optPPPWarnBoot > 0) && (sLeft == optPPPWarnBoot)) {
	    char subject[64];

	    // set the subject
	    sprintf(subject, "Warning: possible disconnection within %d minutes.", optPPPWarnBoot);

	    /* Create tempory file */
	    snprintf(dPath, sizeof(dPath), "/tmp/acua_updated.XXXXXX");
	    if (mkstemp(dPath) == -1) {
		syslog(LOG_ERR, "Could not send mail to [%s]", login);
		errlQuit("Could not create tempory file in /tmp");
	    }

	    // send the mail
	    sendMail(&ur, login, "", subject, LIB"/acua_updated.warn_boot", dPath);
	}
    } else if (effect && WARNBOOT(ur.flags) && !PPPuser) {
	if (idleLeft == 1 && sLeft > 1)
	    userMessage(ttyPath + 5, (char *) msgIdleWarning);
	else if (deductTime || deductSession)
	    for (i = 0; i < nBootWarnTimes; i++)
		if (sLeft == bootWarnTime[i]) {
		    userMessage(ttyPath + 5, (char *) msgTimeWarning, sLeft);
		    break;
		}
    }

    // take idle time into account
    if (!ISMARTBOOT(ur.flags) || systemBusy)
	sLeft = min(sLeft, idleLeft);

    puntDataPeriod = (ur.bLimit && ur.bTx + ur.bRx > ur.bLimit) ||
	(ur.bTxLimit && ur.bTx > ur.bTxLimit) ||
	(ur.bRxLimit && ur.bRx > ur.bRxLimit);
    puntDataSession = (ur.bSlimit && ur.bStx + ur.bSrx > ur.bSlimit) ||
	(ur.bStxLimit && ur.bStx > ur.bStxLimit) ||
	(ur.bSrxLimit && ur.bSrx > ur.bSrxLimit);

    // victimize this user if necessary
    if (sLeft <= 0 || puntDataPeriod || puntDataSession) {
	memcpy(&victim[nVictims].ur, &ur, sizeof(UserRec));
        conninfoResetHost(&ur, myHostAddr);
	//victim[nVictims].tty = devNumFromName(ttyPath + 5);
	//strcpy(victim[nVictims].ttyName, ttyPath + 5);
	victim[nVictims].sLeft = sLeft;
	if ((ur.tLimit >= 0) && (ur.tLeft <= 0))
	    victim[nVictims].reason = PUNT_TIME_PERIOD;
	else if ((ur.sLimit >= 0) && (ur.sLeft <= 0))
	    victim[nVictims].reason = PUNT_TIME_SESSION;
	else if ((curClass >= 0) && (ur.cLimit[curClass] >= 0) && (ur.cLeft[curClass] <= 0))
	    victim[nVictims].reason = PUNT_TIME_CLASS;
	else if (idleLeft <= 0)
	    victim[nVictims].reason = PUNT_TIME_IDLE;
	else if (puntDataPeriod)
	    victim[nVictims].reason = PUNT_DATA_PERIOD;
	else if (puntDataSession)
	    victim[nVictims].reason = PUNT_DATA_SESSION;
	else
	    victim[nVictims].reason = PUNT_BUSY;
	victim[nVictims++].sendMail = EXPLAINBOOT(ur.flags) && ur.tLimit > 0;
    }
  }
  for (i = 0; i < 5; i++)
      usageRec[i].linesBusy[curMinute] += nOnline;
  writeUsageRecords(curMinute);
  userFileClose();

  /* go process-hunting */
  if (effect) {
      for (i = 0; i < nVictims; i++) {
	  if (loggingLevel >= 1) {
	      loginFromUID(login, victim[i].ur.uid);
	      syslog(LOG_INFO, "Punted %s (%s)", login, puntReason[victim[i].reason]);
	  }
	  if (victim[i].sendMail == TRUE) {

	      /* Create tempory file */
	      snprintf(dPath, sizeof(dPath), "/tmp/acua_updated.XXXXXX");
	      if (mkstemp(dPath) == -1) {
		  syslog(LOG_ERR, "Could not send mail to [%s]", login);
		  errlQuit("Could not create tempory file in /tmp");
	      }

	      // send the mail
	      sendMail(&victim[i].ur, login, "", "Connection Terminated",
		       (char*)msgPath[victim[i].reason], dPath);
	  }


	  /* kill all connections from a certain user */
	  for (x = 0; x < MAX_LOGINS; x++)
	  {
	      /* make sure the connection is active */
	      if (victim[i].ur.conninfo.type[x] == CONN_NONE)
		  continue;
	      /* do not try to kill pids from other boxes! */
	      if (victim[i].ur.conninfo.host[x] != myHostAddr)
		  continue;

	      killProcess(victim[i].ur.conninfo.pid[x]);
	  }
      }
  }
}
//char *
//userTTY(char *ttyPath, int uid, PPPuserRec *PPPuser)
//{
//  uid_t                    _uid = uid;
//  LoginRec                *loginRec;
//
//  loginRec = (LoginRec*)hashTableSearch(&loginRecTable, &_uid);
//  while (loginRec) {
//    if ((PPPuser && loginRec->tty == PPPuser->tty) ||
//        (!PPPuser && lineNo(loginRec->tty) >= 0))
//    {
//      sprintf(ttyPath, "/dev/%s", loginRec->ttyName);
//      return ttyPath;
//    }
//    loginRec = (LoginRec*)hashTableSearchNext(&loginRecTable, &_uid);
//  }
//  ttyPath[0] = '\0';
//  return NULL;
//}

void
userMessage(char *ttyName, char *format, ...)
{
  int                      fd;
  FILE                    *f;
  char                     ttyPath[16],
                           line[256];
  struct stat              _stat;
  struct utimbuf           _utime;
  va_list                  ap;

  va_start(ap, format);
  sprintf(ttyPath, "/dev/%s", ttyName);
  if (stat(ttyPath, &_stat))
    goto l0;
  if (!(_stat.st_mode & S_IWGRP))
    goto l0;
  vsprintf(line, format, ap);
  fd = open(ttyPath, O_WRONLY | O_NONBLOCK | O_NOCTTY);
  if (fd >= 0) {
    f = fdopen(fd, "w");
    fputc(7, f);         /* send ^G */
    fprintf(f, "\n\nNOTICE: ");
    fprintf(f, "%s\n\n", line);
    fclose(f);
  }
  if (optIdleLimit) {
    _utime.actime = _stat.st_atime;
    _utime.modtime = _stat.st_mtime;
    utime(ttyPath, &_utime);
  }
l0:
  va_end(ap);
}

int
sendMail(UserRec *ur, char *login, char *cc, char *subject, char *sPath, char *dPath)
{
  int   cstat, timer;
  pid_t cpid;
  char  *dest;
  
  // make the mail to send the user, from the template file in sPath
  preprocessFile(ur, sPath, dPath);
  // take care of where this mail is going
  if (optMailHost[0] != '\0') {
    dest = (char *) malloc(strlen(optMailHost)+strlen(login)+2);
    if (!dest) errQuit("Not enough memory to hold users email address");
    sprintf(dest, "%s@%s", login, optMailHost);
  } else {
    dest = (char *) malloc(strlen(login)+1);
    if (!dest) errQuit("Not enough memory to hold users email address");
    sprintf(dest, "%s", login);
  }
  
  // fork a new process
  cpid = fork();
  
  if (cpid == 0) {
    // child process - replace stdin with the output from preprocessFile()
    // BTW - changing the uid etc after the freopen is not a bug.
    freopen(dPath, "r", stdin);
    // change to a different user if need be
    if (optMailUser) {
      setuid(optMailUser);   seteuid(optMailUser);
    }
    // change to different group if need be
    if (optMailGroup) {
      setgid(optMailGroup);  setegid(optMailGroup);
    }

    // send the email
    if (cc == NULL) execl(optMailProg, "mail", "-s", subject, dest, NULL); else
      execl(optMailProg, "mail", "-s", subject, "-c", cc, dest, NULL);
  } else if (cpid == -1) {
    syslog(LOG_ERR, "Warning - Could not send mail to [%s], fork failed.", login);
    unlink(dPath);
    // free the malloc'd memory.
    free(dest);
    return(1);
  }
  // this is a safety measure, just incase the mail prog crashes.
  timer = time(NULL);
  for (;;) {
    if (waitpid(cpid, &cstat, WNOHANG) > 0) break;
    if (time(NULL) > timer+optMailWait) {
      syslog(LOG_ERR, "Warning - [%s] crashed. Killing dead process", optMailProg);
      kill(cpid, SIGKILL); // go away
      // free the malloc'd memory.
      free(dest);
      return(1);
    }
    usleep(250);
  }
  // free the malloc'd memory.
  free(dest);
  // delete the tempory file we made.
  unlink(dPath);
  return(0);
}

int
PPPidleLeft(UserRec *ur)
{
  int                      i;
  word                     nBytes;

  if ((((time(NULL) - ur->loginTime) / 60) >= ur->PPPidleMinutes)) {
    nBytes = 0;
    for (i = 0; i < ur->PPPidleMinutes; i++)
      nBytes += ur->nBytes[i];
    if (nBytes < ur->PPPidleBytes) return 0;
  }
  return INT_MAX;
}

int
TTYidleLeft(UserRec *ur)
{
    int          res = INT_MAX;
    int          i;
    struct stat  _stat;
    char         ttyPath[32];

    for (i = 0; i < MAX_LOGINS; i++)
    {
        /* skip non-tty entrys */
	if (ur->conninfo.type[i] != CONN_TTY)
	    continue;

	/* skip entrys that do not belong to this host */
	if (ur->conninfo.host[i] != myHostAddr)
            continue;

	snprintf(ttyPath, sizeof(ttyPath), "/dev/%s", ur->conninfo.if_name[i]);

	/* If you can't stat then the user must be gone for pts logins */
	if (stat(ttyPath, &_stat) < 0)
	    continue;

        /* calculate how much idle time the user has remaining */
	res = ur->idleLimit * 60 - (time(NULL) - _stat.st_atime);

    }

    if (res < 0)
	res = 0;
    else
	res = res / 60 + 1;

    return(res);
}

int
PPPupdate(UserRec *ur)
{
    word                     bStx = 0, bSrx = 0;
    int                      s, i, idx;
    struct    ifpppstatsreq  req;
    devStats                 ds;

  s = socket(AF_INET, SOCK_DGRAM, 0);
  if (s < 0)
      perrQuit("acua_updated, PPPupdate: couldn't create socket");

  for (i = 0; i < MAX_LOGINS; i++)
  {
      /* Make sure this record is for us to process */
      if (ur->conninfo.host[i] == myHostAddr)
      {
          /* PPP / PPPoE */
	  if (((ur->conninfo.type[i] == CONN_PPP) ||
	       (ur->conninfo.type[i] == CONN_PPPoE)))
	  {
	      memset(&req, 0, sizeof (req));
	      snprintf(req.ifr__name, sizeof(req.ifr__name), "%s", ur->conninfo.if_name[i]);
	      syslog(LOG_ERR, "Acct: [%s]", req.ifr__name);
	      req.stats_ptr = (caddr_t) &req.stats;

	      bStx += req.stats.p.ppp_obytes;
	      bSrx += req.stats.p.ppp_ibytes;
	  }

          /* Ethernet / CIPE */
	  if (((ur->conninfo.type[i] == CONN_ETH) ||
	       (ur->conninfo.type[i] == CONN_CIPE)))
	  {
              /* Suits Linux 2.2 and Linux 2.4 */
	      if (!getDeviceStats(ur->conninfo.if_name[i], &ds, 0, 1, 9))
	      {
                  syslog(LOG_ERR, "Acct: [%s]", ur->conninfo.if_name[i]);
		  bSrx += ds.ibytes;
		  bStx += ds.obytes;
	      }
	  }
      }
  }

  if (ur->PPPidleMinutes > 0)
  {
      idx = ((time(NULL) - ur->loginTime) / 60) % ur->PPPidleMinutes;
  }
  else
  {
      idx = ((time(NULL) - ur->loginTime) / 60) % 60;
  }
  ur->nBytes[idx] = 0;
  if (bStx > ur->bStx) {
      ur->nBytes[idx] += bStx - ur->bStx;
      ur->bTx += bStx - ur->bStx;
      ur->bStx = bStx;
  }
  if (bSrx > ur->bSrx) {
      ur->nBytes[idx] += bSrx - ur->bSrx;
      ur->bRx += bSrx - ur->bSrx;
      ur->bSrx = bSrx;
  }
  close(s);
  return(0);
}

int
warnExpire(time_t expire) {
  int                      i;
  time_t                   t = time(NULL), low, high;

  for (i = 0; i < nExpireWarnTimes; i++) {
    low = expire - expireWarnTime[i] * 24 * 60 * 60;
    high = expire - (expireWarnTime[i] - 1) * 24 * 60 * 60;
    if (t >= low + 120 && t < high) return 1;
  }
  return 0;
}

int
CPUpriority(int priority) {
  int                     res;

  if (optLowCPUpriority == optHighCPUpriority)
    return 0;

  if (priority <= 4) {
    res = 4 - priority;
    res *= optLowCPUpriority;
    res /= 4;
  } else {
    res = priority - 4;
    res *= optHighCPUpriority;
    res /= 3;
  }
  return res;
}

int
victimCmp(const VictimRec *v1, const VictimRec *v2)
{
  int                      v1Data = v1->reason == PUNT_DATA_PERIOD ||
                                    v1->reason == PUNT_DATA_SESSION;
  int                      v2Data = v2->reason == PUNT_DATA_PERIOD ||
                                    v2->reason == PUNT_DATA_SESSION;

  if (v1Data && !v2Data)
    return -1;
  else if (v2Data && !v1Data)
    return 1;
  else if (PRIORITY(v1->ur.flags) == PRIORITY(v2->ur.flags)) {
    if (v1->sLeft == v2->sLeft)
      return 0;
    else if (v1->sLeft < v2->sLeft)
      return -1;
    else
      return 1;
  } else if (PRIORITY(v1->ur.flags) < PRIORITY(v2->ur.flags))
    return -1;
  else
    return 1;
}

void
readUsageRecords()
{
  int                      i,
                           n;
  time_t                   t;
  struct tm               *tmP;
  FILE                    *f;
  char                     path[256];

  time(&t);
  for (i = 0; i < 5; i++) {
    sprintf(path, LIB"/record/%s", usageRecName[i]);
    f = fopen(path, "rb");
    if (f) {
      n = fread(usageRec + i, sizeof(UsageRec), 1, f);
      fclose(f);
      if (n <= 0) goto create;
      if (usageRec[i].expire && t >= usageRec[i].expire) {
        tmP = localtime(&usageRec[i].expire);
        sprintf(path, LIB"/record/archive/%s.%d-%d-%d", usageRecName[i],
                tmP->tm_year, tmP->tm_mon, tmP->tm_mday);
        f = fopen(path, "wb");
        if (f) {
          fwrite(usageRec + i, sizeof(UsageRec), 1, f);
          fclose(f);
        }
        goto create;
      }
    } else {
create:
      bzero(usageRec + i, sizeof(UsageRec));
      usageRec[i].startTime = t;
      if (usageRecExpire[i][0])
        usageRec[i].expire = addTime(time(NULL), (char*)usageRecExpire[i]);
    }
  }
}

void
writeUsageRecords(int curMinute)
{
  int                      i;
  FILE                    *f;
  char                     path[256];

  for (i = 0; i < 5; i++) {
    usageRec[i].nMinutes[curMinute]++;
    sprintf(path, LIB"/record/%s", usageRecName[i]);
    f = fopen(path, "wb");
    if (f) {
      fwrite(usageRec + i, sizeof(UsageRec), 1, f);
      fclose(f);
    }
  }
}

void
unlinkOldLockFiles()
{
  DIR            *dir = opendir(LIB);
  char            path[256];
  struct dirent  *_dirent;
  struct stat     _stat;

  if (dir == NULL)
    perrQuit("opendir(%s)", LIB);

  while ((_dirent = readdir(dir)) != NULL) {
    if (!strncmp(_dirent->d_name, "LOCK", 4)) {
      sprintf(path, "%s/%s", LIB, _dirent->d_name);
      if (lstat(path, &_stat))
        continue;
      if (time(NULL) - _stat.st_atime >= 3600)
        unlink(path);
    }      
  }
  closedir(dir);
}

void
segv_handler(int junk)
{
  syslog(LOG_ERR, "Segmentation fault - please report.");
  exit(1);
}

